/*
 * JFBTERM - 
 * Copyright (C) 1999  Noritoshi MASUICHI (nmasu@ma3.justnet.ne.jp)
 *
 * KON2 - Kanji ON Console -
 * Copyright (C) 1992-1996 Takashi MANABE (manabe@papilio.tutics.tut.ac.jp)
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *	notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *	notice, this list of conditions and the following disclaimer in the
 *	documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY TAKASHI MANABE ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE TERRENCE R. LAMBERT BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * 
 */

#include	<stdio.h>
#include	<unistd.h>
#include	<string.h>
#include	<termios.h>
#include	<malloc.h>
#include	<sys/types.h>
#include	<sys/ioctl.h>
#include	<sys/kd.h>

#include	"util.h"
#include	"term.h"
#include	"vterm.h"
#include	"font.h"
#include	"csv.h"

#include	"sequence.h"
#include	"message.h"

#define	LEN_REPORT	9

static void tvterm_set_default_encoding(TVterm* p, const char* en);
static void tvterm_esc_start(TVterm* p, u_char ch);
static void tvterm_esc_bracket(TVterm*, u_char);
static void tvterm_esc_traditional_mulitbyte_fix(TVterm* p, u_char ch);
static void tvterm_esc_desginate_font(TVterm* p, u_char ch);
static void tvterm_invoke_gx(TVterm* p, TFontSpec* fs, u_int n);
static void tvterm_re_invoke_gx(TVterm* p, TFontSpec* fs);
void tvterm_set_window_size(TVterm* p);
#if 0
static void tvterm_esc_set_double_code(TVterm* p, u_char ch);
#endif

void tvterm_init(TVterm* p, TTerm* pt, u_int hw, u_int hh, const char* en)
{
	p->term = pt;
	p->xcap = hw;
	p->xcap4 = (hw+7) & ~3;
	p->ycap = hh;

	tpen_init(&(p->pen));
	p->savedPen = NULL;
	p->savedPenSL = NULL;
	p->scroll = 0;
	p->soft = FALSE;
	p->wrap = FALSE;
	p->esc = NULL;

	p->textHead = 0;

	/* iso 2022 's default */
	p->gDefaultL = 0;
	p->gDefaultR = 1;
	p->gDefaultIdx[0] = 0;			/* G0 <== ASCII */
	p->gDefaultIdx[1] = 1;			/* G1 <== JIS X 0208 */
	p->gDefaultIdx[2] = 0;			/* G2 <== ASCII */
	p->gDefaultIdx[3] = 0;			/* G3 <== ASCII */

	tvterm_set_default_encoding(p, en);
}

void tvterm_set_default_encoding(TVterm* p, const char* en)
{
	static const char* table[] = {"G0", "G1", "G2", "G3", NULL}; 
	
	const char *g;
	int ig;
	int i;
	TCsv farg;
	
	tcsv_init(&farg, en);
	if (farg.cap != 6) {
		print_message("ENCODING : Skipped (BAD FORMAT)\n");
		goto FINALIZE;
	}
	/* GL */
	g = tcsv_get_token(&farg);
	ig = util_search_string(g, table);
	if (ig == -1) {
		print_message("ENCODING : GL ? (BAD FORMAT)\n");
		ig = 0;
	}
	p->gDefaultL = ig;
	/* GR */
	g = tcsv_get_token(&farg);
	ig = util_search_string(g, table);
	if (ig == -1) {
		print_message("ENCODING : GR ? (BAD FORMAT)\n");
		ig = 1;
	}
	p->gDefaultR = ig;
	/* G0 .. G3 */
	for (i = 0 ; i < 4 ; i++) {
		g = tcsv_get_token(&farg);
		ig = tfont_ary_search_idx(g);
		if (ig == -1) {
			print_message("ENCODING : G%d ? (BAD FORMAT)\n", i);
			ig = 0;
		}
		p->gDefaultIdx[i] = ig;
	}
	
FINALIZE:	
	tcsv_final(&farg);
}

void tvterm_set_default_invoke_and_designate(TVterm* p)
{
	p->gIdx[0] = p->gDefaultIdx[0];		/* G0 <== ASCII */
	p->gIdx[1] = p->gDefaultIdx[1];		/* G1 <== JIS X 0208 */
	p->gIdx[2] = p->gDefaultIdx[2];		/* G2 <== ASCII */
	p->gIdx[3] = p->gDefaultIdx[3];		/* G3 <== ASCII */
	tvterm_invoke_gx(p, &(p->gl), p->gDefaultL);/* GL <== G0 */
	tvterm_invoke_gx(p, &(p->gr), p->gDefaultR);/* GR <== G1 */
	p->tgl = p->gl;				/* Next char's GL <== GL */
	p->tgr = p->gr;				/* Next char's GR <== GR */
	p->knj1 = 0;
}



void tvterm_start(TVterm* p)
{
	p->pen.x = 0;
	p->pen.y = 0;
	p->xmax = p->xcap;
	p->ymax = p->ycap;
	p->tab = 8;
	p->pen.fcol = 7;
	p->pen.bcol = 0;
	p->pen.attr = 0;
	p->esc = NULL;
	p->soft = FALSE;
	p->ins = FALSE;
	p->wrap = FALSE;
	p->active = TRUE;

	/* ISO-2022 */
	p->escSignature = 0;
	p->escGn = 0;
	tvterm_set_default_invoke_and_designate(p);

	p->cursor.x = 0;
	p->cursor.y = 0;
	p->cursor.on = TRUE;
	p->cursor.shown = FALSE;
	p->cursor.wide = FALSE;
	p->cursor.width = gFontsWidth;
	p->cursor.height = gFontsHeight;

	p->tsize = p->xcap4 * p->ycap;
	p->text = (u_int *)calloc(p->xcap4, p->ycap * sizeof(u_int));
	p->attr = (u_char *)calloc(p->xcap4, p->ycap);
	p->flag = (u_char *)calloc(p->xcap4, p->ycap);

	ioctl(0, KDSETMODE, KD_GRAPHICS);
	/*
	 * ioctl(0, TIOCGWINSZ, &text_win);
	 * cInfo.shown = FALSE;
	 * saved = FALSE;
	 * GraphMode();
	ioctl(0, TIOCGWINSZ, p->winrect);
	 */
        /*  */
	tvterm_register_signal(p);
	tvterm_set_window_size(p);
}

void tvterm_final(TVterm* p)
{
	ioctl(0, KDSETMODE, KD_TEXT);
	tpen_final(&(p->pen));
	if (p->savedPen) {
		tpen_final(p->savedPen);
		free(p->savedPen);
		p->savedPen = NULL;
	}
	if (p->savedPenSL) {
		tpen_final(p->savedPenSL);
		free(p->savedPenSL);
		p->savedPenSL = NULL;
	}
	p->textHead = 0;
	util_free(p->text);
	util_free(p->attr);
	util_free(p->flag);
}

void tvterm_push_current_pen(TVterm* p, TBool b)
{
	TPen* t;
	TPen** base;
	base = b ? &(p->savedPen) : &(p->savedPenSL);
	t = (TPen*)malloc(sizeof(TPen));

	if (!t) {
		return;
	}
	tpen_init(t);
	tpen_copy(t, &(p->pen));
	t->prev = *base;
	*base = t;
}

void tvterm_pop_pen_and_set_currnt_pen(TVterm* p, TBool b)
{
	TPen* t;
	TPen** base;
	base = b ? &(p->savedPen) : &(p->savedPenSL);
	t = *base;

	if (!t) {
		return;
	}
	tpen_copy(&(p->pen), t);

	if (p->pen.y < p->ymin) p->pen.y = p->ymin;
	if (p->pen.y >= p->ymax-1) p->pen.y = p->ymax -2;
	
	*base = t->prev;
	free(t);
}

/*---------------------------------------------------------------------------*/

static inline int IS_GL_AREA(TVterm* p, u_char ch)
{
	return (p->tgl.type & TFONT_FT_96CHAR) ? (0x1F < ch && ch < 0x80) :
						(0x20 < ch && ch < 0x7F);
}
static inline int IS_GR_AREA(TVterm* p, u_char ch)
{
	return (p->tgr.type & TFONT_FT_96CHAR) ? (0x9F < ch) :
						(0xA0 < ch && ch < 0xFF);
}

static inline void INSERT_N_CHARS_IF_NEEDED(TVterm* p, int n)
{
	if (p->ins) {
		tvterm_insert_n_chars(p, n);
	}
}

static inline void SET_WARP_FLAG_IF_NEEDED(TVterm* p)
{
	if (p->pen.x == p->xmax-1) {
		p->wrap = TRUE;
	}
}


static int tvterm_put_normal_char(TVterm* p, u_char ch, const char *buff)
{
	if (p->pen.x == p->xmax) {
		p->wrap = TRUE;
		p->pen.x --;
	}
	if (p->wrap) {
		p->pen.x -= p->xmax -1;
		if (p->pen.y == p->ymax -1) {
			p->scroll++;
		} else {
			p->pen.y ++;
		}
		p->wrap = FALSE;
		return -1;
	}
	if (p->knj1) { /*   2 Х */
		INSERT_N_CHARS_IF_NEEDED(p, 2);
		tvterm_wput(p, p->knj1idx,
			p->knj1h == FH_RIGHT ? p->knj1 | 0x80 : p->knj1 & 0x7F,
			p->knj1h == FH_RIGHT ? ch | 0x80 : ch & 0x7F);
		p->pen.x += 2;
		p->knj1 = 0;
		p->tgr = p->gr; p->tgl = p->gl;
	} else if (IS_GL_AREA(p, ch)) {
		if (p->tgl.type & TFONT_FT_DOUBLE) {
			SET_WARP_FLAG_IF_NEEDED(p);
			p->knj1 = ch;
			p->knj1h = p->tgl.half;
			p->knj1idx = p->tgl.idx;
		} else {
			INSERT_N_CHARS_IF_NEEDED(p, 1);
			tvterm_sput(p, p->tgl.idx,
				p->tgl.half == FH_RIGHT ? ch | 0x80: ch);
			p->pen.x ++;
			p->tgr = p->gr; p->tgl = p->gl;
		}
	} else if (IS_GR_AREA(p, ch)) {
		if (p->tgr.type & TFONT_FT_DOUBLE) {
			SET_WARP_FLAG_IF_NEEDED(p);
			p->knj1 = ch;
			p->knj1h = p->tgr.half;
			p->knj1idx = p->tgr.idx;
		} else {
			INSERT_N_CHARS_IF_NEEDED(p, 1);
			tvterm_sput(p, p->tgr.idx,
				p->tgr.half == FH_LEFT ? ch & ~0x80: ch);
			p->pen.x ++;
			p->tgr = p->gr; p->tgl = p->gl;
		}
	} else if (ch == 0x20) {
		INSERT_N_CHARS_IF_NEEDED(p, 1);
		tvterm_sput(p, 0, 0x20);
		p->pen.x ++;
		p->tgr = p->gr; p->tgl = p->gl;
	}
	return 0;
}

int tvterm_iso_C0_set(TVterm* p, u_char ch)
{
	switch(ch) {
	case ISO_BS:
		if (p->pen.x) {
			p->pen.x --;
		}
		p->wrap = FALSE;
		break;
	case ISO_HT:
		p->pen.x += p->tab - (p->pen.x % p->tab);
		p->wrap = FALSE;
		if (p->pen.x < p->xmax) {
			break;
		}
		p->pen.x -= p->xmax;
		/* fail into next case */
	case ISO_VT:
	case ISO_FF:
		tvterm_set_default_invoke_and_designate(p);
		/* fail into next case */
	case ISO_LF:
		p->wrap = FALSE;
		if (p->pen.y == p->ymax -1) {
			p->scroll ++;
		} else  {
			p->pen.y ++;
		}
		break;
	case ISO_CR:
		p->pen.x = 0;
		p->wrap = FALSE;
		break;
	case UNIVERSAL_ESC:
		p->esc = tvterm_esc_start;
		p->escSignature = 0;
		p->escGn = 0;
		return 1;
	case ISO_LS1:
		tvterm_invoke_gx(p, &(p->gl), 1); /* GL <== G1 */
		p->tgl = p->gl;
		return 1;
	case ISO_LS0:
		tvterm_invoke_gx(p, &(p->gl), 0); /* GL <== G0 */
		p->tgl = p->gl;
		return 1;
	default:
		break;
	}
	return 0;
}

int tvterm_iso_C1_set(TVterm* p, u_char ch)
{
    switch (ch) {
    case ISO_SS2:	/* single shift 2 */
	tvterm_invoke_gx(p, &(p->tgr), 2); /* GR <== G2 */
	return 1;
    case ISO_SS3:	/* single shift 3 */
	tvterm_invoke_gx(p, &(p->tgr), 3); /* GR <== G3 */
	return 1;
    default:
	break;
    }
    return 0;
}

void tvterm_emulate(TVterm* p, const char *buff, int nchars)
{
	u_char	ch;
	int rev;
	int cn;

	while (nchars-- > 0) {
		ch = *(buff++);
		if (!ch) {
			continue;
		} else if (p->esc) {
			p->esc(p, ch);
		} else if (ch < 0x20) {
			cn = tvterm_iso_C0_set(p, ch);
			if (cn) {
				continue;
			}
		} else if ((0x7F < ch) && (ch < 0xA0)) {
			cn = tvterm_iso_C1_set(p, ch);
			if (cn) {
				continue;
			}
		} else if (ch == ISO_DEL) {
			/* nothing to do. */
		} else {
			rev = tvterm_put_normal_char(p, ch, buff);
			if (rev == 0) {
				continue;
			} else if (rev < 0) {
				nchars -= rev;
				buff += rev;
			}
		}
		if (p->scroll > 0) {
			tvterm_text_scroll_up(p, p->scroll);
		} else if (p->scroll < 0) {
			tvterm_text_scroll_down(p, -(p->scroll));
		}
		p->scroll = 0;
	}
}

#define ESC_ISO_GnDx(n, x) \
	{\
		p->escSignature |= (x); p->escGn = (n);\
		p->esc = tvterm_esc_desginate_font;\
	}

#define Fe(x)	((x)-0x40)

static void tvterm_esc_start(TVterm* p, u_char ch)
{
	p->esc = NULL;
	switch(ch) {
	case Fe(ISO_CSI):
		p->esc = tvterm_esc_bracket;
		break;
	case ISO__MBS:
		p->esc = tvterm_esc_traditional_mulitbyte_fix;
		p->escSignature = TFONT_FT_DOUBLE;
		break;
	case ISO_GZD4:
		ESC_ISO_GnDx(0, TFONT_FT_94CHAR);
		break;
	case MULE__GZD6:
		ESC_ISO_GnDx(0, TFONT_FT_96CHAR);
		break;
	case ISO_G1D4:
		ESC_ISO_GnDx(1, TFONT_FT_94CHAR);
		break;
	case ISO_G1D6:
		ESC_ISO_GnDx(1, TFONT_FT_96CHAR);
		break;
#if 0
	case ISO_G2D4:
	case ISO_G2D6:
	case ISO_G3D4:
	case ISO_G3D6:
#endif
	case Fe(ISO_NEL):		/* E */
		p->pen.x = 0;
		p->wrap = FALSE;
	case Fe(TERM_IND):		/* D */
		if (p->pen.y == p->ymax -1) {
			p->scroll ++;
		} else {
			p->pen.y ++;
		}
		break;
	case Fe(ISO_RI):		/* M */
		if (p->pen.y == p->ymin) {
			p->scroll --;
		} else {
			p->pen.y --;
		}
		break;
	case Fe(ISO_SS2):		/* N *//* 7bit single shift 2 */
		tvterm_invoke_gx(p, &(p->tgl), 2); /* GL <== G2 */
		break;
	case Fe(ISO_SS3):		/* O *//* 7bit single shift 3 */
		tvterm_invoke_gx(p, &(p->tgl), 3); /* GL <== G3 */
		break;

	case ISO_RIS:			/* c */
		p->pen.fcol = 7;
		p->pen.bcol = 0;
		p->pen.attr = 0;
		p->wrap = FALSE;
		tvterm_set_default_invoke_and_designate(p);
		/* fail into next case */
	case TERM_CAH:
		p->pen.x = 0;
		p->pen.y = 0;
		p->wrap = FALSE;
		tvterm_text_clear_all(p);
		break;
	case DEC_SC:
		tvterm_push_current_pen(p, TRUE);
		break;
	case DEC_RC:
		tvterm_pop_pen_and_set_currnt_pen(p, TRUE);
		p->wrap = FALSE;
		break;
	}
}

/*---------------------------------------------------------------------------*/
void tvterm_esc_set_attr(TVterm* p, int col)
{
	switch(col) {
	case 0: tpen_off_all_attribute(&(p->pen));
		break;
	case 1: tpen_higlight(&(p->pen));
		break;
	case 21: tpen_dehiglight(&(p->pen));
		break;
	case 4:	tpen_underline(&(p->pen));
		break;
	case 24: tpen_no_underline(&(p->pen));
		break;
	case 7:	tpen_reverse(&(p->pen));
		break;
	case 27: tpen_no_reverse(&(p->pen));
		break;
#if 0 /* FIX ME ! */
	case 10: p->trans = CS_LEFT;
		break;
	case 11: p->trans = CS_GRAPH;
		break;
#else
	case 10:
		p->gIdx[0] = 0;
		tvterm_re_invoke_gx(p, &(p->gl));
		tvterm_re_invoke_gx(p, &(p->gr));
		p->tgl = p->gl; p->tgr = p->gr;
		break;
#if 0
	case 11:
		p->gIdx[1] = 3;
		tvterm_re_invoke_gx(p, &(p->gl));
		tvterm_re_invoke_gx(p, &(p->gr));
		p->tgl = p->gl; p->tgr = p->gr;
		break;
#endif
#endif
	default: tpen_set_color(&(p->pen), col);
		break;
	}
}

static void tvterm_set_mode(TVterm* p, u_char mode, TBool sw)
{
	switch(mode) {
	case 4:
		p->ins = sw;
		break;
	case 25:
		p->sw = sw;
		break;
	}
}

static void tvterm_esc_report(TVterm* p, u_char mode, u_short arg)
{
	p->report[0] = '\0';

	switch(mode) {
	case 'n':
		if (arg == 6) {
			int x = (p->pen.x < p->xmax-1) ? p->pen.x : p->xmax-1;
			int y = (p->pen.y < p->ymax-1) ? p->pen.y : p->ymax-1;
			sprintf(p->report, "\x1B[%d;%dR", y, x);
		} else if (arg == 5) {
			strcpy(p->report, "\x1B[0n\0");
		}
		break;
	case 'c':
		if (arg == 0) {
			strcpy(p->report, "\x1B[?6c\0");
		}
		break;
	}
	write(p->term->ttyfd, p->report, strlen(p->report));
}

static void tvterm_set_region(TVterm* p,int ymin, int ymax)
{
	/* FIX ME ! : 1999/10/30 */
	/* XXX: ESC[?1001r is used for mouse control by w3m. ukai 1999/10/27*/
	/* XXX: ESC[?1001r is used for mouse control by w3m. oike 1999/11/05*/
	if (ymin < 0 || ymin >= p->ycap || ymin > ymax || ymax > p->ycap) {
	        /* ignore */
		return;
	}
	p->ymin = ymin;
	p->ymax = ymax;
	p->pen.x = 0;
	if (p->pen.y < p->ymin || p->pen.y > p->ymax-1) p->pen.y = p->ymin-1;
	p->wrap = FALSE;
	if (p->ymin || p->ymax != p->ycap) {
		p->soft = TRUE;
	} else {
		p->soft = FALSE;
	}
}

void tvterm_set_window_size(TVterm* p)
{
	struct winsize win;

	win.ws_row = p->ymax;
	win.ws_col = p->xcap;
	win.ws_xpixel = 0;
	win.ws_ypixel = 0;
	ioctl(p->term->ttyfd, TIOCSWINSZ, &win);
}


static void tvterm_esc_status_line(TVterm* p, u_char mode)
{
	switch(mode) {
	case 'T':	/* To */
		if (p->sl == SL_ENTER) break;
		if (!p->savedPenSL) {
			tvterm_push_current_pen(p, FALSE);
		}
	case 'S':	/* Show */
		if (p->sl == SL_NONE) {
			p->ymax = p->ycap - 1;
			tvterm_set_window_size(p);
		}
		if (mode == 'T') {
			p->sl = SL_ENTER;
			tvterm_set_region(p, p->ycap-1, p->ycap);
		}
		break;
	case 'F':	/* From */
		if (p->sl == SL_ENTER) {
			p->sl = SL_LEAVE;
			tvterm_set_region(p, 0, p->ycap -1);
			if (p->savedPenSL) {
				tvterm_pop_pen_and_set_currnt_pen(p, FALSE);
			}
			p->savedPenSL = NULL;
		}
		break;
	case 'H':	/* Hide */
	case 'E':	/* Erase */
		if (p->sl == SL_NONE) {
			break;
		}
		tvterm_set_region(p, 0, p->ycap);
		tvterm_set_window_size(p);
		p->sl = SL_NONE;
		break;
	default:
		p->esc = tvterm_esc_bracket;
		tvterm_esc_bracket(p, mode);
		return;
	}
	p->wrap = FALSE;
	p->esc = NULL;
}

#define	MAX_NARG	8

static void tvterm_esc_bracket(TVterm* p, u_char ch)
{
	u_char	n;
	static u_short varg[MAX_NARG], narg, question;

	if (ch >= '0' && ch <= '9') {
		varg[narg] = (varg[narg] * 10) + (ch - '0');
	} else if (ch == ';') {
		/*  MAX_NARG ޤǤݡȤʤ!! */
		if (narg < MAX_NARG) {
			narg ++;
			varg[narg] = 0;
		} else {
			p->esc = NULL;
		}
	} else {
		p->esc = NULL;
		switch(ch) {
		case 'K':
			tvterm_text_clear_eol(p, varg[0]);
			break;
		case 'J':
			tvterm_text_clear_eos(p, varg[0]);
			break;
		case ISO_CS_NO_CUU:
			p->pen.y -= varg[0] ? varg[0]: 1;
			if (p->pen.y < p->ymin) {
				p->scroll -= p->pen.y - p->ymin;
				p->pen.y = p->ymin;
			}
			break;
		case ISO_CS_NO_CUD:
			p->pen.y += varg[0] ? varg[0]: 1;
			if (p->pen.y >= p->ymax) {
				p->scroll += p->pen.y - p->ymin;
				p->pen.y = p->ymax-1;
			}
			break;
		case ISO_CS_NO_CUF:
			p->pen.x += varg[0] ? varg[0]: 1;
			p->wrap = FALSE;
			break;
		case ISO_CS_NO_CUB:
			p->pen.x -= varg[0] ? varg[0]: 1;
			p->wrap = FALSE;
			break;
		case 'G':
			p->pen.x = varg[0] ? varg[0] - 1: 0;
			p->wrap = FALSE;
			break;
		case 'P':
			tvterm_delete_n_chars(p, varg[0] ? varg[0]: 1);
			break;
		case '@':
			tvterm_insert_n_chars(p, varg[0] ? varg[0]: 1);
			break;
		case 'L':
			tvterm_text_move_down(p, p->pen.y, p->ymax,
						varg[0] ? varg[0]: 1);
			break;
		case 'M':
			tvterm_text_move_up(p, p->pen.y, p->ymax,
						varg[0] ? varg[0]: 1);
			break;
		case 'H':
		case 'f':
			if (varg[1]) {
				p->pen.x = varg[1] - 1;
			} else {
				p->pen.x = 0;
			}
			p->wrap = FALSE;
		case 'd':
			p->pen.y = varg[0] ? varg[0] - 1: 0;
			break;
		case 'm':
			for (n = 0; n <= narg; n ++) {
				tvterm_esc_set_attr(p, varg[n]);
			}
			break;
		case 'r':
			n = varg[1];
			if (p->sl != SL_NONE) {
				if (n == p->ycap) {
					 n --;
				}
			}
			tvterm_set_region(p, varg[0] ? (varg[0] - 1): 0, n);
			break;
		case 'l':
			for (n = 0; n <= narg; n ++) {
				tvterm_set_mode(p, varg[n], FALSE);
			}
			break;
		case 'h':
			for (n = 0; n <= narg; n ++) {
				tvterm_set_mode(p, varg[n], TRUE);
			}
			break;
		case '?':
			p->esc = tvterm_esc_status_line;
	#if 0
			question = TRUE;
			p->esc = tvterm_esc_bracket;
	#endif
			break;
		case 's':
			tvterm_push_current_pen(p, TRUE);
			break;
		case 'u':
			tvterm_pop_pen_and_set_currnt_pen(p, TRUE);
			break;
		case 'n':
		case 'c':
			if (question != TRUE) {
				tvterm_esc_report(p, ch, varg[0]);
			}
			break;
		case 'R':
			break;
		}
		if (p->esc == NULL) {
			question = narg = varg[0] = varg[1] = 0;
		}
	}
}

/* Note:
* KON2 Support
* ESC ISO_94_TO_G0 U  ==> GRAPHIC FONT SET
* ESC ISO_94_TO_G1 U  ==> ISO 8859-1 ?
* ESC ISO_94_TO_G1 0  ==> GRAPHIC FONT SET
*/

static void tvterm_invoke_gx(TVterm* p, TFontSpec* fs, u_int n)
{
	fs->invokedGn = n;
	fs->idx = p->gIdx[fs->invokedGn];
	fs->type = gFont[fs->idx].fsignature;
	fs->half = gFont[fs->idx].fhalf;
}

static void tvterm_re_invoke_gx(TVterm* p, TFontSpec* fs)
{
	fs->idx = p->gIdx[fs->invokedGn];
	fs->type = gFont[fs->idx].fsignature;
	fs->half = gFont[fs->idx].fhalf;
}

static void tvterm_esc_desginate_font(TVterm* p, u_char ch)
{
	int i;

	for (i = 0 ; gFont[i].fsignature ; i++) {
		if (gFont[i].fsignature  == (ch | p->escSignature)) {
			p->gIdx[p->escGn] = i;
			tvterm_re_invoke_gx(p, &(p->gl));
			tvterm_re_invoke_gx(p, &(p->gr));
			p->tgl = p->gl; p->tgr = p->gr;
			break;
		}
	}
	p->esc = NULL;
}

static void tvterm_esc_traditional_mulitbyte_fix(TVterm* p, u_char ch)
{
	if (ch == 0x40 || ch == 0x41 || ch == 0x42) {
		tvterm_esc_desginate_font(p, ch);
	} else {
		tvterm_esc_start(p, ch);
	}
}

#if 0
static void tvterm_esc_set_double_code(TVterm* p, u_char ch)
{
	switch(ch) {
	case ISO_GZD4:
		p->escSignature = TFONT_FT_94CHAR | TFONT_FT_DOUBLE;
		p->escGn = 0;
		p->esc = tvterm_esc_desginate_font;
		break;
	case ISO_G1D4:
		p->escSignature = TFONT_FT_94CHAR | TFONT_FT_DOUBLE;
		p->escGn = 1;
		p->esc = tvterm_esc_desginate_font;
		break;
	case ISO_G2D4:
		p->escSignature = TFONT_FT_94CHAR | TFONT_FT_DOUBLE;
		p->escGn = 2;
		p->esc = tvterm_esc_desginate_font;
		break;
	case ISO_G3D4:
		p->escSignature = TFONT_FT_94CHAR | TFONT_FT_DOUBLE;
		p->escGn = 3;
		p->esc = tvterm_esc_desginate_font;
		break;
	case MULE__GZD6:
		p->escSignature = TFONT_FT_96CHAR | TFONT_FT_DOUBLE;
		p->escGn = 0;
		p->esc = tvterm_esc_desginate_font;
		break;
	case ISO_G1D6:
		p->escSignature = TFONT_FT_96CHAR | TFONT_FT_DOUBLE;
		p->escGn = 1;
		p->esc = tvterm_esc_desginate_font;
		break;
	case ISO_G2D6:
		p->escSignature = TFONT_FT_96CHAR | TFONT_FT_DOUBLE;
		p->escGn = 2;
		p->esc = tvterm_esc_desginate_font;
		break;
	case ISO_G3D6:
		p->escSignature = TFONT_FT_96CHAR | TFONT_FT_DOUBLE;
		p->escGn = 3;
		p->esc = tvterm_esc_desginate_font;
		break;
	default:
		tvterm_esc_desginate_font(p, ch);
		p->esc = NULL;
		break;
	}
}
#endif


