/*
 * JFBTERM -
 * Copyright (C) 1999  Noritoshi MASUICHI (nmasu@ma3.justnet.ne.jp)
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *      notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *      notice, this list of conditions and the following disclaimer in the
 *      documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY NORITOSHI MASUICHI ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL NORITOSHI MASUICHI BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

#ifdef HAVE_CONFIG_H
#	include "config.h"
#endif

#include <stdio.h>
#include <stdarg.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/vt.h>
#include <sys/kd.h>
#include <signal.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <getopt.h>


#include <pcf.h>
#include <font.h>
#include <fbcommon.h>
#include <term.h>
#include <vterm.h>
#include <message.h>
#include <mytypes.h>
#include <getcap.h>
#include <util.h>
#include <main.h>


#ifdef SYSCONFDIR
#define	FILE_JFBTERM_CONF	SYSCONFDIR"/jfbterm.conf"
#else
#define	FILE_JFBTERM_CONF	"/etc/jfbterm.conf"
#endif

TFrameBufferMemory gFramebuffer;

static void tapp_get_options(TApplication* p, int argc, char *argv[])
{
	static struct option optList[] = {
		{"shell",	1, NULL, 'e'},
		{"exec",	1, NULL, 'e'},
		{"help",	0, NULL, 'h'},
		{NULL,		0, NULL, 0},
	};
	int c;
	int ix;
	int i;
	
	while (1) {
		c = getopt_long(argc, argv, "e:h", optList, &ix);
		if (c == EOF) {
			break;
		}
		switch (c) {
		case 'e':
			p->gExecShell = optarg;
			break;
		case 'h':
			p->gOptShowHelpQ = TRUE;
			break;
		default:
			break;
		}
	}
	p->gExecShellArgv = (char**)calloc(argc + 2 - optind, sizeof(char*));
	p->gExecShellArgv[0] = p->gExecShell;
	for (i = 0 ; i < argc+1 - optind ; i++) {
		p->gExecShellArgv[i+1] = argv[i + optind];
	}
	
}

void tapp_change_to_original_console(TApplication* p)
{
        int cfd;
	int n = p->gOrigVirtualConsole;

/*
        signal(SIGUSR1, SIG_DFL);
        signal(SIGUSR2, SIG_DFL);
*/

        cfd = open("/dev/console", O_WRONLY);
        if (cfd < 0 && (cfd = open("/dev/console", O_RDONLY)) < 0) {
                print_strerror("/dev/console");
        }
        if (ioctl(cfd, VT_ACTIVATE, n) != 0) {
                fprintf(stderr, "can't activate VC(%d)", n);
        }
        close(cfd);
}

void tapp_final(TApplication* p)
{
	if (p->gCapsQ) {
		tcaps_final(&(p->gCaps));
	}
	if (p->gOrigVirtualConsoleQ) {
		tapp_change_to_original_console(p);
	}
	util_free(p->gExecShellArgv);
}

TApplication gApp;

void tapp_final_at_exit(void)
{
	tapp_final(&gApp);
}

void tapp_init(TApplication* p)
{
	static char shell[128];
	
	p->gOrigVirtualConsoleQ	= FALSE;
	p->gOrigVirtualConsole	= -1;
	tcaps_init(&(p->gCaps));
	p->gCapsQ	= TRUE;
	p->gOptShowHelpQ = FALSE;
	if (getenv("SHELL")) {
		/* This cause a buffer overflow. */
		memset (shell, '\0', sizeof shell);
		strncpy(shell, getenv("SHELL"), sizeof shell - 1);
		p->gExecShell = shell;
	} else {
		p->gExecShell = "/bin/sh";
	}
	p->gExecShellArgv = NULL;

	atexit(tapp_final_at_exit);
}

void tapp_change_to_new_console(TApplication* p)
{
        struct vt_stat vts;
        int cfd;
        int vfd;
        int vtNum;
        int child;
        int parent;
        char vtty[128];
        int mode;

        cfd = open("/dev/console", O_WRONLY);
        if (cfd < 0 && (cfd = open("/dev/console", O_RDONLY)) < 0) {
                die("can't open /dev/console");
        }
#if 1
        ioctl(cfd, KDGETMODE, &mode);
        if (mode == KD_TEXT) {
                close(cfd);
                return;
        }
#endif
        ioctl(cfd, VT_GETSTATE, &vts);
	p->gOrigVirtualConsole	= vts.v_active;
	p->gOrigVirtualConsoleQ	= TRUE;

        ioctl(cfd, VT_OPENQRY, &vtNum);
        if (vtNum < 0) {
                die("can't get free VC");
        }
        parent = getpid();
        if ((child = fork()) == -1) {
                print_strerror_and_exit("fork");
        }
        if (child) {
                fprintf(stderr, "\r\nJFBTERM> switched to new VC\r\n");
		p->gOrigVirtualConsoleQ	= FALSE;
                exit(EXIT_SUCCESS);
        }
        setsid();
        sprintf(vtty, "/dev/tty%d", vtNum);
        if ((vfd = open(vtty, O_RDWR)) < 0) {
                die("can't open %s", vtty);
        }
        if (ioctl(cfd, VT_ACTIVATE, vtNum) != 0) {
                die("can't activate VC(%d)", vtNum);
        }
/*
        atexit(change_to_original_console);
*/
        close(cfd);
        dup2(vfd, 0);
        dup2(vfd, 1);
        dup2(vfd, 2);
}

void ShowCaps(void)
{
	fprintf(stderr,
		"======== Capabilities =====================================\n"
		"[[ COLOR ]]\n"
#ifdef JFB_VGA16FB
			" VGA16"
#endif
#ifdef JFB_PSEUDOCOLOR
			" Pseudo"
#endif
#ifdef JFB_DIRECTCOLOR
			" Direct"
#endif
#ifdef JFB_TRUECOLOR
			" True"
#endif
		"\n"
		"[[ BITS ]]\n"
#ifdef JFB_1BPP
			" 1bpp"
#endif
#ifdef JFB_2BPP
			" 2bpp"
#endif
#ifdef JFB_4BPP
			" 4bpp"
#endif
#ifdef JFB_VGA16FB
			" 4bpp(VGA)"
#endif
#ifdef JFB_8BPP
			" 8bpp"
#endif
#ifdef JFB_15BPP
			" 15bpp"
#endif
#ifdef JFB_16BPP
			" 16bpp"
#endif
#ifdef JFB_24BPP
			" 24bpp"
#endif
#ifdef JFB_32BPP
			" 32bpp"
#endif
		"\n"
	);
	fprintf(stderr, "[[ FONTSET ]]\n");
	tfont_ary_show_list(stderr);

	fprintf(stderr,
		"[[ ENCODING ]]\n"
#ifdef JFB_MINI_JFBTERM
			" ISO-2022-JP EUC-JP"
#else
			" ISO-2022-*"
#if defined(JFB_UNICODE) && defined(JFB_UTF_7)
			" UTF-7"
#endif	/* JFB_UNICODE && JFB_UTF_7 */
#if defined(JFB_UNICODE) && defined(JFB_UTF_8)
			" UTF-8"
#endif	/* JFB_UNICODE && JFB_UTF_8 */
#ifdef JFB_TRON
			" TRON"
#endif	/* JFB_TRON */
#endif	/* JFB_MINI_JFBTERM */
		"\n"
		"[[ MISC ]]\n"
#ifdef DEBUG
			" DEBUG"
#endif
		"\n"
	);

}

int main(int argc, char *argv[])
{
	TCapability* fcap;
	char* cp;
	const char* tn;
	const char* en;

	fprintf(stderr,
		"%s - Kanji on framebuffer console Version %s\n"
		"	Copyright (C) 1999-2000 Noritoshi Masuichi\n"
		"This program is based on KON2\n"
		"	Copyright (C) 1992-1996 Takashi MANABE\n\n",
		PACKAGE, VERSION
	);

	tapp_init(&gApp);

	tapp_get_options(&gApp, argc, argv);

	if (gApp.gOptShowHelpQ) {
		ShowCaps();
		exit(EXIT_SUCCESS);
	}

	if (geteuid() != 0) {
		fprintf(stderr, "permission denied.\n");
		exit(EXIT_FAILURE);
	}

	tcaps_read(&(gApp.gCaps), FILE_JFBTERM_CONF);
	fcap = tcaps_find(&(gApp.gCaps), "fontset"); 
	if (!fcap || !(fcap->values)) {
		fprintf(stderr, "no font specified.\n");
		exit(EXIT_FAILURE);
	}

	if ((cp = tcaps_find_first(&(gApp.gCaps), "gamma")) != NULL) {
		fbgamma = (float)atof(cp);
	}

	tapp_change_to_new_console(&gApp);

	tfbm_init(&gFramebuffer);
	
	tfont_setup_fontlist(fcap->values);

	tfbm_open(&gFramebuffer);

	tn = tcaps_find_first(&(gApp.gCaps), "term");
	if (!tn) {
		tn = "kon";
	}
	en = tcaps_find_first(&(gApp.gCaps), "encoding");
	if (!en) {
		en = "G0,G1,ASCII,ISO8859-1,ASCII,ASCII";
	}
	tterm_start(&gTerm, tn, en);

	tfbm_close(&gFramebuffer);
	tfont_ary_final();

	exit(EXIT_SUCCESS);
}

	
