/************************************************************
 * Copyright 2004 Masahiko SAWAI All Rights Reserved. 
************************************************************/
package say.swing;
import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Container;
import java.awt.Dimension;
import java.awt.Rectangle;
import java.awt.Font;
import java.awt.Frame;
import java.awt.GraphicsEnvironment;
import java.awt.GridLayout;
import java.awt.Point;
import java.awt.FocusTraversalPolicy;
import java.awt.event.ActionListener;
import java.awt.event.ActionEvent;
import java.awt.event.FocusAdapter;
import java.awt.event.FocusEvent;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import javax.swing.*;
import javax.swing.border.Border;
import javax.swing.text.JTextComponent;
import javax.swing.text.Document;
import javax.swing.text.BadLocationException;
import javax.swing.text.Position;
import javax.swing.event.ListSelectionListener;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.DocumentListener;
import javax.swing.event.DocumentEvent;
import java.util.Arrays;
import java.util.List;
import java.util.ResourceBundle;
import java.util.MissingResourceException;
import java.util.regex.Pattern;
import java.util.regex.Matcher;

public class JFontChooser extends JComponent
{
	/**
	*  Return value from showDialog(Component parent).
	**/
	public static final int OK_OPTION = 0;

	/**
	*  Return value from showDialog(Component parent).
	**/
	public static final int CANCEL_OPTION = 1;

	/**
	*  Return value from showDialog(Component parent).
	**/
	public static final int ERROR_OPTION = -1;

	private static final int[] FONT_STYLE_CODES = { 
		Font.PLAIN, Font.BOLD, Font.ITALIC, Font.BOLD|Font.ITALIC
	};
	private static final Font DEFAULT_SELECTED_FONT = new Font("Serif", Font.PLAIN, 12);
	private static final Font DEFAULT_FONT = new Font("Dialog", Font.PLAIN, 10);

	private static final String[] FONT_SIZE_LIST = {
		"8", "9", "10", "11", "12", "14", "16", "18", "20", 
		"22", "24", "26", "28", "36", "48", "72",
	};

	private static ResourceBundle messageCatalog;
	protected static String _(String key)
	{
		String value = key;
		try
		{
			if (messageCatalog == null)
				messageCatalog = ResourceBundle.getBundle(JFontChooser.class.getName() + "Messages");
			value = messageCatalog.getString(key);
		}
		catch (MissingResourceException e) { }
		return value;
	}

	private static String[] fontStyleNames = null;
	protected String[] getFontStyleNames()
	{
		if (fontStyleNames == null)
		{
			int i = 0;
			fontStyleNames = new String[4];
			fontStyleNames[i++] = _("Plain");
			fontStyleNames[i++] = _("Bold");
			fontStyleNames[i++] = _("Italic");
			fontStyleNames[i++] = _("BoldItalic");
		}
		return fontStyleNames;
	}

	protected int dialogResultValue = ERROR_OPTION;

	private String[] fontFamilyNames = null;

	private JTextField fontFamilyTextField = null;
	private JTextField fontStyleTextField = null;
	private JTextField fontSizeTextField = null;

	private JList fontNameList = null;
	private JList fontStyleList = null;
	private JList fontSizeList = null;

	private JPanel fontNamePanel = null;
	private JPanel fontStylePanel = null;
	private JPanel fontSizePanel = null;
	private JPanel samplePanel = null;

	private JTextField sampleText = null;

	public JFontChooser()
	{
		JPanel selectPanel = new JPanel();
		selectPanel.setLayout(new BoxLayout(selectPanel, BoxLayout.X_AXIS));
		selectPanel.add(getFontFamilyPanel());
		selectPanel.add(getFontStylePanel());
		selectPanel.add(getFontSizePanel());

		JPanel contentsPanel = new JPanel();
		contentsPanel.setLayout(new GridLayout(2, 1));
		contentsPanel.add(selectPanel, BorderLayout.NORTH);
		contentsPanel.add(getSamplePanel(), BorderLayout.CENTER);

		this.setLayout(new BoxLayout(this, BoxLayout.X_AXIS));
		this.add(contentsPanel);
		this.setBorder(BorderFactory.createEmptyBorder(5,5,5,5));
		this.setSelectedFont(DEFAULT_SELECTED_FONT);
	}

	public JTextField getFontFamilyTextField() 
	{ 
		if (fontFamilyTextField == null)
		{
			fontFamilyTextField = new JTextField();
			fontFamilyTextField.addFocusListener(
				new TextFieldFocusHandlerForTextSelection(fontFamilyTextField));
			fontFamilyTextField.addKeyListener(
				new TextFieldKeyHandlerForListSelectionUpDown(getFontFamilyList()));
			fontFamilyTextField.getDocument().addDocumentListener(
				new ListSearchTextFieldDocumentHandler(getFontFamilyList()));

		}
		return fontFamilyTextField; 
	}


	public JTextField getFontStyleTextField() 
	{ 
		if (fontStyleTextField == null)
		{
			fontStyleTextField = new JTextField();
			fontStyleTextField.addFocusListener(
				new TextFieldFocusHandlerForTextSelection(fontStyleTextField));
			fontStyleTextField.addKeyListener(
				new TextFieldKeyHandlerForListSelectionUpDown(getFontStyleList()));
			fontStyleTextField.getDocument().addDocumentListener(
				new ListSearchTextFieldDocumentHandler(getFontStyleList()));
		}
		return fontStyleTextField; 
	}

	public JTextField getFontSizeTextField() 
	{ 
		if (fontSizeTextField == null)
		{
			fontSizeTextField = new JTextField();
			fontSizeTextField.addFocusListener(
				new TextFieldFocusHandlerForTextSelection(fontSizeTextField));
			fontSizeTextField.addKeyListener(
				new TextFieldKeyHandlerForListSelectionUpDown(getFontSizeList()));
			fontSizeTextField.getDocument().addDocumentListener(
				new ListSearchTextFieldDocumentHandler(getFontSizeList()));
		}
		return fontSizeTextField; 
	}

	public JList getFontFamilyList() 
	{ 
		if (fontNameList == null)
		{
			fontNameList = new JList(getFontFamilies());
			fontNameList.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
			fontNameList.addListSelectionListener(
				new ListSelectionHandler(getFontFamilyTextField()));
			fontNameList.setSelectedIndex(0);
			fontNameList.setFont(DEFAULT_FONT);
			fontNameList.setFocusable(false);
		}
		return fontNameList; 
	}

	public JList getFontStyleList() 
	{ 
		if (fontStyleList == null)
		{
			fontStyleList = new JList(getFontStyleNames());
			fontStyleList.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
			fontStyleList.addListSelectionListener(
				new ListSelectionHandler(getFontStyleTextField()));
			fontStyleList.setSelectedIndex(0);
			fontStyleList.setFont(DEFAULT_FONT);
			fontStyleList.setFocusable(false);
		}
		return fontStyleList; 
	}

	public JList getFontSizeList() 
	{ 
		if (fontSizeList == null)
		{
			fontSizeList = new JList(FONT_SIZE_LIST);
			fontSizeList.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
			fontSizeList.addListSelectionListener(
				new ListSelectionHandler(getFontSizeTextField()));
			fontSizeList.setSelectedIndex(0);
			fontSizeList.setFont(DEFAULT_FONT);
			fontSizeList.setFocusable(false);
		}
		return fontSizeList; 
	}

	public String getSelectedFontFamily()
	{
		String fontName = (String)getFontFamilyList().getSelectedValue();
		return fontName;
	}

	public int getSelectedFontStyle()
	{
		int index  = getFontStyleList().getSelectedIndex();
		return FONT_STYLE_CODES[index];
	}

	public int getSelectedFontSize()
	{
		int fontSize = 1;
		String fontSizeString = getFontSizeTextField().getText(); 
		while (true)
		{
			try
			{
				fontSize = Integer.parseInt(fontSizeString);
				break;
			}
			catch (NumberFormatException e)
			{
				fontSizeString = (String)getFontSizeList().getSelectedValue();
				getFontSizeTextField().setText(fontSizeString); 
			}
		}
		
		return fontSize;
	}

	public Font getSelectedFont()
	{
		Font font = new Font(getSelectedFontFamily(),
			getSelectedFontStyle(), getSelectedFontSize());
		return font;
	}


	public void setSelectedFontFamily(String name)
	{
		String[] names = getFontFamilies();
		for (int i = 0;i < names.length;i++)
		{
			if (names[i].toLowerCase().equals(name.toLowerCase())) 
			{
				getFontFamilyList().setSelectedIndex(i);
				break;
			}
		}
		updateSampleFont();
	}

	public void setSelectedFontStyle(int style)
	{
		for (int i = 0;i < FONT_STYLE_CODES.length;i++)
		{
			if (FONT_STYLE_CODES[i] == style)
			{
				getFontStyleList().setSelectedIndex(i);
				break;
			}
		}
		updateSampleFont();
	}

	public void setSelectedFontSize(int size)
	{
		String sizeString = String.valueOf(size);
		for (int i = 0;i < FONT_SIZE_LIST.length;i++)
		{
			if (FONT_SIZE_LIST[i].equals(sizeString))
			{
				getFontSizeList().setSelectedIndex(i);
				break;
			}
		}
		getFontSizeTextField().setText(sizeString);
		updateSampleFont();
	}

	public void setSelectedFont(Font font)
	{
		setSelectedFontFamily(font.getFamily());
		setSelectedFontStyle(font.getStyle());
		setSelectedFontSize(font.getSize());
	}

	/**
	*  Show font selection dialog.
	*  @param parent Dialog's Parent component.
	*  @return OK_OPTION or CANCEL_OPTION
	**/
	public int showDialog(Component parent)
	{
		dialogResultValue = ERROR_OPTION;
		JDialog dialog = createDialog(parent);
		dialog.addWindowListener( new WindowAdapter(){
			public void windowClosing(WindowEvent e)
			{
				dialogResultValue = CANCEL_OPTION;
			}
		});
		
		dialog.show();
		dialog.dispose();
		dialog = null;
		return dialogResultValue;
	}


	protected class ListSelectionHandler implements ListSelectionListener
	{
		private JTextComponent textComponent;
		ListSelectionHandler(JTextComponent textComponent)
		{
			this.textComponent = textComponent;
		}

		public void valueChanged(ListSelectionEvent e)
		{
			if (e.getValueIsAdjusting() == false)
			{
				JList list = (JList)e.getSource();
				String fontName = (String)list.getSelectedValue();

				String oldFontName = textComponent.getText();
				textComponent.setText(fontName); 
				if (!oldFontName.equalsIgnoreCase(fontName))
				{
					textComponent.selectAll();
					textComponent.requestFocus();
				}
				
				updateSampleFont();
			}
		}
	}

	protected class TextFieldFocusHandlerForTextSelection extends FocusAdapter
	{
		private JTextComponent textComponent;
		public TextFieldFocusHandlerForTextSelection(JTextComponent textComponent)
		{
			this.textComponent = textComponent;
		}

		public void focusGained(FocusEvent e)
		{
			textComponent.selectAll();
		}

		public void focusLost(FocusEvent e)
		{
			textComponent.select(0, 0);
			updateSampleFont();
		}
	}


	protected class TextFieldKeyHandlerForListSelectionUpDown extends KeyAdapter
	{
		private JList targetList;
		public TextFieldKeyHandlerForListSelectionUpDown(JList list)
		{
			this.targetList = list;
		}

		public void keyPressed(KeyEvent e)
		{
			int i = targetList.getSelectedIndex();
			switch (e.getKeyCode())
			{
				case KeyEvent.VK_UP:
					i = targetList.getSelectedIndex() - 1;
					if (i < 0) i = 0;
					targetList.setSelectedIndex(i);
					break;
				case KeyEvent.VK_DOWN:
					int listSize = targetList.getModel().getSize();
					i = targetList.getSelectedIndex() + 1;
					if (i >= listSize) i = listSize - 1;
					targetList.setSelectedIndex(i);
					break;
				default:
					break;
			}
		}
	}

	protected class ListSearchTextFieldDocumentHandler implements DocumentListener
	{
		JList targetList;
		public ListSearchTextFieldDocumentHandler(JList targetList)
		{
			this.targetList = targetList;
		}
		public void insertUpdate(DocumentEvent e)
		{
			update(e);
		}
		public void removeUpdate(DocumentEvent e)
		{
			update(e);
		}
		public void changedUpdate(DocumentEvent e) 
		{ 
			update(e);
		}

		private void update(DocumentEvent event)
		{
			String newValue = "";
			try
			{
				Document doc = event.getDocument();
				newValue = doc.getText(0, doc.getLength());
			}
			catch (BadLocationException e)
			{
				e.printStackTrace();
			}
			
			if (newValue.length() > 0)
			{
				int index = targetList.getNextMatch(newValue, 0, Position.Bias.Forward);
				if (index < 0) index = 0;
				targetList.ensureIndexIsVisible(index);

				String matchedName = targetList.getModel().getElementAt(index).toString();
				if (newValue.equalsIgnoreCase(matchedName))
				{
					if (index != targetList.getSelectedIndex())
					{
						SwingUtilities.invokeLater(new ListSelector(index));
					}
				}
			}
		}

		public class ListSelector implements Runnable
		{
			private int index;
			public ListSelector(int index)
			{
				this.index = index;
			}
			public void run()
			{
				targetList.setSelectedIndex(this.index);
			}
		}
	}

	protected class DialogButtonActionListener implements ActionListener
	{
		private JDialog dialog;
		private int resultValue; // OK_OPTION, CANCEL_OPTION, 
		protected DialogButtonActionListener(JDialog dialog, int resultValue)
		{
			this.dialog = dialog;
			this.resultValue = resultValue;
		}

		public void actionPerformed(ActionEvent e)
		{
			dialogResultValue = resultValue;
			dialog.hide();
		}
	}

	protected JDialog createDialog(Component parent)
	{
		Frame frame = parent instanceof Frame ? (Frame) parent
			: (Frame)SwingUtilities.getAncestorOfClass(Frame.class, parent);
		JDialog dialog= new JDialog(frame, _("SelectFont"), true);

		ActionListener okActionListener = new DialogButtonActionListener(dialog, OK_OPTION);
		getFontFamilyTextField().addActionListener(okActionListener);
		getFontStyleTextField().addActionListener(okActionListener);
		getFontSizeTextField().addActionListener(okActionListener);
		getSampleTextField().addActionListener(okActionListener);

		JButton okButton = new JButton(_("OK"));
		okButton.setFont(DEFAULT_FONT);
		okButton.addActionListener(okActionListener);
		JButton cancelButton = new JButton(_("Cancel"));
		cancelButton.setFont(DEFAULT_FONT);
		cancelButton.addActionListener(new DialogButtonActionListener(dialog, CANCEL_OPTION));

		JPanel panel = new JPanel();
		panel.setLayout(new GridLayout(2, 1));
		panel.add(okButton);
		panel.add(cancelButton);
		panel.setBorder(BorderFactory.createEmptyBorder(25, 0, 10, 10));

		JPanel buttonPanel = new JPanel();
		buttonPanel.setLayout(new BorderLayout());
		buttonPanel.add(panel, BorderLayout.NORTH);

		getActionMap();

		dialog.getContentPane().add(this, BorderLayout.CENTER);
		dialog.getContentPane().add(buttonPanel, BorderLayout.EAST);
		dialog.pack();
		dialog.setLocationRelativeTo(frame);
		return dialog;
	}


	protected void updateSampleFont()
	{
		Font font = getSelectedFont();
		getSampleTextField().setFont(font);
	}

	protected JPanel getFontFamilyPanel()
	{
		if (fontNamePanel == null)
		{
			fontNamePanel = new JPanel();
			fontNamePanel.setLayout(new BorderLayout());
			fontNamePanel.setBorder(BorderFactory.createEmptyBorder(5,5,5,5));
			fontNamePanel.setPreferredSize(new Dimension(180, 130));

			JScrollPane scrollPane = new JScrollPane(getFontFamilyList());
			scrollPane.getVerticalScrollBar().setFocusable(false);
			scrollPane.setVerticalScrollBarPolicy(JScrollPane.VERTICAL_SCROLLBAR_ALWAYS);

			JPanel p = new JPanel();
			p.setLayout(new BorderLayout());
			p.add(getFontFamilyTextField(), BorderLayout.NORTH);
			p.add(scrollPane, BorderLayout.CENTER);

			JLabel label = new JLabel(_("FontName"));
			label.setHorizontalAlignment(JLabel.LEFT);
			label.setHorizontalTextPosition(JLabel.LEFT);
			label.setLabelFor(getFontFamilyTextField());
			label.setDisplayedMnemonic('F');

			fontNamePanel.add(label, BorderLayout.NORTH);
			fontNamePanel.add(p, BorderLayout.CENTER);

		}
		return fontNamePanel;
	}

	protected JPanel getFontStylePanel()
	{
		if (fontStylePanel == null)
		{
			fontStylePanel = new JPanel();
			fontStylePanel.setLayout(new BorderLayout());
			fontStylePanel.setBorder(BorderFactory.createEmptyBorder(5,5,5,5));
			fontStylePanel.setPreferredSize(new Dimension(140, 130));

			JScrollPane scrollPane = new JScrollPane(getFontStyleList());
			scrollPane.getVerticalScrollBar().setFocusable(false);
			scrollPane.setVerticalScrollBarPolicy(JScrollPane.VERTICAL_SCROLLBAR_ALWAYS);

			JPanel p = new JPanel();
			p.setLayout(new BorderLayout());
			p.add(getFontStyleTextField(), BorderLayout.NORTH);
			p.add(scrollPane, BorderLayout.CENTER);

			JLabel label = new JLabel(_("FontStyle"));
			label.setHorizontalAlignment(JLabel.LEFT);
			label.setHorizontalTextPosition(JLabel.LEFT);
			label.setLabelFor(getFontStyleTextField());
			label.setDisplayedMnemonic('Y');

			fontStylePanel.add(label, BorderLayout.NORTH);
			fontStylePanel.add(p, BorderLayout.CENTER);
		}
		return fontStylePanel;
	}

	protected JPanel getFontSizePanel()
	{
		if (fontSizePanel == null)
		{
			fontSizePanel = new JPanel();
			fontSizePanel.setLayout(new BorderLayout());
			fontSizePanel.setPreferredSize(new Dimension(70, 130));
			fontSizePanel.setBorder(BorderFactory.createEmptyBorder(5,5,5,5));

			JScrollPane scrollPane = new JScrollPane(getFontSizeList());
			scrollPane.getVerticalScrollBar().setFocusable(false);
			scrollPane.setVerticalScrollBarPolicy(JScrollPane.VERTICAL_SCROLLBAR_ALWAYS);

			JPanel p = new JPanel();
			p.setLayout(new BorderLayout());
			p.add(getFontSizeTextField(), BorderLayout.NORTH);
			p.add(scrollPane, BorderLayout.CENTER);

			JLabel label = new JLabel(_("FontSize"));
			label.setHorizontalAlignment(JLabel.LEFT);
			label.setHorizontalTextPosition(JLabel.LEFT);
			label.setLabelFor(getFontSizeTextField());
			label.setDisplayedMnemonic('S');

			fontSizePanel.add(label, BorderLayout.NORTH);
			fontSizePanel.add(p, BorderLayout.CENTER);
		}
		return fontSizePanel;
	}

	protected JPanel getSamplePanel()
	{
		if (samplePanel == null)
		{
			Border titledBorder = BorderFactory.createTitledBorder(
				BorderFactory.createEtchedBorder(), _("Sample"));
			Border empty = BorderFactory.createEmptyBorder(5,10,10,10);
			Border border = BorderFactory.createCompoundBorder(titledBorder,empty);

			samplePanel = new JPanel();
			samplePanel.setLayout(new BorderLayout());
			samplePanel.setBorder(border);

			samplePanel.add(getSampleTextField(), BorderLayout.CENTER);
		}
		return samplePanel;
	}

	protected JTextField getSampleTextField() 
	{ 
		if (sampleText == null)
		{
			Border lowered = BorderFactory.createLoweredBevelBorder();

			sampleText = new JTextField(_("SampleString"));
			sampleText.setBorder(lowered);
			sampleText.setPreferredSize(new Dimension(300, 100));
		}
		return sampleText; 
	}

	protected String[] getFontFamilies()
	{
		if (fontFamilyNames == null)
		{
			GraphicsEnvironment env = GraphicsEnvironment.getLocalGraphicsEnvironment ();
			fontFamilyNames = env.getAvailableFontFamilyNames();
		}
		return fontFamilyNames;
	}
}

