/*
 * brownies and its relative products are published under the terms
 * of the Apache Software License.
 * 
 * Created on 2004/11/20 1:22:21
 */
package org.asyrinx.brownie.core.query;

import java.text.DateFormat;
import java.text.DecimalFormat;
import java.text.NumberFormat;
import java.util.Date;
import java.util.Iterator;

import org.asyrinx.brownie.core.lang.StringUtils;
import org.asyrinx.brownie.core.query.exp.ICompositeExpression;
import org.asyrinx.brownie.core.query.exp.IExpression;
import org.asyrinx.brownie.core.query.exp.IExpressionWrap;
import org.asyrinx.brownie.core.query.exp.IFieldCompare;
import org.asyrinx.brownie.core.query.exp.IFieldValuedExpression;
import org.asyrinx.brownie.core.query.exp.IValuedExpression;
import org.asyrinx.brownie.core.query.model.Field;
import org.asyrinx.brownie.core.query.model.FieldAlias;
import org.asyrinx.brownie.core.query.model.FieldAliasList;
import org.asyrinx.brownie.core.query.model.Table;
import org.asyrinx.brownie.core.query.model.TableAlias;
import org.asyrinx.brownie.core.query.model.TableAliasList;
import org.asyrinx.brownie.core.util.jp.JpDateFormat;

/**
 * @author takeshi
 */
public class BasicSelectBuilder implements SelectBuilder {

    /**
     *  
     */
    public BasicSelectBuilder() {
        this(StatementType.SQL);
    }

    /**
     * @param sql
     */
    public BasicSelectBuilder(StatementType statementType) {
        super();
        this.statementType = statementType;
    }

    private final StatementType statementType;

    /**
     * @param field
     * @return
     */
    public String getFieldName(Field field) {
        return statementType.getFieldName(field);
    }

    /**
     * @param table
     * @return
     */
    public String getTableName(Table table) {
        return statementType.getTableName(table);
    }

    public String buildSelect(Select select) {
        this.needTableAliasName = !hasSinlgleTableAlias(select);
        final StringBuffer result = new StringBuffer();
        doBuildSelect(select, result);
        return result.toString();
    }

    public String buildExpression(IExpression expression) {
        this.needTableAliasName = false;
        return buildExpressionImpl(expression);
    }

    public String buildExpressionImpl(IExpression expression) {
        final StringBuffer result = new StringBuffer();
        doBuildExpression(expression, result);
        return result.toString();
    }

    protected void doBuildSelect(Select select, StringBuffer dest) {
        addWhenNotEmpty("select", dest, doBuildFieldAliasList(select.getFields()));
        addWhenNotEmpty("from", dest, doBuildTableAliasList(select.getFroms()));
        addWhenNotEmpty("where", dest, doBuildExpression(select.getWhere()));
        addWhenNotEmpty("group by", dest, doBuildFieldAliasList(select.getGroupBys()));
        addWhenNotEmpty("having", dest, doBuildExpression(select.getHavings()));
        addWhenNotEmpty("order by", dest, doBuildFieldAliasList(select.getOrderBys()));
    }

    protected void addWhenNotEmpty(String header, StringBuffer dest, StringBuffer source) {
        if (source.length() == 0)
            return;
        if (dest.length() != 0)
            dest.append(" ");
        dest.append(header);
        dest.append(" ");
        dest.append(source);
    }

    protected StringBuffer doBuildFieldAliasList(FieldAliasList aliasList) {
        final StringBuffer result = new StringBuffer();
        for (Iterator i = aliasList.iterator(); i.hasNext();) {
            final FieldAlias alias = (FieldAlias) i.next();
            result.append(alias.getOwner().getName());
            result.append(".");
            result.append(getFieldName(alias.getField()));
            result.append(" as ");
            result.append(alias.getName());
            if (i.hasNext()) {
                result.append(", ");
            }
        }
        return result;
    }

    protected StringBuffer doBuildTableAliasList(TableAliasList aliasList) {
        final StringBuffer result = new StringBuffer();
        for (Iterator i = aliasList.iterator(); i.hasNext();) {
            final TableAlias alias = (TableAlias) i.next();
            result.append(getTableName(alias.getOriginal()));
            if (this.needTableAliasName) {
                result.append(" as ");
                result.append(alias.getName());
            }
            if (i.hasNext()) {
                result.append(", ");
            }
        }
        return result;
    }

    private boolean needTableAliasName = false;

    private boolean hasSinlgleTableAlias(Select select) {
        if (select.getFroms().isEmpty())
            return true;
        if (select.getFroms().size() > 1)
            return false;
        final TableAlias tableAlias = select.getFroms().get(0);
        return StringUtils.equals(tableAlias.getName(), getTableName(tableAlias.getOriginal()));
    }

    private IExpression rootExpression = null;

    private boolean needBracket(ICompositeExpression expression) {
        if (expression != this.rootExpression)
            return true;
        return !isIgnoreRootBracket();
    }

    protected StringBuffer doBuildExpression(IExpression expression) {
        this.rootExpression = expression;
        try {
            final StringBuffer result = new StringBuffer();
            doBuildExpression(expression, result);
            return result;
        } finally {
            this.rootExpression = null;
        }
    }

    protected void doBuildExpression(IExpression expression, StringBuffer dest) {
        if (expression == null)
            return;
        if (expression instanceof ICompositeExpression) {
            doBuildComposite((ICompositeExpression) expression, dest);
        } else if (expression instanceof IExpressionWrap) {
            doBuildWrap((IExpressionWrap) expression, dest);
        } else if (expression instanceof IValuedExpression) {
            doBuildFieldExpression((IValuedExpression) expression, dest);
        } else if (expression instanceof IFieldCompare) {
            doBuildFieldCompare((IFieldCompare) expression, dest);
        }
    }

    protected void doBuildComposite(ICompositeExpression composite, StringBuffer dest) {
        final StringBuffer result = new StringBuffer();
        for (Iterator i = composite.iterator(); i.hasNext();) {
            final IExpression exp = (IExpression) i.next();
            this.doBuildExpression(exp, result);
            if (i.hasNext()) {
                if (needBracket(composite)) {
                    result.append(getBracketEnd());
                    result.append(" ");
                    result.append(composite.getOperator());
                    result.append(" ");
                    result.append(getBracketBegin());
                } else {
                    result.append(" ");
                    result.append(composite.getOperator());
                    result.append(" ");
                }
            }
        }
        if (result.length() > 1) {
            if (needBracket(composite)) {
                dest.append(getBracketBegin());
                dest.append(result.toString());
                dest.append(getBracketEnd());
            } else {
                dest.append(result.toString());
            }
        }
    }

    protected void doBuildWrap(IExpressionWrap wrap, StringBuffer dest) {
        if (StringUtils.isEmpty(wrap.getOperator())) {
            doBuildExpression(wrap.getContent(), dest);
        } else {
            dest.append(wrap.getOperator());
            dest.append(" ");
            dest.append(getBracketBegin());
            doBuildExpression(wrap.getContent(), dest);
            dest.append(getBracketEnd());
        }
    }

    protected void appendExpressionFieldName(FieldAlias field, StringBuffer dest) {
        if (this.needTableAliasName) {
            dest.append(field.getOwner().getName());
            dest.append(".");
        }
        dest.append(getFieldName(field.getField()));
    }

    protected void doBuildFieldExpression(IValuedExpression valuedExpression, StringBuffer dest) {
        if (valuedExpression instanceof IFieldValuedExpression) {
            final IFieldValuedExpression fieldExpression = (IFieldValuedExpression) valuedExpression;
            final FieldAlias field = fieldExpression.getField();
            appendExpressionFieldName(field, dest);
            dest.append(" ");
            dest.append(fieldExpression.getOperator());
            dest.append(" ");
            dest.append(dispatch4Value(field, fieldExpression.getValue()));
        } else {
            dest.append(valuedExpression.getFieldAsObject());
            dest.append(" ");
            dest.append(valuedExpression.getOperator());
            dest.append(" ");
            dest.append(valuedExpression.getValue());
        }
    }

    protected void doBuildFieldCompare(IFieldCompare compare, StringBuffer dest) {
        appendExpressionFieldName(compare.getLeft(), dest);
        dest.append(" ");
        dest.append(compare.getOperator());
        dest.append(" ");
        appendExpressionFieldName(compare.getRight(), dest);
    }

    protected String dispatch4Value(FieldAlias field, Object value) {
        if (field.getType() == String.class) {
            return toStringValue(value);
        } else if (field.getType().isAssignableFrom(Date.class)) {
            final String format = (StringUtils.isEmpty(field.getFormat())) ? this.getDefaultDateFormat() : field
                    .getFormat();
            return toStringValue(toDateValue(format, value));
        } else if (field.getType().isAssignableFrom(Number.class)) {
            final String format = (StringUtils.isEmpty(field.getFormat())) ? this.getDefaultNumberFormat() : field
                    .getFormat();
            return toNumberValue(field.getType(), format, value);
        } else {
            return String.valueOf(value);
        }
    }

    protected String toDateValue(String format, Object value) {
        if (value instanceof Date) {
            final DateFormat dateFormat = new JpDateFormat(format);
            return dateFormat.format((Date) value);
        } else {
            return value.toString();
        }
    }

    protected String toNumberValue(Class type, String format, Object value) {
        if (value instanceof Number) {
            final Number number = (Number) value;
            if (!StringUtils.isEmpty(format)) {
                final NumberFormat numberFormat = new DecimalFormat(format);
                if (type.isAssignableFrom(Double.class)) {
                    return numberFormat.format(number.doubleValue());
                } else {
                    return numberFormat.format(number.longValue());
                }
            }
        }
        return value.toString();
    }

    protected String toStringValue(Object value) {
        return StringUtils.toQuoted(String.valueOf(value), getStringQuote());
    }

    private char stringQuote = '\'';

    /**
     * @return Returns the stringQuote.
     */
    public char getStringQuote() {
        return stringQuote;
    }

    /**
     * @param stringQuote
     *            The stringQuote to set.
     */
    public void setStringQuote(char stringQuote) {
        this.stringQuote = stringQuote;
    }

    private String bracketBegin = "(";

    private String bracketEnd = ")";

    /**
     * @return Returns the bracketBegin.
     */
    public String getBracketBegin() {
        return bracketBegin;
    }

    /**
     * @param bracketBegin
     *            The bracketBegin to set.
     */
    public void setBracketBegin(String bracketBegin) {
        this.bracketBegin = bracketBegin;
    }

    /**
     * @return Returns the bracketEnd.
     */
    public String getBracketEnd() {
        return bracketEnd;
    }

    /**
     * @param bracketEnd
     *            The bracketEnd to set.
     */
    public void setBracketEnd(String bracketEnd) {
        this.bracketEnd = bracketEnd;
    }

    private String defaultDateFormat = "yyyy-MM-dd HH:mm:ss";

    private String defaultNumberFormat = "yyyy-MM-dd HH:mm:ss";

    /**
     * @return Returns the defaultDateFormat.
     */
    public String getDefaultDateFormat() {
        return defaultDateFormat;
    }

    /**
     * @param defaultDateFormat
     *            The defaultDateFormat to set.
     */
    public void setDefaultDateFormat(String defaultDateFormat) {
        this.defaultDateFormat = defaultDateFormat;
    }

    /**
     * @return Returns the defaultNumberFormat.
     */
    public String getDefaultNumberFormat() {
        return defaultNumberFormat;
    }

    /**
     * @param defaultNumberFormat
     *            The defaultNumberFormat to set.
     */
    public void setDefaultNumberFormat(String defaultNumberFormat) {
        this.defaultNumberFormat = defaultNumberFormat;
    }

    private boolean ignoreRootBracket = true;

    /**
     * @return Returns the ignoreRootBracket.
     */
    public boolean isIgnoreRootBracket() {
        return ignoreRootBracket;
    }

    /**
     * @param ignoreRootBracket
     *            The ignoreRootBracket to set.
     */
    public void setIgnoreRootBracket(boolean ignoreRootBracket) {
        this.ignoreRootBracket = ignoreRootBracket;
    }

}