/*
 * Joey and its relative products are published under the terms
 * of the Apache Software License.
 */
package org.asyrinx.brownie.core.util.jp;

import java.text.DateFormat;
import java.text.DateFormatSymbols;
import java.text.FieldPosition;
import java.text.ParsePosition;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;
import java.util.Iterator;
import java.util.Locale;

import org.asyrinx.brownie.core.lang.StringUtils;
import org.asyrinx.brownie.core.util.Era;
import org.asyrinx.brownie.core.util.SimpleDate;

/**
 * @author Akima
 */
public class JpDateFormat extends DateFormat {

	public static final String FORMAT_DATE_TIME_1 = "jEEjyyNMMdd hhmm";

	public static final String FORMAT_DATE_TIME_2 = "jEjyyNMMdd hhmm";

	public static final String FORMAT_DATE_TIME_3 = "jejyy/MM/dd hh:mm";

	public static final String FORMAT_DATE_TIME_4 = "jEjyy/MM/dd hh:mm";

	public static final String FORMAT_DATE_1 = "jEEjyyNMMdd";

	public static final String FORMAT_DATE_2 = "jEjyyNMMdd";

	public static final String FORMAT_DATE_3 = "jejyy/MM/dd";

	public static final String FORMAT_DATE_4 = "jEjyy/MM/dd";

	/**
	 * Constructor for JpDateFormat.
	 * 
	 * @param pattern
	 */
	public JpDateFormat(String pattern) {
		this(pattern, Locale.getDefault());
	}

	/**
	 * Constructor for JpDateFormat.
	 * 
	 * @param pattern
	 * @param loc
	 */
	public JpDateFormat(String pattern, Locale locale) {
		this(pattern, new DateFormatSymbols(locale));
	}

	/**
	 * Constructor for JpDateFormat.
	 * 
	 * @param pattern
	 * @param dateFormatSymbols
	 */
	public JpDateFormat(String pattern, DateFormatSymbols dateFormatSymbols) {
		super();
		this.pattern = pattern;
		this.dateFormatSymbols = dateFormatSymbols;
	}

	private String pattern;

	private DateFormatSymbols dateFormatSymbols;

	public String toPattern() {
		return this.pattern;
	}

	public void applyPattern(String p) {
		this.pattern = p;
	}

	/**
	 * @return
	 */
	public DateFormatSymbols getDateFormatSymbols() {
		return dateFormatSymbols;
	}

	/**
	 * @param symbols
	 */
	public void setDateFormatSymbols(DateFormatSymbols symbols) {
		dateFormatSymbols = symbols;
	}

	private DateFormat newDateFormat(String p) {
		return new SimpleDateFormat(p, dateFormatSymbols);
	}

	/**
	 * 
	 * @param date
	 * @param toAppendTo
	 * @param pos
	 * @return @see java.text.DateFormat#format(java.util.Date,
	 *           java.lang.StringBuffer, java.text.FieldPosition)
	 */
	public StringBuffer format(Date date, StringBuffer toAppendTo,
			FieldPosition pos) {
		if (date == null)
			throw new NullPointerException("failed to format. date was null.");
		final Era era = JpEra.ERA_GROUP.getEra(new SimpleDate(date));
		final StringBuffer patternBuf = new StringBuffer(toPattern());
		for (int i = 0; i < PATTERNS.length; i++) {
			final JpPattern jpPattern = PATTERNS[i];
			if (jpPattern.contained(patternBuf.toString()))
				jpPattern.format(patternBuf, era, date);
		}
		final DateFormat innerFormat = newDateFormat(patternBuf.toString());
		return innerFormat.format(date, toAppendTo, pos);
	}

	/**
	 * 
	 * @param text
	 * @param pos
	 * @return @see java.text.DateFormat#parse(java.lang.String,
	 *           java.text.ParsePosition)
	 */
	public Date parse(String text, ParsePosition pos) {
		final StringBuffer patternBuf = new StringBuffer(toPattern());
		final StringBuffer textBuf = new StringBuffer(text);
		final JpEraPattern eraPattern = findEraPattern(toPattern(), pos
				.getIndex());
		final Era era = (eraPattern == null) ? null : eraPattern.findEra(
				textBuf, patternBuf, pos.getIndex());
		for (int i = 0; i < PATTERNS_YEAR.length; i++)
			StringUtils.replace(patternBuf, PATTERNS_YEAR[i].pattern, "yyyy");
		final DateFormat innerFormat = newDateFormat(patternBuf.toString());
		final Calendar cal = Calendar.getInstance();
		cal.set(Calendar.YEAR, 0);
		final Date superResult = innerFormat.parse(textBuf.toString(), pos);
		if (superResult != null)
			cal.setTime(superResult);
		if (era != null)
			cal.set(Calendar.YEAR, era.toAnnoDomini(cal.get(Calendar.YEAR)));
		return cal.getTime();
	}

	private JpEraPattern findEraPattern(String text, int position) {
		for (int i = 0; i < PATTERNS_ERA.length; i++) {
			final JpEraPattern eraPattern = PATTERNS_ERA[i];
			int pos = text.indexOf(eraPattern.pattern, position);
			if (pos > -1)
				return PATTERNS_ERA[i];
		}
		return null;
	}

	static public final JpEraPattern ERA_ALPH_SHORT = new JpEraPattern("je") {
		public boolean needSingleQuote() {
			return true;
		}

		protected String toEraStr(Era era) {
			return era.getFirstLetter();
		}
	};

	static public final JpEraPattern ERA_KANJI_SHORT = new JpEraPattern("jE") {
		public boolean needSingleQuote() {
			return true;
		}

		protected String toEraStr(Era era) {
			return era.getCaptionShort();
		}
	};

	static public final JpEraPattern ERA_KANJI_LONG = new JpEraPattern("jEE") {
		public boolean needSingleQuote() {
			return true;
		}

		protected String toEraStr(Era era) {
			return era.getCaption();
		}
	};

	static public final JpPattern YEAR_NUMERIC = new JpPattern("jyy") {
		public String newString(Era era, Date d) {
			return String
					.valueOf(JpEra.ERA_GROUP.getEraYear(new SimpleDate(d)));
		}

		public boolean needSingleQuote() {
			return false;
		}
	};

	static private JpPattern[] PATTERNS = new JpPattern[] { ERA_ALPH_SHORT,
			ERA_KANJI_LONG, ERA_KANJI_SHORT, YEAR_NUMERIC, };

	static private JpEraPattern[] PATTERNS_ERA = new JpEraPattern[] {
			ERA_ALPH_SHORT, ERA_KANJI_LONG, ERA_KANJI_SHORT };

	static private JpPattern[] PATTERNS_YEAR = new JpPattern[] { YEAR_NUMERIC };

}

abstract class JpPattern {
	protected String pattern;

	public JpPattern(String pattern) {
		super();
		this.pattern = pattern;
	}

	public boolean contained(String target) {
		return (target.indexOf(pattern) > -1);
	}

	public void format(StringBuffer value, Era era, Date d) {
		StringUtils.replace(value, this.pattern, newPattern(era, d));
	}

	private String newPattern(Era era, Date d) {
		String result = newString(era, d);
		if (needSingleQuote()) {
			result = StringUtils.toQuoted(result, '\'');
		}
		return result;
	}

	abstract public boolean needSingleQuote();

	abstract public String newString(Era era, Date d);
}

abstract class JpEraPattern extends JpPattern {
	public JpEraPattern(String pattern) {
		super(pattern);
	}

	public String newString(Era era, Date d) {
		return toEraStr(era);
	}

	public Era findEra(StringBuffer textBuf, StringBuffer patternBuf,
			int textIndex) {
		if (textBuf.length() < 1)
			return null;
		int eraIndex = patternBuf.toString().indexOf(pattern) + textIndex;
		final Iterator iterator = JpEra.ERA_GROUP.getEras(Locale.JAPAN)
				.iterator();
		while (iterator.hasNext()) {
			final Era result = (Era) iterator.next();
			final String val = toEraStr(result);
			final String textSub = textBuf.toString().substring(textIndex,
					textIndex + val.length());
			if (val.equals(textSub)) {
				patternBuf.delete(eraIndex, eraIndex + pattern.length());
				textBuf.delete(eraIndex, eraIndex + toEraStr(result).length());
				return result;
			}
		}
		return null;
	}

	abstract protected String toEraStr(Era era);
}