/*
 * Joey and its relative products are published under the terms
 * of the Apache Software License.
 */
/*
 * Created on 2004/03/01
 */
package org.asyrinx.brownie.jdbc.logger;

import java.sql.CallableStatement;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.sql.Savepoint;
import java.sql.Statement;

import org.asyrinx.brownie.core.log.CascadeNamedLog;
import org.asyrinx.brownie.core.log.DispatchLog;
import org.asyrinx.brownie.jdbc.wrapper.ConnectionWrapper;

/**
 * @author akima
 */
public class LogConnection extends ConnectionWrapper {

    /**
     * @param wrapped
     */
    public LogConnection(Connection wrapped) {
        this(wrapped, wrapped.getClass().getName());
    }

    /**
     * @param wrapped
     */
    public LogConnection(Connection wrapped, String loggerName) {
        this(wrapped, loggerName, DispatchLog.DEFAULT_LEVEL);
    }

    /**
     * @param wrapped
     */
    public LogConnection(Connection wrapped, String loggerName, String logLevel) {
        this(wrapped, new CascadeNamedLog(loggerName, logLevel));
    }

    /**
     * @param wrapped
     */
    public LogConnection(Connection wrapped, CascadeNamedLog parentLog) {
        super(wrapped);
        this.log = parentLog.subLog(this.wrapped);
    }

    private final CascadeNamedLog log;

    /*
     * (non-Javadoc)
     * 
     * @see java.sql.Connection#createStatement()
     */
    public Statement createStatement() throws SQLException {
        return new LogStatement(super.createStatement(), log);
    }

    /*
     * (non-Javadoc)
     * 
     * @see java.sql.Connection#createStatement(int, int, int)
     */
    public Statement createStatement(int resultSetType, int resultSetConcurrency, int resultSetHoldability)
            throws SQLException {
        final Statement result = super.createStatement(resultSetType, resultSetConcurrency, resultSetHoldability);
        return new LogStatement(result, log.subLog(result));
    }

    /*
     * (non-Javadoc)
     * 
     * @see java.sql.Connection#createStatement(int, int)
     */
    public Statement createStatement(int resultSetType, int resultSetConcurrency) throws SQLException {
        final Statement result = super.createStatement(resultSetType, resultSetConcurrency);
        return new LogStatement(result, log.subLog(result));
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.asyrinx.brownie.jdbc.logger.ConnectionWrapper#nativeSQL(java.lang.String)
     */
    public String nativeSQL(String sql) throws SQLException {
        try {
            final String result = super.nativeSQL(sql);
            log.log(sql);
            return result;
        } catch (SQLException e) {
            log.log((Object) sql, e);
            throw e;
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see java.sql.Connection#prepareCall(java.lang.String, int, int, int)
     */
    public CallableStatement prepareCall(String sql, int resultSetType, int resultSetConcurrency,
            int resultSetHoldability) throws SQLException {
        final CallableStatement result = super.prepareCall(sql, resultSetType, resultSetConcurrency,
                resultSetHoldability);
        return new LogCallableStatement(result, sql, log.subLog(result));
    }

    /*
     * (non-Javadoc)
     * 
     * @see java.sql.Connection#prepareCall(java.lang.String, int, int)
     */
    public CallableStatement prepareCall(String sql, int resultSetType, int resultSetConcurrency) throws SQLException {
        final CallableStatement result = super.prepareCall(sql, resultSetType, resultSetConcurrency);
        return new LogCallableStatement(result, sql, log.subLog(result));
    }

    /*
     * (non-Javadoc)
     * 
     * @see java.sql.Connection#prepareCall(java.lang.String)
     */
    public CallableStatement prepareCall(String sql) throws SQLException {
        final CallableStatement result = super.prepareCall(sql);
        return new LogCallableStatement(result, sql, log.subLog(result));
    }

    /*
     * (non-Javadoc)
     * 
     * @see java.sql.Connection#prepareStatement(java.lang.String, int, int,
     *      int)
     */
    public PreparedStatement prepareStatement(String sql, int resultSetType, int resultSetConcurrency,
            int resultSetHoldability) throws SQLException {
        final PreparedStatement result = super.prepareStatement(sql, resultSetType, resultSetConcurrency,
                resultSetHoldability);
        return new LogPreparedStatement(result, sql, log.subLog(result));
    }

    /*
     * (non-Javadoc)
     * 
     * @see java.sql.Connection#prepareStatement(java.lang.String, int, int)
     */
    public PreparedStatement prepareStatement(String sql, int resultSetType, int resultSetConcurrency)
            throws SQLException {
        final PreparedStatement result = super.prepareStatement(sql, resultSetType, resultSetConcurrency);
        return new LogPreparedStatement(result, sql, log.subLog(result));
    }

    /*
     * (non-Javadoc)
     * 
     * @see java.sql.Connection#prepareStatement(java.lang.String, int)
     */
    public PreparedStatement prepareStatement(String sql, int autoGeneratedKeys) throws SQLException {
        final PreparedStatement result = super.prepareStatement(sql, autoGeneratedKeys);
        return new LogPreparedStatement(result, sql, log.subLog(result));
    }

    /*
     * (non-Javadoc)
     * 
     * @see java.sql.Connection#prepareStatement(java.lang.String, int[])
     */
    public PreparedStatement prepareStatement(String sql, int[] columnIndexes) throws SQLException {
        final PreparedStatement result = super.prepareStatement(sql, columnIndexes);
        return new LogPreparedStatement(result, sql, log.subLog(result));
    }

    /*
     * (non-Javadoc)
     * 
     * @see java.sql.Connection#prepareStatement(java.lang.String,
     *      java.lang.String[])
     */
    public PreparedStatement prepareStatement(String sql, String[] columnNames) throws SQLException {
        final PreparedStatement result = super.prepareStatement(sql, columnNames);
        return new LogPreparedStatement(result, sql, log.subLog(result));
    }

    /*
     * (non-Javadoc)
     * 
     * @see java.sql.Connection#prepareStatement(java.lang.String)
     */
    public PreparedStatement prepareStatement(String sql) throws SQLException {
        final PreparedStatement result = super.prepareStatement(sql);
        return new LogPreparedStatement(result, sql, log.subLog(result));
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.asyrinx.brownie.jdbc.wrapper.ConnectionWrapper#commit()
     */
    public void commit() throws SQLException {
        try {
            super.commit();
            this.log.log("commit()");
        } catch (SQLException e) {
            this.log.error("failed to commit()", e);
            throw e;
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.asyrinx.brownie.jdbc.wrapper.ConnectionWrapper#rollback()
     */
    public void rollback() throws SQLException {
        try {
            super.rollback();
            this.log.log("rollback()");
        } catch (SQLException e) {
            this.log.error("failed to rollback()", e);
            throw e;
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.asyrinx.brownie.jdbc.wrapper.ConnectionWrapper#rollback(java.sql.Savepoint)
     */
    public void rollback(Savepoint savepoint) throws SQLException {
        try {
            super.rollback(savepoint);
            this.log.log("rollback(" + savepoint + ")");
        } catch (SQLException e) {
            this.log.error("failed to rollback(" + savepoint + ")", e);
            throw e;
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.asyrinx.brownie.jdbc.wrapper.ConnectionWrapper#setAutoCommit(boolean)
     */
    public void setAutoCommit(boolean autoCommit) throws SQLException {
        try {
            super.setAutoCommit(autoCommit);
            this.log.log("setAutoCommit(" + autoCommit + ")");
        } catch (SQLException e) {
            this.log.error("failed to setAutoCommit(" + autoCommit + ")", e);
            throw e;
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.asyrinx.brownie.jdbc.wrapper.ConnectionWrapper#setSavepoint()
     */
    public Savepoint setSavepoint() throws SQLException {
        try {
            final Savepoint result = super.setSavepoint();
            this.log.log("setSavepoint(): " + result);
            return result;
        } catch (SQLException e) {
            this.log.error("failed to setSavepoint()", e);
            throw e;
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.asyrinx.brownie.jdbc.wrapper.ConnectionWrapper#setSavepoint(java.lang.String)
     */
    public Savepoint setSavepoint(String name) throws SQLException {
        try {
            final Savepoint result = super.setSavepoint(name);
            this.log.log("setSavepoint(" + name + "): " + result);
            return result;
        } catch (SQLException e) {
            this.log.error("failed to setSavepoint(" + name + ")", e);
            throw e;
        }
    }

}