/*
 * Created on 2004/05/12
 */
package org.asyrinx.brownie.swing;

import java.awt.Color;
import java.awt.Insets;
import java.awt.Point;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ComponentAdapter;
import java.awt.event.ComponentEvent;
import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;

import javax.swing.JButton;
import javax.swing.JDialog;
import javax.swing.JFrame;
import javax.swing.JPanel;
import javax.swing.JTextField;

import org.apache.commons.lang.StringUtils;
import org.asyrinx.brownie.core.util.jp.JpDateFormat;

public final class DateField extends JPanel {

	protected final JTextField dateText = new JTextField();

	private final JButton dropdownButton = new JButton();

	public DateField() {
		super();
		initialize();
	}

	/**
	 * Alternate constructor that initializes the currently selected date to the
	 * specified date.
	 * 
	 * @param initialDate
	 */
	public DateField(final Date initialDate) {
		super();
		initialize();
		dateText.setText(dateToString(initialDate));
	}

	/**
	 * This method initializes this
	 * 
	 * @return void
	 */
	private void initialize() {
		this.setLayout(new javax.swing.BoxLayout(this,
				javax.swing.BoxLayout.X_AXIS));
		this.add(dateText, null);
		this.add(dropdownButton, null);
		dateText.setText("");
		dateText.setEditable(true);
		dateText.setBackground(new Color(255, 255, 255));
		dropdownButton.setText("...");
		dropdownButton.setMargin(new Insets(2, 2, 2, 2));
		dropdownButton.addActionListener(new ActionListener() {
			public void actionPerformed(final ActionEvent evt) {
				onButtonClick(evt);
			}
		});
		this.setSize(201, 37);
		this.setPreferredSize(new java.awt.Dimension(25, 21));
		dateText.setPreferredSize(new java.awt.Dimension(6, 18));
	}

	/**
	 * Returns the currently selected date or null if not set.
	 * 
	 * @return date
	 */
	public Date getDate() {
		return stringToDate(dateText.getText());
	}

	public void setDate(Date date) {
		dateText.setText(dateToString(date));
	}

	/**
	 * Event handler that displays the DatePicker when the button is clicked.
	 * 
	 * @param e
	 */
	protected void onButtonClick(final ActionEvent e) {
		final JDialog dlg = new JDialog(new JFrame(), true);
		final DatePicker dp = (StringUtils.isEmpty(dateText.getText())) ? new DatePicker()
				: new DatePicker(stringToDate(dateText.getText()));
		dp.setHideOnSelect(true);
		dp.addComponentListener(new ComponentAdapter() {
			public void componentHidden(final ComponentEvent evt) {
				final Date dt = ((DatePicker) evt.getSource()).getDate();
				if (null != dt)
					dateText.setText(dateToString(dt));
				dlg.dispose();
			}
		});
		final Point p = dateText.getLocationOnScreen();
		p.setLocation(p.getX(), p.getY() - 1 + dateText.getSize().getHeight());
		dlg.setLocation(p);
		dlg.setResizable(false);
		dlg.setUndecorated(true);
		dlg.getContentPane().add(dp);
		dlg.pack();
		dlg.show();
	}

	/**
	 * Returns a short string representation for the specified date (January 1,
	 * 2003).
	 * 
	 * @param dt
	 * @return short string
	 */
	protected String dateToString(final Date dt) {
		return (dt == null) ? null : dateFormat.format(dt);
	}

	/**
	 * Constructs a Date object from a short date (January 1, 2003).
	 * 
	 * @param s
	 * @return date
	 */
	private Date stringToDate(final String s) {
		if (StringUtils.isEmpty(s))
			return null;
		try {
			return dateFormat.parse(s);
		} catch (ParseException e) {
			return null;
		}
	}

	private DateFormat dateFormat = new JpDateFormat("yyyy/MM/dd HH:mm:ss");

	/**
	 * @return Returns the dateFormat.
	 */
	public DateFormat getDateFormat() {
		return dateFormat;
	}

	/**
	 * @param dateFormat
	 *               The dateFormat to set.
	 */
	public void setDateFormat(DateFormat dateFormat) {
		this.dateFormat = dateFormat;
	}

	/**
	 * @return Returns the pattern.
	 */
	public String getPattern() {
		if (dateFormat instanceof JpDateFormat)
			return ((JpDateFormat) dateFormat).toPattern();
		else if (dateFormat instanceof SimpleDateFormat)
			return ((SimpleDateFormat) dateFormat).toPattern();
		return null;
	}

	/**
	 * @param pattern
	 *               The pattern to set.
	 */
	public void setPattern(String pattern) {
		if (dateFormat instanceof SimpleDateFormat)
			dateFormat = new SimpleDateFormat(pattern);
		else
			dateFormat = new JpDateFormat(pattern);
	}

	/**
	 * @return
	 */
	public String getText() {
		return dateText.getText();
	}

	/**
	 * @return
	 */
	public boolean isEditable() {
		return dateText.isEditable();
	}

	/**
	 * @param b
	 */
	public void setEditable(boolean b) {
		dateText.setEditable(b);
		dropdownButton.setEnabled(b);
	}

	/**
	 * @param t
	 */
	public void setText(String t) {
		dateText.setText(t);
	}
}