/*
 * Joey and its relative products are published under the terms
 * of the Apache Software License.
 */
/*
 * Created on 2004/02/27
 */
package org.asyrinx.joey.om.hibernate;

import java.text.DateFormat;
import java.text.ParseException;
import java.util.Collection;
import java.util.Date;
import java.util.Map;

import net.sf.hibernate.Criteria;
import net.sf.hibernate.Session;
import net.sf.hibernate.exception.NestableRuntimeException;
import net.sf.hibernate.expression.Criterion;
import net.sf.hibernate.expression.Expression;
import net.sf.hibernate.expression.Order;

import org.asyrinx.brownie.core.sql.Operator;
import org.asyrinx.joey.om.ConditionTranslator;
import org.asyrinx.joey.om.SearchCondition;

/**
 * @author akima
 */
public class CriteriaTranslator extends AbstractHibernateTranslator implements
		ConditionTranslator {

	/**
	 *  
	 */
	public CriteriaTranslator(Map columnToProperty, Map tableToClass,
			Session session, Class targetClass) {
		super(columnToProperty, tableToClass);
		this.criteria = session.createCriteria(targetClass);
		setUsePropertyInCondition(true);
	}

	private final Criteria criteria;

	/**
	 * @see org.asyrinx.joey.om.ConditionTranslator#addFrom(java.lang.String)
	 */
	public void addFrom(String tableName) {
		//criteria.
	}

	/**
	 * @see org.asyrinx.joey.om.ConditionTranslator#addInt(java.lang.String,
	 *        java.lang.String, java.lang.Object)
	 */
	public void addInt(String fieldName, Object value, Object operator) {
		fieldName = toFieldName(fieldName);
		final Object val = toNumber(value, Integer.class);
		if (val == null)
			return;
		criteria.add(CriteriaUtils.toExpression(Operator.get(operator),
				fieldName, val));
	}

	/**
	 * @see org.asyrinx.joey.om.ConditionTranslator#addLong(java.lang.String,
	 *        java.lang.String, java.lang.Object)
	 */
	public void addLong(String fieldName, Object value, Object operator) {
		fieldName = toFieldName(fieldName);
		final Object val = toNumber(value, Long.class);
		if (val == null)
			return;
		criteria.add(CriteriaUtils.toExpression(Operator.get(operator),
				fieldName, val));
	}

	/**
	 * @see org.asyrinx.joey.om.ConditionTranslator#addDate(java.lang.String,
	 *        java.lang.String, java.lang.Object, java.text.DateFormat)
	 */
	public void addDate(String fieldName, Object value, Object operator,
			DateFormat dateFormat) {
		fieldName = toFieldName(fieldName);
		try {
			final Date d = (value instanceof Date) ? (Date) value : dateFormat
					.parse(String.valueOf(value));
			criteria.add(CriteriaUtils.toExpression(Operator.get(operator),
					fieldName, d));
		} catch (ParseException e) {
			throw new NestableRuntimeException(e);
		}
	}

	/**
	 * @see org.asyrinx.joey.om.ConditionTranslator#addString(java.lang.String,
	 *        java.lang.String, java.lang.Object)
	 */
	public void addString(String fieldName, Object value, Object operator) {
		fieldName = toFieldName(fieldName);
		criteria.add(CriteriaUtils.toExpression(Operator.get(operator),
				fieldName, value));
	}

	/**
	 * @see org.asyrinx.joey.om.ConditionTranslator#addAscendingOrderByColumn(java.lang.String)
	 */
	public void addAscendingOrderByColumn(String fieldName) {
		fieldName = toFieldName(fieldName);
		criteria.addOrder(Order.asc(fieldName));
	}

	/**
	 * @see org.asyrinx.joey.om.ConditionTranslator#addDescendingOrderByColumn(java.lang.String)
	 */
	public void addDescendingOrderByColumn(String fieldName) {
		fieldName = toFieldName(fieldName);
		criteria.addOrder(Order.desc(fieldName));
	}

	/**
	 * @return
	 */
	public Criteria toCriteria(SearchCondition condition) {
		condition.attach(this);
		return criteria;
	}

}

class CriteriaUtils {
	public static Criterion toExpression(Operator operator, Object targetValue,
			Object value) {
		if (operator == null) {
			throw new UnsupportedOperationException(
					"No operator to translate to Expression");
		} else if (operator == Operator.AND) {
			if (targetValue instanceof Criterion && value instanceof Criterion)
				return Expression.and((Criterion) targetValue,
						(Criterion) value);
			else
				throw new UnsupportedOperationException(
						"target and value must be Criteria.");
		} else if (operator == Operator.EQUAL || operator == Operator.EQUAL2) {
			return Expression.eq(toString(targetValue), value);
		} else if (operator == Operator.IN) {
			if (value instanceof Collection)
				return Expression.in(toString(targetValue), (Collection) value);
			else if (value.getClass().isArray()) {
				if (!value.getClass().getComponentType().isPrimitive())
					return Expression.in(toString(targetValue),
							(Object[]) value);
				throw new UnsupportedOperationException(
						"value must be Collection or array of Object.");
			} else
				throw new UnsupportedOperationException(
						"value must be Collection or array of Object.");
		} else if (operator == Operator.GREATER_EQUAL
				|| operator == Operator.GREATER_EQUAL2) {
			return Expression.ge(toString(targetValue), value);
		} else if (operator == Operator.GREATER_THAN) {
			return Expression.gt(toString(targetValue), value);
		} else if (operator == Operator.LIKE) {
			return Expression.like(toString(targetValue), value);
		} else if (operator == Operator.NOT_EQUAL
				|| operator == Operator.NOT_EQUAL2) {
			return Expression.not(toExpression(Operator.EQUAL,
					toString(targetValue), value));
		} else if (operator == Operator.NOT_IN) {
			return Expression.not(toExpression(Operator.IN,
					toString(targetValue), value));
		} else if (operator == Operator.OR) {
			if (targetValue instanceof Criterion && value instanceof Criterion)
				return Expression
						.or((Criterion) targetValue, (Criterion) value);
			else
				throw new UnsupportedOperationException(
						"target and value must be Criteria.");
		} else if (operator == Operator.LESS_EQUAL
				|| operator == Operator.LESS_EQUAL2) {
			return Expression.le(toString(targetValue), value);
		} else if (operator == Operator.LESS_THAN) {
			return Expression.lt(toString(targetValue), value);
		} else {
			throw new UnsupportedOperationException(
					"Unknown operator to translate to Expression. '"
							+ operator.toString() + "'");
		}
	}

	private static String toString(Object targetValue) {
		if (!(targetValue instanceof String))
			throw new UnsupportedOperationException("target must be String.");
		return (String) targetValue;
	}

}