/*
 * brownies and its relative products are published under the terms
 * of the Apache Software License.
 * 
 * Created on 2005/02/09 13:26:27
 */
package test.org.asyrinx.brownie.validator.prop;

import org.asyrinx.brownie.core.chars.CharactersExpressionParser;
import org.asyrinx.brownie.core.chars.Japanese;
import org.asyrinx.brownie.core.chars.dic.MsgResourceCharacterSetDictionary;
import org.asyrinx.brownie.core.chars.impl.CharactersExpressionParserImpl;
import org.asyrinx.brownie.core.resource.impl.ResourceBundleMessageResource;
import org.asyrinx.brownie.validator.Message;
import org.asyrinx.brownie.validator.ValidationContext;
import org.asyrinx.brownie.validator.impl.CompositeType;
import org.asyrinx.brownie.validator.impl.CompositeValidator;
import org.asyrinx.brownie.validator.prop.StringFieldCharsExp;
import org.asyrinx.brownie.validator.prop.StringFieldToAddChars;

import test.org.asyrinx.brownie.validator.JapaneseTestCase;
import test.org.asyrinx.brownie.validator.TestUtils;

/**
 * @author takeshi
 */
public class StringFieldTest extends JapaneseTestCase {

    public static void main(String[] args) {
        junit.swingui.TestRunner.run(StringFieldTest.class);
    }

    public static class Contacts {
        public Contacts() {
            super();
        }

        public Contacts(String name, String nameKana, String postalCode, String address,
                String telephone, String email) {
            this.name = name;
            this.nameKana = nameKana;
            this.postalCode = postalCode;
            this.address = address;
            this.telephone = telephone;
            this.email = email;
        }

        private String name = null;

        private String nameKana = null;

        private String postalCode = null;

        private String address = null;

        private String telephone = null;

        private String email = null;

        public String getAddress() {
            return address;
        }

        public void setAddress(String address) {
            this.address = address;
        }

        public String getEmail() {
            return email;
        }

        public void setEmail(String email) {
            this.email = email;
        }

        public String getName() {
            return name;
        }

        public void setName(String name) {
            this.name = name;
        }

        public String getPostalCode() {
            return postalCode;
        }

        public void setPostalCode(String postalCode) {
            this.postalCode = postalCode;
        }

        public String getTelephone() {
            return telephone;
        }

        public void setTelephone(String telephone) {
            this.telephone = telephone;
        }

        public String getNameKana() {
            return nameKana;
        }

        public void setNameKana(String nameKana) {
            this.nameKana = nameKana;
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see junit.framework.TestCase#runTest()
     */
    protected void runTest() throws Throwable {
        try {
            super.runTest();
        } catch (RuntimeException e) {
            e.printStackTrace();
            throw e;
        } catch (Error e) {
            e.printStackTrace();
            throw e;
        }
    }

    public void testEvaluate() {
        ValidationContext context;
        Message message;
        //
        final StringFieldToAddChars v = new StringFieldToAddChars("postalCode", "X֔ԍ", true, 8);
        v.addAcceptable(Japanese.HANKAKU_NUMERIC);
        v.addAcceptable("-");
        final Contacts object1 = new Contacts();
        //
        object1.setPostalCode(null);
        context = TestUtils.assertNG(v, object1, 1);
        message = (Message) context.getMessages().get(0);
        assertEquals("X֔ԍ ɂ͕K͂ĂB", message.getText());
        //
        object1.setPostalCode("");
        context = TestUtils.assertNG(v, object1, 1);
        message = (Message) context.getMessages().get(0);
        assertEquals("X֔ԍ ɂ͕K͂ĂB", message.getText());
        //
        object1.setPostalCode("1234-5678");
        context = TestUtils.assertNG(v, object1, 1);
        message = (Message) context.getMessages().get(0);
        assertEquals("X֔ԍ ͔p 8 (SpȂ炻1/2)܂œ͉\łB", message.getText());
        //
        object1.setPostalCode("123-defg");
        context = TestUtils.assertNG(v, object1, 1);
        message = (Message) context.getMessages().get(0);
        assertEquals("X֔ԍ Ɏgpł镶 p,- łB", message.getText());
    }

    public void testTotalValidate() {
        final StringFieldToAddChars v_name = new StringFieldToAddChars("name", "O", true, 30);
        v_name.addRefused(Japanese.HANKAKU_KATAKANA);
        final StringFieldToAddChars v_nameKana = new StringFieldToAddChars("nameKana", "Ji", true,
                50);
        v_nameKana.addAcceptable(Japanese.ZENKAKU_KATAKANA);
        v_nameKana.addAcceptable(" "); //Xy[XĂȂƂȂB
        v_nameKana.addRefused(Japanese.HANKAKU_KATAKANA);
        v_nameKana.addRefused("y"); //y̓JiɎgȂBdlĂƂŁEEE
        final StringFieldToAddChars v_postalCode = new StringFieldToAddChars("postalCode", "X֔ԍ",
                true, 8);
        v_postalCode.setRegExp("([0-9]{3})|([0-9]{3}-[0-9]{4})");
        v_postalCode.addAcceptable(Japanese.HANKAKU_NUMERIC);
        v_postalCode.addAcceptable("-");
        final StringFieldToAddChars v_address = new StringFieldToAddChars("address", "Z", true, 50);
        v_address.addRefused(Japanese.HANKAKU_KATAKANA);
        v_address.addRefused(Japanese.HANKAKU_NUMERIC);
        final StringFieldToAddChars v_telephone = new StringFieldToAddChars("telephone", "dbԍ",
                true, 17);
        v_telephone.addAcceptable(Japanese.HANKAKU_NUMERIC);
        v_telephone.addAcceptable("+-");
        final StringFieldToAddChars v_email = new StringFieldToAddChars("email", "E[", true, 200);
        v_email.addAcceptable(Japanese.HANKAKU_ALPHABET);
        v_email.addAcceptable(Japanese.HANKAKU_NUMERIC);
        v_email.addAcceptable("-_.@");
        //
        final CompositeValidator validator = new CompositeValidator(CompositeType.AND);
        validator.add(v_name);
        validator.add(v_nameKana);
        validator.add(v_postalCode);
        validator.add(v_address);
        validator.add(v_telephone);
        validator.add(v_email);
        //
        final Contacts contacts = new Contacts("߹Rҵ", "߹Rҵ", "0123-4567", "pŏZ0123",
                "telephone-number", "undeliverable/mail");
        TestUtils.assertNG(validator, contacts);
        contacts.setEmail("deliverable@mail");
        TestUtils.assertNG(validator, contacts);
        contacts.setTelephone("+61-03-7890-1234");
        TestUtils.assertNG(validator, contacts);
        contacts.setAddress("SpJiZOPQR");
        TestUtils.assertNG(validator, contacts);
        contacts.setPostalCode("123-0987");
        TestUtils.assertNG(validator, contacts);
        contacts.setNameKana("yP} _I"); //؂͎gȂĎdlEEEE
        TestUtils.assertNG(validator, contacts);
        contacts.setNameKana("oc} _I");
        TestUtils.assertNG(validator, contacts);
        contacts.setName("~R ʖڗY");
        TestUtils.assertOK(validator, contacts);
    }

    public void testTotalValidateByCharsExp() {
        final StringFieldCharsExp v_name = newStringFieldCharsExp("name", "O", true, 30, null,
                "HAN_KATAKANA", null);
        final StringFieldCharsExp v_nameKana = newStringFieldCharsExp("nameKana", "Ji", true, 50,
                "ZEN_KATAKANA,[[ ]]", "HAN_KATAKANA,[[y]]", null);
        final StringFieldCharsExp v_postalCode = newStringFieldCharsExp("postalCode", "X֔ԍ", true,
                8, "HAN_NUMERIC,[[-]]", null, "([0-9]{3})|([0-9]{3}-[0-9]{4})");
        final StringFieldCharsExp v_address = newStringFieldCharsExp("address", "Z", true, 50,
                null, "HAN_KATAKANA,HAN_NUMERIC", null);
        final StringFieldCharsExp v_telephone = newStringFieldCharsExp("telephone", "dbԍ", true,
                17, "HAN_NUMERIC,[[+-]]", null, null);
        final StringFieldCharsExp v_email = newStringFieldCharsExp("email", "E[", true, 200,
                "HAN_ALPHABET,HAN_NUMERIC,[[-_.@]]", null, null);
        //
        final CompositeValidator validator = new CompositeValidator(CompositeType.AND);
        validator.add(v_name);
        validator.add(v_nameKana);
        validator.add(v_postalCode);
        validator.add(v_address);
        validator.add(v_telephone);
        validator.add(v_email);
        //
        final Contacts contacts = new Contacts("߹Rҵ", "߹Rҵ", "0123-4567", "pŏZ0123",
                "telephone-number", "undeliverable/mail");
        TestUtils.assertNG(validator, contacts);
        contacts.setEmail("deliverable@mail");
        TestUtils.assertNG(validator, contacts);
        contacts.setTelephone("+61-03-7890-1234");
        TestUtils.assertNG(validator, contacts);
        contacts.setAddress("SpJiZOPQR");
        TestUtils.assertNG(validator, contacts);
        contacts.setPostalCode("123-0987");
        TestUtils.assertNG(validator, contacts);
        contacts.setNameKana("yP} _I"); //؂͎gȂĎdlEEEE
        TestUtils.assertNG(validator, contacts);
        contacts.setNameKana("oc} _I");
        TestUtils.assertNG(validator, contacts);
        contacts.setName("~R ʖڗY");
        TestUtils.assertOK(validator, contacts);
    }

    private CharactersExpressionParser expressionParser = null;

    /*
     * ۂ͂̕ӂDIReiB
     */
    private CharactersExpressionParser getExpressionParser() {
        if (expressionParser == null) {
            this.expressionParser = new CharactersExpressionParserImpl(
                    new MsgResourceCharacterSetDictionary(new ResourceBundleMessageResource(
                            "org.asyrinx.brownie.core.chars.characters")));
        }
        return expressionParser;
    }

    private StringFieldCharsExp newStringFieldCharsExp(String name, String displayName,
            boolean required, int size, String accepted, String refused, String regExp) {
        final StringFieldCharsExp result = new StringFieldCharsExp(name, displayName, required,
                size);
        result.setAccepted(accepted);
        result.setRefused(refused);
        result.setRegExp(regExp);
        result.setCharacterSetExpressionParser(getExpressionParser());
        return result;
    }

}