/*
 * joey-gen and its relative products are published under the terms
 * of the Apache Software License.
 * 
 * Created on 2004/12/20 13:58:36
 */
package org.asyrinx.joey.gen.ant.task;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.Writer;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import java.util.StringTokenizer;

import org.apache.commons.collections.ExtendedProperties;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.tools.ant.BuildException;
import org.apache.tools.ant.Project;
import org.apache.tools.ant.Task;
import org.apache.velocity.VelocityContext;
import org.apache.velocity.app.VelocityEngine;
import org.apache.velocity.context.Context;
import org.apache.velocity.exception.MethodInvocationException;
import org.apache.velocity.exception.ParseErrorException;
import org.apache.velocity.exception.ResourceNotFoundException;
import org.apache.velocity.texen.Generator;
import org.apache.velocity.util.StringUtils;
import org.asyrinx.joey.gen.core.JoeyGenerateTarget;

/**
 * @author takeshi
 */
public class MultiTargetTexenTask extends Task {

    protected final Log log = LogFactory.getLog(this.getClass());

    private static final String ERR_MSG_FRAGMENT = ". For more information consult the velocity log, or invoke ant "
            + "with the -debug flag.";

    protected String templatePath;

    protected String outputEncoding;

    protected String inputEncoding;

    protected ExtendedProperties contextProperties;

    protected boolean useClasspath;

    public void setTemplatePath(String templatePath) throws Exception {
        StringBuffer resolvedPath = new StringBuffer();
        StringTokenizer st = new StringTokenizer(templatePath, ",");
        while (st.hasMoreTokens()) {
            final File fullPath = project.resolveFile(st.nextToken());
            resolvedPath.append(fullPath.getCanonicalPath());
            if (st.hasMoreTokens()) {
                resolvedPath.append(",");
            }
        }
        this.templatePath = resolvedPath.toString();
        System.out.println(templatePath);
    }

    public String getTemplatePath() {
        return templatePath;
    }

    public void setOutputEncoding(String outputEncoding) {
        this.outputEncoding = outputEncoding;
    }

    public void setInputEncoding(String inputEncoding) {
        this.inputEncoding = inputEncoding;
    }

    public void setContextProperties(String file) {
        final String[] sources = StringUtils.split(file, ",");
        contextProperties = new ExtendedProperties();
        for (int i = 0; i < sources.length; i++) {
            ExtendedProperties source = new ExtendedProperties();
            try {
                final File fullPath = project.resolveFile(sources[i]);
                log("Using contextProperties file: " + fullPath);
                source.load(new FileInputStream(fullPath));
            } catch (Exception e) {
                final ClassLoader classLoader = this.getClass().getClassLoader();
                try {
                    final InputStream inputStream = classLoader.getResourceAsStream(sources[i]);

                    if (inputStream == null) {
                        throw new BuildException("Context properties file " + sources[i]
                                + " could not be found in the file system or on the classpath!");
                    } else {
                        source.load(inputStream);
                    }
                } catch (IOException ioe) {
                    source = null;
                }
            }
            for (Iterator j = source.getKeys(); j.hasNext();) {
                final String name = (String) j.next();
                final String value = source.getString(name);
                contextProperties.setProperty(name, value);
            }
        }
    }

    public ExtendedProperties getContextProperties() {
        return contextProperties;
    }

    public void setUseClasspath(boolean useClasspath) {
        this.useClasspath = useClasspath;
    }

    public Context initControlContext() {
        return new VelocityContext();
    }

    protected void populateInitialContext(Context context) {
        context.put("now", new Date().toString());
    }

    protected void cleanup() throws Exception {
        //
    }

    protected List initTargets() {
        return null;
    }

    protected void checkTargets(JoeyGenerateTarget target) {
        if (target.getControlTemplate() == null)
            throw new BuildException("The control template needs to be defined!");
        if (target.getOutputDirectory() == null)
            throw new BuildException("The output directory needs to be defined!");
        if (target.getOutputFile() == null)
            throw new BuildException("The output file needs to be defined!");
        log.debug("target=" + target.getTargetName());
        log.debug("outputDirectory=" + target.getOutputDirectory());
        log.debug("controlTemplate=" + target.getControlTemplate());
    }

    public void execute() throws BuildException {
        final List targets = initTargets();
        if (templatePath == null && useClasspath == false) {
            throw new BuildException("The template path needs to be defined if you are not using "
                    + "the classpath for locating templates!");
        }
        //
        final Context c = initControlContext();
        populateInitialContext(c);
        try {
            prepareContextProperties(c);
        } catch (IOException e) {
            throw new BuildException("failed to prepareContextProperties", e);
        }
        //
        final VelocityEngine ve = initVelocityEngine();
        for (Iterator j = targets.iterator(); j.hasNext();) {
            final JoeyGenerateTarget target = (JoeyGenerateTarget) j.next();
            checkTargets(target);
            try {
                final Generator generator = initGenerator();
                generator.setVelocityEngine(ve);
                generator.setOutputPath(target.getOutputDirectory());
                generator.setInputEncoding(inputEncoding);
                generator.setOutputEncoding(outputEncoding);

                generate(c, target, generator);
            } catch (BuildException e) {
                throw e;
            } catch (MethodInvocationException e) {
                throw new BuildException("Exception thrown by '" + e.getReferenceName() + "." + e.getMethodName() + "'"
                        + ERR_MSG_FRAGMENT, e.getWrappedThrowable());
            } catch (ParseErrorException e) {
                throw new BuildException("Velocity syntax error" + ERR_MSG_FRAGMENT, e);
            } catch (ResourceNotFoundException e) {
                throw new BuildException("Resource not found" + ERR_MSG_FRAGMENT, e);
            } catch (Exception e) {
                throw new BuildException("Generation failed" + ERR_MSG_FRAGMENT, e);
            }
        }
    }

    /**
     * @param c
     * @param target
     * @param generator
     * @throws Exception
     * @throws IOException
     */
    protected void generate(final Context c, final JoeyGenerateTarget target, final Generator generator)
            throws Exception, IOException {
        if (templatePath != null)
            generator.setTemplatePath(templatePath);
        final File file = new File(target.getOutputDirectory());
        if (!file.exists())
            file.mkdirs();
        final String path = target.getOutputDirectory() + File.separator + target.getOutputFile();
        log.debug("Generating to file " + path);
        final Writer writer = generator.getWriter(path, outputEncoding);
        writer.write(generator.parse(target.getControlTemplate(), c));
        writer.flush();
        writer.close();
        generator.shutdown();
        cleanup();
    }

    /**
     * @return
     */
    protected Generator initGenerator() {
        return Generator.getInstance();
    }

    /**
     * @return
     */
    protected VelocityEngine initVelocityEngine() {
        VelocityEngine ve = new VelocityEngine();
        // Setup the Velocity Runtime.
        if (templatePath != null) {
            log("Using templatePath: " + templatePath, Project.MSG_VERBOSE);
            ve.setProperty(VelocityEngine.FILE_RESOURCE_LOADER_PATH, templatePath);
        }
        if (useClasspath) {
            log("Using classpath");
            ve.addProperty(VelocityEngine.RESOURCE_LOADER, "classpath");
            ve.setProperty("classpath." + VelocityEngine.RESOURCE_LOADER + ".class",
                    "org.apache.velocity.runtime.resource.loader.ClasspathResourceLoader");
            ve.setProperty("classpath." + VelocityEngine.RESOURCE_LOADER + ".cache", "false");
            ve.setProperty("classpath." + VelocityEngine.RESOURCE_LOADER + ".modificationCheckInterval", "2");
        }
        try {
            ve.init();
        } catch (Exception e) {
            throw new BuildException("VelocityEngine#init() failed " + ERR_MSG_FRAGMENT, e);
        }
        return ve;
    }

    /**
     * @param c
     * @throws IOException
     */
    protected void prepareContextProperties(final Context c) throws IOException {
        if (contextProperties != null) {
            final Iterator i = contextProperties.getKeys();

            while (i.hasNext()) {
                String property = (String) i.next();
                String value = contextProperties.getString(property);
                try {
                    c.put(property, new Integer(value));
                } catch (NumberFormatException nfe) {
                    final String booleanString = contextProperties.testBoolean(value);
                    if (booleanString != null) {
                        c.put(property, new Boolean(booleanString));
                    } else {
                        if (property.endsWith("file.contents")) {
                            value = StringUtils.fileContentsToString(project.resolveFile(value).getCanonicalPath());
                            property = property.substring(0, property.indexOf("file.contents") - 1);
                        }
                        c.put(property, value);
                    }
                }
            }
        }
    }

}