package jp.co.nissy.jpicosheet.core;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.regex.Pattern;


/**
 * 複数のシートを持つことのできる"ブック"です。<br>
 * シートの保持だけでなく、保持しているシートの自動再計算の制御やセル間の参照関係を維持する機能も保持しています。
 * @author yusuke nishikawa
 *
 */
public class Book {


	/**
	 * ブック名
	 */
	private String _name;
	/**
	 * 計算機オブジェクト
	 */
	private Calculator _calculator;
	/**
	 * セル参照解決オブジェクト
	 */
	private Resolver _resolver;
	/**
	 * このブックが持つシートのMap
	 */
	private HashMap<String, Sheet> _sheets = new HashMap<String, Sheet>();


	/**
	 * ブック名のパターン文字列
	 */
	static final String BOOK_NAME_PATTERN = "[a-zA-Z_][a-zA-Z0-9_]*";
	/**
	 * ブック名の正規表現パターン
	 */
	private static Pattern _bookNamePattern = Pattern.compile(BOOK_NAME_PATTERN);


	@SuppressWarnings("unused")
	private Book() {}

	/**
	 * ブック名を指定してブックを作成します
	 * @param bookName ブック名
	 * @throws IllegalArgumentException ブック名が正しくない場合
	 */
	public Book(String bookName) throws IllegalArgumentException {

		// ブック名のチェック
		validateBookName(bookName);

		this._name = bookName;
		this._calculator = new Calculator(this);
		this._resolver = new Resolver(this);
	}


	/**
	 * ブック名を返します
	 * @return ブック名
	 */
	public String getName(){
		return this._name;
	}

	/**
	 * ブック名をセットします
	 * @param bookName ブック名
	 * @throws IllegalArgumentException ブック名が正しくない場合
	 */
	public void setName(String bookName) throws IllegalArgumentException {
		// ブック名のチェック
		validateBookName(bookName);
		this._name = bookName;
	}

	/**
	 * 計算オブジェクトを返します。
	 * @return 計算オブジェクト
	 */
	Calculator getCalculator() {
		return this._calculator;
	}

	/**
	 * リゾルバを返します。
	 * @return リゾルバ
	 */
	public Resolver getResolver() {
		return this._resolver;
	}


	/**
	 * シートを追加します。<br>
	 * 指定したシート名がすでに存在する場合、既存のシートオブジェクトを返します。
	 * @param sheetName シート名
	 * @return 追加したシートオブジェクト
	 */
	public Sheet addSheet(String sheetName)   {
		// シートを追加する
		if (this._sheets.containsKey(sheetName)) {
			return this._sheets.get(sheetName);
		} else {
			Sheet sheet = new Sheet(sheetName, this);
			this._sheets.put(sheetName, sheet);
			// リゾルバのデフォルトシートに登録が無い場合、このシートをデフォルトシートにする
			if (this._resolver.getDefaultSheet() == null) {
				this._resolver.setDefaultSheet(sheet);
			}
			return sheet;
		}
	}


	/**
	 * このブックが保持するすべてのシートへの参照を返します
	 * @return このブックが保持するシートへの参照
	 */
	public List<Sheet> getSheets() {
		return new ArrayList<Sheet>(this._sheets.values());
	}

	/**
	 * 既存のシートのシート名を変更します。<br>
	 * 変更後のシート名を持つシートがすでに存在していた場合、そのシートオブジェクトは削除されます。
	 * @param sheetName 変更対象のシート名
	 * @param newSheetName 変更後のシート名
	 * @throws PicosheetException 指定したシートが存在しない場合
	 */
	public void renameSheet(String sheetName, String newSheetName) throws PicosheetException  {
		// シート名を変更する
		if (this._sheets.containsKey(sheetName)) {
			// Sheetオブジェクト内部の名前を変更
			Sheet sheet = this._sheets.get(sheetName);
			sheet.setName(newSheetName);
			// sheets上の管理をnewSheetNameに変える
			this._sheets.put(newSheetName, this._sheets.get(sheetName));
			this._sheets.remove(sheetName);
		}
	}

	/**
	 * 引数に指定したシートを削除します。
	 * @param sheetName 削除対象のシート名
	 * @throws ReferenceNotFoundException  指定したシートが存在しない場合
	 */
	public void deleteSheet(String sheetName) throws ReferenceNotFoundException {
		// シートを削除する
		if (this._sheets.containsKey((sheetName))) {
			this._sheets.remove(sheetName);
		} else {
			throw new ReferenceNotFoundException("no such sheet " + sheetName);
		}
	}

	/**
	 * 引数に指定したシートオブジェクトを返します。
	 * @param sheetName シート名
	 * @return シートオブジェクト
	 * @throws ReferenceNotFoundException 指定したシートが存在しない場合
	 */
	public Sheet sheet(String sheetName) throws ReferenceNotFoundException {
		// シートを返す
		if (this._sheets.containsKey(sheetName)) {
			return this._sheets.get(sheetName);
		} else {
			throw new ReferenceNotFoundException("no such sheet " + sheetName);
		}
	}



	/**
	 * ブックに対する自動再計算を有効にします。<br>
	 * 同時に、すべてのセルを再計算します。
	 * @throws Exception
	 */
	public void recalcEnable() throws Exception {
		this._calculator.recalcEnable();
	}

	/**
	 * ブックに対する自動再計算を無効にします。<br>
	 */
	public void recalcDisable() {
		this._calculator.recalcDisable();
	}

	/**
	 * 渡された文字列がブック名として正しいかチェックします。<br>
	 * 正しくない場合、例外がスローされます。
	 * @param bookName チェックするシート名
	 * @throws IllegalArgumentException ブック名として正しくない場合
	 */
	private void validateBookName(String bookName) throws IllegalArgumentException {
		if (! _bookNamePattern.matcher(bookName).matches()) {
			throw new IllegalArgumentException("invalid book name \"" + bookName + "\"");
		}
	}

}

