package jp.co.nissy.jpicosheet.core;

import java.math.BigDecimal;
import java.math.MathContext;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Set;
import java.util.Stack;
import java.util.logging.Logger;

import jp.co.nissy.jpicosheet.core.Cell.CellStatus;
import jp.co.nissy.jpicosheet.core.Cell.CellType;
import jp.co.nissy.jpicosheet.core.Element.ElementType;
import jp.co.nissy.jpicosheet.core.Element.ErrorType;
import jp.co.nissy.jpicosheet.core.Element.Operator;
import jp.co.nissy.jpicosheet.function.Function;
import jp.co.nissy.jpicosheet.function.Sum;


class Calculator {

	private Book book;
	private boolean recalcEnabled;
	/**
	 * ログオブジェクト
	 */
	private final Logger logger = Logger.getLogger("jp.co.nissy.jpicosheet");

	/**
	 * このオブジェクトを所有するBookオブジェクトを指定して、このオブジェクトを初期化します。
	 * @param book このオブジェクトを所有するBookオブジェクト
	 */
	public Calculator (Book book) {
		this.book = book;
		this.recalcEnabled = true;
	}


	/**
	 * 指定したCellオブジェクトを計算します。<br>
	 * このオブジェクトを参照しているセルがある場合、それらのセルも再計算されます。<br>
	 * <br>
	 * このメソッドを呼び出すセルは、事前に書き込みロックを取得していなければなりません。
	 * @param cellname 計算キューに追加するセル名 ブック名からセル名までを含んだFull Qualified Cell Nameで表します
	 * @throws ReferenceNotFoundException
	 */
	void calc(String cellname) throws ReferenceNotFoundException {

		calc(this.book.getResolver().getCell(cellname));

	}

	/**
	 * 指定したCellオブジェクトを計算します。<br>
	 * このオブジェクトを参照しているセルがある場合、それらのセルも再計算されます。<br>
	 * <br>
	 * このメソッドを呼び出すセルは、事前に書き込みロックを取得していなければなりません。
	 * @param cell 計算キューに追加するCellオブジェクト
	 */
	void calc(Cell cell) {

		// 再計算が許可されていれば即時に再計算する
		if (this.recalcEnabled) {
			doCalc(cell);
			recalcReferencingCells(cell);
		}

	}


	/**
	 * 渡されたSetに含まれるすべてのCellオブジェクトに対して計算を行います。<br>
	 * このメソッドを呼び出すセルは、事前に書き込みロックを取得していなければなりません。
	 * @param cells 計算キューに追加するCellオブジェクトを含むSet
	 */
	void calc(Set<Cell> cells)   {
		for (Cell cell : cells) {
			calc(cell);
		}
	}

	/**
	 * 再計算を許可します。
	 */
	void recalcEnable() {

		this.recalcEnabled = true;

		// 全シートのすべてのセルを計算対象にする
		List<Cell> recalcCells = new ArrayList<Cell>();
		for (Sheet sheet: this.book.getSheets()) {
			for (Cell cell: sheet.getCells()) {
				if (cell.getCellType().equals(Cell.CellType.FORMULA)) {
					cell.setStatus(Cell.CellStatus.REQUIRE_CALCULATION);
					recalcCells.add(cell);
				}
			}
		}
		// 計算対象のセルを計算する
		for (Cell cell: recalcCells) {
			doCalc(cell);
		}
	}

	/**
	 * 再計算を禁止します。
	 */
	void recalcDisable() {
		// 再計算を禁止する
		this.recalcEnabled = false;
	}

	/**
	 * 引数に渡されたセルを参照しているセルを再計算します<br>
	 * Cellオブジェクトのコレクションを引数に取る同名のメソッドと異なり、
	 * 引数に渡されたセルを参照しているセルを再帰的に調査し、すべて再計算します。
	 * @param cell 再計算対象のセル
	 */
	void recalcReferencingCells(Cell cell) {

		// このセルを参照しているセルがある場合、それらも再計算が必要
		// このセルを参照しているすべてのセル(=再計算が必要なセル)を取得して再計算する
		Resolver resolver = this.book.getResolver();
		Set<Cell> recalcCells = resolver.getReferencingCells(cell);
		recalcReferencingCells(recalcCells);
	}

	/**
	 * コレクションに含まれるセルを再計算します。<br>
	 * @param recalcCells 再計算対象のセル
	 */
	void recalcReferencingCells(Collection<Cell> recalcCells) {
		// 再計算対象となったセルを「計算必要」ステータスにする
		for (Cell refCell : recalcCells) {
			refCell.setStatus(CellStatus.REQUIRE_CALCULATION);
		}
		// 再計算対象となったセルを計算する
		for (Cell refCell : recalcCells) {
			doCalc(refCell);
		}

	}
	/**
	 * 渡されたセルを計算します。<br>
	 * @param cell セルオブジェクト
	 */
	private void doCalc(Cell cell) {

		// セルのタイプが式でなければ何もしない
		if (cell.getCellType() != CellType.FORMULA) {
			return;
		}
		// セルのタイプが式であっても、ステータスが「計算必要」でなければ何もしない
		if (cell.getStatus() != CellStatus.REQUIRE_CALCULATION) {
			return;
		}

		//セルの状態を計算中にする
		cell.setStatus(CellStatus.UNDER_CALCULATION);

		// MathContextをSheetオブジェクトから取得
		MathContext mc = cell.getSheet().getMathContext();

		Element token;
		Stack<Element> calcStack = new Stack<Element>();

		// 逆ポーランド記法式のトークンの数ぶんループ
		for (int i = 0; i < cell.getFormulaRPN().length; i++) {

			// 処理対象のトークンを取り出す
			token = cell.getFormulaRPN()[i];

			// トークンの種類に応じて処理を変える
			switch (token.getType()) {
			case NUMBER:
				// 数値の場合、計算スタックにプッシュ
				calcStack.push(token);
				break;

			case STRING:
				// 文字列の場合、計算スタックにプッシュ
				calcStack.push(token);
				break;

			case REFERENCE:
				// セル参照の場合、参照先セルの値を取得して計算スタックにプッシュ
				Element refCelValue = getReferencesCellValue(token);
				// 得た値のタイプがエラー以外の場合
				if (refCelValue.getType() != ElementType.ERROR) {
					// 計算スタックにプッシュ
					calcStack.push(refCelValue);
				} else {
					// エラーの場合、得た値をセル値としてセットし、このセルのステータスをエラーにして計算を中断する
					cell.setValue(refCelValue);
					cell.setStatus(CellStatus.ERROR);
					return;
				}
				break;

			case GROUP_REFERENCE:
				try {
					// トークンが示すグループが存在する場合、グループが保持するすべてのセルに対し、必要なら再計算を行う
					// TODO:セル参照のたびにCollectionを得てすべてのセルの再計算チェックするのは非効率
					Collection<Cell> cells;
					cells = book.getResolver().getCellsFromGroup(token.getGroupReference());
					recalcIfRequired(cells);
					calcStack.push(token);
				} catch (ReferenceNotFoundException e) {
					// トークンが示すグループが存在しなかった場合、このセルのステータスをエラーにして計算を中断する
					cell.setValue(new Element(ElementType.ERROR, ErrorType.INVALID_REFERENCES));
					cell.setStatus(CellStatus.ERROR);
					return;
				}
				break;

			case TABLE_REFERENCE:
				try {
					// トークンが示すテーブルが存在する場合、テーブル(またはテーブルの範囲)が保持するすべてのセルに対し、必要なら再計算を行う
					// TODO:セル参照のたびにCollectionを得てすべてのセルの再計算チェックするのは非効率
					Collection<Cell> cells;
					cells = book.getResolver().getCellsFromTable(token.getTableReference());
					recalcIfRequired(cells);
					calcStack.push(token);
				} catch (ReferenceNotFoundException e) {
					// トークンが示すテーブルが存在しなかった場合、このセルのステータスをエラーにして計算を中断する
					cell.setValue(new Element(ElementType.ERROR, ErrorType.INVALID_REFERENCES));
					cell.setStatus(CellStatus.ERROR);
					return;
				}
				break;

			case OPERATOR:
				// 演算子の場合、演算を行う
				operate(calcStack, token, mc, cell.getSheet().getBook().getResolver());
				break;
			}
		}

		// この時点で計算スタック中のElement数が１でなければならない
		assert calcStack.size() == 1:"計算終了時点で計算スタック中のElement数が1でなければならない";

		// 最後に残ったものが計算結果
		cell.setValue(calcStack.pop());

		// セルの状態を計算済みにする
		cell.setStatus(CellStatus.CALCULATED);
//		logger.info(cell.getFullyQualifiedName() + ":" + cell.getFormula() + " = " + cell.getString());
//		StringBuilder sb = new StringBuilder();
//		for (Element e: cell.getFormulaRPN()) {
//			sb.append(e.toString()).append(" ");
//		}
//		logger.info(new STRING(sb));

	}

	/**
	 * 渡されたReference型トークンが示す参照先セルから値を取得します
	 * @param referenceElement 参照するセルを示したElement
	 * @return 参照先セルの値
	 */
	private Element getReferencesCellValue(Element referenceElement) {

		// 参照するセルオブジェクトを得る
		Cell referCell;
		referCell = getReferencesCell(referenceElement);
		if (referCell == null) {
			return new Element(ElementType.ERROR, ErrorType.INVALID_REFERENCES);
		}

		// このセルが計算必要ステータスである場合、再計算する
		recalcIfRequired(referCell);

		// 参照するセルの値を得る
		return referCell.getValue();
	}

	/**
	 * 渡されたReference型トークンが示す参照先セルから値を取得します。<br>
	 * セルが見つからなかった場合nullを返します
	 * @param referenceElement 参照するセルを示したElement
	 * @return 参照先セル。参照先セルが見つからなかった場合null。
	 */
	private Cell getReferencesCell(Element referenceElement) {
		// 参照するセルオブジェクトを得る
		Cell referCell;
		try {
			referCell = this.book.getResolver().getCell(referenceElement.getCellReference());
		} catch (ReferenceNotFoundException e) {
			// 参照先が見つからない場合、nullを返す
			return null;
		} catch (IllegalStateException e) {
			// referenceElementがセル参照で無いことは無い
			throw new AssertionError(e.getMessage());
		}
		return referCell;
	}


	/**
	 * 渡されたセルのコレクションのステータスをチェックし、必要な場合再計算を行います
	 * @param cells チェックするセルのコレクション
	 */
	private void recalcIfRequired(Collection<Cell> cells) {

		// 渡されたセルオブジェクトの中に計算が必要なセルがある場合、計算を行う
		for (Cell cell: cells) {
			recalcIfRequired(cell);
		}
	}

	private void recalcIfRequired(Cell cell) {
		// 参照するセルのステータスが計算中の場合、循環参照している
		if (cell.getStatus() == CellStatus.UNDER_CALCULATION) {
			// 循環参照エラーの値をセットする
			cell.setValue(new Element(ElementType.ERROR, ErrorType.CIRCULER_REFERENCE));
			cell.setStatus(CellStatus.ERROR);
		}

		// 参照するセルのステータスが計算必要の場合、計算を行う
		if (cell.getStatus() == CellStatus.REQUIRE_CALCULATION) {
			doCalc(cell);
		}
	}

	/**
	 * 計算を行います
	 * @param calcStack 計算用スタックオブジェクト
	 * @param operatorElem 演算子エレメント
	 * @param mc MathContextオブジェクト
	 * @param resolver リゾルバオブジェクト
	 */
	private boolean operate(Stack<Element> calcStack, Element operatorElem, MathContext mc, Resolver resolver) {
		BigDecimal resultValue;
		BigDecimal rValue;
		BigDecimal lValue;
		switch (operatorElem.getOperator()) {
		case PLUS:
			// スタックから取り出す際は右辺→左辺の順に取り出される。他のオペレータの場合も同様。
			rValue = calcStack.pop().getNumber();
			lValue = calcStack.pop().getNumber();
			resultValue = lValue.add(rValue, mc);
			calcStack.push(new Element(ElementType.NUMBER, resultValue));
			break;
		case MINUS:
			rValue = calcStack.pop().getNumber();
			lValue = calcStack.pop().getNumber();
			resultValue = lValue.subtract(rValue, mc);
			calcStack.push(new Element(ElementType.NUMBER, resultValue));
			break;
		case TIMES:
			rValue = calcStack.pop().getNumber();
			lValue = calcStack.pop().getNumber();
			resultValue = lValue.multiply(rValue, mc);
			calcStack.push(new Element(ElementType.NUMBER, resultValue));
			break;
		case DIVIDE:
			rValue = calcStack.pop().getNumber();
			lValue = calcStack.pop().getNumber();
			if (rValue.signum() == 0) {
				//割る数が0の場合、結果を0除算エラーとする
				calcStack.push(new Element(ElementType.ERROR, ErrorType.DIVIDE_BY_ZERO));
			} else {
				resultValue = lValue.divide(rValue, mc);
				calcStack.push(new Element(ElementType.NUMBER, resultValue));
			}
			break;
		case UNARY_PLUS:
			// 単項演算子＋は何もしない
			break;
		case UNARY_MINUS:
			rValue = calcStack.pop().getNumber();
			resultValue = rValue.negate();
			calcStack.push(new Element(ElementType.NUMBER, resultValue));
			break;
		case FUNCTION:
			// 関数に応じて必要な引数の数が異なる。
			// スタックには0個以上のカンマが積まれている。カンマをすべてpopし、popしたカンマの数だけ
			// 引数をさらにpopして引数とする…カンマの数は、中置記法表記の関数に使われたカンマの数+1個がRPN式に入っている。
			int argCount = 0;
			while(true) {
				// 計算スタックからポップ可能で、タイプがオペレータの場合
				if (calcStack.size() > 0) {
					// 計算スタックの一番上にあるオペレータの種類をチェックする
					Element topElem = calcStack.peek();
					if (topElem.getType() == ElementType.OPERATOR && topElem.getOperator() == Operator.COMMA) {
						// カンマだった場合、argCount数に+1し、計算スタックからポップし捨てる
						argCount++;
						calcStack.pop();
					} else {
						// カンマ以外の場合、argCount数が確定
						break;
					}
				} else {
					// スタックが空の場合、argCount数が確定
					break;
				}
			}

			// argCountの数だけスタックから値を取得する
			Element[] args = new Element[argCount];
			for (int i = argCount-1; 0 <= i; i--) {
				args[i] = calcStack.pop();
			}
			// TODO:まともな関数呼び出しを実装する
			Function func = new Sum();
			// 関数を実行し、結果をスタックにプッシュ
			calcStack.push(func.call(args, mc, resolver));
			break;
		case COMMA:
			// そのままスタックに積む
			calcStack.push(operatorElem);
			break;
		default:
			// すべてのオペレータはなんらかの処理がされなければならない。
			assert true;
		}

		return true;
	}


}

