package jp.co.nissy.jpicosheet.core;

import java.math.BigDecimal;
import java.util.logging.Logger;
import java.util.regex.Pattern;

import jp.co.nissy.jpicosheet.core.Element.ElementType;

/**
 * 値の保持、他のセルの参照や計算を行う"セル"です。<br>
 *
 * セルにはいくつかの種類および状態があります。
 * セルの種類には空、数値、文字列、式があります。種類の情報は<code>getCellType()</code>で得ることができます。<br>
 * セルの状態には計算必要、計算中、計算済み、エラーがあります。状態の情報は<code>getCellStatus()</code>で得ることができます。<br>
 *
 * セルの値は保持しているデータの種類や式の計算結果によって変化しますが、<code>getValue()</code>で得ることができるのはElement型オブジェクトです。
 * Element型オブジェクトはセルの値を表す汎用のデータ型です。<br>
 *
 * セルは必ずいずれかのシートに属します。アプリケーションがセルオブジェクトを単体で作成することはできず、Sheetクラスの
 * <code>addCell()</code>メソッドによって作成します。
 *
 * @author yusuke nishikawa
 *
 */
public class Cell implements Comparable<Cell> {

	static {
		// 空のセル値のため、空のエレメントを作成
		emptyElement = new Element(ElementType.EMPTY);
	}

	/**
	 * 空を表すエレメント
	 */
	private static Element emptyElement;

	private static final String EMPTY_LABEL = "";
	/**
	 * ログオブジェクト
	 */
	private final Logger logger = Logger.getLogger("jp.co.nissy.jpicosheet");

	// セルの状態を表す
	public enum CellStatus {
		/**
		 * 再計算が必要であることを示すセルステータスです
		 */
		REQUIRE_CALCULATION,
		/**
		 * 計算中であることを示すセルステータスです。あるセルの計算処理中に参照したセルがこのステータスであった場合、一連の参照は循環参照しています。
		 */
		UNDER_CALCULATION,
		/**
		 * 計算済みであることを示すセルステータスです。
		 */
		CALCULATED,
		/**
		 * エラーであることを示すセルステータスです。このセルを参照するセルはすべてエラーとなります。
		 */
		ERROR
	}

	// セルのタイプを表す
	public enum CellType {
		/**
		 * セルが空であることを示します。セルの作成直後はこのタイプになっています。
		 */
		EMPTY,
		/**
		 * セルが計算式を持っていることを示します。
		 */
		FORMULA,
		/**
		 * セルが数値を持っていることを示します。
		 */
		NUMBER,
		/**
		 * セルが文字列を持っていることを示します。
		 */
		STRING
	}

	/**
	 * 式(中置記法)
	 */
	private String _formula;
	/**
	 * 式(逆ポーランド式、Element配列)
	 */
	private Element[] _formulaRPN;
	/**
	 * セル名
	 */
	private String _name;

	/**
	 * セルのラベル
	 */
	private String _label;

	/**
	 * このセルを持っているSheetオブジェクトへの参照
	 */
	private Sheet _sheet;

	/**
	 * このセルのタイプ
	 */
	private CellType _cellType = CellType.EMPTY;
	/**
	 * このセルの状態
	 */
	private CellStatus _status = CellStatus.CALCULATED;

	/**
	 * このセルの値
	 */
	private Element _value;

	/**
	 * セル名のパターン文字列
	 */

	static final String CELL_NAME_PATTERN = "[a-zA-Z_][a-zA-Z0-9_]*";
	/**
	 * セル名の正規表現パターン
	 */
	private static Pattern _cellNamePattern = Pattern.compile(CELL_NAME_PATTERN);

	/**
	 * 完全修飾セル名のパターン文字列
	 */
	static final String FULLY_QUALIFIED_CELL_NAME_PATTERN = Sheet.SHEET_NAME_PATTERN + "!" + "[a-zA-Z_][a-zA-Z0-9_]*";

	/**
	 * 完全修飾セル名の正規表現パターン
	 */
	private static Pattern _fullyQualifiedCellNamePattern = Pattern.compile(FULLY_QUALIFIED_CELL_NAME_PATTERN);

	/**
	 * セル名、シートを指定してオブジェクトを作成します
	 *
	 * @param cellName セル名
	 * @param sheet このセルが属するシートオブジェクト
	 * @throws IllegalArgumentException セル名が正しくない場合
	 */
	Cell(String cellName, Sheet sheet) throws IllegalArgumentException {
		this(cellName, sheet, false);
	}

	/**
	 * セル名、シートを指定してオブジェクトを作成します
	 * @param cellName セル名
	 * @param sheet このセルが属するシートオブジェクト
	 * @param createForTableCell テーブル用のセルを作成する場合true、そうでない場合falseを指定
	 * @throws IllegalArgumentException
	 *             セル名が正しくない場合
	 */
	Cell(String cellName, Sheet sheet, boolean createForTableCell) throws IllegalArgumentException {

		// createForTableCellによってTable用の名前チェックを行うか否か決定する
		if (createForTableCell) {
			if (!Table.isValidTableNameWithAddress(cellName)) {
				throw new IllegalArgumentException("invalid cell name for Table \"" + cellName + "\"");
			}
		} else {
			if (!isValidCellName(cellName)) {
				throw new IllegalArgumentException("invalid cell name \"" + cellName + "\"");
			}
		}

		// メンバに値をセット
		this._name = cellName;
		this._label = EMPTY_LABEL;
		this._formula = "";
		this._value = emptyElement;
		this._sheet = sheet;
	}

	/**
	 * 式を返します。返される文字列にはこの文字列が式であることを明示する"="が先頭に付けられます。
	 *
	 * @return 式
	 */
	public String getFormula() {
		//
		return "=" + _formula;
	}

	/**
	 * 式を逆ポーランド記法で表現したElement配列を返します
	 *
	 * @return 式の逆ポーランド表現
	 */
	Element[] getFormulaRPN() {
		return _formulaRPN;
	}

	/**
	 * このセルの名前を返します
	 *
	 * @return セルの名前
	 */
	public String getName() {
		return _name;
	}

	/**
	 * このセルのラベルを返します
	 * @return セルのラベル
	 */
	public String getLabel() {
		return _label;
	}
	/**
	 * このセルの完全修飾名を返します
	 * @return セルの完全修飾名
	 */
	public String getFullyQualifiedName() {
		return _sheet.getName() + "!" + _name;
	}

	/**
	 * このセルを持っているシートへの参照を返します
	 *
	 * @return このセルを持っているシートへの参照
	 */
	public Sheet getSheet() {
		return this._sheet;
	}

	/**
	 * このセルのステータスを返します
	 *
	 * @return このセルのステータス
	 */
	public CellStatus getStatus() {
		return _status;
	}

	/**
	 * このセルの値を文字列として返します
	 *
	 * @return セルの値の文字列表現
	 */
	public String getString() {
		switch (_value.getType()) {
		case NUMBER:
			return _value.getNumber().toString();
		case STRING:
			return _value.getString();
		case DATE:
			return "implementation has not yet.";
		}
		return _value.toString();
	}

	/**
	 * このセルの値のタイプを返します
	 *
	 * @return このセルの、現在の値のタイプ
	 */
	public ElementType getValueType() {
		return this._value.getType();
	}

	/**
	 * このセルのタイプを返します
	 *
	 * @return このセルの、現在のタイプ
	 */
	public CellType getCellType() {
		return this._cellType;
	}

	/**
	 * セルの値を返します
	 *
	 * @return セルの値
	 */
	public Element getValue() {
		return _value;
	}

	public String getValueString() {
		switch(_value.getType()) {
		case EMPTY:
			return "";
		case DATE:
			return "DATE string will showing...";
		case ERROR:
			return _value.toString();
		case NUMBER:
			return _value.getNumber().toString();
		case STRING:
			return _value.getString();
			default:
				assert false: "unexpected value type: " + _value.toString();
			return "";
		}
	}

	/**
	 * 式を逆ポーランド記法で表現したElement配列をセットします
	 *
	 * @param formulaRPN
	 *            式の逆ポーランド表現
	 */
	void setFormulaRPN(Element[] formulaRPN) {
		this._formulaRPN = formulaRPN;
	}


	/**
	 * セル名をセットします
	 *
	 * @param cellName 新しいセル名
	 * @throws IllegalArgumentException セル名として正しくない名前が渡された場合
	 */
	void setName(String cellName) {
		setName(cellName, false);
	}

	/**
	 * セル名をセットします
	 *
	 * @param cellName 新しいセル名
	 * @param createForTableCell テーブル用のセルを作成する場合true、そうでない場合falseを指定
	 * @throws IllegalArgumentException セル名として正しくない名前が渡された場合
	 */
	void setName(String cellName, boolean createForTableCell) throws IllegalArgumentException {
		// 正しいセル名かチェック
		// createForTableCellによってTable用の名前チェックを行うか否か決定する
		if (createForTableCell) {
			if (!Table.isValidTableNameWithAddress(cellName)) {
				throw new IllegalArgumentException("invalid cell name for Table \"" + cellName + "\"");
			}
		} else {
			if (!isValidCellName(cellName)) {
				throw new IllegalArgumentException("invalid cell name \"" + cellName + "\"");
			}
		}
		_name = cellName;
	}

	/**
	 * ラベルをセットします
	 * @param newLabel 新しいラベル
	 * @return このセルオブジェクトへの参照
	 */
	public Cell setLabel(String newLabel) {
		_label = newLabel;
		return this;
	}

	/**
	 * セルのステータスをセットします
	 *
	 * @param status
	 *            セルのステータス
	 */
	void setStatus(CellStatus status) {
		this._status = status;
	}

	/**
	 * セルの値をセットします
	 *
	 * @param value
	 *            セルの値
	 */
	void setValue(Element value) {
		this._value = value;
	}

	public void setValue(String value) {

		if (value == null || value.equals("")) {
			setValue(Cell.emptyElement);
		} else if (isNumString(value)) {
			setNumberValue(value);
		} else if (isFormulaString(value)){
			setFormula(value);
		} else {
			setStringValue(value);
		}
		// TODO:数値と式以外にも判断できないといけない
	}


	/**
	 * 文字列が数値を表す文字列かどうかを返します
	 * @param numString チェックする文字列
	 * @return 文字列が数値を表す場合true、そうでない場合false
	 */
	private boolean isNumString(String numString) {
		try {
			new BigDecimal(numString);
		} catch (NumberFormatException nfe) {
			return false;
		}
		return true;
	}

	/**
	 * 文字列が式かどうかを返します
	 * @param formulaString チェックする文字列
	 * @return 文字列が式の場合true、そうでない場合false
	 */
	private boolean isFormulaString(String formulaString) {
		if (formulaString != null && formulaString.length() > 0 && formulaString.substring(0, 1).equals("=")) {
			return true;
		} else {
			return false;
		}
	}

	/**
	 * セルに数値をセットします
	 *
	 * @param numString
	 *            数値の文字列表現
	 * @return このセルオブジェクトへの参照
	 */
	public Cell setNumberValue(String numString) {
		// セルのタイプを数値にセット
		this._cellType = CellType.NUMBER;
		// このセルが行っていたセル参照をすべて削除
		removeReferenceFromResolver();

		// 値をセット
		this._value = new Element(ElementType.NUMBER, new BigDecimal(numString, this._sheet.getMathContext()));
		// このセルを参照しているセルについて再計算が必要
		getCalculator().calc(this);

		// 計算式関連のメンバをクリア
		this.clearForumlaMembers();

		return this;

	}

	/**
	 * セルに文字列をセットします
	 *
	 * @param string
	 *            セルの値
	 * @return このセルオブジェクトへの参照
	 */
	public Cell setStringValue(String string) {
		// セルのタイプを文字列にセット
		this._cellType = CellType.STRING;
		// このセルが行っていたセル参照をすべて削除
		removeReferenceFromResolver();

		// 値をセット
		this._value = new Element(ElementType.STRING,
				string);

		// このセルを参照しているセルについて再計算が必要
		getCalculator().calc(this);

		// 計算式関連のメンバをクリア
		this.clearForumlaMembers();

		return this;

	}

	/**
	 * セルに式をセットします。 式が入力されると、このセルは再計算が必要な状態となります。
	 *
	 * @param formula
	 *            式
	 * @return このセルオブジェクトへの参照
	 */
	public Cell setFormula(String formula) {
		// セルのタイプを式にセット
		this._cellType = CellType.FORMULA;

		// このセルが行っていたセル参照をすべて削除
		removeReferenceFromResolver();

		// 式の先頭に"="が付いている場合、これを取り除く
		if (formula != null && formula.length() > 0 && formula.substring(0, 1).equals("=")) {
			formula = formula.substring(1);
		}

		// 式を格納
		this._formula = formula;
		// 式をElementの配列に変換
		Element[] infixElem = FormulaParser.split(this._formula, this._sheet.getMathContext());
		// 要素数1で、0番目のElementがエラーの場合、変換時にエラー
		if (infixElem.length == 1 && infixElem[0].getType() == ElementType.ERROR) {
			this._status = CellStatus.ERROR;
			this._value = infixElem[0];
			this._sheet.addErrorCell(this);
			getCalculator().recalcReferencingCells(this);
			return this;
		}

		// 中置記法式として正しいかチェック
		Element errorElem = FormulaParser.checkInfixNotation(infixElem);
		// errorElemがnullでなければチェックNG
		if (errorElem != null) {
			this._status = CellStatus.ERROR;
			this._value = errorElem;
			this._sheet.addErrorCell(this);
			getCalculator().recalcReferencingCells(this);
			return this;
		}

		// 式を逆ポーランド記法に変換して格納する
		this._formulaRPN = FormulaParser.infixToRPN(infixElem, this._sheet.getMathContext());
		// 要素数1で、0番目のElementがエラーの場合、変換時にエラー
		if (this._formulaRPN.length == 1 && this._formulaRPN[0].getType() == ElementType.ERROR) {
			this._status = CellStatus.ERROR;
			this._value = this._formulaRPN[0];
			this._formulaRPN = null;
			this._sheet.addErrorCell(this);
			getCalculator().recalcReferencingCells(this);
			return this;
		}

		// 式の中にセル参照が含まれる場合、参照していることを登録する
		Resolver resolver = this._sheet.getBook().getResolver();
		for (Element elem : this._formulaRPN) {
			switch (elem.getType()) {
			case REFERENCE:
				resolver.registReferences(this, elem.getCellReference());
				break;
			case GROUP_REFERENCE:
				resolver.registGroupReferences(this, elem.getGroupReference());
				break;
			case TABLE_REFERENCE:
				resolver.registTableReferences(this, elem.getTableReference());
				break;
			}
		}

		// セルの状態を「計算必要」とし、セルの計算を行う
		this._status = CellStatus.REQUIRE_CALCULATION;
		getCalculator().calc(this);

		// 計算済み…シートのエラーセルからこのセルを削除(もしあれば)
		this._sheet.removeErrorCell(this);
//		logger.info("Cell: " + this._name + " Value: " + this._value.toString());
		return this;
	}

	/**
	 * このセルにすでに式が格納されていて、その中に参照(グループ参照も含む)が含まれている場合はリゾルバからセル参照を削除します。<br>
	 */
	private void removeReferenceFromResolver() {
		//
		Resolver resolver = this.getSheet().getBook().getResolver();
		if (this._formulaRPN != null) {
			for (Element elem : this._formulaRPN) {
				switch(elem.getType()) {
				case REFERENCE:
					resolver.removeReferences(this, elem.getCellReference());
					break;
				case GROUP_REFERENCE:
					// TODO:resolver.removeGroupReferences
//					resolver.removeGroupReferences(this, elem.getGroupReference());
				}
			}
		}
	}

	/**
	 * 渡された文字列がセル名として正しいかチェックします。<br>
	 * 正しくない場合、例外がスローされます。
	 * @param cellName チェックするセル名
	 * @return セル名として正しい場合true、そうでない場合false
	 */
	static boolean isValidCellName(String cellName) {

		return _cellNamePattern.matcher(cellName).matches();

	}

	/**
	 * 渡された文字列が完全修飾セル名として正しいかチェックします。<br>
	 * 正しくない場合、例外がスローされます。
	 * @param cellName チェックするセル名
	 * @return セル名として正しい場合true、そうでない場合false
	 */
	static boolean isValidFullyQualifiedCellName(String fullyQualifiedCellName) {

		return _fullyQualifiedCellNamePattern.matcher(fullyQualifiedCellName).matches();

	}

	/**
	 * 計算機オブジェクトを返します
	 * @return 計算機オブジェクト
	 */
	private Calculator getCalculator() {
		return this._sheet.getBook().getCalculator();
	}

	/**
	 * このセルの、式関連のメンバを初期化します。
	 */
	private void clearForumlaMembers() {
		this._formula = null;
		this._formulaRPN = null;
	}

	/* (非 Javadoc)
	 * @see java.lang.Object#toString()
	 */
	public String toString() {
		return _name + "=" + _value.toString();
	}

	/* (非 Javadoc)
	 * @see java.lang.Comparable#compareTo(java.lang.Object)
	 */
	public int compareTo(Cell o) {
		return (this.getFullyQualifiedName().compareTo(o.getFullyQualifiedName()));
	}

	/* (非 Javadoc)
	 * @see java.lang.Object#equals(java.lang.Object)
	 */
	@Override
	public boolean equals(Object obj) {
		if (obj instanceof Cell) {
			if (this.hashCode() == ((Cell)obj).hashCode()) {
				return true;
			}
		}
		return false;
	}

	/* (非 Javadoc)
	 * @see java.lang.Object#hashCode()
	 */
	@Override
	public int hashCode() {
		return this.getFullyQualifiedName().hashCode();
	}
}
