package jp.co.nissy.jpicosheet.core;

import java.math.BigDecimal;
import java.util.Date;


/**
 * 計算式の各要素を表す不変のオブジェクトです。<br>
 * CellクラスからgetValue()で値を取得した際に渡されるのもこのクラスのオブジェクトです。
 * @author yusuke nishikawa
 *
 */
public class Element {

	/**
	 * このエレメントの種類を表します
	 * @author yusuke nishikawa
	 *
	 */
	public enum ElementType {
		/**
		 * 空を表します。
		 */
		EMPTY,
		/**
		 * 数値を表します。<code>getNumber()<code>でBigDecimal型のオブジェクトを得ることができます。
		 */
		NUMBER,
		/**
		 *文字列を表します。<code>getString()<code>でString型のオブジェクトを得ることができます。
		 */
		STRING,
		/**
		 * 日時を表します。
		 */
		DATE,
		/**
		 * エラーを表します。<code>getErrorType()<code>でErrorType型の列挙型オブジェクトを得ることができます。
		 */
		ERROR,
		/**
		 * オペレータを表します。<code>getOperator()<code>でOperator型の列挙型オブジェクトを得ることができます。
		 */
		OPERATOR,
		/**
		 * セル参照を表します。
		 */
		REFERENCE,
		/**
		 * グループ参照を表します。
		 */
		GROUP_REFERENCE,
		/**
		 * テーブル参照を表します。
		 */
		TABLE_REFERENCE
	}

	/**
	 * エレメントのタイプがErrorの場合に、エラーの種類を表します
	 * @author yusuke nishikawa
	 *
	 */
	public enum ErrorType {
		/**
		 * ゼロ除算
		 */
		DIVIDE_BY_ZERO,
		/**
		 * 不正な値
		 */
		WRONG_VALUE,
		/**
		 * 不正な参照
		 */
		INVALID_REFERENCES,
		/**
		 * 不正な式
		 */
		INVALID_FORMULA,
		/**
		 * 不正な名前
		 */
		UNRECOGNIZED_NAME,
		/**
		 * 不正な数値
		 */
		INVALID_NUMBER,
		/**
		 * 有効でない値
		 */
		NOT_AVAILABLE_VALUE,
		/**
		 * 循環参照
		 */
		CIRCULER_REFERENCE
	}

	/**
	 * オペレータの種類を表します。オペレータ同士の優先順位情報と、それを比較するメソッドも持っています。
	 * @author yusuke nishikawa
	 *
	 */
	public enum Operator {
		/**
		 * 関数
		 */
		FUNCTION(-1),
		/**
		 * カンマ
		 */
		COMMA(0),
		/**
		 * 左括弧
		 */
		LEFT_PARENTHESIS(0),
		/**
		 * 右括弧
		 */
		RIGHT_PARENTHESIS(0),
		/**
		 * 加算
		 */
		PLUS(1),
		/**
		 * 減算
		 */
		MINUS(1),
		/**
		 * 乗算
		 */
		TIMES(2),
		/**
		 * 除算
		 */
		DIVIDE(2),
		/**
		 * 単項演算子 プラス
		 */
		UNARY_PLUS(3),
		/**
		 * 単項演算子 マイナス
		 */
		UNARY_MINUS(3);

		private int _priority;

		private Operator (int priority) {
			this._priority = priority;
		}

		/**
		 * 引数に渡されたオペレータの優先順位をこのオペレータと比較します。<br>
		 * このオペレータの優先順位のほうが高い場合1以上の値を返し、低い場合は-1以下の値を返します。<br>
		 * 2つのオペレータの優先順位が等しい場合、0を返します。
		 *
		 * @param arg1 比較するオペレータ
		 * @return オペレータの優先順位の比較結果
		 */
		public int evalPriority(Operator arg1) {
			return this._priority - arg1._priority;
		}
	}


	/**
	 * エレメントのタイプ
	 */
	private final ElementType _type;
	/**
	 * エレメントの値
	 */
	private final Object _value;

	private int _hashCode = 0;


	/**
	 * トークンのタイプと値を使ってオブジェクトを初期化します
	 * 値にはトークンのタイプにより以下を与える必要があります。<br>
	 * <ol>
	 * <li>EMPTY	nullがセットされます。渡された値は無視されます。
	 * <li>NUMBER	BigDecimal型オブジェクト
	 * <li>STRING	String型オブジェクト
	 * <li>DATE	Date型オブジェクト
	 * <li>ERROR	String型オブジェクト(TODO:再考必要)
	 * <li>Uncalc	nullがセットされます。渡された値は無視されます。
	 * <li>OPERATOR	列挙型 OPERATOR
	 * <li>REFERENCE	String型オブジェクト
	 * </ol>
	 * オブジェクトのセットが必要なトークンのタイプの場合、指定された型のオブジェクト以外の
	 * オブジェクトをセットした場合は例外がスローされます
	 *
	 * @param tokenType トークン種類を示す列挙型
	 * @param value 値
	 * @exception IllegalArgumentException 指定したトークン種類に合わない値を指定した場合
	 */
	public Element(ElementType tokenType, Object value) {
//		this._type = tokenType;
//		this._value = value;


		this._type = tokenType;
		switch(tokenType) {
		case EMPTY:
			// 引数valueは無視する
			this._value = null;
			break;
		case NUMBER:
			if (value instanceof BigDecimal){
				this._value = value;
			} else {
				throw new IllegalArgumentException("Numberの場合、BigDecimal型オブジェクトが必要");
			}
			break;
		case STRING:
			if (value instanceof String){
				this._value = value;
			} else {
				throw new IllegalArgumentException("Stringの場合、String型オブジェクトが必要");
			}
			break;
		case DATE:
			if (value instanceof Date){
				this._value = value;
			} else {
				throw new IllegalArgumentException("Dateの場合、Date型オブジェクトが必要");
			}
			break;
		case ERROR:
			if (value instanceof ErrorType){
				this._value = value;
			} else {
				throw new IllegalArgumentException("Errorの場合、ErrorType型オブジェクトが必要");
			}
			break;
		case OPERATOR:
			if (value instanceof Operator){
				this._value = value;
			} else {
				throw new IllegalArgumentException("Operatorの場合、列挙型 Operatorが必要");
			}
			break;
		case REFERENCE:
			if (value instanceof String){
				// valueはアドレス付きテーブル名もしくは完全修飾アドレス付きテーブル名、セル名もしくは
				// 完全修飾セル名に準拠した名前でなければならない
				if (!Table.isValidTableNameWithAddress((String)value) &&
						!Table.isValidFullyQualifiedTableNameWithAddress((String)value) &&
						!Cell.isValidCellName((String)value) &&
						!Cell.isValidFullyQualifiedCellName((String)value)) {
					throw new IllegalArgumentException("不適切なセル名:" + (String)value);
				}
				this._value = value;
			} else {
				throw new IllegalArgumentException("Referenceの場合、String型オブジェクトが必要");
			}
			break;
		case GROUP_REFERENCE:
			// valueはグループ名もしくは完全修飾グループ名に準拠した名前でなければならない
			if (!Group.isValidGroupName((String)value) &&
					!Group.isValidFullyQualifiedGroupName((String)value)&&
					!Table.isValidTableName((String)value) &&
					!Table.isValidFullyQualifiedTableName((String)value)) {
				throw new IllegalArgumentException("不適切なグループ名:" + (String)value);
			}
			if (value instanceof String) {
				this._value = value;
			} else {
				throw new IllegalArgumentException("GroupReferenceの場合、String型オブジェクトが必要");
			}
			break;
		case TABLE_REFERENCE:
			// valueはテーブル名もしくは完全修飾テーブル名、範囲付きテーブル名もしくは
			// 完全修飾範囲付きテーブル名でなければならない
			if (!Table.isValidTableName((String)value) &&
					!Table.isValidFullyQualifiedTableName((String)value) &&
					!Table.isValidTableNameWithRange((String)value) &&
					!Table.isValidFullyQualifiedTableNameWithRange((String)value)) {
				throw new IllegalArgumentException("不適切なテーブル名:" + (String)value);
			}
			if (value instanceof String) {
				this._value = value;
			} else {
				throw new IllegalArgumentException("TableReferenceの場合、String型オブジェクトが必要");
			}
			break;
		default:
			this._value = null;
			// case文ですべてのTokenTypeについて処理されていなければならない
			assert true;
		}



	}

	/**
	 * トークンのタイプを使ってオブジェクトを初期化します。<br>
	 * 値にはトークンのタイプにより以下がセットされます。<br>
	 * <ol>
	 * <li>EMPTY	null
	 * <li>NUMBER	0
	 * <li>STRING	""
	 * <li>DATE	エポック
	 * <li>ERROR	IllegalArgumentExceptionがスローされます
	 * <li>Uncalc	null
	 * <li>OPERATOR	IllegalArgumentExceptionがスローされます
	 * <li>REFERENCE	IllegalArgumentExceptionがスローされます
	 * </ol>
	 *
	 * @param tokenType
	 */
	public Element(ElementType tokenType) {
		this._type = tokenType;
		switch(tokenType) {
		case EMPTY:
			this._value = null;
			break;
		case NUMBER:
			this._value = new BigDecimal("0");
			break;
		case STRING:
			this._value = "";
			break;
		case DATE:
			this._value = new Date(0);
			break;
		case ERROR:
			throw new IllegalArgumentException("Errorの場合、valueの指定が必要");
		case OPERATOR:
			throw new IllegalArgumentException("Operatorの場合、valueの指定が必要");
		case REFERENCE:
			throw new IllegalArgumentException("Referenceの場合、valueの指定が必要");
		case GROUP_REFERENCE:
			throw new IllegalArgumentException("GroupReferenceの場合、valueの指定が必要");
		default:
			this._value = null;
			// case文ですべてのTokenTypeについて処理されていなければならない
			assert true;
		}
	}

	/**
	 * タイプを返します
	 * @return エレメントのタイプ
	 */
	public ElementType getType() {
		return _type;
	}

	/**
	 * 値を返します
	 * このメソッドはequals(Object obj)の実装のために利用するだけであるため、デフォルトアクセスのメソッドです。
	 * @return エレメントの値オブジェクト
	 */
	Object getValue() {
		return _value;
	}
	/**
	 * エラーのタイプを返します
	 * @return エレメントのエラーのタイプ
	 * @throws IllegalStateException エレメントのタイプがエラー以外でこのメソッドを呼び出した場合
	 */
	public ErrorType getErrorType() throws IllegalStateException {
		if (this._type != ElementType.ERROR) {
			throw new IllegalStateException("Element type is not ERROR.");
		}
		return (ErrorType)_value;
	}

	/**
	 * セル参照情報を返します
	 * @return セル参照情報
	 * @throws IllegalStateException エレメントのタイプが参照以外でこのメソッドを呼び出した場合
	 */
	public String getCellReference() throws IllegalStateException {

		switch(this._type) {
		case REFERENCE:
			if (this._value instanceof String) {
				return (String) this._value;
			} else {
				throw new IllegalStateException("value is not CellReference");
			}
		default:
			// 型がEmptyの場合もエラー
			throw new IllegalStateException("invalid element type: " + this._type);
		}
	}

	/**
	 * グループ参照情報を返します
	 * @return グループ参照情報
	 * @throws IllegalStateException エレメントのタイプがグループ参照以外でこのメソッドを呼び出した場合
	 */
	public String getGroupReference() {
		switch(this._type) {
		case GROUP_REFERENCE:
			if (this._value instanceof String) {
				return (String) this._value;
			} else {
				throw new IllegalStateException("value is not GROUP_REFERENCE");
			}
		default:
			// 型がEmptyの場合もエラー
			throw new IllegalStateException("invalid element type: " + this._type);
		}
	}

	/**
	 * テーブル参照情報を返します
	 * @return グループ参照情報
	 * @throws IllegalStateException エレメントのタイプがテーブル参照以外でこのメソッドを呼び出した場合
	 */
	public String getTableReference() {
		switch(this._type) {
		case TABLE_REFERENCE:
			if (this._value instanceof String) {
				return (String) this._value;
			} else {
				throw new IllegalStateException("value is not TABLE_REFERENCE");
			}
		default:
			// 型がEmptyの場合もエラー
			throw new IllegalStateException("invalid element type: " + this._type);
		}
	}

	/**
	 * オペレータ情報を返します
	 * @return オペレータ情報
	 * @throws IllegalStateException エレメントのタイプがオペレータ以外でこのメソッドを呼び出した場合
	 */
	public Operator getOperator() throws IllegalStateException {

		switch(this._type) {
		case OPERATOR:
			if (this._value instanceof Operator) {
				return (Operator) this._value;
			} else {
				throw new IllegalStateException("value is not OPERATOR");
			}
		default:
			// 型がEmptyの場合もエラー
			throw new IllegalStateException("invalid element type: " + this._type);
		}
	}

	/**
	 * 数値情報を返します
	 * @return 数値情報
	 * @throws IllegalStateException エレメントのタイプが空もしくは数値情報以外でこのメソッドを呼び出した場合
	 */
	public BigDecimal getNumber() throws IllegalStateException {
		switch(this._type){
		case EMPTY:
			// 0を返す
			return new BigDecimal("0");
		case NUMBER:
			if (this._value instanceof BigDecimal) {
				return(BigDecimal)this._value;
			} else {
				throw new IllegalStateException("value is not number");
			}
		default:
			throw new IllegalStateException("invalid element type: " + this._type);
		}
	}


	/**
	 * 文字列情報を返します
	 * @return 文字列情報
	 * @throws IllegalStateException エレメントのタイプが空、文字列、参照、エラー以外でこのメソッドを呼び出した場合
	 */
	public String getString() throws IllegalStateException {

		switch (this._type) {
		case EMPTY:
			// 空文字列を返す
			return "";
		case STRING:
			if (this._value instanceof String) {
				return (String) this._value;
			} else {
				throw new IllegalStateException("value is not STRING");
			}
		case REFERENCE:
				return this._type.toString() + ": " + this._value.toString();
		case ERROR:
				return this._type.toString() + ": " +  this._value.toString();
		default:
			throw new IllegalStateException("invalid element type: " + this._type);
		}
	}

	/**
	 * 日時情報を返します
	 * @return 日時情報
	 * @throws IllegalStateException エレメントのタイプが空もしくは日時情報以外でこのメソッドを呼び出した場合
	 */
	public Date getDate() throws IllegalStateException {

		switch (this._type) {
		case EMPTY:
			// エポックのDateオブジェクトを返す
			return new Date(0);
		case DATE:
			if (this._value instanceof Date) {
				// 不変性の維持のため、新しいDateオブジェクトをnewして渡す
				return new Date(((Date)this._value).getTime());
			} else {
				throw new IllegalStateException("value is not DATE");
			}
		default:
			throw new IllegalStateException("invalid element type: " + this._type);
		}
	}

	@Override
	public String toString() {
		// Emptyの場合空文字を返す
		if (this._type == ElementType.EMPTY) {
			return "";
		}
		try {
			return this._type.toString() + ":" + this._value.toString();
		} catch (IllegalStateException e) {
			return e.getMessage();
		}
	}

	/* (非 Javadoc)
	 * @see java.lang.Object#equals(java.lang.Object)
	 */
	@Override
	public boolean equals(Object obj) {
		return this.hashCode() == obj.hashCode();
	}

	/* (非 Javadoc)
	 * @see java.lang.Object#hashCode()
	 */
	@Override
	public int hashCode() {
		// _typeのと_valueのhashcodeの文字列表現を結合し、その文字列のhashcodeを返す
		if (this._hashCode == 0) {
			String concatStr = Integer.toString(_type.hashCode()) + Integer.toString(_value==null? 0 :_value.hashCode());
			this._hashCode = concatStr.hashCode();
		}
		return this._hashCode;
	}
}
