package test;

import java.math.BigDecimal;

import jp.co.nissy.jpicosheet.core.Book;
import jp.co.nissy.jpicosheet.core.Cell;
import jp.co.nissy.jpicosheet.core.Element;
import jp.co.nissy.jpicosheet.core.ReferenceNotFoundException;
import jp.co.nissy.jpicosheet.core.Sheet;
import jp.co.nissy.jpicosheet.core.Cell.CellType;
import jp.co.nissy.jpicosheet.core.Element.ElementType;
import junit.framework.TestCase;

public class CellTest extends TestCase {

	Book book;
	Sheet sheet;
	Cell cell;

	public CellTest(String name) {
		super(name);
	}

	protected void setUp() throws Exception {
		super.setUp();
		book= new Book("myBook");
		sheet = book.addSheet("mySheet");
		cell = sheet.addCell("targetCell");

	}

	protected void tearDown() throws Exception {
		super.tearDown();
	}

	/**
	 * セル作成時のテスト(主にセル名のテスト)
	 * @throws Exception
	 */
	public void testCell() throws Exception {
		sheet.addCell("newcell");
		sheet.addCell("NEWCELL");
		sheet.addCell("newCell");
		sheet.addCell("newCell123");
		sheet.addCell("a");
		sheet.addCell("_newcell");
		sheet.addCell("_newcell_");
		sheet.addCell("_________________newcell");
		sheet.addCell("new_cell_name_with_underscore");
		sheet.addCell("very_long_cell_name_aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa");
		try {
			sheet.addCell("");
			fail("空文字のセル名でエラーにならない");
		} catch (Exception e) {
			// OK
		}
		try {
			sheet.addCell("new cell");
			fail("中にスペースを含むセル名でエラーにならない");
		} catch (Exception e) {
			// OK
		}
		try {
			sheet.addCell("new	cell");
			fail("中にタブを含むセル名でエラーにならない");
		} catch (Exception e) {
			// OK
		}
		try {
			sheet.addCell(" newcell ");
			fail("前後にスペースを含むセル名でエラーにならない");
		} catch (Exception e) {
			// OK
		}
		try {
			sheet.addCell("	newcell	");
			fail("前後にタブを含むセル名でエラーにならない");
		} catch (Exception e) {
			// OK
		}
		try {
			sheet.addCell("日本語");
			fail("日本語のセル名でエラーにならない");
		} catch (Exception e) {
			// OK
		}
		try {
			sheet.addCell("1122");
			fail("数字のみのセル名でエラーにならない");
		} catch (Exception e) {
			// OK
		}
		try {
			sheet.addCell("1abc");
			fail("先頭文字が数字のセル名でエラーにならない");
		} catch (Exception e) {
			// OK
		}
		try {
			sheet.addCell("=abc");
			fail("先頭文字が記号のセル名でエラーにならない");
		} catch (Exception e) {
			// OK
		}
	}

	public void testGetFormula() throws Exception {
		cell.setFormula("1+1");
		assertEquals("=1+1", cell.getFormula());

		cell.setValue("=1+1");
		assertEquals("=1+1", cell.getFormula());

		cell.setValue("=1+1*");
		assertEquals(Element.ElementType.ERROR, cell.getValue().getType());
		assertEquals("=1+1*", cell.getFormula());
		cell.setValue("=1+1-nonexistenceref");
		assertEquals(Element.ElementType.ERROR, cell.getValue().getType());
		assertEquals("=1+1-nonexistenceref", cell.getFormula());
	}

	public void testGetName() {
		assertEquals("targetCell", cell.getName());
		assertEquals("mySheet!targetCell", cell.getFullyQualifiedName());
	}


	public void testGetSheet() {
		assertEquals(cell.getSheet(), sheet);
	}

	public void testGetString() {
		cell.setStringValue("string value");
		assertEquals(cell.getString(), "string value");
	}


	public void testGetValue() {
		cell.setNumberValue("1234");
		assertEquals(cell.getValue(), new Element(ElementType.NUMBER, new BigDecimal("1234")));
	}

	public void testGetValueString() {
		cell.setNumberValue("1234");
		assertEquals(cell.getValueString(), "1234");
		cell.setNumberValue("1234.567");
		assertEquals(cell.getValueString(), "1234.567");
		cell.setFormula("1+2+3");
		assertEquals(cell.getValueString(), "6");
	}

	public void testCompareEqual() {
		cell.setNumberValue("1");
		assertEquals(cell.getValue(), new Element(ElementType.NUMBER, new BigDecimal("1")));
		cell.setNumberValue("12345677890");
		assertEquals(cell.getValue(), new Element(ElementType.NUMBER, new BigDecimal("12345677890")));
		cell.setNumberValue("1234.56789");
		assertEquals(cell.getValue(), new Element(ElementType.NUMBER, new BigDecimal("1234.56789")));
		cell.setStringValue("abcdefgABCDEFG");
		assertEquals(cell.getValue(), new Element(ElementType.STRING, "abcdefgABCDEFG"));

		sheet.addCell("emptyCell1");
		try {
			assertEquals(sheet.getCell("emptyCell1").getValue(), new Element(ElementType.EMPTY));
		} catch (ReferenceNotFoundException e1) {
			fail();
		}
		sheet.addCell("emptyCell2");
		try {
			assertEquals(sheet.getCell("emptyCell1").getValue(), sheet.getCell("emptyCell2").getValue());
		} catch (ReferenceNotFoundException e) {
			fail();
		}
	}

	public void testSetNumberValue() {
		cell.setNumberValue("10");
		assertEquals(cell.getValue().getNumber(), new BigDecimal("10"));
	}

	public void testSetStringValue() {
		cell.setStringValue("abcde");
		assertEquals(cell.getValue().getString(), "abcde");
	}

	public void testEquals() {
		Cell c1 = sheet.addCell("compareCell1");
		Cell c2 = sheet.addCell("compareCell2");
		assertEquals(true, c1.equals(c1));
		assertEquals(false, c1.equals(c2));
		assertEquals(false, c2.equals(c1));

		c1.setNumberValue("1");
		c2.setNumberValue("1");
		assertEquals(true, c1.equals(c1));
		assertEquals(false, c1.equals(c2));
		assertEquals(false, c2.equals(c1));

		c1.setNumberValue("123");
		c2.setNumberValue("456");
		assertEquals(true, c1.equals(c1));
		assertEquals(false, c1.equals(c2));
		assertEquals(false, c2.equals(c1));

	}

	public void testCompare() {
		Cell c1 = sheet.addCell("compareCell1");
		Cell c2 = sheet.addCell("compareCell2");

		assertEquals(0, c1.compareTo(c1));
		assertEquals(-1, c1.compareTo(c2));
		assertEquals(1, c2.compareTo(c1));
	}

	public void testSetValue() {

		Cell c1 = sheet.addCell("c1");
		Cell c2 = sheet.addCell("c2");
		c1.setValue("");
		assertEquals(CellType.EMPTY, c1.getCellType());
		c1.setValue("123");
		assertEquals(CellType.NUMBER, c1.getCellType());
		c1.setValue("-123");
		assertEquals(CellType.NUMBER, c1.getCellType());
		c1.setValue("123.456");
		assertEquals(CellType.NUMBER, c1.getCellType());
		c1.setValue(".000234");
		assertEquals(CellType.NUMBER, c1.getCellType());

		c1.setValue("=1+2");
		assertEquals(CellType.FORMULA, c1.getCellType());
		assertEquals(Element.ElementType.NUMBER, c1.getValue().getType());
		c1.setValue("=0+0");
		assertEquals(CellType.FORMULA, c1.getCellType());
		assertEquals(Element.ElementType.NUMBER, c1.getValue().getType());
		c1.setValue("=1+*/1");
		assertEquals(CellType.FORMULA, c1.getCellType());
		assertEquals(Element.ElementType.ERROR, c1.getValue().getType());
		assertEquals(Element.ErrorType.INVALID_FORMULA, c1.getValue().getErrorType());
		c1.setValue("=1+1+");
		assertEquals(CellType.FORMULA, c1.getCellType());
		assertEquals(Element.ElementType.ERROR, c1.getValue().getType());
		assertEquals(Element.ErrorType.INVALID_FORMULA, c1.getValue().getErrorType());
		c1.setValue("=1+1(");
		assertEquals(CellType.FORMULA, c1.getCellType());
		assertEquals(Element.ElementType.ERROR, c1.getValue().getType());
		assertEquals(Element.ErrorType.INVALID_FORMULA, c1.getValue().getErrorType());

		c1.setValue("=c1+1");
		assertEquals(CellType.FORMULA, c1.getCellType());
		assertEquals(Element.ElementType.ERROR, c1.getValue().getType());
		assertEquals(Element.ErrorType.CIRCULER_REFERENCE, c1.getValue().getErrorType());
		c1.setValue("=c2+1");
		assertEquals(CellType.FORMULA, c1.getCellType());
		assertEquals(Element.ElementType.NUMBER, c1.getValue().getType());
		c1.setValue("=nonexistenceCell+1");
		assertEquals(CellType.FORMULA, c1.getCellType());
		assertEquals(Element.ElementType.ERROR, c1.getValue().getType());
		assertEquals(Element.ErrorType.INVALID_REFERENCES, c1.getValue().getErrorType());

		c1.setValue("abcde");
		assertEquals(CellType.STRING, c1.getCellType());
		c1.setValue("あいうえお");
		assertEquals(CellType.STRING, c1.getCellType());

	}

	public void testLabel() {
		Cell c1 = sheet.addCell("c1");
		assertEquals("", c1.getLabel());

		c1.setLabel("");
		assertEquals("", c1.getLabel());
		c1.setLabel("abcde");
		assertEquals("abcde", c1.getLabel());
		c1.setLabel("ラベルには日本語なども使用可能!?/ <>_+-*/^\"");
		assertEquals("ラベルには日本語なども使用可能!?/ <>_+-*/^\"", c1.getLabel());

		Cell c2 = sheet.addCell("c2");
		assertEquals("", c2.getLabel());

		c2.setLabel("c2's new label");
		assertEquals("c2's new label", c2.getLabel());
		assertEquals("ラベルには日本語なども使用可能!?/ <>_+-*/^\"", c1.getLabel());
	}


}
