#
#  The pkgbuild build engine
#
#  Copyright (C) 2004, 2005 Sun Microsystems, Inc.
#
#  pkgbuild is free software; you can redistribute it and/or
#  modify it under the terms of the GNU General Public License 
#  version 2 published by the Free Software Foundation.
#
#  pkgbuild is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
#  General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software
#  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#
#  As a special exception to the GNU General Public License, if you
#  distribute this file as part of a program that contains a
#  configuration script generated by Autoconf, you may include it under
#  the same distribution terms that you use for the rest of that program.
#
#  Authors:  Laszlo Peter  <laca@sun.com>
#

use strict;
use warnings;
use Text::Wrap;

package config;

# Create a new config object.
sub new ($) {
    my $class = shift;
    my $self = {};
    # default values of keys
    my %defvals = ();
    # values read from rc files
    my %rcvals = ();
    # values set using the set() method, typically values from the command line
    my %clvals = ();
    # a type assigned to each value key
    # valid types: 's' = string, 'n' = number, '!' = bool
    my %valid_keys = ();
    # descriptions of the keys
    my %docs = ();
    $self->{'valid_keys'} = \%valid_keys;
    $self->{'defvals'} = \%defvals;
    $self->{'rcvals'} = \%rcvals;
    $self->{'clvals'} = \%clvals;
    $self->{'docs'} = \%docs;

    return (bless $self, $class);
}

# check that a given value is valid for a given key type
sub _is_type ($$) {
    my $type = shift;
    my $val = shift;

    if ($type eq 's') {
	if (defined ($val)) {
	    return 1;
	}
	return 0;
    } elsif ($type eq 'n') {
	if ($val =~ /^[0-9]+$/) {
	    return 1;
	}
	return 0;
    } elsif ($type eq '!') {
	if ($val eq '0' or $val eq '1') {
	    return 1;
	}
	return 0;
    }
}

# define a key that cannot be changed in the rc file, only
# programmatically
sub define ($$$) {
    my $self = shift;
    my $key = shift;
    my $val = shift;

    if (defined ($self->{'valid_keys'}->{$key})) {
	print "ERROR: option $key already defined as an rc setting\n";
	return 0;
    }

    $self->{defines}->{$key} = $val;
}

# define a new setting:
#   key     = the key of the setting
#   type    = valid type of value for this setting:
#             's' = string, 'n' = number, '!' = bool
#   docs    = description of the setting
#  [defval] = optional default value
sub add ($$$$;$) {
    my $self = shift;
    my $key = shift;
    my $type = shift;
    my $docs = shift;
    my $defval = shift;

    if (defined ($self->{'valid_keys'}->{$key})) {
	print "ERROR: option $key already defined\n";
	return 0;
    }

    if (not $type =~ /^[sn!]$/) {
	print "ERROR: unknown type \"$type\" for option $key\n";
	return 0;
    }

    $self->{'valid_keys'}->{$key} = $type;
    $self->{'docs'}->{$key} = $docs;

    if (defined ($defval)) {
	if (_is_type ($type, $defval)) {
	    $self->{'defvals'}->{$key} = $defval;
	    return 1;
	} else {
	    print "ERROR: Default value for option $key is invalid.\n";
	}
    }
}

# get the value of the key
# values set using 'set' are considered first,
# then values set using an rc file
# finally the default values
sub get ($$) {
    my $self = shift;
    my $key = shift;

    if (not defined ($self->{'valid_keys'}->{$key})) {
	return undef;
    }

    if (defined ($self->{'clvals'}->{$key})) {
	return ($self->_deref_keys($self->{'clvals'}->{$key}));
    }

    if (defined ($self->{'rcvals'}->{$key})) {
	return ($self->_deref_keys($self->{'rcvals'}->{$key}));
    }

    if (defined ($self->{'defvals'}->{$key})) {
	return ($self->_deref_keys($self->{'defvals'}->{$key}));
    }

    return undef;
}

# set the value for a given key
sub set ($$$) {
    my $self = shift;
    my $key = shift;
    my $value = shift;

    if (defined $self->{'valid_keys'}->{$key}) {
	if (_is_type ($self->{'valid_keys'}->{$key}, $value)) {
	    $self->{'clvals'}->{$key} = $value;
	    return 1;
	} else {
	    print "WARNING: Value for option $key is invalid.\n";
	}
    } else {
	print "ERROR: Unknown option: $key\n";
    }
    return 0;
}

# forget all values read from rc files
sub norc ($) {
    my $self = shift;
    my %rcvals = ();
    $self->{'rcvals'} = \%rcvals;
}

# substitutes the values of variables and keys
sub _deref_vars ($$$) {
    my $self = shift;
    my $str = shift;
    my $varref = shift;

    if (not defined ($str)) {
	return undef;
    }

    foreach my $var (keys %$varref) {
	next if not defined $var;
	my $val = $$varref{$var};
	$str =~ s/\${$var}/$val/g;
    }

    my $all_keys = $self->{valid_keys};
    foreach my $key (keys %$all_keys) {
	next if not defined $key;
	my $val;
	if (defined ($self->{'clvals'}->{$key})) {
	    $val = $self->{'clvals'}->{$key};
	} elsif (defined ($self->{'rcvals'}->{$key})) {
	    $val = $self->{'rcvals'}->{$key};
	} elsif (defined ($self->{'defvals'}->{$key})) {
	    $val = $self->{'defvals'}->{$key};
	}

	$str =~ s/\${$key}/$val/g;
    }
 
   return $str;
}

# substitutes the values of keys into default values
# variables in default values have this form: %{foo}
sub _deref_keys ($$) {
    my $self = shift;
    my $str = shift;

    if (not defined ($str)) {
	return undef;
    }

    while ($str =~ /%{([a-zA-Z]+)}/) {
	my $var = $1;
	if (defined ($self->{valid_keys}->{$var})) {
	    my $val = $self->get ($var);
	    $str =~ s/%{$var}/$val/g;
	} elsif (defined ($self->{defines}->{$var})) {
	    my $val = $self->{defines}->{$var};
	    $str =~ s/%{$var}/$val/g;
	} else {
	    $str =~ s/%{$var}/%PeRcEnT{$var}/g;
	}
    }
    $str =~ s/%PeRcEnT/%/g;

    return $str;
}

# readrc (obj, file_name [, ignore_unknown])
#
# read values of settings from an rc file (file_name)
# rc files look like this:
#    ,-----------------------------------.
#    |#comment line                      |
#    |key1: value1                       |
#    |key2: value2                       |
#    |VARIABLE1 = foo                    |
#    |key3: ${VARIABLE1}                 |
#    |key4: ${value2}:bar                |
#    `-----------------------------------'
#
# ignore_unknown is boolean, if undefined or 0 report errors about
# unknown setting in the rc file, otherwise ignore them
sub readrc ($$;$) {
    my $self = shift;
    my $fname = shift;
    my $ignore_unknown = shift;

    if (! -f $fname) {
	return 0;
    }

    my %vars;

    if (! open RCFILE, "<$fname") {
	print ("WARNING: Failed to open file $fname\n");
	return 0;
    }

    my $rcdir = `/usr/bin/dirname $fname`;
    chomp ($rcdir);
    $rcdir = `cd $rcdir; pwd`;
    chomp ($rcdir);
    $vars{'MYDIR'} = $rcdir;

    my $line = <RCFILE>;
    while ($line) {
	if ($line =~ /^\s*([a-zA-Z][a-zA-Z_0-9]*)\s*:\s*"([^"]*)"\s*$/) {
            my $key = lc ($1);
            my $value0 = $2;
            my $value = $2;
            $value =~ s/%/%PeRcEnT/g;
            $value = $self->_deref_vars ($value, \%vars);
            if (defined ($self->{'valid_keys'}->{$key})) {
                if (_is_type ($self->{'valid_keys'}->{$key}, $value)) {
	            $self->{'rcvals'}->{$key} = $value;
                } else {
                    print "WARNING: $fname: Incorrect value \"$value0\" for option $key\n";
                }
            } else {
		if (not $ignore_unknown) {
		    print "WARNING: $fname: Unknown option \"$key\"\n";
		}
            }
        } elsif ($line =~ /^\s*([a-zA-Z][a-zA-Z_0-9]*)\s*:\s*(.+)\s*$/) {
	    my $key = lc ($1);
            my $value0 = $1;
            my $value = $2;
            $value =~ s/%/%PeRcEnT/g;
            $value = $self->_deref_vars ($value, \%vars);
            if (defined ($self->{'valid_keys'}->{$key})) {
                if (_is_type ($self->{'valid_keys'}->{$key}, $value)) {
	            $self->{'rcvals'}->{$key} = $value;
                } else {
                    print "WARNING: $fname: Incorrect value \"$value0\" for option $key\n";
                }
            } else {
		if (not $ignore_unknown) {
		    print "WARNING: $fname: Unknown option \"$key\"\n";
		}
            }
        } elsif ($line =~ /^\s*([A-Z_]+)\s*=\s*"([^"]*)"\s*$/) {
	    my $var = $1;
	    my $val = $2;
	    $val =~ s/%/%PeRcEnT/g;
	    $vars{$var} = $self->_deref_vars ($val, \%vars);
        } elsif ($line =~ /^\s*([A-Z_]+)\s*=\s*(\S+)\s*$/) {
	    my $var = $1;
	    my $val = $2;
	    $val =~ s/%/%PeRcEnT/g;
	    $vars{$var} = $self->_deref_vars ($val, \%vars);
	} elsif ($line =~ /^\s*no([a-zA-Z][a-zA-Z0-9_]*)\s*$/){
	    my $key = lc ($1);
            if (defined ($self->{'valid_keys'}->{$key})) {
                if ($self->{'valid_keys'}->{$key} eq '!') {
	            $self->{'rcvals'}->{$key} = 0;
                } else {
                    print "WARNING: $fname: option $key is not boolean\n";
                }
            } else {
		if (not $ignore_unknown) {
		    print "WARNING: $fname: Unknown option \"$key\"\n";
		}
            }
	} elsif ($line =~ /^\s*([a-zA-Z][a-zA-Z0-9_]*)\s*$/){
	    my $key = lc ($1);
            if (defined ($self->{'valid_keys'}->{$key})) {
                if ($self->{'valid_keys'}->{$key} eq '!') {
	            $self->{'rcvals'}->{$key} = 1;
                } else {
                    print "WARNING: $fname: option $key is not boolean\n";
                }
            } else {
		if (not $ignore_unknown) {
		    print "WARNING: $fname: Unknown option \"$key\"\n";
		}
	    }
	} elsif ($line =~ /^\s*$/) {
	    1;
	} elsif ($line =~ /^\s*#/) {
	    1;
	} else {
	    print "WARNING: $fname: Syntax error in rc file: \"$line\"\n";
	}
	$line = <RCFILE>;
    }
    return 1;
}

# return the default value for a key
sub get_default ($$) {
    my $self = shift;
    my $key = shift;

    if (defined ($self->{'valid_keys'}->{$key})) {
	return $self->_deref_keys ($self->{'defvals'}->{$key});
    }

    return undef;
}

# return 1 if the setting equals to its default value, 0 if different,
# undef if the setting is not defined
sub is_default ($$) {
    my $self = shift;
    my $key = shift;

    my $value = $self->get ($key);
    if (not defined ($self->{'valid_keys'}->{$key})) {
	return undef;
    }
    if (defined ($self->{'clvals'}->{$key})) {
	return 0;
    }
    if (defined ($self->{'rcvals'}->{$key})) {
	return 0;
    }
    return 1;
}

# print all keys and values in a rc file format suitable for loading with
# the readrc method
sub dumprc ($) {
    my $self = shift;

    my $keysref = $self->{'valid_keys'};
    foreach my $key (keys %$keysref) {
	my $type = $self->{'valid_keys'}->{$key};
	if ($type eq 's') {
	    $type = 'string';
	} elsif ($type eq 'n') {
	    $type = 'integer';
	} elsif ($type eq '!') {
	    $type = 'boolean';
	}

        # Break option description over multiple comment lines for readability.
        print Text::Wrap::wrap( '# ', '# ', "$key [$type]: " . $self->{'docs'}->{$key} ), "\n";

	my $val = $self->get ($key);
	$val = '' if not defined ($val);
	if ($self->is_default ($key)) {
	    print "# $key:\t$val\n";
	} else {
	    print "$key:\t$val\n";
	}
	print "\n";
    }
}

1;
