/*

  pkgdbgrep, part of the pkgbuild build engine

  Copyright 2008 Sun Microsystems, Inc.

  pkgbuild is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License 
  version 2 published by the Free Software Foundation.

  pkgbuild is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

  As a special exception to the GNU General Public License, if you
  distribute this file as part of a program that contains a
  configuration script generated by Autoconf, you may include it under
  the same distribution terms that you use for the rest of that program.

  Authors:  Laszlo Peter  <laca@sun.com>

*/

#include <stdio.h>
#include <strings.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdlib.h>
#include <errno.h>

/* must be longer than the longest line in contents */
#define READ_SIZE (32768)   
#define MYNAME "pkgdbgrep"

#ifndef PKGBUILD_VERSION
#error PKGBUILD_VERSION undefined.  Should be defined in the Makefile
#endif

static int n_files_to_find = 0;
static char **files_to_find = NULL;
static char *contents = "/var/sadm/install/contents";

static void usage () {
  printf (MYNAME " [options] /path/to/file...\n"
	  "\n"
	  "Find the given files in the package database and print the matching lines\n\n"
	  "Options:\n"
	  "   -c file , --contents file\n"
	  "                        Read the package contents from file\n"
          "                        default: /var/sadm/install/contents\n"
	  "   -h, --help           Print this usage information and exit\n"
	  "   -v, --version        Print the version of this program and exit\n");
}

static int process_args (int argc, char *argv[]) {
  int i;

  if (argc == 1) {
    usage ();
    exit (1);
  }
  for (i = 1; i < argc; i++) {
    if (argv[i][0] == '-') {
      if ((argv[i][1] == '-') && (argv[i][2] == '\0')) { 
	/* -- end of options */
	i++;
	break;
      }
      if (!strcmp (argv[i], "-h") || !strcmp (argv[i], "--help")) {
	usage ();
	exit (0);
      }
      if (!strcmp (argv[i], "-c") || !strcmp (argv[i], "--contents")) {
	if ((i+1) < argc) {
	  i++;
	  contents = argv[i];
	  continue;
	} else {
	  fprintf (stderr, MYNAME ": Option %s requires an argument\n",
		   argv[i]);
	  exit (1);
	}
      }
      if (!strcmp (argv[i], "--version") || !strcmp (argv[i], "-v")) {
	printf ("pkgbuild version " PKGBUILD_VERSION "\n");
	exit (0);
      }
      /* unknown option */
      fprintf (stderr, MYNAME ": invalid argument %s\n", argv[1]);
      exit (1);
    } else {
      break;
    }
  }
  if (i < argc) {
    files_to_find = argv + i;
    n_files_to_find = argc - i;
  } else {
    usage ();
    return 1;
  }

  return 0;
}

/* print str to the end of the line and return the pointer to the 1st char
   of the next line, unless the end of the string is reached before a \n,
   in which case return str */
static char *print_line (char *str) {
  char *str0 = str;

  while (*str && *str != '\n') str++;
  if (*str == '\n') {
    *str = '\0';
    puts (str0);
    return str + 1;
  } else {
    /* we are at the end of the buffer, the line is not fully read */
    return str0;
  }
}

/* read the contents file and print that lines that match files_to_find */
static int read_contents () {
  static char buf [2 * READ_SIZE], *buf_start, *buf_end, *p, *p_new;
  int *fname_lengths;
  size_t len;
  int fd;
  int i, n;

  /* calculate the length of each path name, so we can later
     use strncmp() for comparing with the contents line */
  if ((fname_lengths = malloc (n_files_to_find)) == NULL) {
    return 1;
  }
  for (i = 0; i < n_files_to_find; i++) {
    fname_lengths[i] = strlen (files_to_find[i]);
  }

  /* open contents file */
  if ((fd = open (contents, O_RDONLY)) == -1) {
    fprintf (stderr, MYNAME ": failed to open file %s for reading: %s\n",
	     contents, strerror (errno));
    return 1;
  }
  buf_start = buf;
  p = NULL;
  /* read contents file until EOF reached */
  while (n_files_to_find && (len = read (fd, buf_start, READ_SIZE)) > 0) {
    p_new = buf;  /* start processing at buf */
    buf_end = buf_start + len;
    *buf_end = '\0';
    while (n_files_to_find && (p != p_new) && (p_new < buf_end)) {
      p = p_new;
      /* p is at the beginning of a line */
      for (i = 0; i < n_files_to_find; i++) {
	if ((buf_end - p) <= fname_lengths[i]) {
	  /* need to read more */
	  memcpy (buf, p, buf_end - p);
	  buf_start = buf + (buf_end - p);
	  break;
	}
	if (memcmp (p, files_to_find[i], fname_lengths[i]) == 0) {
	  if ((p[fname_lengths[i]] == ' ') || (p[fname_lengths[i]] == '=')) {
	    /* found what we are looking for */
	    p_new = print_line (p);  
	    /* moves p to the beginning of the next line unless
	       the line has not been fully read */	  
	    if (p_new == p) {
	      /* need to read more from contents */
	      memcpy (buf, p, buf_end - p);
	      buf_start = buf + (buf_end - p);
	    } else {
	      /* don't look for this file anymore */
	      for (n = i; n + 1 < n_files_to_find; n++) {
		files_to_find[n] = files_to_find[n+1];
		fname_lengths[n] = fname_lengths[n+1];
	      }
	      n_files_to_find --;
	    }
	    break;
	  }
	}
      }
      if ((p == p_new) && (p < buf_end)) {
	/* no match for any of the file names, find the next line */
	if ((p_new = memchr (p, '\n', buf_end - p)) == NULL) {
	  p_new = p;
	} else {
	  p_new ++;
	}
      }
    }
  }
  close (fd);
  return 0;
}

int main (int argc, char *argv[]) {
  if (process_args (argc, argv)) {
    exit (1);
  }

  return read_contents ();
}
