package org.maachang.reflect ;

import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;

/**
 * 高速リフレクションUtil.
 *
 * @version 2009/03/29
 * @author  masahito suzuki
 * @since   FastReflect 1.00
 */
abstract class FastReflectUtil {
    private FastReflectUtil() {}
    
    /** プリミティブ系. **/
    private static final Map<Class,Class> CONV_PRIMITIVE = FastReflectPrimitive.CONV_PRIMITIVE ;
    private static final Set<Class> PRIMITIVE = FastReflectPrimitive.PRIMITIVE ;
    private static final Object OBJECT = FastReflectPrimitive.OBJECT ;
    private static final Object STRING = FastReflectPrimitive.STRING ;
    private static final Set<Class> NUMBER_PRIMITIVE = FastReflectPrimitive.NUMBER_PRIMITIVE ;
    
    /** 変換テーブル. **/
    private static final Map<Class,Integer> CONV_TABLE ;
    
    static {
        Map<Class,Integer> m = new HashMap<Class,Integer>() ;
        m.put( Byte.class,0 ) ;
        m.put( Character.class,1 ) ;
        m.put( Short.class,2 ) ;
        m.put( Integer.class,3 ) ;
        m.put( Long.class,4 ) ;
        m.put( Float.class,5 ) ;
        m.put( Double.class,6 ) ;
        CONV_TABLE = m ;
    }
    
    /**
     * パラメータに対する型を取得.
     * @param args パラメータを設定します.
     * @return Class[] パラメータ型が返されます.
     */
    public static final Class[] getParamsType( Object... args ) {
        int argsLen ;
        if( args == null || ( argsLen = args.length ) <= 0 ) {
            return FastReflectPrimitive.NO_PARAM_CLASS ;
        }
        Class[] c = new Class[ argsLen ] ;
        for( int i = 0 ; i < argsLen ; i ++ ) {
            if( args[ i ] != null ) {
                c[ i ] = args[ i ].getClass() ;
                Class pc = CONV_PRIMITIVE.get( c[ i ] ) ;
                if( pc != null ) {
                    c[ i ] = pc ;
                }
            }
            else {
                c[ i ] = null ;
            }
        }
        return c ;
    }
    
    /**
     * パラメータに対して、一致点数を取得.
     * @param src チェック元を設定します.
     * @param dest チェック先を設定します.
     * @param cl 対象のクラスローダーを設定します.
     * @return int 一致点数が返されます.
     * @exception 例外.
     */
    public static final int parmasScore( Class[] src,Class[] dest,ClassLoader cl )
        throws Exception {
        // １つの引数が一致した場合は、100点が加算される.
        // １つのsrc引数がObject型の場合は、60点が加算される.
        // １つの引数が数値のプリミティブ型同士の場合は、50点が加算される.
        // １つの引数の関係が、プリミティブ - 文字列関係の場合は、40点が加算される.
        // ただし、引数のどれか１つが一致しなかった場合は、-1を返す.
        int ret = 0 ;
        int lenJ ;
        Class s,d ;
        boolean one = true ;
        int len = src.length ;
        for( int i = 0 ; i < len ; i ++ ) {
            s = src[ i ] ;
            d = dest[ i ] ;
            // チェック先がNULLの場合は、相互チェックしない.
            if( d != null ) {
                // チェック元と、チェック先が一致している.
                if( s == d ) {
                    ret += 100 ;
                }
                // チェック元がObjectの場合は、相互チェックしない.
                else if( s == OBJECT ) {
                    ret += 60 ;
                }
                // ・チェック元・先が、数値プリミティブ値の場合.
                else if( NUMBER_PRIMITIVE.contains( s ) && NUMBER_PRIMITIVE.contains( d ) ) {
                    ret += 50 ;
                }
                // ・チェック元が文字列で、チェック先が、文字列か、プリミティブ型の場合.
                // ・チェック元がプリミティブか、文字列で、チェック先が、文字列の場合.
                else if( ( s == STRING && ( PRIMITIVE.contains( d ) ) ) ||
                    ( d == STRING && ( PRIMITIVE.contains( s ) ) ) ) {
                    ret += 40 ;
                }
                else {
                    // チェック元に対して、チェック先の継承クラス／インターフェイスを
                    // 掘り下げてチェックする.
                    FastClassElement em = FastReflectClass.getClass( d.getName() ) ;
                    one = false ;
                    Class o ;
                    String[] ifce ;
                    // チェック元がインターフェイス属性の場合は、インターフェイス内容と、
                    // スーパークラスのチェックを行う.
                    if( s.isInterface() ) {
                        String sname = s.getName() ;
                        while( true ) {
                            ifce = em.getInterfaseNames() ;
                            if( ifce != null && ( lenJ = ifce.length ) > 0  ) {
                                // インターフェース名群と、チェック元のクラス名が一致.
                                if( Arrays.binarySearch( ifce,sname ) != -1 ) {
                                    one = true ;
                                    ret += 100 ;
                                }
                                // 継承インターフェイスが１つの場合.
                                else if( lenJ == 1 ) {
                                    if( isInterface( sname,ifce[ 0 ],cl ) ) {
                                        one = true ;
                                        ret += 100 ;
                                    }
                                }
                                // 継承インターフェイスが複数の場合.
                                else {
                                    for( int j = 0 ; j < lenJ ; j ++ ) {
                                        if( isInterface( sname,ifce[ i ],cl ) ) {
                                            one = true ;
                                            ret += 100 ;
                                            break ;
                                        }
                                    }
                                }
                                if( one ) {
                                    break ;
                                }
                            }
                            // スーパークラスを取得.
                            em = FastReflectClass.getClass( em.getSuperClassName() ) ;
                            // スーパークラスがオブジェクトの場合.
                            if( (o = em.getClassObject()) == OBJECT ) {
                                return -1 ;
                            }
                            // スーパークラスと、チェック元が一致する場合.
                            else if( o == s ) {
                                one = true ;
                                ret += 100 ;
                                break ;
                            }
                        }
                    }
                    // チェック元がオブジェクトの場合は、スーパークラスのみチェック.
                    else {
                        while( true ) {
                            // スーパークラスを取得.
                            em = FastReflectClass.getClass( em.getSuperClassName() ) ;
                            // スーパークラスがオブジェクトの場合.
                            if( (o = em.getClassObject()) == OBJECT ) {
                                return -1 ;
                            }
                            // スーパークラスと、チェック元が一致する場合.
                            else if( o == s ) {
                                one = true ;
                                ret += 100 ;
                                break ;
                            }
                        }
                    }
                    // 一致条件が存在しない.
                    if( !one ) {
                        return -1 ;
                    }
                }
            }
        }
        return ret ;
    }
    
    /** interface比較. **/
    private static final boolean isInterface( String sname,String name,ClassLoader cl )
        throws Exception {
        FastClassElement em = FastReflectClass.getClass( cl,name ) ;
        while( em != null ) {
            String superClassName = em.getSuperClassName() ;
            if( superClassName == sname ) {
                return true ;
            }
            em = FastReflectClass.getClass( cl,superClassName ) ;
        }
        return false ;
    }
    
    /**
     * パラメータ変換.
     * @param args 変換対象のパラメータを設定します.
     * @param types 変換対象のパラメータタイプを設定します.
     * @return Object[] 変換されたオブジェクトが返されます.
     */
    public static final Object[] convertParams( Object[] args,Class[] types ) {
        int len = args.length ;
        Class s,d ;
        Object[] ret ;
        if( len > 0 ) {
            ret = new Object[ len ] ;
            System.arraycopy( args,0,ret,0,len ) ;
        }
        else {
            return FastReflectPrimitive.NO_PARAM ;
        }
        for( int i = 0 ; i < len ; i ++ ) {
            if( ret[ i ] != null ) {
                s = types[ i ] ;
                d = ret[ i ].getClass() ;
                if( s != d ) {
                    if( NUMBER_PRIMITIVE.contains( s ) && NUMBER_PRIMITIVE.contains( d ) ) {
                        ret[ i ] = convertNumberPrimitive( ret[ i ],CONV_TABLE.get( s ),CONV_TABLE.get( d ) ) ;
                    }
                    else if( s == STRING && ( PRIMITIVE.contains( d ) ) ) {
                        ret[ i ] = ret[ i ].toString() ;
                    }
                    else if( d == STRING && ( PRIMITIVE.contains( s ) ) ) {
                        if( s == Boolean.class ) {
                            String str = (( String )args[ i ]).toLowerCase() ;
                            if( str.equals( "true" ) ) {
                                ret[ i ] = Boolean.TRUE ;
                            }
                            else if( str.equals( "false" ) ) {
                                ret[ i ] = Boolean.FALSE ;
                            }
                            else {
                                throw new ClassCastException( "第" + (i+1) + "引数のキャストに失敗しました" ) ;
                            }
                        }
                        else {
                            Integer o = CONV_TABLE.get( s ) ;
                            if( o != null ) {
                                ret[ i ] = convertNumber( o,(String)ret[ i ] ) ;
                            }
                        }
                    }
                }
            }
        }
        return ret ;
    }
    
    /** 指定タイプに対して文字列から、数値変換. **/
    private static final Object convertNumber( int type,String s ) {
        s = s.trim().toLowerCase() ;
        if( s.endsWith( "f" ) || s.endsWith( "l" ) ) {
            s = s.substring( 0,s.length()-1 ) ;
        }
        if( type == 5 || type == 6 ) {
            return convertType( type,s ) ;
        }
        int p = s.indexOf( "." ) ;
        if( p == -1 ) {
            if( s.startsWith( "0x" ) ) {
                s = s.substring( 2 ) ;
                int len = s.length() ;
                if( len > 8 ) {
                    if( len > 16 ) {
                        throw new NumberFormatException( "数値変換に失敗しました" ) ;
                    }
                    long ret = 0L ;
                    for( int i = 0 ; i < len ; i ++ ) {
                        char c = s.charAt( i ) ;
                        if( c >= '1' && c <= '9' ) {
                            ret |= ( (int)(c-'0') << i ) ;
                        }
                        else if( c >= 'a' && c <= 'f' ) {
                            ret |= ( (int)((c-'a')+10) << i ) ;
                        }
                    }
                    switch( type ) {
                        case 0 : return ( byte )(ret&0x00000000000000ffL) ;
                        case 1 : return ( char )(ret&0x000000000000ffffL) ;
                        case 2 : return ( short )(ret&0x000000000000ffffL) ;
                        case 3 : return ( int )(ret&0x00000000ffffffffL) ;
                        case 4 : return ret ;
                    }
                    return null ;
                }
                else {
                    int ret = 0 ;
                    for( int i = 0 ; i < len ; i ++ ) {
                        char c = s.charAt( i ) ;
                        if( c >= '1' && c <= '9' ) {
                            ret |= ( (int)(c-'0') << i ) ;
                        }
                        else if( c >= 'a' && c <= 'f' ) {
                            ret |= ( (int)((c-'a')+10) << i ) ;
                        }
                    }
                    switch( type ) {
                        case 0 : return ( byte )(ret&0x000000ff) ;
                        case 1 : return ( char )(ret&0x0000ffff) ;
                        case 2 : return ( short )(ret&0x0000ffff) ;
                        case 3 : return ret ;
                        case 4 : return ( long )ret ;
                    }
                    return null ;
                }
            }
            return convertType( type,s ) ;
        }
        return convertType( type,s.substring( 0,p ) ) ;
    }
    
    /** 文字列に対して、プリミティブタイプ変換. **/
    private static final Object convertType( int type,String s ) {
        switch( type ) {
            case 0 : return Byte.parseByte( s ) ;
            case 1 : return ( s.length() == 1 ) ? s.charAt( 0 ) :
                (char)( Integer.parseInt( s ) & 0x0000ffff ) ;
            case 2 : return Short.parseShort( s ) ;
            case 3 : return Integer.parseInt( s ) ;
            case 4 : return Long.parseLong( s ) ;
            case 5 : return Float.parseFloat( s ) ;
            case 6 : return Double.parseDouble( s ) ;
        }
        return s ;
    }
    
    /** 数値系プリミティブから、数値系プリミティブに対して、キャスト. **/
    private static final Object convertNumberPrimitive( Object o,int srcType,int destType ) {
        switch( destType ) {
            case 0 :
            {
                byte x = (Byte)o ;
                switch( srcType ) {
                    case 0 :return x ;
                    case 1 :return (char)x;
                    case 2 :return (short)x ;
                    case 3 :return (int)x ;
                    case 4 :return (long)x ;
                    case 5 :return (float)x ;
                    case 6 :return (double)x ;
                }
            }
            case 1 :
            {
                char x = ( Character )o ;
                switch( srcType ) {
                    case 0 :return (byte)x ;
                    case 1 :return x ;
                    case 2 :return (short)x ;
                    case 3 :return (int)x ;
                    case 4 :return (long)x ;
                    case 5 :return (float)x ;
                    case 6 :return (double)x ;
                }
            }
            case 2 :
            {
                short x = (Short)o ;
                switch( srcType ) {
                    case 0 :return (byte)x ;
                    case 1 :return (char)x;
                    case 2 :return x ;
                    case 3 :return (int)x ;
                    case 4 :return (long)x ;
                    case 5 :return (float)x ;
                    case 6 :return (double)x ;
                }
            }
            case 3 :
            {
                int x = (Integer)o ;
                switch( srcType ) {
                    case 0 :return (byte)x ;
                    case 1 :return (char)x;
                    case 2 :return (short)x ;
                    case 3 :return x ;
                    case 4 :return (long)x ;
                    case 5 :return (float)x ;
                    case 6 :return (double)x ;
                }
            }
            case 4 :
            {
                long x = (Long)o ;
                switch( srcType ) {
                    case 0 :return (byte)x ;
                    case 1 :return (char)x;
                    case 2 :return (short)x ;
                    case 3 :return (int)x ;
                    case 4 :return x ;
                    case 5 :return (float)x ;
                    case 6 :return (double)x ;
                }
            }
            case 5 :
            {
                float x = (Float)o ;
                switch( srcType ) {
                    case 0 :return (byte)x ;
                    case 1 :return (char)x;
                    case 2 :return (short)x ;
                    case 3 :return (int)x ;
                    case 4 :return (long)x ;
                    case 5 :return x ;
                    case 6 :return (double)x ;
                }
            }
            case 6 :
            {
                double x = (Double)o ;
                switch( srcType ) {
                    case 0 :return (byte)x ;
                    case 1 :return (char)x;
                    case 2 :return (short)x ;
                    case 3 :return (int)x ;
                    case 4 :return (long)x ;
                    case 5 :return (float)x ;
                    case 6 :return x ;
                }
            }
        }
        return o ;
    }
}

