/*
 * Copyright (c) 2007, to-do. All rights reserved.
 */
#include "util.h"
#include "var.h"
#include "_sym.h"
#include "_ext.h"

/************************************************************
 *
 ************************************************************/
void* array_alloc(exec_t* ex)
{
	return alloc_link(ex, sizeof(array_t), &array_proto);
}
void array_free(exec_t* ex, void *v)
{
	free_link(ex, v);
}
int array_push(exec_t* ex, array_t* arr, var_t* v)
{
	int i = arr->n++;
	arr->v = realloc(arr->v, arr->n * sizeof(var_t));
	copy_var(arr->v + i, v);
	lock_var(ex, arr->v + i);
	return 1;
}
int array_pop(exec_t* ex, array_t* arr, var_t* v)
{
	int i;
	if (arr->n <= 0) {
		if (v) new_null(v);
		return 0;
	}
	i = --arr->n;
	unlock_var(ex, arr->v + i);
	if (v) {
		copy_var(v, arr->v + i);
	} else {
		delete_var(ex, arr->v + i);
	}
	return 1;
}

/************************************************************
 *
 ************************************************************/
int array_get(exec_t* ex, array_t* arr, int i, var_t* v)
{
	if (i < 0) i += arr->n;
	if ((i < 0) || (i >= arr->n)) {
		new_null(v);
		return 0;
	}
	copy_var(v, arr->v + i);
	return 1;
}
int array_del(exec_t* ex, array_t* arr, int i, var_t* v)
{
	if (i < 0) i += arr->n;
	if ((i < 0) || (i >= arr->n)) {
		if (v) new_null(v);
		return 0;
	}
	unlock_var(ex, arr->v + i);
	if (v) {
		copy_var(v, arr->v + i);
	} else {
		delete_var(ex, arr->v + i);
	}
	if (--arr->n > i) {
		memcpy(arr->v + i, arr->v + i + 1,
			(arr->n - i) * sizeof(var_t));
	}
	return 1;
}
int array_set(exec_t* ex, array_t* arr, int i, var_t* v)
{
	if (i < 0) i += arr->n + 1;
	if (i < 0) i = 0;
	if (i >= arr->n) {
		arr->v = realloc(arr->v, (i + 1) * sizeof(var_t));
		if (i > arr->n) {
			memset(arr->v + arr->n, 0, (i - arr->n) * sizeof(var_t));
		}
		arr->n = i + 1;
	} else {
		unlock_var(ex, arr->v + i);
		delete_var(ex, arr->v + i);
	}
	copy_var(arr->v + i, v);
	lock_var(ex, arr->v + i);
	return 1;
}
int array_add(exec_t* ex, array_t* arr, int i, var_t* v)
{
	if (i < 0) i += arr->n + 1;
	if (i < 0) i = 0;
	if (i >= arr->n) {
		arr->v = realloc(arr->v, (i + 1) * sizeof(var_t));
		if (i > arr->n) {
			memset(arr->v + arr->n, 0, (i - arr->n) * sizeof(var_t));
		}
		arr->n = i + 1;
	} else {
		unlock_var(ex, arr->v + i);
		delete_var(ex, arr->v + i);
		arr->v = realloc(arr->v, (arr->n + 1) * sizeof(var_t));
		memmove(arr->v + i + 1, arr->v, (arr->n - i) * sizeof(var_t));
		arr->n++;
	}
	copy_var(arr->v + i, v);
	lock_var(ex, arr->v + i);
	return 1;
}

/************************************************************
 *
 ************************************************************/
void array_trace(void *p)
{
	array_t* arr = p;
	int i;
	for (i = 0; i < arr->n; i++) {
		trace(" %d: %s\n", i, var_toChars(arr->v + i));
	}
}
void array_clear(exec_t* ex, void *p)
{
	array_t* arr = p;
	int i;
	for (i = 0; i < arr->n; i++) {
		unlock_var(ex, arr->v + i);
		delete_var(ex, arr->v + i);
	}
	if (arr->v) {
		free(arr->v);
		arr->v = NULL;
	}
	arr->n = 0;
}
int array_getItem(exec_t* ex, void *p, var_t* k, var_t* v)
{
	return array_get(ex, p, var_toInt(k), v);
}
int array_delItem(exec_t* ex, void *p, var_t* k, var_t* v)
{
	return array_del(ex, p, var_toInt(k), v);
}
int array_setItem(exec_t* ex, void *p, var_t* k, var_t* v)
{
	return array_set(ex, p, var_toInt(k), v);
}
int array_eachItem(exec_t* ex, void *p, int i, var_t* k, var_t* v)
{
	array_t* arr = p;
	if ((i < 0) || (i >= arr->n)) {
		return 0;
	}
	if (k) new_int(k, i);
	if (v) copy_var(v, arr->v + i);
	return 1;
}
int array_setLength(exec_t* ex, void *p, int n)
{
	array_t* arr = p;
	if (n < 0) n += arr->n;
	if (n < 0) n = 0;
	if (n > arr->n) {
		arr->v = realloc(arr->v, n + sizeof(var_t));
		while (n < arr->n) {
			new_null(arr->v + arr->n++);
		}
	} else if (n < arr->n) {
		while (arr->n-- > n) {
			unlock_var(ex, arr->v + arr->n);
			delete_var(ex, arr->v + arr->n);
		}
	}
	return 1;
}
int array_getLength(exec_t* ex, void *p)
{
	return ((array_t*)p)->n;
}
int array_toBool(var_t* v)
{
	return (((array_t*)v->u.p)->n > 0);
}
int array_toChar(var_t* v)
{
	return (((array_t*)v->u.p)->n & 0xFF);
}
int array_toInt(var_t* v)
{
	return ((array_t*)v->u.p)->n;
}
double array_toFloat(var_t* v)
{
	return ((array_t*)v->u.p)->n;
}
int array_toString(var_t* v, char **s, int *n)
{
	*s = "";
	*n = 0;
	return 0;
}

/************************************************************
 *
 ************************************************************/
static int _array_pop(exec_t* ex, array_t* arr, int argc, var_t* argv, var_t* res)
{
	warnif(argc != 0,  "invalid arguments");
	return array_pop(ex, arr, res);
}
static int _array_push(exec_t* ex, array_t* arr, int argc, var_t* argv, var_t* res)
{
	int i, n=0;
	warnif(argc < 1,  "invalid arguments");
	for (i = 0; i < argc; i++) {
		n += array_push(ex, arr, argv + i);
	}
	new_int(res, n);
	return 1;
}
static int _array_shift(exec_t* ex, array_t* arr, int argc, var_t* argv, var_t* res)
{
	warnif(argc != 0,  "invalid arguments");
	return array_del(ex, arr, 0, res);
}
static int _array_unshift(exec_t* ex, array_t* arr, int argc, var_t* argv, var_t* res)
{
	int i, n=0;
	warnif(argc < 1,  "invalid arguments");
	for (i = argc - 1; i >= 0; i--) {
		n += array_add(ex, arr, 0, argv + i);
	}
	new_int(res, n);
	return 1;
}
static int _array_splice(exec_t* ex, array_t* arr, int argc, var_t* argv, var_t* res)
{
	int i,j,n;
	warnif(argc < 1,  "invalid arguments");
	i = var_toInt(argv);
	if (i < 0) i += arr->n;
	warnif ((i < 0) || (i >= arr->n), "invalid argument-0") ;
	if (argc >= 2) {
		n = var_toInt(argv + 1);
	} else {
		n = arr->n - i;
	}
	warnif(n <= 0,  "invalid argument-1");
	new_array(ex, res, 0);
	while (n-- > 0) {
		var_t v;
		array_del(ex, arr, i, &v);
		array_push(ex, res->u.p, &v);
	}
	for (j = 2; j < argc; j++) {
		array_add(ex, arr, i++, argv + j);
	}
	return 1;
}
static int _array_join(exec_t* ex, array_t* arr, int argc, var_t* argv, var_t* res)
{
	int i, n1, n2;
	char *s1, *s2;
	n1 = 0;
	warnif(argc > 1,  "invalid arguments");
	if (argc >= 1) {
		var_toString(argv, &s1, &n1);
	}
	warnif(arr->n <= 0,  "invalid argument-0");
	new_string(ex, res, NULL);
	for (i=0; i < arr->n; i++) {
		if (i && n1) {
			string_add(res->u.p, s1, n1);
		}
		if (var_toString(arr->v + i, &s2, &n2)) {
			string_add(res->u.p, s2, n2);
		}
	}
	return 1;
}
static int _array_reverse(exec_t* ex, array_t* arr, int argc, var_t* argv, var_t* res)
{
	array_t* ar2;
	int i, j, n;
	warnif((n = arr->n) <= 0, "invalid data");
	ar2 = array_alloc(ex);
	ar2->v = malloc((ar2->n = n) * sizeof(var_t));
	for (i = 0, j = n - 1; i < n; i++, j--) {
		copy_var(ar2->v + j, arr->v + i);
		lock_var(ex, ar2->v + j);
	}
	new_array(ex, res, ar2);
	return 1;
}

/************************************************************
 *
 ************************************************************/
static int cmp_N_L(const void *v1, const void *v2)
{
	return var_cmpAsNumber((var_t*)v1, (var_t*)v2);
}
static int cmp_N_R(const void *v1, const void *v2)
{
	return var_cmpAsNumber((var_t*)v2, (var_t*)v1);
}
static int cmp_S_L(const void *v1, const void *v2)
{
	return var_cmpAsString((var_t*)v1, (var_t*)v2);
}
static int cmp_S_R(const void *v1, const void *v2)
{
	return var_cmpAsString((var_t*)v2, (var_t*)v1);
}
static int _array_sort(exec_t* ex, array_t* arr, int argc, var_t* argv, var_t* res)
{
	array_t* ar2;
	char *s;
	int i, c, n, rev, num;
	
	warnif(arr->n <= 0, "invalid data");
	
	rev = num = 0;
	if (argc > 0) {
		warnif (!(s = var_toChars(argv)), "invalid argument-0");
		while ((c = *s++) != 0) {
			switch(tolower(c)) {
			case 'r': rev = 1; break;
			case 'n': num = 1; break;
			case 's': num = 0; break;
			}
		}
	}
	
	ar2 = array_alloc(ex);
	n = ar2->n = arr->n;
	ar2->v = malloc(n * sizeof(var_t));
	memcpy(ar2->v, arr->v, n * sizeof(var_t));
	for (i = 0; i < n; i++) {
		lock_var(ex, ar2->v + i);
	}
	
	if (num) {
		if (rev) {
			qsort(ar2->v, n, sizeof(var_t), cmp_N_R);
		} else {
			qsort(ar2->v, n, sizeof(var_t), cmp_N_L);
		}
	} else {
		if (rev) {
			qsort(ar2->v, n, sizeof(var_t), cmp_S_R);
		} else {
			qsort(ar2->v, n, sizeof(var_t), cmp_S_L);
		}
	}
	
	new_array(ex, res, ar2);
	return 1;
}

/************************************************************
 *
 ************************************************************/
typedef int (*array_func)(exec_t*, array_t*, int, var_t*, var_t*);
static array_func is_array_func(int sym)
{
	switch (sym) {
#include "_array_call.h"
	}
	return NULL;
}
int array_callFunc(exec_t* ex, void* p, int sym, int argc, var_t* argv, var_t* res)
{
	array_func f = is_array_func(sym);
	if (f) {
		return (*f)(ex, p, argc, argv, res);
	}
	return -1;
}
