/*
 * Copyright (c) 2007, to-do. All rights reserved.
 */
#include "util.h"
#include "var.h"
#include "_sym.h"
#include "_ext.h"

typedef struct DB_t  DB_t;
typedef struct DBI_t DBI_t;

struct DBI_t {
	size_t dn;
	size_t up;
	size_t gt;
	size_t lt;
	size_t name;
	size_t data;
};

struct DB_t {
	size_t top;
	size_t bot;
	size_t min;
	size_t max;
	FILE* fp;
	size_t sz;
	size_t cmpi;
	size_t seqi;
	int (*cmpp)(DB_t*, size_t, char*, int);
	int (*seqp)(DB_t*, size_t);
	size_t entry;
	size_t dump;
	size_t tail;
};

/************************************************************
 *
 ************************************************************/
static void* dballoc(DB_t* db, size_t sz)
{
	size_t old, off, prev, next;
	
	if (sz < 4) sz = 4;
	
	if (!db->dump) {
		db->dump = db->tail;
		memset((void*)db + db->dump, 0, 8);
	}
	off = db->dump;
	prev = next = 0;
	
	do {
		if (next) {
			prev = off;
			off = next;
		}
		old  = *((size_t*)((void*)db + off));
		next = *((size_t*)((void*)db + off + 4));
	} while (old && old < sz);
	
	if (!next) {
		assert((old == 0) && (off == db->tail));
		*((size_t*)((void*)db + off)) = sz;
		db->tail = next = off + 4 + sz;
	} else {
		assert((old && old >= sz) && (off != db->tail));
	}
	if (!prev) {
		db->dump = next;
	} else {
		*((size_t*)((void*)db + prev + 4)) = next;
	}
	
	return ((void*)db + off + 4);
}
static void dbfree(DB_t* db, void *p)
{
	*((size_t*)p) = db->dump;
	db->dump = (p - 4) - (void*)db;
}

static void dbread(DB_t* db, size_t off, char **data, int *len)
{
	void *p = (void*)db + off;
	if (len) *len = *((int*)p);
	if (data) *data = p + 4;
}
static size_t dbwrite(DB_t* db, char *data, int len)
{
	void *p = dballoc(db, 4 + len + 1);
	*((int*)p) = len;
	/* strncpy */
	memcpy(p + 4, data, len);
	*(char*)(p + 4 + len) = 0;
	return ((void*)p - (void*)db);
}

/************************************************************
 *
 ************************************************************/
static int dbseq_0(DB_t* db, size_t off)
{
	return (!off ? db->top : ((DBI_t*)((void*)db + off))->dn);
}
static int dbseq_1(DB_t* db, size_t off)
{
	return (!off ? db->bot : ((DBI_t*)((void*)db + off))->up);
}
static int dbseq_2(DB_t* db, size_t off)
{
	return (!off ? db->min : ((DBI_t*)((void*)db + off))->gt);
}
static int dbseq_3(DB_t* db, size_t off)
{
	return (!off ? db->max : ((DBI_t*)((void*)db + off))->lt);
}

static int dbreset(DB_t* db)
{
	if (db->seqi == 0) {
		return 0;
	}
	db->seqi = 0;
	return 1;
}
static int dbseq(DB_t* db, char** name, int *nlen, char** data, int *dlen)
{
	DBI_t* di;
	
	if (!(db->seqi = (*db->seqp)(db, db->seqi))) {
		return 0;
	}
	
	di = (void*)db + db->seqi;
	dbread(db, di->name, name, nlen);
	dbread(db, di->data, data, dlen);
	
	return 1;
}

/************************************************************
 *
 ************************************************************/
static int dbcmp_0(DB_t* db, size_t off, char* s2, int n2)
{
	char *s1;
	int n1,i,r;
	dbread(db, off, &s1, &n1);
	for (i=0; (i < n1) && (i < n2); i++) {
		if ((r = s1[i] - s2[i]) != 0) return r;
	}
	return (n1 - n2);
}
static int dbcmp_1(DB_t* db, size_t off, char* s2, int n2)
{
	char *s1;
	int n1,i,r;
	dbread(db, off, &s1, &n1);
	if ((r = n1 - n2) != 0) return r;
	for (i=0; i < n1; i++) {
		if ((r = s1[i] - s2[i]) != 0) return r;
	}
	return 0;
}
static int dbcmp_2(DB_t* db, size_t off, char* s2, int n2)
{
	char *s1;
	int n1,i,r;
	dbread(db, off, &s1, &n1);
	for (i=0; (i < n1) && (i < n2); i++) {
		if ((r = tolower(s1[i]) - tolower(s2[i])) != 0) return r;
	}
	return (n1 - n2);
}
static int dbcmp_3(DB_t* db, size_t off, char* s2, int n2)
{
	char *s1;
	int n1,i,r;
	dbread(db, off, &s1, &n1);
	if ((r = n1 - n2) != 0) return r;
	for (i=0; i < n1; i++) {
		if ((r = tolower(s1[i]) - tolower(s2[i])) != 0) return r;
	}
	return 0;
}

/************************************************************
 *
 ************************************************************/
static int dbget(DB_t* db, char *name, int nlen, char** data, int *dlen)
{
	DBI_t* di;
	size_t off;
	int r;
	off = db->entry;
	while (off) {
		di = (void*)db + off;
		r = (*db->cmpp)(db, di->name, name, nlen);
		if (r == 0) {
			dbread(db, di->data, data, dlen);
			return 1;
		} else if (r < 0) {
			off = di->gt;
		} else {
			off = di->lt;
		}
	}
	return 0;
}
static int dbset(DB_t* db, char *name, int nlen, char *data, int dlen)
{
	DBI_t *di;
	size_t off, gt, lt;
	int r;
	off = db->entry;
	gt = lt = 0;
	while (off) {
		di = (void*)db + off;
		r = (*db->cmpp)(db, di->name, name, nlen);
		if (!r) {
			break;
		} else if (r < 0) {
			lt = off;
			off = gt ? 0 : di->gt;
		} else {
			gt = off;
			off = lt ? 0 : di->lt;
		}
	}
	if (!off) {
		di = dballoc(db, sizeof(DBI_t));
		off = (void*)di - (void*)db;
		di->lt = lt;
		di->gt = gt;
		
		if (!db->entry) db->entry = off;
		
		if (gt) ((DBI_t*)((void*)db + gt))->lt = off;
		else db->max = off;
		if (lt) ((DBI_t*)((void*)db + lt))->gt = off;
		else db->min = off;
		
		di->up = db->bot;
		di->dn = 0;
		if (di->up) ((DBI_t*)(size_t*)((void*)db + di->up))->dn = off;
		else db->top = off;
		db->bot = off;
		
		di->name = dbwrite(db, name, nlen);
	} else {
		dbfree(db, (void*)db + di->data);
	}
	di->data = dbwrite(db, data, dlen);
	return 1;
}
static int dbdel(DB_t* db, char *name, int nlen)
{
	DBI_t *di;
	size_t off;
	int r;
	off = db->entry;
	while (off) {
		di = (void*)db + off;
		r = (*db->cmpp)(db, di->name, name, nlen);
		if (!r) {
			break;
		} else if (r < 0) {
			off = di->gt;
		} else {
			off = di->lt;
		}
	}
	if (!off) {
		return 0;
	}
	
	if (off == db->entry) db->entry = di->lt ? di->lt : di->gt;
	
	if (di->gt) ((DBI_t*)((void*)db + di->gt))->lt = di->lt;
	else db->max  = di->lt;
	if (di->lt) ((DBI_t*)((void*)db + di->lt))->gt = di->gt;
	else db->min = di->gt;
	
	if (di->up) ((DBI_t*)((void*)db + di->up))->dn = di->dn;
	else db->top = di->dn;
	if (di->dn) ((DBI_t*)((void*)db + di->dn))->up = di->up;
	else db->bot = di->up;
	
	dbfree(db, (void*)db + di->data);
	dbfree(db, (void*)db + di->name);
	dbfree(db, di);
	return 1;
}
/************************************************************
 *
 ************************************************************/
static int dbtrace(DB_t* db)
{
	size_t off = db->dump;
	int i = 0;
	trace("dump:\n");
	while (off) {
		trace("%d: %d: %d\n",
			i++, (int)off, (int)*((size_t*)((void*)db + off)));
		off = *((size_t*)((void*)db + off + 4));
	}
	return i;
}
static int dblength(DB_t* db)
{
	size_t off;
	int len = 0;
	off = db->min;
	while (off) {
		off = ((DBI_t*)((void*)db + off))->gt;
		len++;
	}
	return len;
}
static int dbsync(DB_t* db)
{
	size_t sz;
	int r;
	sz = db->sz;
	r = msync(db, sz, MS_SYNC) == 0;
	return r;
}
static int dbclose(DB_t* db)
{
	FILE* fp;
	size_t sz;
	int r1,r2,r3;
	
	if (0) {
		dbtrace(db);
		dblength(db);
		dbsync(db);
	}
	
	fp = db->fp;
	sz = db->sz;
	r1 = msync(db, sz, MS_SYNC) == 0;
	r2 = munmap(db, sz) == 0;
	r3 = fclose(fp) == 0;
	return (r1 && r2 && r3);
}

/************************************************************
 *
 ************************************************************/
static DB_t* dbopen(char *path, char* mode)
{
	FILE* fp;
	DB_t* db;
	size_t sz;
	int ex, cmp, seq;
	
	sz = 0x10000;
	cmp = seq = 0;
	if (mode) {
		while (*mode != 0) {
			switch (*mode++) {
			case 'n': cmp |= 1; break; /* cmp as number */
			case 'i': cmp |= 2; break; /* cmp ignore case */
			case 'r': seq |= 1; break; /* seq reverse */
			case 's': seq |= 2; break; /* seq sort */
			case 'm': sz = 0x00100000; break; /* 1M */
			case 'f': sz = 0x01000000; break; /* 16M */
			case 'q': sz = 0x10000000; break; /* 250M */
			case 'g': sz = 0x40000000; break; /* 1G */
			}
		}
	}
	
	ex = 0;
	if ((fp = fopen(path, "r+")) != NULL) {
		ex = 1;
		fseek(fp, 0, SEEK_END);
		sz = ftell(fp);
	} else {
		fp = fopen(path, "w+");
		if (!fp) {
			warns("fopen error: %s %s\n", path, strerror(errno));
			return 0;
		}
		fseek(fp, sz + 1, SEEK_SET);
		fputc(0, fp);
	}
	rewind(fp);
	
	db = mmap(0, sz, PROT_READ|PROT_WRITE, MAP_SHARED, fileno(fp), 0);
	if (!db || db==MAP_FAILED) {
		fclose(fp);
		warns("mmap error: %s %s\n", path, strerror(errno));
		return 0;
	}
	
	if (!ex) {
		memset(db, 0, sizeof(DB_t));
		db->tail = sizeof(DB_t);
		db->cmpi = cmp;
	} else {
		cmp = db->cmpi;
		db->seqi = 0;
	}
	
	db->fp = fp;
	db->sz = sz;
	
	switch (cmp) {
	case 0: db->cmpp = dbcmp_0; break;
	case 1: db->cmpp = dbcmp_1; break;
	case 2: db->cmpp = dbcmp_2; break;
	case 3: db->cmpp = dbcmp_3; break;
	}
	
	switch (seq) {
	case 0: db->seqp = dbseq_0; break;
	case 1: db->seqp = dbseq_1; break;
	case 2: db->seqp = dbseq_2; break;
	case 3: db->seqp = dbseq_3; break;
	}
	
	return db;
}

/************************************************************
 *
 ************************************************************/
void* dbm_open(char* path, char* mode)
{
	return dbopen(path, mode);
}

/************************************************************
 *
 ************************************************************/
int dbm_getItem(exec_t* ex, void* p, var_t* k, var_t* v)
{
	char *name, *data;
	int nlen, dlen;
	
	warnif (!var_toString(k, &name, &nlen), "invalid key");
	
	if (!dbget(p, name, nlen, &data, &dlen)) {
		return 0;
	}
	/* new_string_copy(val, data, dlen); */
	unserial_from_bytes(ex, v, data, dlen);
	return 1;
}
int dbm_setItem(exec_t* ex, void* p, var_t* k, var_t* v)
{
	char *name, *data;
	int nlen, dlen, r;
	
	warnif (!var_toString(k, &name, &nlen), "invalid key");
	
	data = NULL;
	dlen = 0;
	r = serial_to_bytes(ex, v, &data, &dlen);
	if (!r) {
		/* case "db[name] = null" */
		r = dbdel(p, name, nlen);
	} else {
		r = dbset(p, name, nlen, data, dlen);
	}
	if (data) free(data);
	return r;
}
int dbm_delItem(exec_t* ex, void* p, var_t* k, var_t* v)
{
	char *name, *data;
	int nlen, dlen;
	
	warnif (!var_toString(k, &name, &nlen), "invalid key") ;
	
	if (v) {
		if (!dbget(p, name, nlen, &data, &dlen)) {
			return 0;
		}
		/* new_string_copy(val, data, dlen); */
		unserial_from_bytes(ex, v, data, dlen);
	}
	return dbdel(p, name, nlen);
}
int dbm_eachItem(exec_t* ex, void* p, int i, var_t* k, var_t* v)
{
	char *name, *data;
	int nlen, dlen;
	if (!i) {
		dbreset(p);
	}
	if (!dbseq(p, &name, &nlen, &data, &dlen)) {
		return 0;
	}
	if (k) {
		new_string_copy(ex, k, name, nlen);
	}
	if (v) {
		/* new_string_copy(v, data, dlen); */
		unserial_from_bytes(ex, v, data, dlen);
	}
	return 1;
}

/************************************************************
 *
 ************************************************************/
static int _dbm_close(exec_t* ex, void* p, int argc, var_t* argv, var_t* res)
{
	new_bool(res, dbclose(p) != 0);
	return 1;
}

/************************************************************
 *
 ************************************************************/
typedef int (*dbm_func)(exec_t*, void*, int, var_t*, var_t*);
static dbm_func is_dbm_func(int sym)
{
	switch (sym) {
#include "_dbm_call.h"
	}
	return NULL;
}
int dbm_callFunc(exec_t* ex, void* p, int sym, int argc, var_t* argv, var_t* res)
{
	dbm_func f = is_dbm_func(sym);
	if (f) {
		return (*f)(ex, p, argc, argv, res);
	}
	return -1;
}

