/*
 * Copyright (c) 2007, to-do. All rights reserved.
 */
#include "util.h"
#include "var.h"
#include "_sym.h"
#include "_ext.h"
#include "_op.h"

/************************************************************
 *
 ************************************************************/
static int read_ushort(char **p)
{
	int v = *((unsigned short*)(*p));
	(*p) += sizeof(unsigned short);
	return v;
}
static int read_int(char **p)
{
	int v = *((int*)(*p));
	(*p) += sizeof(int);
	return v;
}
static double read_float(char **p)
{
	double f = *((double*)(*p));
	(*p) += sizeof(double);
	return f;
}
static int read_chars(char **p, char **s)
{
	(*s) = (*p);
	(*p) = strchr(*p, 0);
	assert((*p) != NULL);
	return ((*p)++ - (*s));
}
static int read_string(char **p, char **s)
{
	int n;
	n = *((int*)(*p));
	(*p) += sizeof(int);
	(*s) = (*p);
	(*p) += n;
	return n;
}
static char* read_offset(char **p)
{
	char *j = (*p);
	j += *((int*)(*p));
	(*p) += sizeof(int);
	return j;
}

/************************************************************
 *
 ************************************************************/
static int ex_sp_push(exec_t* ex, var_t* v)
{
	return array_push(ex, ex->sp, v);
}
static int ex_sp_pop(exec_t* ex, var_t* v)
{
	return array_pop(ex, ex->sp, v);
}
static int ex_sp_get(exec_t* ex, int i, var_t* v)
{
	return array_get(ex, ex->sp, i, v);
}
static int ex_set_res(exec_t* ex, var_t* v)
{
	if (ex->res->type) {
		delete_var(ex, ex->res);
	}
	copy_var(ex->res, v);
	return 1;
}
static int ex_del_res(exec_t* ex, var_t* v)
{
	if (!ex->res->type) {
		v->type = 0;
	} else {
		copy_var(v, ex->res);
		ex->res->type = 0;
	}
	return 1;
}
static int ex_on_goto(exec_t* ex)
{
	var_t v;
	if (!frame_getLocal(ex, ex->fp, ex->req, &v)) {
		return 0;
	}
	ex->req = 0;
	errif(v.type != V_label, "invalid type");
	ex->bc = v.u.p;
	return 1;
}

/************************************************************
 *
 ************************************************************/
static int ex_void(exec_t* ex)
{
	return 1;
}
static int ex_file(exec_t* ex)
{
	read_chars(&ex->bc, &ex->file);
	return 1;
}
static int ex_line(exec_t* ex)
{
	ex->line = read_ushort(&ex->bc);
	return 1;
}
static int ex_names(exec_t* ex)
{
	char *s;
	int n, m, k;
	n = read_int(&ex->bc);
	while (n-- > 0) {
		m = read_chars(&ex->bc, &s);
		k = string_to_symbol(s, m);
	}
	return 1;
}
static int ex_var(exec_t* ex)
{
	var_t v;
	int k;
	k = read_int(&ex->bc);
	new_null(&v);
	frame_setLocal(ex, ex->fp, k, &v);
	return 1;
}
static int ex_label(exec_t* ex)
{
	var_t v;
	int k;
	char* j;
	k = read_int(&ex->bc);
	j = read_offset(&ex->bc);
	new_label(&v, j);
	frame_setLocal(ex, ex->fp, k, &v);
	return 1;
}

/************************************************************
 *
 ************************************************************/
static int ex_jump(exec_t* ex)
{
	char *j;
	j = read_offset(&ex->bc);
	ex->bc = j;
	return 1;
}
static int ex_and(exec_t* ex)
{
	var_t v;
	char* j;
	j = read_offset(&ex->bc);
	ex_sp_pop(ex, &v);
	if (var_toBool(&v) == 0) {
		ex->bc = j;
	}
	return 1;
}
static int ex_or(exec_t* ex)
{
	var_t v;
	char* j;
	j = read_offset(&ex->bc);
	ex_sp_pop(ex, &v);
	if (var_toBool(&v) != 0) {
		ex->bc = j;
	}
	return 1;
}
static int ex_case(exec_t* ex)
{
	var_t v1, v2;
	char *j;
	
	j = read_offset(&ex->bc);
	
	ex_sp_pop(ex, &v2);
	ex_sp_get(ex, -1, &v1);
	
	if (var_cmp(&v1, &v2) != 0) {
		ex->bc = j;
	}
	
	return 1;
}
static int ex_loop(exec_t* ex)
{
	var_t v;
	char *e;
	int i;
	
	e = read_offset(&ex->bc);
	
	ex_sp_pop(ex, &v);
	
	i = ex->lc++;
	
	errif(i < 0, "out of loop range");
	
	if (var_toBool(&v) == 0) {
		ex->bc = e;
	}
	return 1;
}
static int ex_in(exec_t* ex)
{
	var_t v1,v2,v3;
	int i, j, k;
	char *e;
	
	e = read_offset(&ex->bc);
	
	copy_var(&v1, ex->sp->v + ex->sp->n - 1);
	j = (ex->sp->v + ex->sp->n - 2)->u.i;
	k = (ex->sp->v + ex->sp->n - 3)->u.i;
	
	i = ex->lc++;
	
	errif(i < 0, "out of loop range");
	
	if (var_eachItem(ex, &v1, i, &v2, &v3) <= 0) {
		ex->bc = e;
	} else {
		if (k > 0) {
			frame_set(ex, ex->fp, k, &v2);
		}
		if (j > 0) {
			frame_set(ex, ex->fp, j, &v3);
		}
	}
	return 1;
}

/************************************************************
 *
 ************************************************************/
static int ex_args(exec_t* ex)
{
	var_t v;
	int i, j, n;
	n = read_int(&ex->bc);
	errif(ex->argc < n, "invalid arguments");
	for (i = -n, j = 0; (i < 0) && (j < ex->argc); i++, j++) {
		ex_sp_get(ex, i, &v);
		errif(v.type != V_symbol, "invalid type");
		frame_setLocal(ex, ex->fp, v.u.i, ex->argv + j);
	}
	while (n-- > 0) {
		ex_sp_pop(ex, &v);
	}
	return 1;
}
static int ex_catch(exec_t* ex)
{
	var_t v;
	int k;
	k = read_int(&ex->bc);
	ex_del_res(ex, &v);
	if (!k) {
		delete_var(ex, &v);
	} else {
		frame_setLocal(ex, ex->fp, k, &v);
	}
	return 1;
}

/************************************************************
 *
 ************************************************************/
static int ex_goto(exec_t* ex)
{
	ex->req = read_int(&ex->bc);
	if (ex_on_goto(ex)) {
		return 1;
	}
	return K_goto;
}
static int ex_throw(exec_t* ex)
{
	var_t v;
	ex_sp_pop(ex, &v);
	ex_set_res(ex, &v);
	ex->req = K_throw;
	if (ex_on_goto(ex)) {
		return 1;
	}
	return K_throw;
}
static int ex_exit(exec_t* ex)
{
	var_t v;
	ex_sp_pop(ex, &v);
	ex_set_res(ex, &v);
	return K_exit;
}
static int ex_return(exec_t* ex)
{
	var_t v;
	ex_sp_pop(ex, &v);
	ex_set_res(ex, &v);
	return K_return;
}

/************************************************************
 *
 ************************************************************/
static int ex_frameOut(exec_t* ex)
{
	return K_break;
}
static int ex_frameIn(exec_t* ex)
{
	int r;
	
	r = call_frame(ex);
	
	switch (r) {
	case -1:
	case K_throw:
		ex->req = K_throw;
	case K_goto:
		assert(ex->req != 0);
		if (ex_on_goto(ex)) {
			r = 1;
		}
		break;
	case K_break:
		r = 1;
		break;
	}
	
	return r;
}

static int ex_call(exec_t* ex)
{
	var_t ref, res, *argv;
	int i, r, argc;
	
	argc = read_int(&ex->bc);
	errif(argc >= 256, "too large arguments");
	for (i = argc; i > 0; i--) {
		ex_sp_pop(ex, &ref);
	}
	argv = ex->sp->v + ex->sp->n;
	ex_sp_pop(ex, &ref);
	
	res.type = 0;
	
	r = refer_call(ex, &ref, argc, argv, &res);
	
	switch (r) {
	case -1:
	case K_throw:
		ex->req = K_throw;
	case K_goto:
		assert(ex->req != 0);
		ex_set_res(ex, &res);
		if (ex_on_goto(ex)) {
			r = 1;
		}
		break;
	case K_exit:
		ex_set_res(ex, &res);
		break;
	case K_return:
	default:
		ex_sp_push(ex, &res);
		r = 1;
		break;
	}
	
	delete_var(ex, &ref);
	
	return r;
}

/************************************************************
 *
 ************************************************************/
static int ex_class(exec_t* ex)
{
	var_t v;
	int k;
	char* j;
	k = read_int(&ex->bc);
	j = read_offset(&ex->bc);
	new_class(&v, j);
	frame_setLocal(ex, ex->gp, k, &v);
	return 1;
}
static int ex_function(exec_t* ex)
{
	var_t v;
	int k;
	char* j;
	k = read_int(&ex->bc);
	j = read_offset(&ex->bc);
	new_function(&v, j);
	frame_setLocal(ex, ex->gp, k, &v);
	return 1;
}
static int ex_this(exec_t* ex)
{
	var_t v;
	errif( !ex->gp->p , "this global has not parent");
	new_frame(ex, &v, ex->gp);
	ex_sp_push(ex, &v);
	return 1;
}

/************************************************************
 *
 ************************************************************/
static int ex_store(exec_t* ex)
{
	var_t ref, val;
	
	ex_sp_pop(ex, &val);
	ex_sp_get(ex, -1, &ref);
	
	if (!refer_set(ex, &ref, &val)) {
		;
	}
	return 1;
}
static int ex_load(exec_t* ex)
{
	var_t ref, res;
	
	ex_sp_pop(ex, &ref);
	
	if (!refer_get(ex, &ref, &res)) {
		new_null(&res);
	}
	ex_sp_push(ex, &res);
	
	delete_var(ex, &ref);
	return 1;
}
static int ex_delete(exec_t* ex)
{
	var_t ref, res;
	
	ex_sp_pop(ex, &ref);
	if (!refer_del(ex, &ref, &res)) {
		new_null(&res);
	}
	
	ex_sp_push(ex, &res);
	
	delete_var(ex, &ref);
	return 1;
}
static int ex_join(exec_t* ex)
{
	var_t v1, v2, v3, v4;
	ex_sp_pop(ex, &v2);
	ex_sp_pop(ex, &v1);
	if (v1.type != V_refer) {
		new_refer(ex, &v3, 0);
		if (v1.type == V_symbol) {
			new_frame(ex, &v4, ex->fp);
			array_push(ex, v3.u.p, &v4);
		}
		array_push(ex, v3.u.p, &v1);
		copy_var(&v1, &v3);
	}
	array_push(ex, v1.u.p, &v2);
	ex_sp_push(ex, &v1);
	return 1;
}
static int ex_new(exec_t* ex)
{
	return 1;
}

/************************************************************
 *
 ************************************************************/
static int ex_sand(exec_t* ex)
{
	var_t v;
	copy_var(&v, ex->sp->v + (ex->sp->n - 1));
	copy_var(ex->sp->v + (ex->sp->n - 1), ex->sp->v + (ex->sp->n - 2));
	copy_var(ex->sp->v + (ex->sp->n - 2), &v);
	ex_sp_push(ex, &v);
	return 1;
}
static int ex_copy(exec_t* ex)
{
	var_t v;
	ex_sp_get(ex, -1, &v);
	ex_sp_push(ex, &v);
	return 1;
}
static int ex_pop(exec_t* ex)
{
	var_t v;
	ex_sp_pop(ex, &v);
	delete_var(ex, &v);
	return 1;
}
static int ex_echo(exec_t* ex)
{
	var_t v;
	char *s;
	int c, n;
	c = *ex->bc++;
	ex_sp_pop(ex, &v);
	if (var_toString(&v, &s, &n)) {
		fwrite(s, 1, n, g_stdout);
	}
	if (c) {
		fputc(c, g_stdout);
	}
	delete_var(ex, &v);
	return 1;
}
static int ex_doc(exec_t* ex)
{
	char *s;
	int n;
	n = read_string(&ex->bc, &s);
	fwrite(s, 1, n, g_stdout);
	return 1;
}

/************************************************************
 *
 ************************************************************/
static int ex_string(exec_t* ex)
{
	var_t v;
	char *s;
	int n;
	n = read_string(&ex->bc, &s);
	new_string_copy(ex, &v, s, n);
	ex_sp_push(ex, &v);
	return 1;
}
static int ex_regex(exec_t* ex)
{
	var_t v;
	char *s;
	int i,n;
	i = read_int(&ex->bc);
	n = read_chars(&ex->bc, &s);
	new_regex(ex, &v, s, n, i);
	ex_sp_push(ex, &v);
	return 1;
}
static int ex_char(exec_t* ex)
{
	var_t v;
	int c;
	c = read_int(&ex->bc);
	new_char(&v, c);
	ex_sp_push(ex, &v);
	return 1;
}
static int ex_int(exec_t* ex)
{
	var_t v;
	int i;
	i = read_int(&ex->bc);
	new_int(&v, i);
	ex_sp_push(ex, &v);
	return 1;
}
static int ex_float(exec_t* ex)
{
	var_t v;
	double f;
	f = read_float(&ex->bc);
	new_float(&v, f);
	ex_sp_push(ex, &v);
	return 1;
}
static int ex_true(exec_t* ex)
{
	var_t v;
	new_bool(&v, 1);
	ex_sp_push(ex, &v);
	return 1;
}
static int ex_false(exec_t* ex)
{
	var_t v;
	new_bool(&v, 0);
	ex_sp_push(ex, &v);
	return 1;
}
static int ex_null(exec_t* ex)
{
	var_t v;
	new_null(&v);
	ex_sp_push(ex, &v);
	return 1;
}
static int ex_symbol(exec_t* ex)
{
	var_t v;
	int k;
	k = read_int(&ex->bc);
	new_symbol(&v, k);
	ex_sp_push(ex, &v);
	return 1;
}

/************************************************************
 *
 ************************************************************/
static int ex_array(exec_t* ex)
{
	var_t v1, v2;
	int i,n;
	n = read_int(&ex->bc);
	new_array(ex, &v1, 0);
	for (i = -n; i < 0; i++) {
		ex_sp_get(ex, i, &v2);
		array_push(ex, v1.u.p, &v2);
	}
	while (n-- > 0) {
		ex_sp_pop(ex, &v2);
	}
	ex_sp_push(ex, &v1);
	return 1;
}
static int ex_object(exec_t* ex)
{
	var_t v1, v2, v3;
	int i,n;
	n = 2 * read_int(&ex->bc);
	new_object(ex, &v1, 0);
	for (i = -n; i < 0; i += 2) {
		ex_sp_get(ex, i+0, &v2);
		ex_sp_get(ex, i+1, &v3);
		object_set(ex, v1.u.p, var_toSymbol(&v2), &v3);
	}
	while (n-- > 0) {
		ex_sp_pop(ex, &v2);
	}
	ex_sp_push(ex, &v1);
	return 1;
}

/************************************************************
 *
 ************************************************************/
static int ex_match(exec_t* ex)
{
	var_t v1,v2;
	char *s;
	int n;
	ex_sp_pop(ex, &v2);
	ex_sp_pop(ex, &v1);
	if (var_toString(&v1, &s, &n)) {
		n = string_match(ex, s, n, &v2, &v1);
	}
	if (!n) {
		new_bool(&v1, 0);
	}
	ex_sp_push(ex, &v1);
	return 1;
}
static int ex_notmatch(exec_t* ex)
{
	var_t v1,v2;
	char *s;
	int n;
	ex_sp_pop(ex, &v2);
	ex_sp_pop(ex, &v1);
	if (var_toString(&v1, &s, &n)) {
		n = string_match(ex, s, n, &v2, NULL);
	} else {
		n = 0;
	}
	new_bool(&v1, n == 0);
	ex_sp_push(ex, &v1);
	return 1;
}

/************************************************************
 *
 ************************************************************/
static int ex_typeof(exec_t* ex)
{
	var_t v;
	char *s;
	ex_sp_pop(ex, &v);
	s = typeof_var(&v);
	new_string_copy(ex, &v, s, strlen(s));
	ex_sp_push(ex, &v); 
	return 1;
}
static int ex_toBool(exec_t* ex)
{
	var_t v;
	int r;
	ex_sp_pop(ex, &v);
	r = var_toBool(&v);
	new_bool(&v, r);
	ex_sp_push(ex, &v); 
	return 1;
}
static int ex_toChar(exec_t* ex)
{
	var_t v;
	int r;
	ex_sp_pop(ex, &v);
	r = var_toChar(&v);
	new_char(&v, r);
	ex_sp_push(ex, &v); 
	return 1;
}
static int ex_toInt(exec_t* ex)
{
	var_t v;
	int r;
	ex_sp_pop(ex, &v);
	r = var_toInt(&v);
	new_int(&v, r);
	ex_sp_push(ex, &v); 
	return 1;
}
static int ex_toFloat(exec_t* ex)
{
	var_t v;
	double f;
	f = var_toFloat(&v);
	new_float(&v, f);
	ex_sp_push(ex, &v); 
	return 1;
}
static int ex_toString(exec_t* ex)
{
	var_t v;
	char *s;
	int n;
	ex_sp_pop(ex, &v);
	if (var_toString(&v, &s, &n)) {
		new_string_copy(ex, &v, s, n);
	} else {
		new_null(&v);
	}
	ex_sp_push(ex, &v); 
	return 1;
}
static int ex_toNumber(exec_t* ex)
{
	var_t v;
	double f;
	int i;
	ex_sp_pop(ex, &v);
	if (var_toNumber(&v, &f, &i) > 1) {
		new_float(&v, f);
	} else {
		new_int(&v, i);
	}
	ex_sp_push(ex, &v); 
	return 1;
}
static int ex_plus(exec_t* ex)
{
	var_t v;
	double f;
	int i;
	ex_sp_pop(ex, &v);
	if (var_toNumber(&v, &f, &i) > 1) {
		new_float(&v, f);
	} else {
		new_int(&v, i);
	}
	ex_sp_push(ex, &v); 
	return 1;
}
static int ex_minus(exec_t* ex)
{
	var_t v;
	double f;
	int i;
	ex_sp_pop(ex, &v);
	if (var_toNumber(&v, &f, &i) > 1) {
		new_float(&v, - f);
	} else {
		new_int(&v, - i);
	}
	ex_sp_push(ex, &v); 
	return 1;
}
static int ex_not(exec_t* ex)
{
	var_t v;
	int r;
	ex_sp_pop(ex, &v);
	r = ! var_toBool(&v);
	new_int(&v, r);
	ex_sp_push(ex, &v); 
	return 1;
}
static int ex_bitnot(exec_t* ex)
{
	var_t v;
	int r;
	ex_sp_pop(ex, &v);
	r = ~ var_toInt(&v);
	new_int(&v, r);
	ex_sp_push(ex, &v); 
	return 1;
}

/************************************************************
 *
 ************************************************************/
static int ex_inc(exec_t* ex)
{
	var_t v;
	double f;
	int i;
	ex_sp_pop(ex, &v);
	if (var_toNumber(&v, &f, &i) > 1) {
		new_float(&v, f + 1);
	} else {
		new_int(&v, i + 1);
	}
	ex_sp_push(ex, &v);
	return 1;
}
static int ex_dec(exec_t* ex)
{
	var_t v;
	double f;
	int i;
	ex_sp_pop(ex, &v);
	if (var_toNumber(&v, &f, &i) > 1) {
		new_float(&v, f - 1);
	} else {
		new_int(&v, i - 1);
	}
	ex_sp_push(ex, &v);
	return 1;
}

/************************************************************
 *
 ************************************************************/
static int ex_cmp(exec_t* ex)
{
	var_t v1,v2;
	int r;
	ex_sp_pop(ex, &v2);
	ex_sp_pop(ex, &v1);
	r = var_cmp(&v1, &v2);
	new_int(&v1, r);
	ex_sp_push(ex, &v1);
	return 1;
}
static int ex_eq(exec_t* ex)
{
	var_t v1,v2;
	int r;
	ex_sp_pop(ex, &v2);
	ex_sp_pop(ex, &v1);
	r = var_cmp(&v1, &v2) == 0;
	new_int(&v1, r);
	ex_sp_push(ex, &v1);
	return 1;
}
static int ex_ne(exec_t* ex)
{
	var_t v1,v2;
	int r;
	ex_sp_pop(ex, &v2);
	ex_sp_pop(ex, &v1);
	r = var_cmp(&v1, &v2) != 0;
	new_int(&v1, r);
	ex_sp_push(ex, &v1);
	return 1;
}
static int ex_gt(exec_t* ex)
{
	var_t v1,v2;
	int r;
	ex_sp_pop(ex, &v2);
	ex_sp_pop(ex, &v1);
	r = var_cmp(&v1, &v2) > 0;
	new_int(&v1, r);
	ex_sp_push(ex, &v1);
	return 1;
}
static int ex_lt(exec_t* ex)
{
	var_t v1,v2;
	int r;
	ex_sp_pop(ex, &v2);
	ex_sp_pop(ex, &v1);
	r = var_cmp(&v1, &v2) < 0;
	new_int(&v1, r);
	ex_sp_push(ex, &v1);
	return 1;
}

static int ex_mod(exec_t* ex)
{
	var_t v1, v2;
	int r;
	ex_sp_pop(ex, &v2);
	ex_sp_pop(ex, &v1);
	r = var_toInt(&v1) % var_toInt(&v2);
	new_int(&v1, r);
	ex_sp_push(ex, &v1);
	return 1;
}
static int ex_bitand(exec_t* ex)
{
	var_t v1, v2;
	int r;
	ex_sp_pop(ex, &v2);
	ex_sp_pop(ex, &v1);
	r = var_toInt(&v1) & var_toInt(&v2);
	new_int(&v1, r);
	ex_sp_push(ex, &v1);
	return 1;
}
static int ex_bitor(exec_t* ex)
{
	var_t v1, v2;
	int r;
	ex_sp_pop(ex, &v2);
	ex_sp_pop(ex, &v1);
	r = var_toInt(&v1) | var_toInt(&v2);
	new_int(&v1, r);
	ex_sp_push(ex, &v1);
	return 1;
}
static int ex_bitsub(exec_t* ex)
{
	var_t v1, v2;
	int r;
	ex_sp_pop(ex, &v2);
	ex_sp_pop(ex, &v1);
	r = var_toInt(&v1) & (((unsigned int)-1) ^ var_toInt(&v2));
	new_int(&v1, r);
	ex_sp_push(ex, &v1);
	return 1;
}
static int ex_bitxor(exec_t* ex)
{
	var_t v1, v2;
	int r;
	ex_sp_pop(ex, &v2);
	ex_sp_pop(ex, &v1);
	r = var_toInt(&v1) ^ var_toInt(&v2);
	new_int(&v1, r);
	ex_sp_push(ex, &v1);
	return 1;
}
static int ex_bitshR(exec_t* ex)
{
	var_t v1, v2;
	int r;
	ex_sp_pop(ex, &v2);
	ex_sp_pop(ex, &v1);
	r = var_toInt(&v1) >> var_toInt(&v2);
	new_int(&v1, r);
	ex_sp_push(ex, &v1);
	return 1;
}
static int ex_bitshL(exec_t* ex)
{
	var_t v1, v2;
	int r;
	ex_sp_pop(ex, &v2);
	ex_sp_pop(ex, &v1);
	r = var_toInt(&v1) << var_toInt(&v2);
	new_int(&v1, r);
	ex_sp_push(ex, &v1);
	return 1;
}

static int ex_mul(exec_t* ex)
{
	var_t v1, v2;
	double f1,f2;
	int i1,i2,r1,r2;
	ex_sp_pop(ex, &v2);
	ex_sp_pop(ex, &v1);
	r1 = var_toNumber(&v1, &f1, &i1);
	r2 = var_toNumber(&v2, &f2, &i2);
	if ((r1 > 1) || (r2 > 1)) {
		new_float(&v1, f1 * f2);
	} else {
		new_int(&v1, i1 * i2);
	}
	ex_sp_push(ex, &v1);
	return 1;
}
static int ex_div(exec_t* ex)
{
	var_t v1, v2;
	double f1,f2;
	int i1,i2,r1,r2;
	ex_sp_pop(ex, &v2);
	ex_sp_pop(ex, &v1);
	r1 = var_toNumber(&v1, &f1, &i1);
	r2 = var_toNumber(&v2, &f2, &i2);
	if ((r1 > 1) || (r2 > 1)) {
		new_float(&v1, f1 / f2);
	} else {
		new_int(&v1, i1 / i2);
	}
	ex_sp_push(ex, &v1);
	return 1;
}
static int ex_pow(exec_t* ex)
{
	var_t v1, v2;
	double f1,f2;
	int i1,i2,r1,r2;
	ex_sp_pop(ex, &v2);
	ex_sp_pop(ex, &v1);
	r1 = var_toNumber(&v1, &f1, &i1);
	r2 = var_toNumber(&v2, &f2, &i2);
	if ((r1 > 1) || (r2 > 1)) {
		new_float(&v1, pow(f1, f2));
	} else {
		new_int(&v1, (int)pow(i1, i2));
	}
	ex_sp_push(ex, &v1);
	return 1;
}
static int ex_sub(exec_t* ex)
{
	var_t v1, v2;
	double f1,f2;
	int i1,i2,r1,r2;
	ex_sp_pop(ex, &v2);
	ex_sp_pop(ex, &v1);
	r1 = var_toNumber(&v1, &f1, &i1);
	r2 = var_toNumber(&v2, &f2, &i2);
	if ((r1 > 1) || (r2 > 1)) {
		new_float(&v1, f1 - f2);
	} else {
		new_int(&v1, i1 - i2);
	}
	ex_sp_push(ex, &v1);
	return 1;
}
static int ex_add(exec_t* ex)
{
	var_t v1, v2, v3;
	ex_sp_pop(ex, &v2);
	ex_sp_pop(ex, &v1);
	var_add(ex, &v1, &v2, &v3);
	ex_sp_push(ex, &v3);
	return 1;
}

/************************************************************
 *
 ************************************************************/
typedef int (*ex_proc)(exec_t*);

#include "_op2ex.h"

static int ex_main(exec_t* ex)
{
	array_t *oldsp;
	ex_proc p;
	int oldlc;
	int r;
	op_t c;
	
	oldsp = ex->sp;
	oldlc = ex->lc;
	ex->sp = array_alloc(ex);
	ex->lc = 0;
	
	do {
		r = 0;
		if (!(c = *ex->bc++)) {
			/* EOF */
			r = 1;
			break;
		}
		p = op2ex[c];
		if (!p) {
			warns("unknown operand: `%c`\n", c);
		} else {
			r = (*p)(ex);
		}
	} while (r == 1);
	
	array_free(ex, ex->sp);
	ex->sp = oldsp;
	ex->lc = oldlc;
	
	return r;
}

int call_frame(exec_t* ex)
{
	object_t* fp;
	int r;
	
	fp = frame_alloc(ex);
	fp->p = ex->fp;
	ex->fp = fp;
	
	assert(ex->res->type == 0);
	
	r = ex_main(ex);
	
	ex->fp = fp->p;
	
	lock_var(ex, ex->res);
	free_links(ex, fp);
	unlock_var(ex, ex->res);
	
	return r;
}

int call_bytes(exec_t* ex, char* func, char *bc, int argc, var_t* argv, var_t* res)
{
	exec_t old;
	int r;
	
	memcpy(&old, ex, sizeof(exec_t));
	ex->bc   = bc;
	ex->argc = argc;
	ex->argv = argv;
	ex->res  = res;
	ex->func = func;
	
	r = ex_main(ex);
	
	ex->bc   = old.bc;
	ex->argc = old.argc;
	ex->argv = old.argv;
	ex->res  = old.res;
	ex->func = old.func;
	
	return r;
}

int call_function(exec_t* ex, object_t* gp, int sym, char *bc, 
	int argc, var_t* argv, var_t* res)
{
	object_t *oldgp, *oldfp;
	char *s;
	int r;
	
	oldgp = ex->gp;
	oldfp = ex->fp;
	ex->gp = gp;
	ex->fp = gp;
	s = symbol_to_chars(sym);
	
	r = call_bytes(ex, s, bc, argc, argv, res);
	
	ex->gp = oldgp;
	ex->fp = oldfp;
	
	return r;
}

int eval_bytes(exec_t* ex, char *sc, int sz, var_t* res)
{
	char *bc;
	int bn;
	int r;
	
	res->type = 0;
	
	bc = NULL;
	bn = 0;
	
	r = compile_code_to_bytes(NULL, sc, sz, &bc, &bn);
	if (r != 0) {
		if (bc) free(bc);
		/* compile error */
		return 0;
	}
	
	r = call_bytes(ex, "eval", bc, 0, NULL, res);
	
	free(bc);
	
	return r;
}

/************************************************************
 *
 ************************************************************/
void ex_trace(exec_t* ex, char *s, ...)
{
	va_list va;
	
	va_start(va, s);
	vfprintf(g_stdout, s, va);
	va_end(va);
	
	if (ex->func) {
		fprintf(g_stdout, " in %s", ex->func);
	}
	if (ex->file) {
		fprintf(g_stdout, " [%s:%d]", ex->file, ex->line);
	}
	fputc('\n', g_stdout);
}
int exec_bytes(char *bc, int bn)
{
	exec_t ex;
	var_t res;
	object_t* fp;
	char *s;
	int n, r;
	
	/* skip for first void */
	while (*bc && *bc != 1) bc++;
	if (!*bc) {
		/* invalid bytecode */
		return 0;
	}
	
	memset(&res, 0, sizeof(var_t));
	
	memset(&ex, 0, sizeof(exec_t));
	fp = frame_alloc(&ex);
	ex.gp = fp;
	ex.fp = fp;
	
	r = call_bytes(&ex, "main", bc, 0, NULL, &res);
	
	switch (r) {
	case K_exit:
	case K_return:
		if (res.type == V_int) {
			r = res.u.i;
		} else if (var_toString(&res, &s, &n)) {
			fprintf(g_stdout, "%.*s\n", n, s);
			r = -1;
		} else {
			r = 0;
		}
		break;
	case 1:
		r = 0;
		break;
	case K_goto:
		assert(ex.req != 0);
		ex_trace(&ex, "invalid label: %s", symbol_to_chars(ex.req));
		r = -1;
		break;
	case -1:
	case K_throw:
		if (var_toString(&res, &s, &n)) {
			ex_trace(&ex, "throwed: %.*s", n, s);
		} else {
			ex_trace(&ex, "uncatched throw");
		}
		r = -1;
		break;
	case 0:
	default:
		ex_trace(&ex, "invalid result: %d", r);
		r = -1;
		break;
	}
	
	free_links(&ex, fp);
	if (ex.gc != NULL) {
		trace_links(&ex);
	}
	
	return r;
}

int exec_file(char *bcfile)
{
	FILE *fp;
	char *bc;
	int bn, r;
	
	if (!(bc = vmopen(bcfile, &bn, &fp))) {
		warns("vmopen error: %s %s\n", bcfile, strerror(errno));
		return -1;
	}
	
	r = exec_bytes(bc, bn);
	
	vmclose(bc, bn, fp);
	
	return r;
}

int exec_string(char *sc, int sz)
{
	char *bc;
	int bn;
	int r;
	
	bc = NULL;
	bn = 0;
	
	r = compile_code_to_bytes(NULL, sc, sz, &bc, &bn);
	if (r != 0) {
		if (bc) free(bc);
		/* compile error */
		return r;
	}
	
	r = exec_bytes(bc, bn);
	
	free(bc);
	
	return r;
}

int exec_source(char *scfile)
{
	char *bc;
	int bn, r;
	
	bc = NULL;
	bn = 0;
	r = compile_file_to_bytes(scfile, &bc, &bn);
	if (r != 0) {
		if (bc) free(bc);
		return r;
	}
	
	r = exec_bytes(bc, bn);
	
	free(bc);
	
	return r;
}
