/*
 * Copyright (c) 2007, to-do. All rights reserved.
 */
#include "util.h"
#include "var.h"
#include "_sym.h"
#include "_ext.h"

/************************************************************
 *
 ************************************************************/
int string_add(string_t* str, char *s, int n)
{
	if (!s || (n <= 0)) {
		return 0;
	}
	str->s = realloc(str->s, str->n + n + 1);
	/* strn0cpy */
	memcpy(str->s + str->n, s, n);
	str->s[str->n += n] = 0;
	return 1;
}
int string_get(exec_t* ex, string_t* str, int i, var_t* v)
{
	if (i < 0) i += str->n;
	if ((i < 0) || (i >= str->n)) {
		new_null(v);
		return 0;
	}
	new_char(v, str->s[i]);
	return 1;
}
int string_del(exec_t* ex, string_t* str, int i, var_t* v)
{
	if (i < 0) i += str->n;
	if ((i < 0) || (i >= str->n)) {
		new_null(v);
		return 0;
	}
	if (v) {
		new_char(v, str->s[i]);
	}
	memmove(str->s + i, str->s + i + 1, --str->n - i);
	return 1;
}
int string_set(exec_t* ex, string_t* str, int i, var_t* v)
{
	int c;
	c = var_toChar(v);
	if (i < 0) i += str->n + 1;
	if (i < 0) i = 0;
	if (i >= str->n) {
		str->s = realloc(str->s, i + 2);
		if (i > str->n) {
			memset(str->s + str->n, ' ', i - str->n);
		}
		str->s[str->n = i + 1] = 0;
	}
	str->s[i] = c;
	return 1;
}
int string_match(exec_t* ex, char *s1, int n1, var_t* val, var_t* res)
{
	int so[10], eo[10];
	char *s2;
	int i, n2, m;
	var_t v;
	
	if (val->type != V_regex) {
		warnif (!s1 || !n1, "invalid data");
		warnif (!var_toString(val, &s2, &n2), "invalid argument-0") ;
		for (i = 0; (i + n2) <= n1; i++) {
			if (!strncmp(s1 + i, s2, n2)) {
				/* what to return ? */
				if (res) {
					new_string_copy(ex, res, s1 + i, n1 - i);
				}
				return 1;
			}
		}
		return 0;
	}
	
	i = m = 0;
	while ((i = regex_exec(val->u.p, s1, n1, i, so, eo)) > 0) {
		m = i;
		i = eo[0] + (eo[0] == so[0]);
	}
	if (!m) {
		return 0;
	}
	if (!res) {
		return m;
	}
	new_array(ex, res, NULL);
	for (i = 0; i < m; i++) {
		if (eo[i] > so[i]) {
			new_string_copy(ex, &v, s1 + so[i], eo[i] - so[i]);
		} else {
			new_null(&v);
		}
		array_push(ex, res->u.p, &v);
	}
	return m;
}

/************************************************************
 *
 ************************************************************/
void* string_alloc(exec_t* ex)
{
	return alloc_link(ex, sizeof(string_t), &string_proto);
}
void string_free(exec_t* ex, void *v)
{
	free_link(ex, v);
}

/************************************************************
 *
 ************************************************************/
void string_trace(void *p)
{
	string_t* str = p;
	trace(" %.*s\n", str->n, str->s ? str->s : "");
}
void string_clear(exec_t* ex, void *p)
{
	string_t* str = p;
	if (str->s) {
		free(str->s);
		str->s = NULL;
	}
	str->n = 0;
}
int string_getItem(exec_t* ex, void *p, var_t* k, var_t* v)
{
	return string_get(ex, p, var_toInt(k), v);
}
int string_delItem(exec_t* ex, void *p, var_t* k, var_t* v)
{
	return string_del(ex, p, var_toInt(k), v);
}
int string_setItem(exec_t* ex, void *p, var_t* k, var_t* v)
{
	return string_set(ex, p, var_toInt(k), v);
}
int string_eachItem(exec_t* ex, void *p, int i, var_t* k, var_t* v)
{
	string_t* str = p;
	if ((i < 0) || (i >= str->n)) {
		return 0;
	}
	if (k) new_int(k, i);
	if (v) new_char(v, str->s[i]);
	return 1;
}
int string_setLength(exec_t* ex, void *p, int n)
{
	string_t* str = p;
	if (n < 0) n += str->n;
	if (n < 0) n=0;
	if (n > str->n) {
		str->s = realloc(str->s, n + 1);
		memset(str->s + str->n, ' ', n - str->n);
		str->s[str->n = n] = 0;
	} else if (n < str->n) {
		str->s[str->n = n] = 0;
	}
	return 1;
}
int string_getLength(exec_t* ex, void *p)
{
	return ((string_t*)p)->n;
}
int string_toBool(var_t* v)
{
	string_t* str = v->u.p;
	return (str->s && str->n);
}
int string_toChar(var_t* v)
{
	string_t* str = v->u.p;
	if (str->s && str->n) {
		return *str->s;
	}
	return 0;
}
int string_toInt(var_t* v)
{
	string_t* str = v->u.p;
	if (str->s && str->n) {
		return strtol(str->s, NULL, 0);
	}
	return 0;
}
double string_toFloat(var_t* v)
{
	string_t* str = v->u.p;
	if (str->s && str->n) {
		return strtod(str->s, NULL);
	}
	return 0;
}
int string_toString(var_t* v, char **s, int *n)
{
	string_t* str = v->u.p;
	if (str->s && str->n) {
		*s = str->s;
		*n = str->n;
		return *n;
	}
	*s = NULL;
	*n = 0;
	return 0;
}
int string_toNumber(var_t* v, double *f, int *i)
{
	string_t* str = v->u.p;
	if (str->s && str->n) {
		return strtonum(str->s, str->n, f, i);
	}
	*f = *i = 0;
	return 0;
}

/************************************************************
 *
 ************************************************************/
static int _string_trim(exec_t* ex, string_t* str, int argc, var_t* argv, var_t* res)
{
	char *s, *e;
	char *mode = NULL;
	warnif(argc > 1, "invalid arguments");
	if (argc >= 1) {
		mode = var_toChars(argv);
	}
	if (!str->s || !str->n) return 0;
	s = str->s;
	e = s + str->n;
	while ((s < e) && isspace(*s)) s++;
	while ((e > s) && isspace(*(e-1))) e--;
	if (e <= s) return 0;
	new_string_copy(ex, res, s, e - s);
	return 1;
}
static int _string_substr(exec_t* ex, string_t* str, int argc, var_t* argv, var_t* res)
{
	int i, n, m;
	warnif (argc < 1, "invalid arguments") ;
	warnif (!str->s || !str->n, "invalid data");
	i = var_toInt(argv);
	if (i < 0) i += str->n;
	warnif ((i < 0) || (i >= str->n), "invalid argument-0");
	m = n = str->n - i;
	if (argc >= 2) {
		n = var_toInt(argv + 1);
		if (n < 0) n += m;
		if (n > m) n = m;
	} 
	warnif (n <= 0, "invalid argument-1") ;
	new_string_copy(ex, res, str->s + i, n);
	return 1;
}
static int _string_index(exec_t* ex, string_t* str, int argc, var_t* argv, var_t* res)
{
	char *s;
	int n;
	int i;
	new_int(res, -1);
	
	warnif(argc < 1, "invalid arguments");
	warnif(!str->s || !str->n, "invalid data");
	i = 0;
	if (argc >= 2) {
		i = var_toInt(argv + 1);
		if (i < 0) i += str->n;
		warnif((i < 0) || (i >= str->n), "invalid argument-1");
	}
	warnif(!var_toString(argv, &s, &n), "invalid argument-0") ;
	warnif(n > (str->n - i), "invalid argument-0") ;
	
	for (; i <= (str->n - n); i++) {
		if (!strncmp(str->s + i, s, n)) {
			new_int(res, i);
			return 1;
		}
	}
	/* not found */
	return 1;
}
static int _string_rindex(exec_t* ex, string_t* str, int argc, var_t* argv, var_t* res)
{
	char *s;
	int n;
	int i;
	new_int(res, -1);
	
	warnif(argc < 1, "invalid arguments");
	warnif(!str->s || !str->n, "invalid data");
	i = 0;
	if (argc >= 2) {
		i = var_toInt(argv + 1);
		if (i < 0) i += str->n;
		warnif((i < 0) || (i >= str->n), "invalid argument-1");
	}
	warnif(!var_toString(argv, &s, &n), "invalid argument-0") ;
	warnif(n > (str->n - i), "invalid argument-0") ;
	
	i = (str->n - n) - i;
	for (; i >= 0; i--) {
		if (!strncmp(str->s + i, s, n)) {
			new_int(res, i);
			return 1;
		}
	}
	/* not found */
	return 1;
}

static int _string_match(exec_t* ex, string_t* str, int argc, var_t* argv, var_t* res)
{
	int n;
	
	warnif (argc < 1, "invalid arguments") ;
	
	n = string_match(ex, str->s, str->n, argv, res);
	
	return (n > 0);
}
static int _string_replace(exec_t* ex, string_t* str, int argc, var_t* argv, var_t* res)
{
	array_t *arr;
	var_t v;
	int so[10], eo[10];
	char *s, *r;
	int i,j,k,n,m;
	
	warnif (argc < 1, "invalid arguments") ;
	
	if ((argc < 2) || !var_toString(argv + 1, &r, &m)) {
		r = NULL;
		m = 0;
	}
	
	if (argv->type != V_regex)
	{
		warnif (!str->s || !str->n, "invalid data") ;
		warnif (!var_toString(argv, &s, &n), "invalid argument-0") ;
		new_string(ex, res, NULL);
		i = 0;
		for (j = 0; (j + n) <= str->n; j++) {
			if (!strncmp(str->s + j, s, n)) {
				if (j > i) {
					string_add(res->u.p, str->s + i, j - i);
				}
				if (m) {
					string_add(res->u.p, r, m);
				}
				j += n - 1;
				i = j + 1;
			}
		}
		if (i < str->n) {
			string_add(res->u.p, str->s + i, str->n - i);
		}
	}
	else
	{
		new_string(ex, res, NULL);
		i = n = 0;
		while ((j = regex_exec(argv->u.p, str->s, str->n, i, so, eo)) > 0) {
			n = j;
			if (so[0] > i) {
				string_add(res->u.p, str->s + i, so[0] - i);
			}
			i = 0;
			for (j = 0; j < (m - 1); j++) {
				if ((r[j] == '$') && (isdigit(r[j+1]) || (r[j+1]=='+'))) {
					if (j > i) {
						string_add(res->u.p, r + i, j - i);
					}
					k = isdigit(r[++j]) ? (r[j] - '0') : (n - 1);
					if ((k < n) && (eo[k] > so[k])) {
						string_add(res->u.p, str->s + so[k], eo[k] - so[k]);
					}
					i = j + 1;
				}
			}
			if (i < m) {
				string_add(res->u.p, r + i, m - i);
			}
			i = eo[0] + (eo[0] == so[0]);
		}
		if (i < str->n) {
			string_add(res->u.p, str->s + i, str->n - i);
		}
		if (n <= 0) {
			string_free(ex, res->u.p);
			new_null(res);
			return 0;
		}
		arr = array_alloc(ex);
		for (i = 0; i < n; i++) {
			if (eo[i] > so[i]) {
				new_string_copy(ex, &v, str->s + so[i], eo[i] - so[i]);
			} else {
				new_null(&v);
			}
			array_push(ex, arr, &v);
		}
		if (argc >= 3) {
			refer_set(ex, argv + 2, res);
		} else {
			free(str->s);
			memcpy(str, res->u.p, sizeof(string_t));
			memset(res->u.p, 0, sizeof(string_t));
			string_free(ex, res->u.p);
		}
		new_array(ex, res, arr);
	}
	
	return 1;
}
static int _string_split(exec_t* ex, string_t* str, int argc, var_t* argv, var_t* res)
{
	int so[10], eo[10];
	char *s;
	int n,i,j;
	var_t v;
	
	warnif (argc < 1, "invalid arguments");
	warnif (!str->s || !str->n, "invalid data") ;
	
	if (argv->type != V_regex)
	{
		warnif (!var_toString(argv, &s, &n), "invalid argument-0") ;
		new_array(ex, res, NULL);
		i = 0;
		for (j = 0; (j + n) <= str->n; j++) {
			if (!strncmp(str->s + j, s, n)) {
				if (j > i) {
					new_string_copy(ex, res->u.p, str->s + i, j - i);
				} else {
					new_null(&v);
				}
				j += n - 1;
				i = j + 1;
			}
		}
		if (i < str->n) {
			new_string_copy(ex, &v, str->s + i, str->n - i);
			array_push(ex, res->u.p, &v);
		}
	}
	else
	{
		new_array(ex, res, NULL);
		i = 0;
		while ((n = regex_exec(argv->u.p, str->s, str->n, i, so, eo)) > 0) {
			if (so[0] > i) {
				new_string_copy(ex, &v, str->s + i, so[0] - i);
			} else {
				new_null(&v);
			}
			array_push(ex, res->u.p, &v);
			i = eo[0] + (eo[0] == so[0]);
		}
		if (i < str->n) {
			new_string_copy(ex, &v, str->s + i, str->n - i);
			array_push(ex, res->u.p, &v);
		}
	}
	return 1;
}

/************************************************************
 *
 ************************************************************/
typedef int (*string_func)(exec_t*, string_t*, int, var_t*, var_t*);
static string_func is_string_func(int sym)
{
	switch (sym) {
#include "_string_call.h"
	}
	return NULL;
}
int string_callFunc(exec_t* ex, void* p, int sym, int argc, var_t* argv, var_t* res)
{
	string_func f = is_string_func(sym);
	if (f) {
		return (*f)(ex, p, argc, argv, res);
	}
	return -1;
}
