/*
 * Copyright (c) 2007, to-do. All rights reserved.
 */
#include "util.h"
#include "var.h"
#include "_sym.h"
#include "_ext.h"

typedef struct DB_t  DB_t;
typedef struct DBM_t DBM_t;
typedef struct DBI_t DBI_t;

struct DBI_t {
	size_t dn;
	size_t up;
	size_t gt;
	size_t lt;
	size_t name;
	size_t data;
};

struct DBM_t {	
	size_t cmpi;
	size_t seqi;
	
	size_t tail;
	size_t dump;
	size_t entry;
	
	size_t top;
	size_t bot;
	size_t min;
	size_t max;
};

struct DB_t {
	DBM_t* mp;
	FILE* fp;
	size_t sz;
	int (*cmpp)(DBM_t*, size_t, char*, int);
	int (*seqp)(DBM_t*, size_t);
};

/************************************************************
 *
 ************************************************************/
static void* dballoc(DBM_t* mp, size_t sz)
{
	size_t old, off, prev, next;
	
	if (sz < 4) sz = 4;
	
	if (!mp->dump) {
		mp->dump = mp->tail;
		memset((void*)mp + mp->dump, 0, 8);
	}
	off = mp->dump;
	prev = next = 0;
	
	do {
		if (next) {
			prev = off;
			off = next;
		}
		old  = *((size_t*)((void*)mp + off));
		next = *((size_t*)((void*)mp + off + 4));
	} while (old && old < sz);
	
	if (!next) {
		assert((old == 0) && (off == mp->tail));
		*((size_t*)((void*)mp + off)) = sz;
		mp->tail = next = off + 4 + sz;
	} else {
		assert((old && old >= sz) && (off != mp->tail));
	}
	if (!prev) {
		mp->dump = next;
	} else {
		*((size_t*)((void*)mp + prev + 4)) = next;
	}
	
	return ((void*)mp + off + 4);
}
static void dbfree(DBM_t* mp, void *p)
{
	*((size_t*)p) = mp->dump;
	mp->dump = (p - 4) - (void*)mp;
}
static void dbread(DBM_t* mp, size_t off, char **data, int *len)
{
	void *p = (void*)mp + off;
	if (len) *len = *((int*)p);
	if (data) *data = p + 4;
}
static size_t dbwrite(DBM_t* mp, char *data, int len)
{
	void *p = dballoc(mp, 4 + len + 1);
	*((int*)p) = len;
	/* strncpy */
	memcpy(p + 4, data, len);
	*(char*)(p + 4 + len) = 0;
	return ((void*)p - (void*)mp);
}

/************************************************************
 *
 ************************************************************/
static int dbseq_0(DBM_t* mp, size_t off)
{
	return (!off ? mp->top : ((DBI_t*)((void*)mp + off))->dn);
}
static int dbseq_1(DBM_t* mp, size_t off)
{
	return (!off ? mp->bot : ((DBI_t*)((void*)mp + off))->up);
}
static int dbseq_2(DBM_t* mp, size_t off)
{
	return (!off ? mp->min : ((DBI_t*)((void*)mp + off))->gt);
}
static int dbseq_3(DBM_t* mp, size_t off)
{
	return (!off ? mp->max : ((DBI_t*)((void*)mp + off))->lt);
}

static int dbreset(DB_t* db)
{
	if (db->mp->seqi == 0) {
		return 0;
	}
	db->mp->seqi = 0;
	return 1;
}
static int dbseq(DB_t* db, char** name, int *nlen, char** data, int *dlen)
{
	DBI_t* ip;
	
	if (!(db->mp->seqi = (*db->seqp)(db->mp, db->mp->seqi))) {
		return 0;
	}
	
	ip = (void*)db->mp + db->mp->seqi;
	dbread(db->mp, ip->name, name, nlen);
	dbread(db->mp, ip->data, data, dlen);
	
	return 1;
}

/************************************************************
 *
 ************************************************************/
static int dbcmp_0(DBM_t* mp, size_t off, char* s2, int n2)
{
	char *s1;
	int n1,i,r;
	dbread(mp, off, &s1, &n1);
	for (i=0; (i < n1) && (i < n2); i++) {
		if ((r = s1[i] - s2[i]) != 0) return r;
	}
	return (n1 - n2);
}
static int dbcmp_1(DBM_t* mp, size_t off, char* s2, int n2)
{
	char *s1;
	int n1,i,r;
	dbread(mp, off, &s1, &n1);
	if ((r = n1 - n2) != 0) return r;
	for (i=0; i < n1; i++) {
		if ((r = s1[i] - s2[i]) != 0) return r;
	}
	return 0;
}
static int dbcmp_2(DBM_t* mp, size_t off, char* s2, int n2)
{
	char *s1;
	int n1,i,r;
	dbread(mp, off, &s1, &n1);
	for (i=0; (i < n1) && (i < n2); i++) {
		if ((r = tolower(s1[i]) - tolower(s2[i])) != 0) return r;
	}
	return (n1 - n2);
}
static int dbcmp_3(DBM_t* mp, size_t off, char* s2, int n2)
{
	char *s1;
	int n1,i,r;
	dbread(mp, off, &s1, &n1);
	if ((r = n1 - n2) != 0) return r;
	for (i=0; i < n1; i++) {
		if ((r = tolower(s1[i]) - tolower(s2[i])) != 0) return r;
	}
	return 0;
}

/************************************************************
 *
 ************************************************************/
static int dbget(DB_t* db, char *name, int nlen, char** data, int *dlen)
{
	DBI_t* ip;
	size_t off;
	int r;
	off = db->mp->entry;
	while (off) {
		ip = (void*)db->mp + off;
		r = (*db->cmpp)(db->mp, ip->name, name, nlen);
		if (r == 0) {
			dbread(db->mp, ip->data, data, dlen);
			return 1;
		} else if (r < 0) {
			off = ip->gt;
		} else {
			off = ip->lt;
		}
	}
	return 0;
}
static int dbset(DB_t* db, char *name, int nlen, char *data, int dlen)
{
	DBI_t *ip;
	size_t off, gt, lt;
	int r;
	off = db->mp->entry;
	gt = lt = 0;
	while (off) {
		ip = (void*)db->mp + off;
		r = (*db->cmpp)(db->mp, ip->name, name, nlen);
		if (!r) {
			break;
		} else if (r < 0) {
			lt = off;
			off = gt ? 0 : ip->gt;
		} else {
			gt = off;
			off = lt ? 0 : ip->lt;
		}
	}
	if (!off) {
		ip = dballoc(db->mp, sizeof(DBI_t));
		off = (void*)ip - (void*)db->mp;
		ip->lt = lt;
		ip->gt = gt;
		
		if (!db->mp->entry) db->mp->entry = off;
		
		if (gt) ((DBI_t*)((void*)db->mp + gt))->lt = off;
		else db->mp->max = off;
		if (lt) ((DBI_t*)((void*)db->mp + lt))->gt = off;
		else db->mp->min = off;
		
		ip->up = db->mp->bot;
		ip->dn = 0;
		if (ip->up) ((DBI_t*)(size_t*)((void*)db->mp + ip->up))->dn = off;
		else db->mp->top = off;
		db->mp->bot = off;
		
		ip->name = dbwrite(db->mp, name, nlen);
	} else {
		dbfree(db->mp, (void*)db->mp + ip->data);
	}
	ip->data = dbwrite(db->mp, data, dlen);
	return 1;
}
static int dbdel(DB_t* db, char *name, int nlen)
{
	DBI_t *ip;
	size_t off;
	int r;
	off = db->mp->entry;
	while (off) {
		ip = (void*)db->mp + off;
		r = (*db->cmpp)(db->mp, ip->name, name, nlen);
		if (!r) {
			break;
		} else if (r < 0) {
			off = ip->gt;
		} else {
			off = ip->lt;
		}
	}
	if (!off) {
		return 0;
	}
	
	if (off == db->mp->entry) db->mp->entry = ip->lt ? ip->lt : ip->gt;
	
	if (ip->gt) ((DBI_t*)((void*)db->mp + ip->gt))->lt = ip->lt;
	else db->mp->max  = ip->lt;
	if (ip->lt) ((DBI_t*)((void*)db->mp + ip->lt))->gt = ip->gt;
	else db->mp->min = ip->gt;
	
	if (ip->up) ((DBI_t*)((void*)db->mp + ip->up))->dn = ip->dn;
	else db->mp->top = ip->dn;
	if (ip->dn) ((DBI_t*)((void*)db->mp + ip->dn))->up = ip->up;
	else db->mp->bot = ip->up;
	
	dbfree(db->mp, (void*)db->mp + ip->data);
	dbfree(db->mp, (void*)db->mp + ip->name);
	dbfree(db->mp, ip);
	return 1;
}

/************************************************************
 *
 ************************************************************/
static int dbtrace(DB_t* db, FILE* fp)
{
	size_t off = db->mp->dump;
	int i = 0;
	while (off) {
		fprintf(fp, "%d: %d: %d\n",
			i++, (int)off, (int)*((size_t*)((void*)db->mp + off)));
		off = *((size_t*)((void*)db->mp + off + 4));
	}
	return i;
}
static int dblength(DB_t* db)
{
	size_t off;
	int len = 0;
	off = db->mp->min;
	while (off) {
		off = ((DBI_t*)((void*)db->mp + off))->gt;
		len++;
	}
	return len;
}
static int dbsync(DB_t* db)
{
	int r;
	r = msync(db->mp, db->sz, MS_SYNC) == 0;
	return r;
}
static int dbclose(DB_t* db)
{
	int r1,r2,r3;
	
	r1 = r2 = r3 = 0;
	
	if (db->mp) {
		if (0) {
			dblength(db);
			dbsync(db);
		}
		
		r1 = msync(db->mp, db->sz, MS_SYNC) == 0;
		r2 = munmap(db->mp, db->sz) == 0;
		r3 = fclose(db->fp) == 0;
		
		memset(db, 0, sizeof(DB_t));
	}
	
	return (r1 && r2 && r3);
}

/************************************************************
 *
 ************************************************************/
static int dbopen(DB_t* db, char *path, char* mode)
{
	FILE* fp;
	DBM_t* mp;
	size_t sz;
	int ex, cmp, seq;
	
	memset(db, 0, sizeof(DB_t));
	
	sz = 0x10000;
	cmp = seq = 0;
	if (mode) {
		while (*mode != 0) {
			switch (*mode++) {
			case 'n': cmp |= 1; break; /* cmp as number */
			case 'i': cmp |= 2; break; /* cmp ignore case */
			case 'r': seq |= 1; break; /* seq reverse */
			case 's': seq |= 2; break; /* seq sort */
			case 'm': sz = 0x00100000; break; /* 1M */
			case 'f': sz = 0x01000000; break; /* 16M */
			case 'q': sz = 0x10000000; break; /* 250M */
			case 'g': sz = 0x40000000; break; /* 1G */
			}
		}
	}
	
	ex = 0;
	if ((fp = fopen(path, "r+")) != NULL) {
		ex = 1;
		fseek(fp, 0, SEEK_END);
		sz = ftell(fp);
	} else {
		fp = fopen(path, "w+");
		if (!fp) {
			return 0;
		}
		fseek(fp, sz + 1, SEEK_SET);
		fputc(0, fp);
	}
	rewind(fp);
	
	mp = mmap(0, sz, PROT_READ|PROT_WRITE, MAP_SHARED, fileno(fp), 0);
	assert(mp && (mp != MAP_FAILED));
	
	if (!ex) {
		memset(mp, 0, sizeof(DBM_t));
		mp->tail = sizeof(DBM_t);
		mp->cmpi = cmp;
	} else {
		cmp = mp->cmpi;
		mp->seqi = 0;
	}
	
	db->mp = mp;
	db->fp = fp;
	db->sz = sz;
	
	switch (cmp) {
	case 0: db->cmpp = dbcmp_0; break;
	case 1: db->cmpp = dbcmp_1; break;
	case 2: db->cmpp = dbcmp_2; break;
	case 3: db->cmpp = dbcmp_3; break;
	}
	
	switch (seq) {
	case 0: db->seqp = dbseq_0; break;
	case 1: db->seqp = dbseq_1; break;
	case 2: db->seqp = dbseq_2; break;
	case 3: db->seqp = dbseq_3; break;
	}
	
	return 1;
}

/************************************************************
 *
 ************************************************************/
int new_dbm(exec_t* ex, var_t* v, char* path, char* mode)
{
	DB_t *db;
	v->p = NULL;
	db = (DB_t*)alloc_link(ex, sizeof(DB_t), dbm_proto);
	if (!dbopen(db, path, mode)) {
		fprintf(ex->err, "fopen error: %s %s\n", path, strerror(errno));
		free_link(ex, (void*)db);
		return 0;
	}
	v->p = dbm_proto;
	v->u.p = (void*)db;
	return 1;
}

/************************************************************
 *
 ************************************************************/
void dbm_clear(exec_t* ex, void *p)
{
	dbclose(p);
}
void dbm_trace(exec_t* ex, void *p)
{
	dbtrace(p, ex->out);
}

/************************************************************
 *
 ************************************************************/
int dbm_getItem(exec_t* ex, void* p, var_t* k, var_t* v)
{
	char *name, *data;
	int nlen, dlen;
	
	if (!var_toString(ex, k, &name, &nlen)) {
		fprintf(ex->err, "bad key: %.*s\n", nlen, name);
		return 0;
	}
	
	if (!dbget(p, name, nlen, &data, &dlen)) {
		return 0;
	}
	/* new_string_copy(val, data, dlen); */
	unserial_from_bytes(ex, v, data, dlen);
	return 1;
}
int dbm_setItem(exec_t* ex, void* p, var_t* k, var_t* v)
{
	char *name, *data;
	int nlen, dlen, r;
	
	if (!var_toString(ex, k, &name, &nlen)) {
		fprintf(ex->err, "bad key: %.*s\n", nlen, name);
		return 0;
	}
	
	data = NULL;
	dlen = 0;
	r = serial_to_bytes(ex, v, &data, &dlen);
	if (!r) {
		/* case "db[name] = null" */
		r = dbdel(p, name, nlen);
	} else {
		r = dbset(p, name, nlen, data, dlen);
	}
	if (data) free(data);
	return r;
}
int dbm_delItem(exec_t* ex, void* p, var_t* k, var_t* v)
{
	char *name, *data;
	int nlen, dlen;
	
	if (!var_toString(ex, k, &name, &nlen)) {
		fprintf(ex->err, "bad key: %.*s\n", nlen, name);
		return 0;
	}
	
	if (v) {
		if (!dbget(p, name, nlen, &data, &dlen)) {
			return 0;
		}
		/* new_string_copy(val, data, dlen); */
		unserial_from_bytes(ex, v, data, dlen);
	}
	return dbdel(p, name, nlen);
}
int dbm_eachItem(exec_t* ex, void* p, int i, var_t* k, var_t* v)
{
	char *name, *data;
	int nlen, dlen;
	if (!i) {
		dbreset(p);
	}
	if (!dbseq(p, &name, &nlen, &data, &dlen)) {
		return 0;
	}
	if (k) {
		new_string_copy(ex, k, name, nlen);
	}
	if (v) {
		/* new_string_copy(v, data, dlen); */
		unserial_from_bytes(ex, v, data, dlen);
	}
	return 1;
}

/************************************************************
 *
 ************************************************************/
static int _dbm_trace(exec_t* ex, void* p, int argc, var_t* argv, var_t* res)
{
	new_int(res, dbtrace(p, ex->out));
	return 1;
}
static int _dbm_close(exec_t* ex, void* p, int argc, var_t* argv, var_t* res)
{
	new_bool(res, dbclose(p) != 0);
	return 1;
}

/************************************************************
 *
 ************************************************************/
int dbm_callFunc(exec_t* ex, void* p, int sym, int argc, var_t* argv, var_t* res)
{
	switch (sym) {
#include "_dbm_call.h"
	}
	return -1;
}
