/*
 * Copyright (c) 2007, to-do. All rights reserved.
 */
#include "util.h"

/* signal */
#include <signal.h>

/* struct timeval */
#include <sys/time.h>

/* socket */
#include <sys/socket.h>

/* sockaddr_in */
#include <netinet/in.h>

/* inet_aton */
#include <arpa/inet.h>

/* resover */
/* require `-lresolv` flag on Linux. */
#ifdef HAVE_ARPA_NAMESER_COMPAT_H
#include <arpa/nameser_compat.h>
#else
#include <arpa/nameser.h>
#endif

#include <resolv.h>

/* gethostbyname (hostent) */
#include <netdb.h>

/////////////////////////////////////////////////////////
// socket proc
static int g_sockfd = 0;

void connect_timeout(int sig)
{
	if (g_sockfd > 0) close(g_sockfd);
	g_sockfd = 0;
}

static FILE* open_inet(char* host, int port)
{
	static int timeout = 3;
	struct sockaddr_in addr;
	struct hostent *he;
	struct timeval tv; // int32, int32
	int fd, r;
	FILE* fp;
	
	memset(&addr, 0, sizeof(struct sockaddr_in));
	addr.sin_family = AF_INET;
	addr.sin_port = htons(port);
	
	if (inet_aton(host, (void*)&addr.sin_addr)) {
		;
	} else if ((he = gethostbyname(host)) != NULL) {
		addr.sin_addr = *(struct in_addr*)he->h_addr;
	} else {
		return NULL;
	}
	
	assert((fd = socket(AF_INET, SOCK_STREAM, 0)) > 0);
	
	/* set timeout for `read` */
	memset(&tv, 0, sizeof(struct timeval));
	tv.tv_sec = timeout;
	assert(setsockopt(fd, SOL_SOCKET, SO_RCVTIMEO, &tv, sizeof(struct timeval)) == 0);
	
	g_sockfd = fd;
	signal(SIGALRM, connect_timeout);
	alarm(timeout);
	r = connect(fd, (struct sockaddr*)&addr, sizeof(struct sockaddr_in));
	alarm(0);
	signal(SIGALRM, SIG_DFL);
	g_sockfd = 0;
	if (r != 0) {
		close(fd);
		return NULL;
	}
	
	assert((fp = fdopen(fd, "r+b")) != NULL);
	setvbuf(fp, NULL, _IONBF, 0);
	return fp;
}

/**********************************************************************
 *
 **********************************************************************/
static char* read_pop(FILE* fp, char buf[BUFSIZ])
{
	char *s, *e;
	do {
		*buf = 0;
		if (!(s = fgets(buf, BUFSIZ, fp))) {
			sprintf(buf, "fgets error: %s", strerror(errno));
			return NULL;
		}
	} while (*s && (*s != '+') && (*s != '-'));
	e = s + strlen(s);
	while ((e > s) && isspace(*(e - 1))) *(--e) = 0;
	if (*s++ != '+') return NULL;
	while (*s && !isspace(*s)) s++;
	while (isspace(*s)) s++;
	return s;
}

int popmail(char *user, char *pass, FILE* out, FILE* err)
{
	char buf[BUFSIZ];
	char host[MAXDNAME];
	char *s, *e;
	int userLen, passLen, port, n;
	FILE *fp;
	
	port = 110;
	if (!(s = strchr(user, '@'))) {
		fputs("no user\n", err);
		return 0;
	} else {
		userLen = s++ - user;
	}
	if ((e = memchr(user, ':', userLen)) != NULL) {
		if (!pass) {
			pass = e + 1;
			passLen = (user + userLen) - pass;
		}
		userLen = e - user;
	} else {
		if (!pass) {
			fputs("no pass\n", err);
			return 0;
		}
	}
	n = strlen(s);
	if ((e = strchr(s, ':')) != NULL) {
		n = e++ - s;
		port = atoi(e);
	}
	if (n >= MAXDNAME) {
		fprintf(err, "too large domain: %.*s\n", n, s);
		return 0;
	}
	memcpy(host, s, n);
	host[n] = 0;
	
	if (!(fp = open_inet(host, port))) {
		fprintf(err, "open_inet error: %s\n", host);
		return 0;
	}
	
	fprintf(fp, "USER %.*s\r\n", userLen, user);
	if (!read_pop(fp, buf)) {
		fprintf(err, "`user` error: %s\n", buf);
		fclose(fp);
		return 0;
	}
	fprintf(fp, "PASS %.*s\r\n", passLen, pass);
	if (!read_pop(fp, buf)) {
		fprintf(err, "`pass` error: %s\n", buf);
		fclose(fp);
		return 0;
	}
	
	fputs("STAT\r\n", fp);
	if (!(s = read_pop(fp, buf))) {
		fprintf(err, "`stat` error: %s\n", buf);
		fclose(fp);
		return 0;
	}
	if ((n = atoi(s)) <= 0) {
		fputs("QUIT\r\n", fp);
		fclose(fp);
		return 0;
	}
	
	fputs("RETR 1\r\n", fp);
	if (!read_pop(fp, buf)) {
		fprintf(err, "`retr` error: %s\n", buf);
		fputs("RSET\r\n", fp);
		fputs("QUIT\r\n", fp);
		fclose(fp);
		return 0;
	}
	
	while ((s = fgets(buf, BUFSIZ, fp)) != NULL) {
		if ((*s++ == '.') && ((*s == '\r') || (*s == '\n'))) {
			break;
		}
		fputs(buf, out);
	}
	
	fputs("DELE 1\r\n", fp);
	if (!read_pop(fp, buf)) {
		fprintf(err, "`dele` error: %s\n", buf);
		fputs("RSET\r\n", fp);
		fputs("QUIT\r\n", fp);
		fclose(fp);
		return 0;
	}
	
	fputs("QUIT\r\n", fp);
	fclose(fp);
	
	return 1;
}

/**********************************************************************
 *
 **********************************************************************/
static int getmxhost(char* host, char res[MAXDNAME])
{
	unsigned char pk[PACKETSZ];
	unsigned char *s, *e;
	int type, n, c;
	
	if (inet_aton(host, (void*)pk)) {
		strcpy(res, host);
		return 1;
	}
	n = res_query(host, C_IN, T_MX, pk, PACKETSZ);
	if (n < sizeof(HEADER)) {
		return 0;
	}
	e = pk + n;
	s = pk + sizeof(HEADER);
	c = ntohs(((HEADER*)pk)->qdcount);
	if (c <= 0) return 0;
	while (c-- > 0) {
		if ((n = dn_skipname(s, e)) < 0) {
			return 0;
		}
		s += n;
		s += INT16SZ; // type
		s += INT16SZ; // class
	}
	c = ntohs(((HEADER*)pk)->ancount);
	if (c <= 0) return 0;
	while (c-- > 0) {
		if ((n = dn_skipname(s, e)) < 0) {
			return 0;
		}
		s += n;
		GETSHORT(type, s);
		s += INT16SZ; // class
		s += INT32SZ; // TTL
		GETSHORT(n, s);
		if (type == T_MX) {
			assert(n >= INT16SZ);
			s += INT16SZ;
			return (dn_expand(pk, e, s, res, MAXDNAME) > 0);
		}
		s += n;
	}
	return 0;
}

static int read_smtp(FILE* fp, char buf[BUFSIZ], int req)
{
	char *e;
	int res = 0;
	do {
		*buf = 0;
		if (!fgets(buf, BUFSIZ, fp)) {
			sprintf(buf, "fgets error: %s", strerror(errno));
			return 0;
		}
		res = strtol(buf, &e, 10) / 100;
		if (*e == '-') res = - res;
	} while (res == -req);
	e = buf + strlen(buf);
	while ((e > buf) && isspace(*(e - 1))) *(--e) = 0;
	return (res == req);
}

#define MAXMADDR  (MAXDNAME * 2)

static int parse_address(char *s, int n, char addr[MAXMADDR], char host[MAXDNAME])
{
	char *e, *f;
	e = s + n;
	if ((f = memchr(s, '<', e - s)) != NULL) {
		s = ++f;
		while ((s < e) && (*s != '>')) s++;
		n = s - f;
		if (*s == '>') s++;
	} else {
		f = s;
		while ((s < e) && !isspace(*s) && !strchr(";,", *s)) s++;
		n = s - f;
	}
	if ((n <= 0) || (n >= MAXMADDR)) return 0;
	memcpy(addr, f, n);
	addr[n] = 0;
	if (!(f = strchr(addr, '@')) || !*(++f)) return 0;
	if (!getmxhost(f, host)) return 0;
	return 1;
}

int sendmail(char *to, char* data, FILE* err)
{
	char buf[BUFSIZ];
	char toAddr[MAXMADDR], fromAddr[MAXMADDR];
	char toHost[MAXDNAME], fromHost[MAXDNAME];
	char *from, *head, *eoHead, *s, *e, *v;
	int toLen, fromLen;
	int istype, issubj, isto;
	int n;
	FILE* fp;
	
	if (*to == '-') to = NULL;
	toLen = to ? strlen(to) : 0;
	from = NULL;
	fromLen = 0;
	istype = issubj = isto = 0;
	eoHead = NULL;
	
	if ((s = head = data) != NULL) {
		e = s;
		while ((n = strcspn(s, "\r\n")) > 0) {
			e = s + n;
			n = strcspn(s, ": \t\v\f\r\n");
			v = s + n + strspn(s + n, ": \t\v\f");
			if (!strncasecmp("to", s, n)) {
				isto = 1;
				if (!to) toLen = e - (to = v);
			} else if (!strncasecmp("from", s, n)) {
				if (!from) fromLen = e - (from = v);
			} else if (!strncasecmp("subject", s, n)) {
				issubj = 1;
			} else if (!strncasecmp("content-type", s, n)) {
				istype = 1;
			}
			s = e;
			if ((*s == '\r') || (*s == '\n')) {
				if (*(s + 1) == (*s ^ 7)) s++;
				if ((*(++s) == '\r') || (*s == '\n')) {
					break;
				}
			}
		}
		eoHead = e;
		while (isspace(*s)) s++;
		data = s;
	}
	
	if ((fromLen <= 0) || (toLen <= 0)) {
		fputs("bad address\n", err);
		return 0;
	}
	if (data && !issubj) {
		fputs("no subject\n", err);
		return 0;
	}
	
	if (!parse_address(from, fromLen, fromAddr, fromHost)) {
		fprintf(err, "bad sender address: %.*s\n", fromLen, from);
		return 0;
	}
	
	if (!parse_address(to, toLen, toAddr, toHost)) {
		fprintf(err, "bad address: %.*s\n", toLen, to);
		return 0;
	}
	
	if (!(fp = open_inet(toHost, 25))) {
		fprintf(err, "connect error: %s (%s)\n", toHost, strerror(errno));
		return 0;
	}
	
	if (!read_smtp(fp, buf, 2)) {
		fprintf(err, "smtp error: %s\n", buf);
		fclose(fp);
		return 0;
	}
	
	fprintf(fp, "HELO %s\r\n", fromHost);
	if (!read_smtp(fp, buf, 2)) {
		fprintf(err, "`helo` error: %s\n", buf);
		fclose(fp);
		return 0;
	}
	
	fprintf(fp, "MAIL FROM: <%s>\r\n", fromAddr);
	if (!read_smtp(fp, buf, 2)) {
		fprintf(err, "`mail` error: %s\n", buf);
		fclose(fp);
		return 0;
	}
	
	fprintf(fp, "RCPT TO: <%s>\r\n", toAddr);
	if (!read_smtp(fp, buf, 2)) {
		fprintf(err, "`rcpt` error: %s\n", buf);
		fputs("RSET\r\n", fp);
		fputs("QUIT\r\n", fp);
		fclose(fp);
		return 0;
	}
	
	fputs("DATA\r\n", fp);
	if (!read_smtp(fp, buf, 3)) {
		fprintf(err, "`data` error: %s\n", buf);
		fputs("RSET\r\n", fp);
		fputs("QUIT\r\n", fp);
		fclose(fp);
		return 0;
	}
	
	if (!data) {
		fputs("RSET\r\n", fp);
		fputs("QUIT\r\n", fp);
		fclose(fp);
		return 1;
	}
	
	if (!isto) {
		fprintf(fp, "To: %s\r\n", toAddr);
	}
	s = head;
	while ((s < eoHead) && ((n = strcspn(s, "\r\n")) > 0)) {
		fprintf(fp, "%.*s\r\n", n, s);
		s += n;
		while (isspace(*s)) s++;
	}
	if (!istype) {
		if (strchr(data, 0x1b)) {
			fputs("Content-Type: text/plain; charse=iso-2022-jp\r\n", fp);
			fputs("Content-Transfer-Encoding: 7bit\r\n", fp);
		}
	}
	n = strlen(s = data);
	while ((n > 0) && isspace(s[n - 1])) n--;
	fprintf(fp, "Content-Length: %d\r\n", n + 2);
	fputs("\r\n", fp);
	fwrite(s, 1, n, fp);
	fputs("\r\n.\r\n", fp);
	
	if (!read_smtp(fp, buf, 2)) {
		fprintf(err, "`.` error: %s\n", buf);
		fputs("RSET\r\n", fp);
		fputs("QUIT\r\n", fp);
		return 0;
	}
	
	fputs("QUIT\r\n", fp);
	if (!read_smtp(fp, buf, 2)) {
		fprintf(err, "`quit` error: %s\n", buf);
		fclose(fp);
		return 0;
	}
	
	fclose(fp);
	
	return 1;
}

/**********************************************************************
 *
 **********************************************************************/
#ifdef HAVE_OPENSSL_SSL_H
/* ssl client `-lssl` */
#include <openssl/ssl.h>
#endif

int httpget(char *url, char *head, char* data, FILE* out, FILE* err)
{
	char buf[BUFSIZ];
	char host[MAXDNAME];
	FILE *fp, *tmp;
	char *path, *type, *s, *e;
	int port, https, n;
	
	port = 80;
	https = 0;
	
	s = url;
	if (!strncmp(s, "http://", 7)) {
		s += 7;
	} else if (!strncmp(s, "https://", 8)) {
		s += 8;
		port = 443;
		https = 1;
	}
	
	n = strlen(s);
	if ((path = strchr(s, '/')) != NULL) {
		n = path++ - s;
	} else {
		path = "";
	}
	if ((e = memchr(s, ':', n)) != NULL) {
		n = e++ - s;
		port = atoi(e);
	}
	if (n >= MAXDNAME) {
		fprintf(err, "too large host: %s\n", s);
		return 0;
	}
	memcpy(host, s, n);
	host[n] = 0;
	
	if (!(fp = open_inet(host, port))) {
		fprintf(err, "connect error: %s (%s)\n", host, strerror(errno));
		return 0;
	}
	
	tmp = fp;
#ifdef HAVE_OPENSSL_SSL_H
	if (https) {
		tmp = tmpfile();
	}
#endif
	
	fprintf(tmp, "%s /%s HTTP/1.0\r\n", data ? "POST" : "GET", path);
	fprintf(tmp, "Host: %s\r\n", host);
	
	type = "application/x-www-urlencoded";
	if ((s = head) != NULL) {
		while ((n = strcspn(s, "\r\n")) > 0) {
			fprintf(tmp, "%.*s\r\n", n, s);
			if (type && !strncasecmp("content-type", s, strcspn(s, ": \t\v\f\r\n"))) {
				type = NULL;
			}
			s += n;
			while (isspace(*s)) s++;
		}
	}
	if (data) {
		if (!type) {
			fprintf(tmp, "Content-Type: %s\r\n", type);
		}
		fprintf(tmp, "Content-Length: %d\r\n", (int)strlen(data));
		fputs("\r\n", tmp);
		fputs(data, tmp);
	}
	fputs("\r\n", tmp);
	
#ifdef HAVE_OPENSSL_SSL_H
	if (https) {
		SSL_CTX *ctx;
		SSL     *ssx;
		
		SSL_library_init();
		ctx = SSL_CTX_new(SSLv23_client_method());
		ssx = SSL_new(ctx);
		SSL_set_fd(ssx, fileno(fp));
		SSL_connect(ssx);
		
		rewind(tmp);
		while ((n = fread(buf, 1, BUFSIZ - 1, tmp)) > 0) {
			SSL_write(ssx, buf, n);
		}
		fclose(tmp);
		
		while ((n = SSL_read(ssx, buf, BUFSIZ - 1)) > 0) {
			fwrite(buf, 1, n, out);
		}
		SSL_shutdown(ssx);
		SSL_free(ssx);
		SSL_CTX_free(ctx);
		
	} else
#endif
	{
		while ((n = fread(buf, 1, BUFSIZ - 1, fp)) > 0) {
			fwrite(buf, 1, n, out);
		}
	}
	
	fclose(fp);
	
	return 1;
}

/**********************************************************************
 *
 **********************************************************************/
static int read_ftp(FILE* fp, char buf[BUFSIZ], int req)
{
	char *e;
	int res = 0;
	do {
		*buf = 0;
		if (!fgets(buf, BUFSIZ, fp)) {
			sprintf(buf, "fgets error: %s", strerror(errno));
			return 0;
		}
		res = strtol(buf, &e, 10) / 100;
		if (*e == '-') res = - res;
	} while (res == -req);
	e = buf + strlen(buf);
	while ((e > buf) && isspace(*(e - 1))) *(--e) = 0;
	return (res == req);
}

int ftpsite(char *user, char* pass, char *cmd, FILE* err)
{
	char buf[BUFSIZ];
	char host[MAXDNAME];
	char *dir, *arg, *s, *e;
	int userLen, passLen, port, n;
	FILE *fp;
	
	port = 21;
	
	s = cmd;
	while (*s && !isspace(*s)) s++;
	n = s - cmd;
	while (isspace(*s)) s++;
	if (!strncasecmp("mkdir", cmd, n)) {
		cmd = "MKD";
	} else if (!strncasecmp("rmdir", cmd, n)) {
		cmd = "RMD";
	} else if (!strncasecmp("chmod", cmd, n)) {
		cmd = "SITE CHMOD";
	} else if (!strncasecmp("rename", cmd, n)) {
		cmd = NULL; /* rename */
	} else {
		fprintf(err, "unknown command: %s\n", cmd);
	}
	arg = s;
	
	if (!(s = strchr(user, '@'))) {
		s = user;
		userLen = strlen(user = "anonymous");
	} else {
		userLen = s++ - user;
	}
	if ((e = memchr(user, ':', userLen)) != NULL) {
		if (!pass) {
			pass = e + 1;
			passLen = (user + userLen) - pass;
		}
		userLen = e - user;
	} else {
		if (!pass) {
			passLen = strlen(pass = "user@host");
		}
	}
	
	n = strlen(s);
	if ((dir = strchr(s, '/')) != NULL) {
		n = dir++ - s;
	}
	if ((e = memchr(s, ':', n)) != NULL) {
		n = e++ - s;
		port = atoi(e);
	}
	
	if (n >= MAXDNAME) {
		fprintf(err, "too large domain: %.*s\n", n, s);
		return 0;
	}
	memcpy(host, s, n);
	host[n] = 0;
	
	if (!(fp = open_inet(host, port))) {
		fprintf(err, "open_inet error: %s\n", host);
		return 0;
	}
	
	if (!read_ftp(fp, buf, 2)) {
		fprintf(err, "ftp error: %s\n", buf);
		fclose(fp);
		return 0;
	}
	
	fprintf(fp, "USER %.*s\r\n", userLen, user);
	if (!read_ftp(fp, buf, 3)) {
		fprintf(err, "`user` error: %s\n", buf);
		fclose(fp);
		return 0;
	}
	
	fprintf(fp, "PASS %.*s\r\n", passLen, pass);
	if (!read_ftp(fp, buf, 2)) {
		fprintf(err, "`pass` error: %s\n", buf);
		fclose(fp);
		return 0;
	}
	
	if (dir) {
		fprintf(fp, "CWD %s\r\n", dir);
		if (!read_ftp(fp, buf, 2)) {
			fprintf(err, "`chdr` error: %s\n", buf);
			fclose(fp);
			return 0;
		}
	}
	
	if (!cmd) { /* renmae */
		s = arg;
		while (*s && !isspace(*s)) s++;
		n = s - arg;
		fprintf(fp, "RNFR %.*s\r\n", n, arg);
		if (!read_ftp(fp, buf, 3)) {
			fprintf(err, "`RNFR` error: %s\n", buf);
			fclose(fp);
			return 0;
		}
		while (isspace(*s)) s++;
		fprintf(fp, "RNTO %s\r\n", s);
		if (!read_ftp(fp, buf, 2)) {
			fprintf(err, "`RNTO` error: %s\n", buf);
			fclose(fp);
			return 0;
		}
	} else {
		fprintf(fp, "%s %s\r\n", cmd, arg);
		if (!read_ftp(fp, buf, 2)) {
			fprintf(err, "`%s` error: %s\n", cmd, buf);
			fclose(fp);
			return 0;
		}
	}
	
	fputs("QUIT\r\n", fp);
	fclose(fp);
	
	return 1;
}
