/*
 * Copyright (c) 2007, to-do. All rights reserved.
 */
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>

#include "jis.h"

/************************************************************
 *
 ************************************************************/
static unsigned short _sjis_to_jis(int c)
{
	int a,b;
	a = c >> 8;
	b = c & 0xff;
	if (a >= 0xe0) a -= 0x40;
	if (b >= 0x80) b--;
	c = ((a - 0x81) * 0xbc) + (b - 0x40);
	a = (c / 0x5e) + 0x21;
	b = (c % 0x5e) + 0x21;
	return ((a << 8) | b);
}
static unsigned short _jis_to_sjis(int c)
{
	int a,b;
	a = c >> 8;
	b = c & 0xff;
	c = ((a - 0x21) * 0x5e) + (b - 0x21);
	a = (c / 0xbc) + 0x81;
	b = (c % 0xbc) + 0x40;
	if (a >= 0xa0) a += 0x40;
	if (b >= 0x7f) b++;
	return ((a << 8) | b);
}
static inline unsigned short _jis_to_euc(int c)
{
	return (c | 0x8080);
}
static inline unsigned short _euc_to_jis(int c)
{
	return (c & 0x7f7f);
}

/************************************************************
 *
 ************************************************************/
static unsigned short _sjis_to_ucs(int c)
{
	return _jis_to_ucs[_sjis_to_jis(c)];
}
static inline unsigned short _ucs_to_sjis(int c)
{
	return _jis_to_sjis(_ucs_to_jis[c]);
}
static inline unsigned short _ucs_to_euc(int c)
{
	return _jis_to_euc(_ucs_to_jis[c]);
}
static inline unsigned short _euc_to_ucs(int c)
{
	return _jis_to_ucs[_euc_to_jis(c)];
}

/************************************************************
 *
 ************************************************************/
static int is_utf8(char *s, char *e)
{
	int c, n = 0;
	for (; (s < e) && ((c = *s) != 0); s++) {
		if (c <= 0x7e) break;
		if (c >= 0xe0) {
			c = *(++s);
			if (!(c >= 0x80 && c<0xc0)) return 0;
			c = *(++s);
			if (!(c >= 0x80 && c<0xc0)) return 0;
			n++;
		} else if (c >= 0xc0) {
			c = *(++s);
			if (!(c >= 0x80 && c<0xc0)) return 0;
			n++;
		} else {
			return 0;
		}
	}
	return (n > 0);
}
static int is_euc(char *s, char *e)
{
	int c, n = 0;
	for (; (s < e) && ((c = *s) != 0); s++) {
		if (c <= 0x7e) break;
		if ((c >= 0xa1) && (c <= 0xfe)) {
			c = *(++s);
			if (!(c >= 0xa1 && c<=0xfe)) return 0;
			n++;
		} else if (c == 0x8e) {
			c = *(++s);
			if (!(c >= 0xa1 && c<=0xdf)) return 0;
			;
		} else {
			return 0;
		}
	}
	return (n > 0);
}
static int is_sjis(char *s, char *e)
{
	int c, n = 0;
	for (; (s < e) && ((c = *s) != 0); s++) {
		if (c <= 0x7e) break;
		if ((c>=0x81 && c<=0x9f) || (c>=0xe0 && c<=0xfc)) {
			c = *(++s);
			if (!(c>=0x40 && c<=0xfc)) return 0;
			n++;
		} else if (c >= 0xa1 && c<=0xdf) {
			;
		} else {
			return 0;
		}
	}
	return (n > 0);
}

/************************************************************
 *
 ************************************************************/
static unsigned short get_utf8(char **s, int *j)
{
	unsigned short c;
	c = *(*s)++;
	if (c >= 0xe0) {
		c = ((c & 0x0f) << 12) | ((**s & 0x3f) << 6) | (*(*s + 1) & 0x3f);
		(*s) += 2;
	} else if (c >= 0xc0) {
		c = ((c & 0x1f) << 6) | (*(*s)++ & 0x3f);
	}
	return c;
}
static unsigned short get_sjis(char **s, int *j)
{
	unsigned short c;
	c = *(*s)++;
	if ((c >= 0xa1) && (c <= 0xdf)) {
		c = 0xff40 | (c & 0x7f);
	} else if (c >= 0x81) {
		c = _sjis_to_ucs((c << 8) | *(*s)++);
	}
	return c;
}
static unsigned short get_euc(char **s, int *j)
{
	unsigned short c;
	c = *(*s)++;
	if (c == 0x8e) {
		c = 0xff40 | (*(*s)++ & 0x7f);
	} else if (c >= 0xa1) {
		c = _euc_to_ucs((c << 8) | *(*s)++);
	}
	return c;
}
static unsigned short get_jis(char **s, int *j)
{
	unsigned short c;
	c = *(*s)++;
	if (c == 0x1b) {
		*j = (**s == '$') ? 0x80 : (*(*s + 1) == 'I');
		(*s) += 2;
		c = *(*s)++;
	}
	if (*j & 0x80) {
		c = _jis_to_ucs[(c << 8) | *(*s)++];
	} else if (*j) {
		c = 0xff40 | c;
	}
	return c;
}

/************************************************************
 *
 ************************************************************/
static int put_utf8(char *d, unsigned short c, int *j)
{
	if (!c) return 0;
	if (c >= 0x800) {
		*d++ = 0xe0 | ((c >> 12) & 0x0f);
		*d++ = 0x80 | ((c >> 6) & 0x3f);
		*d++ = 0x80 | (c & 0x3f);
		return 3;
	} else if (c >= 0x80) {
		*d++ = 0xc0 | ((c >> 6) & 0x1f);
		*d++ = 0x80 | (c & 0x3f);
		return 2;
	}
	*d++ = c;
	return 1;
}
static int put_sjis(char *d, unsigned short c, int *j)
{
	if (!c) return 0;
	if ((c >= 0xff61) && (c <= 0xff9f)) {
		*d++ = 0x80 | (c & 0x3f);
		return 1;
	} else if (c >= 0x80) {
		c = _ucs_to_sjis(c);
		*d++ = c >> 8;
		*d++ = c & 0xff;
		return 2;
	}
	*d++ = c;
	return 1;
}
static int put_euc(char *d, unsigned short c, int *j)
{
	if (!c) return 0;
	if ((c >= 0xff61) && (c <= 0xff9f)) {
		*d++ = 0x8e;
		*d++ = 0x80 | (c & 0x3f);
		return 2;
	} else if (c >= 0x80) {
		c = _ucs_to_euc(c);
		*d++ = c >> 8;
		*d++ = c & 0xff;
		return 2;
	}
	*d++ = c;
	return 1;
}
static int put_jis(char *d, unsigned short c, int *j)
{
	int n = 0;
	if ((c >= 0xff61) && (c <= 0xff9f)) {
		if (!(*j & 0x7f)) {
			*d++ = 0x1b; *d++ = '('; *d++ = 'B';
			*j = 1;
			n += 3;
		}
		if (*j == 1) {
			*j = c & 0x3f;
		} else {
			*d++ = *j;
			*d++ = c & 0x3f;
			*j = 1;
			n += 2;
		}
	} else if (c >= 0x80) {
		if (!(*j & 0x80)) {
			*d++ = 0x1b; *d++ = '$'; *d++ = 'B';
			*j = 0x80;
			n += 3;
		}
		c = _ucs_to_jis[c];
		*d++ = c >> 8;
		*d++ = c & 0xff;
		n += 2;
	} else {
		if (*j) {
			*d++ = 0x1b; *d++ = '('; *d++ = 'B';
			*j = 0;
			n += 3;
		}
		if (c) {
			*d++ = c;
			n++;
		}
	}
	return n;
}

/************************************************************
 *
 ************************************************************/
int jcodeof(char *s, int n)
{
	char *e = s + n;
	int c;
	for (; (s < e) && ((c = *s) != 0); s++) {
		if (c == 0x1b) return 'j';
		else if (c >= 0x80) break;
	}
	if (!*s) return 0;
	if (is_utf8(s, e)) return 'u';
	if (is_euc(s, e)) return 'e';
	if (is_sjis(s, e)) return 's';
	return 's';
}

int jconv(char **d, char *s, int n, int j)
{
	int (*pc)(char*, unsigned short, int*);
	unsigned short (*gc)(char**, int*);
	char *e;
	int c;
	
	if (!s || !*s || (n<=0)) {
		*d = NULL;
		return 0;
	}
	
	c = jcodeof(s, n);
	switch (c) {
	case 'e': gc = get_euc;  break;
	case 'j': gc = get_jis;  break;
	case 's': gc = get_sjis; break;
	case 'u': gc = get_utf8; break;
	default:  gc = NULL; c = 0; break;
	}
	
	j = tolower(j);
	switch (j) {
	case 'j': pc = put_jis;  break;
	case 'e': pc = put_euc;  break;
	case 's': pc = put_sjis; break;
	default:  pc = put_utf8; j = 'u'; break;
	}
	
	if (!c || (c == j)) {
		memcpy(*d = calloc(n + 1, 1), s, n);
		return n;
	}
	
	*d = calloc((n * 4) + 1, 1);
	e = s + n;
	n = j = 0;
	while ((s < e) && ((c = (*gc)(&s, &j)) != 0)) {
		n += (*pc)(*d + n, c, &j);
	}
	n += (*pc)(*d + n, 0, &j);
	*(*d + n) = 0;
	
	return n;
}

int jtrim(char **d, char *s, int n)
{
	unsigned short (*gc)(char**, int*);
	char *e;
	int c, j;
	
	if (!s || !*s || (n<=0)) {
		*d = NULL;
		return 0;
	}
	
	c = jcodeof(s, n);
	switch (c) {
	case 'e': gc = get_euc;  break;
	case 'j': gc = get_jis;  break;
	case 's': gc = get_sjis; break;
	default:  gc = get_utf8; break;
	}
	
	*d = calloc((n * 2) + 1, 1);
	e = s + n;
	n = j = 0;
	while ((s < e) && ((c = (*gc)(&s, &j)) != 0)) {
		if (c == 0x3000) {
			c = ' ';
		} else if ((c >= 0xff00) && (c <= 0xff5e)) {
			c = (c & 0xff) + 0x20;
		}
		if (!n && isspace(c)) continue;
		else if ((c == 0xa) || (c == 0xd)) {
			if (*s == (c ^ 7)) s++;
			c = 0xa;
		}
		n += put_utf8(*d + n, c, &j);
	}
	while ((n > 0) && isspace(*(*d + (n - 1)))) n--;
	*(*d + n) = 0;
	
	return n;
}

