/*
 * Copyright (c) 2007, to-do. All rights reserved.
 */
#include "util.h"
#include "var.h"
#include "_sym.h"
#include "_ext.h"

/************************************************************
 *
 ************************************************************/
void* array_alloc(exec_t* ex)
{
	return alloc_link(ex, sizeof(array_t), array_proto);
}
void array_free(exec_t* ex, void *v)
{
	free_link(ex, v);
}
int array_push(exec_t* ex, array_t* arr, var_t* v)
{
	int i = arr->n++;
	arr->v = realloc(arr->v, arr->n * sizeof(var_t));
	copy_var(arr->v + i, v);
	lock_var(ex, arr->v + i);
	return 1;
}
int array_pop(exec_t* ex, array_t* arr, var_t* v)
{
	int i;
	if (arr->n <= 0) {
		if (v) new_null(v);
		return 0;
	}
	i = --arr->n;
	unlock_var(ex, arr->v + i);
	if (v) {
		copy_var(v, arr->v + i);
	} else {
		delete_var(ex, arr->v + i);
	}
	return 1;
}

/************************************************************
 *
 ************************************************************/
int array_get(exec_t* ex, array_t* arr, int i, var_t* v)
{
	if (i < 0) i += arr->n;
	if ((i < 0) || (i >= arr->n)) {
		new_null(v);
		return 0;
	}
	copy_var(v, arr->v + i);
	return 1;
}
int array_del(exec_t* ex, array_t* arr, int i, var_t* v)
{
	if (i < 0) i += arr->n;
	if ((i < 0) || (i >= arr->n)) {
		if (v) new_null(v);
		return 0;
	}
	unlock_var(ex, arr->v + i);
	if (v) {
		copy_var(v, arr->v + i);
	} else {
		delete_var(ex, arr->v + i);
	}
	if (--arr->n > i) {
		memcpy(arr->v + i, arr->v + i + 1,
			(arr->n - i) * sizeof(var_t));
	}
	return 1;
}
int array_set(exec_t* ex, array_t* arr, int i, var_t* v)
{
	if (i < 0) i += arr->n + 1;
	if (i < 0) i = 0;
	if (i >= arr->n) {
		arr->v = realloc(arr->v, (i + 1) * sizeof(var_t));
		if (i > arr->n) {
			memset(arr->v + arr->n, 0, (i - arr->n) * sizeof(var_t));
		}
		arr->n = i + 1;
	} else {
		unlock_var(ex, arr->v + i);
		delete_var(ex, arr->v + i);
	}
	copy_var(arr->v + i, v);
	lock_var(ex, arr->v + i);
	return 1;
}
int array_add(exec_t* ex, array_t* arr, int i, var_t* v)
{
	if (i < 0) i += arr->n + 1;
	if (i < 0) i = 0;
	if (i >= arr->n) {
		arr->v = realloc(arr->v, (i + 1) * sizeof(var_t));
		if (i > arr->n) {
			memset(arr->v + arr->n, 0, (i - arr->n) * sizeof(var_t));
		}
		arr->n = i + 1;
	} else {
		unlock_var(ex, arr->v + i);
		delete_var(ex, arr->v + i);
		arr->v = realloc(arr->v, (arr->n + 1) * sizeof(var_t));
		memmove(arr->v + i + 1, arr->v, (arr->n - i) * sizeof(var_t));
		arr->n++;
	}
	copy_var(arr->v + i, v);
	lock_var(ex, arr->v + i);
	return 1;
}

/************************************************************
 *
 ************************************************************/
void array_trace(exec_t* ex, void *p)
{
	array_t* arr = p;
	int i;
	for (i = 0; i < arr->n; i++) {
		fprintf(ex->out, " %d: %s\n", i, var_toChars(ex, arr->v + i));
	}
}
void array_clear(exec_t* ex, void *p)
{
	array_t* arr = p;
	int i;
	for (i = 0; i < arr->n; i++) {
		unlock_var(ex, arr->v + i);
		delete_var(ex, arr->v + i);
	}
	if (arr->v) {
		free(arr->v);
		arr->v = NULL;
	}
	arr->n = 0;
}
int array_getItem(exec_t* ex, void *p, var_t* k, var_t* v)
{
	return array_get(ex, p, var_toInt(ex, k), v);
}
int array_delItem(exec_t* ex, void *p, var_t* k, var_t* v)
{
	return array_del(ex, p, var_toInt(ex, k), v);
}
int array_setItem(exec_t* ex, void *p, var_t* k, var_t* v)
{
	return array_set(ex, p, var_toInt(ex, k), v);
}
int array_eachItem(exec_t* ex, void *p, int i, var_t* k, var_t* v)
{
	array_t* arr = p;
	if ((i < 0) || (i >= arr->n)) {
		return 0;
	}
	if (k) new_int(k, i);
	if (v) copy_var(v, arr->v + i);
	return 1;
}
int array_setLength(exec_t* ex, void *p, int n)
{
	array_t* arr = p;
	if (n < 0) n += arr->n;
	if (n < 0) n = 0;
	if (n > arr->n) {
		arr->v = realloc(arr->v, n + sizeof(var_t));
		while (n < arr->n) {
			new_null(arr->v + arr->n++);
		}
	} else if (n < arr->n) {
		while (arr->n-- > n) {
			unlock_var(ex, arr->v + arr->n);
			delete_var(ex, arr->v + arr->n);
		}
	}
	return 1;
}
int array_getLength(exec_t* ex, void *p)
{
	return ((array_t*)p)->n;
}
int array_toBool(exec_t* ex, var_t* v)
{
	return (((array_t*)v->u.p)->n > 0);
}
int array_toChar(exec_t* ex, var_t* v)
{
	return (((array_t*)v->u.p)->n & 0xFF);
}
int array_toInt(exec_t* ex, var_t* v)
{
	return ((array_t*)v->u.p)->n;
}
double array_toFloat(exec_t* ex, var_t* v)
{
	return ((array_t*)v->u.p)->n;
}
int array_toString(exec_t *ex, var_t* v, char **s, int *n)
{
	*s = "";
	*n = 0;
	return 0;
}

/************************************************************
 *
 ************************************************************/
static int _array_pop(exec_t* ex, array_t* arr, int argc, var_t* argv, var_t* res)
{
	exerrif(ex, argc != 0,  "bad arguments");
	return array_pop(ex, arr, res);
}
static int _array_push(exec_t* ex, array_t* arr, int argc, var_t* argv, var_t* res)
{
	int i, n=0;
	exerrif(ex, argc < 1,  "bad arguments");
	for (i = 0; i < argc; i++) {
		n += array_push(ex, arr, argv + i);
	}
	new_int(res, n);
	return 1;
}
static int _array_shift(exec_t* ex, array_t* arr, int argc, var_t* argv, var_t* res)
{
	exerrif(ex, argc != 0,  "bad arguments");
	return array_del(ex, arr, 0, res);
}
static int _array_unshift(exec_t* ex, array_t* arr, int argc, var_t* argv, var_t* res)
{
	int i, n=0;
	exerrif(ex, argc < 1,  "bad arguments");
	for (i = argc - 1; i >= 0; i--) {
		n += array_add(ex, arr, 0, argv + i);
	}
	new_int(res, n);
	return 1;
}
static int _array_splice(exec_t* ex, array_t* arr, int argc, var_t* argv, var_t* res)
{
	int i,j,n;
	exerrif(ex, argc < 1,  "bad arguments");
	i = var_toInt(ex, argv);
	if (i < 0) i += arr->n;
	exerrif(ex, (i < 0) || (i >= arr->n), "bad argument-0") ;
	if (argc >= 2) {
		n = var_toInt(ex, argv + 1);
	} else {
		n = arr->n - i;
	}
	exerrif(ex, n <= 0,  "bad argument-1");
	new_array(ex, res, 0);
	while (n-- > 0) {
		var_t v;
		array_del(ex, arr, i, &v);
		array_push(ex, res->u.p, &v);
	}
	for (j = 2; j < argc; j++) {
		array_add(ex, arr, i++, argv + j);
	}
	return 1;
}
static int _array_join(exec_t* ex, array_t* arr, int argc, var_t* argv, var_t* res)
{
	int i, n1, n2;
	char *s1, *s2;
	n1 = 0;
	exerrif(ex, argc > 1,  "bad arguments");
	if (argc >= 1) {
		var_toString(ex, argv, &s1, &n1);
	}
	exerrif(ex, arr->n <= 0,  "bad argument-0");
	new_string(ex, res, NULL);
	for (i=0; i < arr->n; i++) {
		if (i && n1) {
			string_add(res->u.p, s1, n1);
		}
		if (var_toString(ex, arr->v + i, &s2, &n2)) {
			string_add(res->u.p, s2, n2);
		}
	}
	return 1;
}
static int _array_reverse(exec_t* ex, array_t* arr, int argc, var_t* argv, var_t* res)
{
	array_t* ar2;
	int i, j, n;
	exerrif(ex, (n = arr->n) <= 0, "bad data");
	ar2 = array_alloc(ex);
	ar2->v = malloc((ar2->n = n) * sizeof(var_t));
	for (i = 0, j = n - 1; i < n; i++, j--) {
		copy_var(ar2->v + j, arr->v + i);
		lock_var(ex, ar2->v + j);
	}
	new_array(ex, res, ar2);
	return 1;
}

/************************************************************
 *
 ************************************************************/
static void qsort_vars(exec_t* ex, var_t* vars, int first, int last,
	int (*cmp)(exec_t*, var_t*, var_t*), int rev)
{
	var_t mid, tmp;
	int low, high;
	if (first >= last) {
		return;
	}
	low  = first;
	high = last;
	memcpy(&mid, vars + ((first + last) / 2), sizeof(var_t));
	do {
		if (rev) {
			while ((*cmp)(ex, vars + low,  & mid) > 0) low++;
			while ((*cmp)(ex, vars + high, & mid) < 0) high--;
		} else {
			while ((*cmp)(ex, vars + low,  & mid) < 0) low++;
			while ((*cmp)(ex, vars + high, & mid) > 0) high--;
		}
		if (low <= high) {
			if (low < high) {
				memcpy(&tmp, vars + low, sizeof(var_t));
				memcpy(vars + low,  vars + high, sizeof(var_t));
				memcpy(vars + high, &tmp, sizeof(var_t));
			}
			low++;
			high--;
		}
	} while (low <= high);
	qsort_vars(ex, vars, first, high, cmp, rev);
	qsort_vars(ex, vars, low,   last, cmp, rev);
}
static int _array_sort(exec_t* ex, array_t* arr, int argc, var_t* argv, var_t* res)
{
	array_t* ar2;
	char *s;
	int i, c, n, rev, num;
	int (*cmp)(exec_t*, var_t*, var_t*);
	
	exerrif(ex, arr->n <= 0, "bad data");
	
	rev = num = 0;
	if (argc > 0) {
		exerrif(ex, !(s = var_toChars(ex, argv)), "bad argument-0");
		while ((c = *s++) != 0) {
			switch(tolower(c)) {
			case 'r': rev = 1; break;
			case 'n': num = 1; break;
			case 's': num = 0; break;
			}
		}
	}
	
	if (num) {
		cmp = cmp_vars_asNumber;
	} else {
		cmp = cmp_vars_asString;
	}
	
	ar2 = array_alloc(ex);
	n = ar2->n = arr->n;
	ar2->v = malloc(n * sizeof(var_t));
	memcpy(ar2->v, arr->v, n * sizeof(var_t));
	for (i = 0; i < n; i++) {
		lock_var(ex, ar2->v + i);
	}
	
	qsort_vars(ex, ar2->v, 0, n - 1, cmp, rev);
	
	new_array(ex, res, ar2);
	return 1;
}

/************************************************************
 *
 ************************************************************/
int array_callFunc(exec_t* ex, void* p, int sym, int argc, var_t* argv, var_t* res)
{
	switch (sym) {
#include "_array_call.h"
	}
	return -1;
}
