/*
 * Copyright (c) 2007, to-do. All rights reserved.
 */
#include "util.h"
#include "var.h"
#include "_sym.h"
#include "_ext.h"

/************************************************************
 *
 ************************************************************/
static int skip_brace(char **s, char *e, int q);

static int issymbols(char *s, int n)
{
	if (!n-- || !(isalpha(*s) || (*s == '_'))) return 0;
	for (s++; n-- > 0; s++) {
		if (!(isalnum(*s) || (*s == '_'))) return 0;
	}
	return 1;
}
static int skip_symbol(char **s, char *e)
{
	int c;
	if ((*s >= e) || !(isalpha(**s) || (**s == '_'))) return 0;
	while ((*s < e) && (isalnum(**s) || (**s == '_'))) (*s)++;
	if ((*s < e) && ((c = **s) != 0)) {
		if (c == '.') {
			(*s)++;
			if (!skip_symbol(s, e)) return 0;
			return c;
		}
		else if ((c == '[') || (c =='(')) {
			(*s)++;
			if (!skip_brace(s, e, c)) return 0;
			return c;
		}
	}
	return 1;
}
static int skip_quote(char **s, char *e, int q)
{
	while ((*s < e) && **s && (**s != q)) {
		if ((*(*s)++ == '\\') && **s) (*s)++;
	}
	if ((*s >= e) || (**s != q)) return 0;
	(*s)++;
	return 1;
}
static int skip_number(char **s, char *e)
{
	if (strchr("+-.", **s)) (*s)++;
	if ((*s >= e) || !isdigit(**s)) return 0;
	(*s)++;
	while ((*s < e) && (isalnum(**s) || (**s == '.'))) (*s)++;
	return 1;
}
static int skip_space(char** s, char *e)
{
	while ((*s < e) && isspace(**s)) (*s)++;
	return ((*s < e) ? **s : 0);
}
static int skip_brace(char **s, char *e, int q)
{
	int c;
	switch (q) {
	case '{': q='}'; break;
	case '[': q=']'; break;
	case '(': q=')'; break;
	}
	if (!skip_space(s, e)) return 0;
	else if ((c == '"') || (c == '\'')) {
		(*s)++;
		if (!skip_quote(s, e, c)) return 0;
	}
	else if ((c == '{') || (c == '(')) {
		(*s)++;
		if (!skip_brace(s, e, c)) return 0;
	}
	else if ((q == ')') && (c == ',')) {
		(*s)++;
		if (!skip_brace(s, e, q)) return 0;
	}
	else if (isalpha(c) || (c == '_')) {
		if (!skip_symbol(s, e)) return 0;
	}
	else if (isdigit(c) || strchr("+-.", c)) {
		if (!skip_number(s, e)) return 0;
	}
	if (!skip_space(s, e) || (**s != q)) return 0;
	(*s)++;
	return 1;
}

/************************************************************
 *
 ************************************************************/
typedef struct {
	FILE* f;
	char **s;
	int n;
} stream_t;

static int stream_putc(stream_t* st, int c)
{
	if (st->f) {
		if (fputc(c, st->f) != c) return 0;
		st->n++;
	} else {
		*st->s = realloc(*st->s, st->n + 2);
		*(*st->s + st->n++) = c;
	}
	return 1;
}
static int stream_putcn(stream_t* st, int c, int n)
{
	int i;
	if (n <= 0) {
		return 0;
	} else if (st->f) {
		for (i = 0; i < n; i++) {
			if (fputc(c, st->f) != c) return 0;
		}
		st->n += n;
	} else {
		*st->s = realloc(*st->s, st->n + n + 1);
		for (i = 0; i < n; i++) {
			*(*st->s + st->n++) = c;
		}
	}
	return n;
}
static int stream_write(stream_t* st, char *s, int n)
{
	if (n <= 0) {
		return 0;
	} else if (st->f) {
		if (fwrite(s, 1, n, st->f) != n) return 0;
	} else {
		*st->s = realloc(*st->s, st->n + n + 1);
		memcpy(*st->s + st->n, s, n);
	}
	st->n += n;
	return n;
}
static int stream_seek(stream_t* st, int n)
{
	if (!n) {
		return 0;
	} else if (st->f) {
		if (fseek(st->f, n, SEEK_CUR) != 0) return 0;
	} else {
		if (n > 0) {
			*st->s = realloc(*st->s, st->n + n + 1);
		}
	}
	st->n += n;
	return n;
}
static int stream_flush(stream_t* st)
{
	if (st->f) {
		if (fflush(st->f) != 0) return 0;
	} else {
		if (!*st->s || !st->n) {
			if (*st->s) free(*st->s);
			*st->s = NULL;
			st->n = 0;
			return 0;
		}
		*(*st->s + st->n) = 0;
	}
	return st->n;
}

/************************************************************
 *
 ************************************************************/
static int echo_to_stream(exec_t* ex, stream_t* st, char *s, int n)
{
	var_t v;
	char *e, *f;
	int c, m, r;
	
	e = s + n;
	f = s;
	n = 0;
	while (skip_quote(&f, e, '$')) {
		if ((m = (f - 1) - s) > 0) {
			n += stream_write(st, s, m);
		}
		v.p = NULL;
		r = 0;
		c = *f++;
		if (c == '$') {
			new_int(&v, getpid());
			r = 1;
		} else if (isalpha(c) || (c == '_')) {
			s = --f;
			if (skip_symbol(&f, e)) {
				assert((m = f - s) > 0);
				if (issymbols(s, m)) {
					c = string_to_symbol(ex, s, m);
					r = frame_get(ex, ex->fp, c, &v);
				} else {
					r = eval_bytes(ex, s, m, &v);
				}
			}
		} else if ((c == '{') || (c == '(')) {
			s = f;
			if (skip_brace(&f, e, c)) {
				if ((m = (f - 1) - s) > 0) {
					r = eval_bytes(ex, s, m, &v);
				}
			}
		}
		if (r) {
			if (var_toString(ex, &v, &s, &m)) {
				n += stream_write(st, s, m);
			}
		}
		s = f;
	}
	if ((m = e - s) > 0) {
		n += stream_write(st, s, m);
	}
	assert(stream_flush(st) == n);
	return n;
}

int echo_to_file(exec_t* ex, FILE* fp, char *s, int n)
{
	stream_t st;
	memset(&st, 0, sizeof(stream_t));
	st.f = fp;
	return echo_to_stream(ex, &st, s, n);
}
int echo_to_bytes(exec_t* ex, char **d, char *s, int n)
{
	stream_t st;
	memset(&st, 0, sizeof(stream_t));
	*d = NULL;
	st.s = d;
	return echo_to_stream(ex, &st, s, n);
}

/************************************************************
 * PRINTF
 ************************************************************/

#define PF_MINUS   1
#define PF_ZERO    (1 << 1)
#define PF_PLUS    (1 << 3)
#define PF_SPACE   (1 << 4)
#define PF_COMMA   (1 << 5)
#define PF_UPPER   (1 << 6)

/************************************************************
 *
 ************************************************************/
static int pf_int_to_str(char *dst, unsigned int u, int base, int opt)
{
	char *ns = (opt & PF_UPPER) ? "0123456789ABCDEF" : "0123456789abcdef";
	char tmp[256];
	int n, a;
	for (n=a=0; u > 0; n++) {
		if (opt & PF_COMMA) {
			if (n && !(n % 3)) tmp[a++] = ',';
		}
		tmp[a++] = ns[u % base];
		u /= base;
	}
	if (!a) tmp[a++] = '0';
	for (n=0; a > 0; n++) {
		dst[n] = tmp[--a];
	}
	dst[n] = 0;
	return n;
}
static int pf_float_to_str(char *s, double f, int max, int opt)
{
	int n, c, i;
	n = pf_int_to_str(s, f, 10, opt);
	c = (int)(f *= 10) % 10;
	for (i = 0; max ? (i < max) : (c && (i < 6)); i++) {
		if (!i) s[n++] = '.';
		s[n++] = c + '0';
		c = (int)(f *= 10) % 10;
	}
	s[n] = 0;
	return n;
}

/************************************************************
 *
 ************************************************************/
static int pf_put_string(stream_t* st, char *s, int n, int min, int max, int opt)
{
	int m = 0;
	if ((max > 0) && (n > max)) n = max;
	if (!(opt & PF_MINUS)) {
		m += stream_putcn(st, (opt & PF_ZERO) ? '0' : ' ', min - n);
	}
	if (n > 0) {
		m += stream_write(st, s, n);
	}
	if (opt & PF_MINUS) {
		m += stream_putcn(st, (opt & PF_ZERO) ? '0' : ' ', min - m);
	}
	return m;
}
static int pf_put_char(stream_t* st, int c, int min, int max, int opt)
{
	char s[1];
	return pf_put_string(st, s, ((s[0] = c) != 0) ? 1 : 0, min, max, opt);
}
static int pf_put_int(stream_t* st, int v, int type, int min, int max, int opt)
{
	char s[256];
	int n=0, base=10;
	
	if (type == 'x') {
		base = 16;
	} else if (type == 'x') {
		base = 8;
	} else if (type == 'u') {
		;
	} else {
		if (v < 0) {
			s[n++] = '-';
			v = - v;
		} else {
			if (opt & PF_PLUS) {
				s[n++] = '+';
			} else if (opt & PF_SPACE) {
				s[n++] = ' ';
			}
		}
	}
	
	n += pf_int_to_str(s + n, v, base, opt);
	
	return pf_put_string(st, s, n, min, max, opt);
}
static int pf_put_float(stream_t* st, double f, int type, int min, int max, int opt)
{
	char *es = (opt & PF_UPPER) ? "EKMGTPEZY" : "ekmgtpezy";
	char s[256];
	int n=0, e;
	
	if (f < 0) {
		s[n++] = '-';
		f = -f;
	} else {
		if (opt & PF_PLUS) {
			s[n++] = '+';
		} else if (opt & PF_SPACE) {
			s[n++] = ' ';
		}
	}
	
	e = f ? log10(f) : 0;
	
	if (type == 'k') {
		e = (e >= 3) ? ((e / 3) * 3) : 0;
	} else if (type == 'g') {
		if (!max) max = 6;
		if ((e < max) && (e > - max)) {
			e = max = 0;
		} else {
			max--;
		}
	} else if (type == 'f') {
		e = 0;
	}
	
	if (e) {
		f /= pow(10, e);
	}
	
	n += pf_float_to_str(s + n, f, max, opt);
	
	if (e) {
		if (type == 'k') {
			s[n++] = es[e / 3];
		} else {
			s[n++] = *es;
			s[n++] = (e < 0) ? '-' : '+';
			if (e < 0) e = -e;
			s[n++] = (e / 10) + '0';
			s[n++] = (e % 10) + '0';
		}
	}
	
	return pf_put_string(st, s, n, min, 0, opt);
}
static int pf_parse(char **s, int *min, int *max, int *opt)
{
	if (**s == '%') {
		(*s)++;
		return 0;
	}
	*min = *max = *opt = 0;
	while (**s != 0) {
			 if (**s == '0') *opt |= PF_ZERO;
		else if (**s == '-') *opt |= PF_MINUS;
		else if (**s == '+') *opt |= PF_PLUS;
		else if (**s == ' ') *opt |= PF_SPACE;
		else if (**s == ',') *opt |= PF_COMMA;
		else break;
		(*s)++;
	}
	if (**s == '*') {
		(*s)++;
		*min = -1;
	} else {
		while (isdigit(**s)) {
			*min = (*min * 10) + (*(*s)++ - '0');
		}
	}
	if (**s == '.') {
		(*s)++;
		if (**s == '*') {
			(*s)++;
			*max = -1;
		} else {
			while (isdigit(**s)) {
				*max = (*max * 10) + (*(*s)++ - '0');
			}
		}
	}
	if (!isalpha(**s)) return 0;
	if (isupper(**s)) *opt |= PF_UPPER;
	return tolower(*(*s)++);
}

/************************************************************
 *
 ************************************************************/
static int printf_to_stream(exec_t* ex, stream_t* st, char *s, var_t* va, int vn)
{
	var_t* v;
	char *e;
	int i, m, n;
	int type, min, max, opt;
	n = 0;
	i = 0;
	e = s;
	while ((e = strchr(e, '%')) != NULL) {
		m = e++ - s;
		if (!(type = pf_parse(&e, &min, &max, &opt))) continue;
		if (m > 0) {
			n += stream_write(st, s, m);
		}
		if (min < 0) min = (i < vn) ? var_toInt(ex, va + i++) : 0;
		if (max < 0) max = (i < vn) ? var_toInt(ex, va + i++) : 0;
		v = (i < vn) ? (va + i++) : NULL;
		switch (type) {
		case 'i': case 'd': case 'u': case 'o': case 'x':
			n += pf_put_int(st, v ? var_toInt(ex, v) : 0, type, min, max, opt);
			break;
		case 'k': case 'f': case 'e': case 'g': 
			n += pf_put_float(st, v ? var_toFloat(ex, v) : 0, type, min, max, opt);
			break;
		case 'c':
			n += pf_put_char(st, v ? var_toChar(ex, v) : 0, min, max, opt);
			break;
		case 's':
			if (!v || !var_toString(ex, v, &s, &m)) m = 0;
			n += pf_put_string(st, s, m, min, max, opt);
			break;
		}
		s = e;
	}
	if ((m = strlen(s)) > 0) {
		n += stream_write(st, s, m);
	}
	assert(stream_flush(st) == n);
	return n;
}

int printf_to_file(exec_t* ex, FILE* fp, char *s, var_t* va, int vn)
{
	stream_t st;
	memset(&st, 0, sizeof(stream_t));
	st.f = fp;
	return printf_to_stream(ex, &st, s, va, vn);
}
int printf_to_bytes(exec_t* ex, char **d, char *s, var_t* va, int vn)
{
	stream_t st;
	memset(&st, 0, sizeof(stream_t));
	*d = NULL;
	st.s = d;
	return printf_to_stream(ex, &st, s, va, vn);
}

/************************************************************
 * TRACE
 ************************************************************/
static int char_to_trace(int c, stream_t* st);
static int string_to_trace(string_t* str, stream_t* st);
static int var_to_trace(exec_t* ex, var_t* v, stream_t* st, int d);
static int symbol_to_trace(exec_t* ex, int k, stream_t* st);

static int symbol_to_trace(exec_t* ex, int k, stream_t* st)
{
	string_t str;
	if (!symbol_to_string(ex, k, &str.s, &str.n)) {
		return 0;
	}
	if (!issymbols(str.s, str.n)) {
		return string_to_trace(&str, st);
	}
	return stream_write(st, str.s, str.n);
}
static int object_to_trace(exec_t* ex, object_t* obj, stream_t* st, int d)
{
	each_t e;
	int i, j, n, a, b;
	n = stream_putc(st, '{');
	for (i = j = 0; i < obj->n; i++) {
		a = 0;
		if (j > 0) {
			a += stream_putc(st, ',');
		}
		a += stream_putc(st, '\n');
		a += stream_putcn(st, '\t', d+1);
		if (object_each(obj, i, &e)) {
			if ((b = symbol_to_trace(ex, e.k, st)) > 0) {
				a += b + stream_putc(st, ':');
				if ((b = var_to_trace(ex, &e.v, st, d+1)) > 0) {
					n += a + b;
					j++;
					continue;
				}
			}
		}
		stream_seek(st, -a);
	}
	if (j > 0) {
		n += stream_putc(st, '\n');
		n += stream_putcn(st, '\t', d);
	}
	n += stream_putc(st, '}');
	return n;
}
static int array_to_trace(exec_t* ex, array_t* arr, stream_t* st, int d)
{
	int i, j, n, a, b;
	n = stream_putc(st, '[');
	for (i = j = 0; i < arr->n; i++) {
		a = 0;
		if (j > 0) {
			a += stream_putc(st, ',');
		}
		a += stream_putc(st, '\n');
		a += stream_putcn(st, '\t', d+1);
		if ((b = var_to_trace(ex, arr->v + i, st, d+1)) > 0) {
			n += a + b;
			j++;
		} else {
			stream_seek(st, -a);
		}
	}
	if (j > 0) {
		n += stream_putc(st, '\n');
		n += stream_putcn(st, '\t', d);
	}
	n += stream_putc(st, ']');
	return n;
}
static int string_to_trace(string_t* str, stream_t* st)
{
	int i, c, n;
	n = stream_putc(st, '"');
	for (i = 0; i < str->n; i++) {
		c = str->s[i];
		if (c_to_meta(&c, '"')) {
			n += stream_putc(st, '\\');
		}
		n += stream_putc(st, c);
	}
	n += stream_putc(st, '"');
	return n;
}
static int char_to_trace(int c, stream_t* st)
{
	int n;
	n = stream_putc(st, '\'');
	if (c_to_meta(&c, '\'')) {
		n += stream_putc(st, '\\');
	}
	n += stream_putc(st, c);
	n += stream_putc(st, '\'');
	return n;
}
static int float_to_trace(double f, stream_t* st)
{
	char s[256];
	return stream_write(st, s, sprintf(s, "%g", f));
}
static int int_to_trace(int v, stream_t* st)
{
	char s[256];
	return stream_write(st, s, sprintf(s, "%d", v));
}
static int var_to_trace(exec_t* ex, var_t* v, stream_t* st, int d)
{
	if (v->p == object_proto) {
		return object_to_trace(ex, v->u.p, st, d);
	} else if (v->p == array_proto) {
		return array_to_trace(ex, v->u.p, st, d);
	} else if (v->p == string_proto) {
		return string_to_trace(v->u.p, st);
	} else if (v->p == char_proto) {
		return char_to_trace(v->u.i, st);
	} else if (v->p == float_proto) {
		return float_to_trace(v->u.f, st);
	} else if (v->p == int_proto) {
		return int_to_trace(v->u.i, st);
	} else if (v->p == bool_proto) {
		return int_to_trace(v->u.i, st);
	}
	return 0;
}
static int trace_to_stream(exec_t* ex, var_t* v, stream_t *st)
{
	int n;
	n = var_to_trace(ex, v, st, 0);
	if ((n > 0) && (st->f != NULL)) {
		n += stream_putc(st, '\n');
	}
	assert(stream_flush(st) == n);
	return n;
}
int trace_to_file(exec_t* ex, var_t* v, FILE* fp)
{
	stream_t st;
	memset(&st, 0, sizeof(stream_t));
	st.f = fp;
	return trace_to_stream(ex, v, &st);
}
int trace_to_bytes(exec_t* ex, var_t* v, char **d)
{
	stream_t st;
	memset(&st, 0, sizeof(stream_t));
	*d = NULL;
	st.s = d;
	return trace_to_stream(ex, v, &st);
}

/************************************************************
 *
 ************************************************************/
static int trace_to_var(exec_t* ex, var_t* v, char** s, char *e);

static int trace_to_object(exec_t* ex, var_t* v, char** s, char *e)
{
	object_t* obj;
	int k;
	new_object(ex, v, NULL);
	obj = v->u.p;
	while (trace_to_var(ex, v, s, e)) {
		if (skip_space(s, e) != ':') break;
		(*s)++;
		k = var_toSymbol(ex, v);
		if (!trace_to_var(ex, v, s, e)) break;
		object_set(ex, obj, k, v);
		if (skip_space(s, e) != ',') break;
		(*s)++;
	}
	if (*(*s)++ != '}') {
		object_free(ex, obj);
		return 0;
	}
	new_object(ex, v, obj);
	return 1;
}
static int trace_to_array(exec_t* ex, var_t* v, char** s, char* e)
{
	array_t* arr;
	new_array(ex, v, NULL);
	arr = v->u.p;
	while (trace_to_var(ex, v, s, e)) {
		array_push(ex, arr, v);
		if (skip_space(s, e) != ',') break;
		(*s)++;
	}
	if (*(*s)++ != ']') {
		array_free(ex, arr);
		return 0;
	}
	new_array(ex, v, arr);
	return 1;
}
static int trace_to_string(exec_t* ex, var_t* v, char** s, char* e)
{
	char *f = (*s);
	int c;
	string_t* str;
	while (((*s) < e) && (*(*s) != '"')) {
		if (*(*s)++ == '\\') (*s)++;
	}
	if (*(*s)++ != '"') {
		return 0;
	}
	new_string(ex, v, NULL);
	str = v->u.p;
	str->s = calloc((*s) - f, 1);
	while ((f < e) && (*f != '"')) {
		if ((c = *f++) == '\\') {
			c = meta_to_c(&f, 0);
		}
		str->s[str->n++] = c;
	}
	str->s[str->n] = 0;
	return 1;
}
static int trace_to_char(var_t* v, char** s, char* e)
{
	int c = 0;
	if (((*s) < e) && (*(*s) != '\'')) {
		if ((c = *(*s)++) == '\\') {
			c = meta_to_c(s, 0);
		}
	}
	if (*(*s)++ != '\'') {
		return 0;
	}
	new_char(v, c);
	return 1;
}
static int trace_to_number(var_t* v, char** s, char *e)
{
	char *f = *s;
	if ((**s == '+') || (**s == '-')) (*s)++;
	while ((*s < e) && isalnum(**s)) (*s)++;
	if (**s == '.') {
		new_float(v, strtod(f, s));
	} else {
		new_int(v, strtol(f, s, 0));
	}
	return 1;
}
static int trace_to_symbol(exec_t* ex, var_t* v, char** s, char *e)
{
	char *f = (*s);
	int n;
	while (((*s) < e) && (isalnum(*(*s)) || (*(*s)=='_'))) (*s)++;
	n = *s - f;
	if ((n == 4) && !strncmp("null", f, n)) new_null(v);
	else if ((n == 4) && !strncmp("true", f, n)) new_bool(v, 1);
	else if ((n == 5) && !strncmp("false", f, n)) new_bool(v, 0);
	else new_symbol(v, string_to_symbol(ex, f, n));
	return 1;
}
static int trace_to_var(exec_t* ex, var_t* v, char** s, char *e)
{
	int c = skip_space(s, e);
	if (c == '{') {
		(*s)++;
		return trace_to_object(ex, v, s, e);
	} else if (c == '[') {
		(*s)++;
		return trace_to_array(ex, v, s, e);
	} else if (c == '"') {
		(*s)++;
		return trace_to_string(ex, v, s, e);
	} else if (c == '\'') {
		(*s)++;
		return trace_to_char(v, s, e);
	} else if (isdigit(c) || ((c > 0) && strchr("+-.",c) && isdigit(*(*s + 1)))) {
		return trace_to_number(v, s, e);
	} else if (isalpha(c) || (c == '_')) {
		return trace_to_symbol(ex, v, s, e);
	}
	return 0;
}

int untrace_from_bytes(exec_t* ex, var_t* v, char *s, int n)
{
	int r;
	char *p = s;
	r = trace_to_var(ex, v, &p, s + n);
	if (!r) {
		new_null(v);
	}
	return r;
}
