/*
 * Copyright (c) 2007, to-do. All rights reserved.
 */
#include "util.h"
#include "var.h"
#include "_sym.h"
#include "_ext.h"

/************************************************************
 *
 ************************************************************/
/*
sleep
alarm
fork
*/

/************************************************************
 *
 ************************************************************/
#include <sys/wait.h>

static int _global_exec(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	var_t *ref;
	char buf[BUFSIZ + 2];
	char *cmd, *s;
	int pid, rc, sz, n, i;
	int out[2], in[2];
	
	exerrif(ex, argc < 1, "bad arguments");
	exerrif(ex, !(cmd = var_toChars(ex, argv)), "bad commands");
	
	ref = NULL;
	s = NULL;
	n = 0;
	for (i = 1; i < argc; i++) {
		if (is_refer_var(argv + i)) {
			if (ref != NULL) {
				fprintf(ex->err, "bad argument %d\n", i);
				return 0;
			}
			ref = argv + i;
		} else {
			if ((s != NULL) || !var_toString(ex, argv + i, &s, &n)) {
				fprintf(ex->err, "bad argument %d\n", i);
				return 0;
			}
		}
	}
	
	assert(pipe(out) == 0);
	
	if ((pid = fork()) == 0) {
		close(out[0]);
		if (s && n) {
			assert(pipe(in) == 0);
			write(in[1], s, n);
			close(in[1]);
			dup2(in[0], 0);
		}
		dup2(out[1], 1);
		dup2(out[1], 2);
		execlp("sh", "sh", "-c", cmd, NULL);
		exit(-1);
	}
	
	close(out[1]);
	if (waitpid(pid, &rc, 0) != pid) {
		rc = -1; /* waitpid error */
	} else if (!WIFEXITED(rc)) {
		rc = -1; /* not exited */
	} else {
		rc = WEXITSTATUS(rc);
	}
	if (ref != NULL) {
		new_int(res, rc);
		refer_set(ex, ref, res);
	}
	
	s = NULL;
	n = 0;
	if (rc >= 0) {
		sz = BUFSIZ;
		while ((sz >= BUFSIZ) && ((sz = read(out[0], buf, BUFSIZ)) > 0)) {
			s = realloc(s, n + sz + 1);
			memcpy(s + n, buf, sz);
			n += sz;
		}
	}
	close(out[0]);
	
	if (s) {
		s[n] = 0;
		new_string_with(ex, res, s, n);
	} else {
		new_null(res);
	}
	
	return 1;
}

/************************************************************
 *
 ************************************************************/
static int _global_getenv(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	char *s;
	exerrif(ex, argc != 1, "bad arguments");
	exerrif(ex, !(s = var_toChars(ex, argv)), "bad argument-0");
	if (!(s = getenv(s))) {
		return 0;
	}
	new_string_copy(ex, res, s, strlen(s));
	return 1;
}
static int _global_setenv(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	char *name, *val = NULL;
	exerrif(ex, argc < 1, "bad arguments");
	exerrif(ex, !(name = var_toChars(ex, argv)), "bad argument-0");
	if (argc >= 2) {
		val = var_toChars(ex, argv + 1);
	}
	if (!val) {
		unsetenv(name);
		return 0;
	}
	new_bool(res, setenv(name, val, 1) == 0);
	return 1;
}

static int _global_system(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	char *s;
	exerrif(ex, argc != 1,  "bad arguments");
	exerrif(ex, !(s = var_toChars(ex, argv)),  "bad argument-0");
	new_int(res, system(s));
	return 1;
}

static int _global_popen(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	FILE* fp;
	char *cmd, *mode="r";
	exerrif(ex, argc < 1, "bad arguments");
	exerrif(ex, !(cmd = var_toChars(ex, argv)), "bad argument-0");
	exerrif(ex, (argc >= 2) && !(mode = var_toChars(ex, argv)), "bad argument-1");
	exerrif(ex, !(fp = popen(cmd, mode)), "popen error");
	new_file(ex, res, fp, pclose);
	return 1;
}

/************************************************************
 *
 ************************************************************/
static int _global_load(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	var_t v;
	char *fn;
	char *bc;
	int bn, k, r;
	
	exerrif(ex, argc != 1,  "bad arguments");
	exerrif(ex, !(fn = var_toChars(ex, argv)),  "bad argument-0");
	
	k = strtoh(fn, strlen(fn));
	
	if (frame_getLocal(ex, ex->fp, k, &v)) {
		/* loaded already */
		new_bool(res, 1);
		return 1;
	}
	
	bc = NULL;
	bn = 0;
	r = compile_file(ex, fn, &bc, &bn);
	if (r != 0) {
		if (bc) free(bc);
		/* compile error */
		return 0;
	}
	
	r = call_bytes(ex, "load", bc, 0, NULL, res);
	
	new_string_with(ex, &v, bc, bn);
	frame_setLocal(ex, ex->fp, k, &v);
	
	return r;
}
static int _global_eval(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	char *s;
	int n, r;
	
	exerrif(ex, argc != 1,  "bad arguments");
	exerrif(ex, !var_toString(ex, argv, &s, &n), "bad argument-0");
	
	r = eval_bytes(ex, s, n, res);
	
	return r;
}

/************************************************************
 *
 ************************************************************/
static int _global_print(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	char *s;
	int i,n,m=0;
	exerrif(ex, argc < 1,  "bad arguments");
	for (i = 0; i < argc; i++) {
		if (var_toString(ex, argv + i, &s, &n)) {
			m += fwrite(s, 1, n, ex->out);
		}
	}
	new_int(res, m);
	return 1;
}
static int _global_println(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	char *s;
	int i,n,m=0;
	exerrif(ex, argc < 1,  "bad arguments");
	for (i = 0; i < argc; i++) {
		if (var_toString(ex, argv + i, &s, &n)) {
			m += fwrite(s, 1, n, ex->out);
		}
		m += fputc('\n', ex->out) > 0;
	}
	new_int(res, m);
	return 1;
}
static int _global_printf(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	char *s;
	int n;
	exerrif(ex, argc < 1,  "bad arguments");
	exerrif(ex, !(s = var_toChars(ex, argv)),  "bad argument-0");
	n = printf_to_file(ex, ex->out, s, argv + 1, argc - 1);
	new_int(res, n);
	return 1;
}
static int _global_sprintf(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	string_t* str;
	char *s;
	exerrif(ex, argc < 1,  "bad arguments");
	exerrif(ex, !(s = var_toChars(ex, argv)),  "bad argument-0");
	str = string_alloc(ex);
	str->n = printf_to_bytes(ex, &str->s, s, argv + 1, argc - 1);
	new_string(ex, res, str);
	return 1;
}
static int _global_echo(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	char *s;
	int n;
	
	exerrif(ex, argc != 1, "bad arguments");
	exerrif(ex, !var_toString(ex, argv, &s, &n), "bad argument-0");
	
	n = echo_to_file(ex, ex->out, s, n);
	
	new_int(res, n);
	
	return 1;
}
static int _global_printfile(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	FILE* fp;
	char buf[BUFSIZ + 1];
	char *s;
	int n, m;
	
	exerrif(ex, argc != 1, "bad arguments");
	exerrif(ex, !(s = var_toChars(ex, argv)), "bad argument-0");
	
	exerrif(ex, !(fp = fopen(s, "r")), "fopen error");
	
	n = 0;
	while ((m = fread(buf, 1, BUFSIZ, fp)) > 0) {
		fwrite(buf, 1, m, ex->out);
		n += m;
	}
	fclose(fp);
	new_int(res, n);
	return 1;
}

/************************************************************
 *
 ************************************************************/
static int _global_serial(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	char *s;
	int n;
	
	exerrif(ex, argc != 1,  "bad arguments");
	
	if (!serial_to_bytes(ex, argv, &s, &n)) {
		return 0;
	}
	new_string_with(ex, res, s, n);
	return 1;
}

static int _global_unserial(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	char *s;
	int n;
	
	exerrif(ex, argc != 1,  "bad arguments");
	exerrif(ex, !var_toString(ex, argv, &s, &n),  "bad argument-0");
	
	n = unserial_from_bytes(ex, res, s, n);
	return (n > 0);
}

static int _global_tracef(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	char *fn;
	FILE* fp;
	fp = ex->out;
	fn = NULL;
	
	exerrif(ex, argc < 1,  "bad arguments");
	if (argc >= 2) {
		if ((argv + 1)->p == file_proto) {
			fp = *(FILE**)((argv + 1)->u.p);
		} else {
			exerrif(ex, !(fn = var_toChars(ex, argv + 1)), "bad argument 1");
			exerrif(ex, !(fp = fopen(fn, "w")), "fopen error");
		}
	}
	
	new_int(res, trace_to_file(ex, argv, fp));
	if (fn) {
		fclose(fp);
	}
	return 1;
}
static int _global_traces(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	char *s;
	int n;
	
	exerrif(ex, argc != 1,  "bad arguments");
	
	if (!(n = trace_to_bytes(ex, argv, &s))) {
		return 0;
	}
	new_string_with(ex, res, s, n);
	
	return 1;
}
static int _global_untrace(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	char *s;
	int n;
	exerrif(ex, argc != 1,  "bad arguments");
	exerrif(ex, !var_toString(ex, argv, &s, &n),  "bad argument-0");
	n = untrace_from_bytes(ex, res, s, n);
	return (n > 0);
}

/************************************************************
 *
 ************************************************************/
static int _global_pack(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	char *fmt, *dst;
	int len;
	
	exerrif(ex, argc < 2,  "bad arguments");
	exerrif(ex, !(fmt = var_toChars(ex, argv)),  "bad argument-0");
	
	len = pack_to_bytes(ex, &dst, fmt, argv + 1, argc - 1);
	if (!len) {
		return 0;
	}
	new_string_with(ex, res, dst, len);
	return 1;
}
static int _global_unpack(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	char *fmt, *src;
	int len;
	
	exerrif(ex, argc < 2,  "bad arguments");
	exerrif(ex, !(fmt = var_toChars(ex, argv)),  "bad argument-0");
	exerrif(ex, !var_toString(ex, argv + 1, &src, &len),  "bad argument-1");
	
	len = unpack_from_bytes(ex, fmt, src, len, res);
	
	return (len > 0);
}

static int _global_jconv(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	char *s, *d;
	int n, q;
	q = 0;
	exerrif(ex, argc < 1,  "bad arguments");
	exerrif(ex, !var_toString(ex, argv, &s, &n),  "bad argument-0");
	if (argc >= 2) {
		q = var_toChar(ex, argv + 1);
	}
	if (!(n = jconv(&d, s, n, q))) {
		return 0;
	}
	new_string_with(ex, res, d, n);
	return 1;
}
static int _global_quote(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	char *s, *d;
	int n,q;
	q = 0;
	exerrif(ex, argc < 1,  "bad arguments");
	exerrif(ex, !var_toString(ex, argv, &s, &n),  "bad argument-0");
	if (argc >= 2) {
		q = var_toChar(ex, argv + 1);
	}
	if (!(n = quote(&d, s, n, q))) {
		return 0;
	}
	new_string_with(ex, res, d, n);
	return 1;
}
static int _global_unquote(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	char *s, *d;
	int n;
	exerrif(ex, argc != 1,  "bad arguments");
	exerrif(ex, !var_toString(ex, argv, &s, &n),  "bad argument-0");
	if (!(n = unquote(&d, s, n))) {
		return 0;
	}
	new_string_with(ex, res, d, n);
	return 1;
}
static int _global_entity(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	char *s, *d;
	int n;
	exerrif(ex, argc < 1,  "bad arguments");
	exerrif(ex, !var_toString(ex, argv, &s, &n),  "bad argument-0");
	if (!(n = entity(&d, s, n))) {
		return 0;
	}
	new_string_with(ex, res, d, n);
	return 1;
}
static int _global_unentity(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	char *s, *d;
	int n;
	exerrif(ex, argc != 1,  "bad arguments");
	exerrif(ex, !var_toString(ex, argv, &s, &n),  "bad argument-0");
	if (!(n = unentity(&d, s, n))) {
		return 0;
	}
	new_string_with(ex, res, d, n);
	return 1;
}
static int _global_encode(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	char *s, *d;
	int n,q;
	q = 0;
	exerrif(ex, argc < 1,  "bad arguments");
	exerrif(ex, !var_toString(ex, argv, &s, &n),  "bad argument-0");
	if (argc >= 2) {
		q = var_toChar(ex, argv + 1);
	}
	if (!(n = encode(&d, s, n, q))) {
		return 0;
	}
	new_string_with(ex, res, d, n);
	return 1;
}
static int _global_decode(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	char *s, *d;
	int n,q;
	q = 0;
	exerrif(ex, argc < 1,  "bad arguments");
	exerrif(ex, !var_toString(ex, argv, &s, &n),  "bad argument-0");
	if (argc >= 2) {
		q = var_toChar(ex, argv + 1);
	}
	if (!(n = decode(&d, s, n, q))) {
		return 0;
	}
	new_string_with(ex, res, d, n);
	return 1;
}
static int _global_crypt(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	char *s, *k;
	char buf[10];
	exerrif(ex, argc < 1,  "bad arguments");
	exerrif(ex, !(s = var_toChars(ex, argv)),  "bad argument-0");
	if (argc >= 2) {
		k = var_toChars(ex, argv + 1);
	} else {
		strncpy(k = buf, s, 2);
		k[2] = 0;
	}
	exerrif(ex, !(s = strcrypt(s, k)), "crypt error");
	
	new_string_copy(ex, res, s, strlen(s));
	return 1;
}
static int _global_md5(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	char buf[33];
	char *s;
	int n;
	exerrif(ex, argc != 1,  "bad arguments");
	exerrif(ex, !var_toString(ex, argv, &s, &n),  "bad argument-0");
	exerrif(ex, !(n = strnmd5(buf, s, n)), "md5 error");
	new_string_copy(ex, res, buf, n);
	return 1;
}

/************************************************************
 *
 ************************************************************/
static int _global_atan2(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	exerrif(ex, argc != 2,  "bad arguments");
	new_float(res, atan2(var_toFloat(ex, argv), var_toFloat(ex, argv + 1)));
	return 1;
}
static int _global_pow(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	exerrif(ex, argc != 2,  "bad arguments");
	new_float(res, pow(var_toFloat(ex, argv), var_toFloat(ex, argv + 1)));
	return 1;
}
static int _global_sin(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	exerrif(ex, argc != 1,  "bad arguments");
	new_float(res, sin(var_toFloat(ex, argv)));
	return 1;
}
static int _global_asin(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	exerrif(ex, argc != 1,  "bad arguments");
	new_float(res, asin(var_toFloat(ex, argv)));
	return 1;
}
static int _global_cos(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	exerrif(ex, argc != 1,  "bad arguments");
	new_float(res, cos(var_toFloat(ex, argv)));
	return 1;
}
static int _global_acos(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	exerrif(ex, argc != 1,  "bad arguments");
	new_float(res, acos(var_toFloat(ex, argv)));
	return 1;
}
static int _global_tan(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	exerrif(ex, argc != 1,  "bad arguments");
	new_float(res, tan(var_toFloat(ex, argv)));
	return 1;
}
static int _global_atan(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	exerrif(ex, argc != 1,  "bad arguments");
	new_float(res, atan(var_toFloat(ex, argv)));
	return 1;
}
static int _global_sqrt(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	exerrif(ex, argc != 1,  "bad arguments");
	new_float(res, sqrt(var_toFloat(ex, argv)));
	return 1;
}
static int _global_exp(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	exerrif(ex, argc != 1,  "bad arguments");
	new_float(res, exp(var_toFloat(ex, argv)));
	return 1;
}
static int _global_log(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	exerrif(ex, argc != 1,  "bad arguments");
	new_float(res, log(var_toFloat(ex, argv)));
	return 1;
}
static int _global_log10(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	exerrif(ex, argc != 1,  "bad arguments");
	new_float(res, log10(var_toFloat(ex, argv)));
	return 1;
}
static int _global_ceil(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	exerrif(ex, argc != 1,  "bad arguments");
	new_int(res, (int)ceil(var_toFloat(ex, argv)));
	return 1;
}
static int _global_floor(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	exerrif(ex, argc != 1,  "bad arguments");
	new_int(res, (int)floor(var_toFloat(ex, argv)));
	return 1;
}
static int _global_round(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	exerrif(ex, argc != 1,  "bad arguments");
/*	new_int(res, (int)round(var_toFloat(ex, argv))); */
	new_int(res, (int)rint(var_toFloat(ex, argv)));
	return 1;
}
static int _global_abs(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	double f;
	int i;
	exerrif(ex, argc != 1,  "bad arguments");
	if (var_toNumber(ex, argv, &f, &i) > 1) {
		new_float(res, fabs(f));
	} else {
		new_int(res, abs(i));
	}
	return 1;
}
static int _global_rand(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	double f, r;
	int i;
	static long seed = 0;
	if (!seed) {
		srand(seed = 1);
	}
	exerrif(ex, argc > 1,  "bad arguments");
	r = rand() / (double)RAND_MAX;
	if (argc < 1) {
		new_float(res, r);
	} else {
		if (var_toNumber(ex, argv, &f, &i) > 1) {
			new_float(res, r * f);
		} else {
			new_int(res, (int)(r * i));
		}
	}
	return 1;
}
static int _global_min(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	int i;
	exerrif(ex, argc < 2,  "bad arguments");
	copy_var(res, argv);
	for (i = 1; i < argc; i++) {
		if (cmp_vars_asNumber(ex, res, argv + i) > 0) {
			copy_var(res, argv + i);
		}
	}
	return 1;
}
static int _global_max(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	int i;
	exerrif(ex, argc < 2,  "bad arguments");
	copy_var(res, argv);
	for (i = 1; i < argc; i++) {
		if (cmp_vars_asNumber(ex, res, argv + i) < 0) {
			copy_var(res, argv + i);
		}
	}
	return 1;
}
static int _global_isfinite(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	exerrif(ex, argc != 1,  "bad arguments");
	new_bool(res, (argv->p == int_proto) || ((argv->p == float_proto) && finite(argv->u.f)));
	return 1;
}
static int _global_isinf(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	exerrif(ex, argc != 1,  "bad arguments");
	new_bool(res, (argv->p == float_proto) && isinf(argv->u.f));
	return 1;
}
static int _global_isnan(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	exerrif(ex, argc != 1,  "bad arguments");
	new_bool(res, (argv->p == float_proto) && isnan(argv->u.f));
	return 1;
}

/************************************************************
 *
 ************************************************************/
static int _global_readfile(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	FILE* fp;
	char buf[BUFSIZ + 1];
	char *s;
	int n, m;
	exerrif(ex, argc != 1,  "bad arguments");
	exerrif(ex, !(s = var_toChars(ex, argv)), "bad argument-0");
	exerrif(ex, !(fp = fopen(s, "r")), "fopen error");
	s = NULL;
	n = 0;
	while ((m = fread(buf, 1, BUFSIZ, fp)) > 0) {
		s = realloc(s, n + m + 1);
		memcpy(s + n, buf, m);
		n += m;
	}
	fclose(fp);
	if (!n) {
		return 0;
	}
	s[n] = 0;
	new_string_with(ex, res, s, n);
	return 1;
}
static int _global_writefile(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	FILE* fp;
	char *s;
	int n;
	exerrif(ex, argc != 2,  "bad arguments");
	exerrif(ex, !(s = var_toChars(ex, argv)), "bad argument-0");
	exerrif(ex, !(fp = fopen(s, "w")), "fopen error");
	if (!var_toString(ex, argv + 1, &s, &n)) {
		fprintf(ex->err, "bad argument: %s\n", typeof_var(ex, argv + 1));
		n = 0;
	} else {
		n = fwrite(s, 1, n, fp);
	}
	fclose(fp);
	new_int(res, n);
	return 1;
}
static int _global_fopen(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	FILE* fp;
	char *path=NULL, *mode = "r";
	
	exerrif(ex, argc < 1,  "bad arguments");
	exerrif(ex, !(path = var_toChars(ex, argv + 0)), "bad argument-0");
	exerrif(ex, (argc >= 2) && !(mode = var_toChars(ex, argv + 1)), "bad argument-1");
	
	exerrif(ex, !(fp = fopen(path, mode)), "fopen error");
	new_file(ex, res, fp, fclose);
	return 1;
}
static int _global_tmpfile(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	FILE* fp;
	exerrif(ex, argc != 0,  "bad arguments");
	assert((fp = tmpfile()) != NULL);
	new_file(ex, res, fp, fclose);
	return 1;
}
static int _global_tempnam(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	char buf[FILENAME_MAX];
	char *s1, *s2, *s3;
	struct stat st;
	int i;
	s1 = s2 = s3 = NULL;
	
	if (argc >= 1) s1 = var_toChars(ex, argv + 0);
	if (argc >= 2) s2 = var_toChars(ex, argv + 1);
	if (argc >= 3) s3 = var_toChars(ex, argv + 2);
	if (!s1) s1 = P_tmpdir;
	if (!s1) s1 = "";
	if (!s2) s2 = "";
	if (!s3) s3 = "";
	
	exerrif(ex,
		*s1 && ((stat(s1, &st) != 0) || !(st.st_mode & S_IFDIR)),
		"bad directory");
	
	exerrif(ex,
		(strlen(s1) + strlen(s2) + strlen(s3) + 10) >= FILENAME_MAX,
		"too large file-name");
	
	s1 = strchr(strcpy(buf, s1), 0);
	if (*buf && (*(s1 - 1) != '/')) {
		*s1++ = '/';
	}
	
	if (*s2) {
		s1 = strchr(strcpy(s1, s2), 0);
	}
	
	i = 0;
	do {
		sprintf(s1, "%04x%s", i++, s3);
	} while ((stat(buf, &st) == 0) && (i < 0xFFFF));
	
	exerrif(ex, errno != ENOENT, "tempnam error");
	
	new_string_copy(ex, res, buf, strlen(buf));
	
	return 1;
}
static int _global_chmod(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	char *path;
	int mode = 0777;
	exerrif(ex, argc < 1,  "bad arguments");
	exerrif(ex, !(path = var_toChars(ex, argv)),  "bad argument-0");
	if (argc > 1) {
		mode = var_toInt(ex, argv + 1);
	}
	exerrif(ex, chmod(path, mode) != 0, "chmod error");
	new_bool(res, 1);
	return 1;
}
static int _global_mkdir(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	char *path;
	int mode = 0777;
	exerrif(ex, argc < 1,  "bad arguments");
	exerrif(ex, !(path = var_toChars(ex, argv)),  "bad argument-0");
	if (argc > 1) {
		mode = var_toInt(ex, argv + 1);
	}
	exerrif(ex, mkdir(path, mode) != 0, "mkdir error");
	new_bool(res, 1);
	return 1;
}
static int _global_touch(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	FILE* fp;
	char *path;
	exerrif(ex, argc != 1,  "bad arguments");
	exerrif(ex, !(path = var_toChars(ex, argv)),  "bad argument-0");
	if (!(fp = fopen(path, "r"))) {
		exerrif(ex, !(fp = fopen(path, "w")), "fopen error");
		fclose(fp);
	} else {
		fclose(fp);
		if (utimes(path, NULL) != 0) {
			fprintf(ex->err, "utime error: %s %s\n", path, strerror(errno));
		}
	}
	new_bool(res, 1);
	return 1;
}
static int _global_chdir(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	char *path;
	exerrif(ex, argc != 1,  "bad arguments");
	exerrif(ex, !(path = var_toChars(ex, argv)),  "bad argument-0");
	exerrif(ex, chdir(path) != 0, "mkdir error");
	new_bool(res, 1);
	return 1;
}
static int _global_getcwd(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	char *s;
	exerrif(ex, argc != 0,  "bad arguments");
	exerrif(ex, !(s = getcwd(NULL, 0)), "getcwd error");
	new_string_with(ex, res, s, strlen(s));
	return 1;
}
static int _global_rename(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	char *f1, *f2;
	exerrif(ex, argc < 2,  "bad arguments");
	exerrif(ex, !(f1 = var_toChars(ex, argv + 0)),  "bad argument-0");
	exerrif(ex, !(f2 = var_toChars(ex, argv + 1)),  "bad argument-1");
	exerrif(ex, rename(f1, f2) != 0, "rename error");
	new_bool(res, 1);
	return 1;
}
static int _global_symlink(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	char *f1, *f2;
	exerrif(ex, argc < 2,  "bad arguments");
	exerrif(ex, !(f1 = var_toChars(ex, argv + 0)),  "bad argument-0");
	exerrif(ex, !(f2 = var_toChars(ex, argv + 1)),  "bad argument-1");
	exerrif(ex, symlink(f1, f2) != 0, "symlink error");
	new_bool(res, 1);
	return 1;
}
static int _global_flink(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	char *f1, *f2;
	exerrif(ex, argc < 2,  "bad arguments");
	exerrif(ex, !(f1 = var_toChars(ex, argv + 0)),  "bad argument-0");
	exerrif(ex, !(f2 = var_toChars(ex, argv + 1)),  "bad argument-1");
	exerrif(ex, link(f1, f2) != 0, "link error");
	new_bool(res, 1);
	return 1;
}
static int _global_fcopy(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	FILE *p1, *p2;
	char *f1, *f2;
	char buf[BUFSIZ];
	size_t sz;
	exerrif(ex, argc < 2,  "bad arguments");
	exerrif(ex, !(f1 = var_toChars(ex, argv + 0)),  "bad argument-0");
	exerrif(ex, !(f2 = var_toChars(ex, argv + 1)),  "bad argument-1");
	exerrif(ex, !(p1 = fopen(f1, "r")), "fopen error");
	if (!(p2 = fopen(f2, "w"))) {
		fclose(p1);
		fprintf(ex->err, "fopen error: %s %s\n", f2, strerror(errno));
		return 0;
	}
	while ((sz = fread(buf, 1, BUFSIZ, p1)) > 0) {
		fwrite(buf, 1, sz, p2);
	}
	fclose(p2);
	fclose(p1);
	new_bool(res, 1);
	return 1;
}
static int _global_truncate(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	char *path;
	size_t sz = 0;
	exerrif(ex, argc < 1,  "bad arguments");
	exerrif(ex, !(path = var_toChars(ex, argv)),  "bad argument-0");
	if (argc > 1) {
		sz = var_toInt(ex, argv + 1);
	}
	exerrif(ex, truncate(path, sz) != 0, "truncate error");
	new_bool(res, 1);
	return 1;
}

/************************************************************
 *
 ************************************************************/
static int _global_glob(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	glob_t gl;
	var_t tmp;
	char *s;
	int i;
	
	exerrif(ex, argc != 1,  "bad arguments");
	exerrif(ex, !(s = var_toChars(ex, argv)),  "bad argument-0");
	
	memset(&gl, 0, sizeof(glob_t));
	/* append '/' if directory */
	exerrif(ex, glob(s, GLOB_MARK, NULL, &gl) != 0, "glob error");
	
	new_array(ex, res, NULL);
	for (i = 0; i < gl.gl_pathc; i++) {
		s = gl.gl_pathv[i];
		new_string_copy(ex, &tmp, s, strlen(s));
		array_push(ex, res->u.p, &tmp);
	}
	globfree(&gl);
	return 1;
}
static int _global_remove(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	glob_t gl;
	char *s;
	int i, j, n;
	
	exerrif(ex, argc < 1,  "bad arguments");
	
	n = 0;
	for (i = 0; i < argc; i++) {
		if (!(s = var_toChars(ex, argv + i))) {
			fprintf(ex->err, "bad argument: %d %s\n", i, typeof_var(ex, argv + i));
		} else if (strchr(s, '*')) {
			memset(&gl, 0, sizeof(glob_t));
			/* do not sort */
			if (glob(s, GLOB_NOSORT, NULL, &gl) != 0) {
				fprintf(ex->err, "glob error: %s %s\n", s, strerror(errno));
			} else {
				for (j = 0; j < gl.gl_pathc; j++) {
					if (remove(s = gl.gl_pathv[j]) != 0) {
						fprintf(ex->err, "remove error: %s %s\n", s, strerror(errno));
					} else {
						n++;
					}
				}
				globfree(&gl);
			}
		} else {
			if (remove(s) != 0) {
				fprintf(ex->err, "remove error: %s %s\n", s, strerror(errno));
			} else {
				n++;
			}
		}
	}
	new_int(res, n);
	return 1;
}
static int _global_unlink(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	glob_t gl;
	char *s;
	int i, j, n;
	
	exerrif(ex, argc < 1,  "bad arguments");
	
	n = 0;
	for (i = 0; i < argc; i++) {
		if (!(s = var_toChars(ex, argv + i))) {
			fprintf(ex->err, "bad argument: %d %s\n", i, typeof_var(ex, argv + i));
		} else if (strchr(s, '*')) {
			memset(&gl, 0, sizeof(glob_t));
			/* do not sort */
			if (glob(s, GLOB_NOSORT, NULL, &gl) != 0) {
				fprintf(ex->err, "glob error: %s %s\n", s, strerror(errno));
			} else {
				for (j = 0; j < gl.gl_pathc; j++) {
					if (unlink(s = gl.gl_pathv[j]) != 0) {
						fprintf(ex->err, "unlink error: %s %s\n", s, strerror(errno));
					} else {
						n++;
					}
				}
				globfree(&gl);
			}
		} else {
			if (unlink(s) != 0) {
				fprintf(ex->err, "unlink error: %s %s\n", s, strerror(errno));
			} else {
				n++;
			}
		}
	}
	new_int(res, n);
	return 1;
}

/************************************************************
 *
 ************************************************************/
static int _global_fmode(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	struct stat st;
	char *path;
	exerrif(ex, argc != 1,  "bad arguments");
	exerrif(ex, !(path = var_toChars(ex, argv)), "bad argument-0");
	exerrif(ex, stat(path, &st) != 0, "stat error");
	new_int(res, st.st_mode);
	return 1;
}
static int _global_fsize(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	struct stat st;
	char *path;
	exerrif(ex, argc != 1,  "bad arguments");
	exerrif(ex, !(path = var_toChars(ex, argv)), "bad argument-0");
	exerrif(ex, stat(path, &st) != 0, "stat error");
	new_int(res, st.st_size);
	return 1;
}
static int _global_fmtime(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	struct stat st;
	char *path;
	exerrif(ex, argc != 1,  "bad arguments");
	exerrif(ex, !(path = var_toChars(ex, argv)), "bad argument-0");
	exerrif(ex, stat(path, &st) != 0, "stat error");
	new_int(res, st.st_mtime);
	return 1;
}
static int _global_fctime(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	struct stat st;
	char *path;
	exerrif(ex, argc != 1,  "bad arguments");
	exerrif(ex, !(path = var_toChars(ex, argv)), "bad argument-0");
	exerrif(ex, stat(path, &st) != 0, "stat error");
	new_int(res, st.st_ctime);
	return 1;
}
static int _global_fatime(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	struct stat st;
	char *path;
	exerrif(ex, argc != 1,  "bad arguments");
	exerrif(ex, !(path = var_toChars(ex, argv)), "bad argument-0");
	exerrif(ex, stat(path, &st) != 0, "stat error");
	new_int(res, st.st_atime);
	return 1;
}

/************************************************************
 *
 ************************************************************/
#include <pwd.h>
#include <grp.h>
static int _global_chown(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	char *path, *us, *gs;
	struct passwd *pw;
	struct group *gr;
	path = us = gs = NULL;
	pw = NULL;
	gr = NULL;
	exerrif(ex, argc < 1,  "bad arguments");
	exerrif(ex, !(path = var_toChars(ex, argv)), "bad argument-0");
	exerrif(ex, (argc >= 2) && !(us = var_toChars(ex, argv + 1)), "bad argument-1");
	exerrif(ex, (argc >= 3) && !(gs = var_toChars(ex, argv + 2)), "bad argument-2");
	exerrif(ex, us && !(pw = getpwnam(us)), "getpwnam error");
	exerrif(ex, gs && !(gr = getgrnam(gs)), "getgrnamm error");
	exerrif(ex, chown(path, pw ? pw->pw_uid : -1, gr ? gr->gr_gid : -1) != 0, "chown error");
	new_bool(res, 1);
	return 1;
}
static int _global_fuid(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	struct stat st;
	char *path;
	struct passwd *pw;
	exerrif(ex, argc != 1,  "bad arguments");
	exerrif(ex, !(path = var_toChars(ex, argv)), "bad argument-0");
	exerrif(ex, stat(path, &st) != 0, "stat error");
	exerrif(ex, !(pw = getpwuid(st.st_uid)), "getpwuid error");
	new_string_copy(ex, res, pw->pw_name, strlen(pw->pw_name));
	return 1;
}
static int _global_fgid(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	struct stat st;
	char *path;
	struct group *gr;
	
	exerrif(ex, argc != 1,  "bad arguments");
	exerrif(ex, !(path = var_toChars(ex, argv)), "bad argument-0");
	exerrif(ex, stat(path, &st) != 0, "stat error");
	exerrif(ex, !(gr = getgrgid(st.st_gid)), "getgrgid error");
	new_string_copy(ex, res, gr->gr_name, strlen(gr->gr_name));
	return 1;
}

/************************************************************
 *
 ************************************************************/
static int _global_dbopen(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	char *path=NULL, *mode = NULL;
	
	exerrif(ex, argc < 1,  "bad arguments");
	exerrif(ex, !(path = var_toChars(ex, argv)), "bad argument-0");
	exerrif(ex, (argc >= 2) && !(mode = var_toChars(ex, argv)), "bad argument-1");
	
	return new_dbm(ex, res, path, mode);
}

/************************************************************
 *
 ************************************************************/
static int _global_xmlopen(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	char *path=NULL, *mode = NULL;
	
	exerrif(ex, argc < 1,  "bad arguments");
	exerrif(ex, !(path = var_toChars(ex, argv)), "bad argument-0");
	exerrif(ex, (argc >= 2) && !(mode = var_toChars(ex, argv)), "bad argument-1");
	
	return new_xml_from_file(ex, res, path, mode);
}

/************************************************************
 *
 ************************************************************/
static int vars_toTime(exec_t* ex, int argc, var_t* argv, int g)
{
	if (argc < 1) {
		return time(NULL);
	} else if (argc >= 2) {
		int y,m,d,h,i,s;
		y = m = d = h = i = s = 0;
		if (argc > 5) s = var_toInt(ex, argv + 5);
		if (argc > 4) i = var_toInt(ex, argv + 4);
		if (argc > 3) h = var_toInt(ex, argv + 3);
		if (argc > 2) d = var_toInt(ex, argv + 2);
		if (argc > 1) m = var_toInt(ex, argv + 1);
		if (argc > 0) y = var_toInt(ex, argv + 0);
		return datetotime(y, m, d, h, i, s, g);
	} else if (argv->p == string_proto) {
		char *s;
		int n;
		exerrif(ex, !var_toString(ex, argv, &s, &n),  "bad argument-0");
		return strtotime(s, n, g);
	}
	return var_toInt(ex, argv);
}
static int _global_mktime(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	int t;
	
	t = vars_toTime(ex, argc, argv, 0);
	
	new_int(res, t);
	return 1;
}
static int _global_timegm(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	int t;
	
	t = vars_toTime(ex, argc, argv, 1);
	
	new_int(res, t);
	return 1;
}
static int _global_time(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	int t;
	
	t = vars_toTime(ex, argc, argv, 0);
	
	new_int(res, t);
	return 1;
}
static int _global_date(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	char dst[BUFSIZ];
	char *s;
	int t, n;
	t = vars_toTime(ex, argc - 1, argv + 1, 0);
	if (argc < 1) {
		s = "f";
		n = 1;
	} else {
		exerrif(ex, !(s = var_toChars(ex, argv)),  "bad argument-0");
	}
	n = timetostr(dst, s, t, 0);
	if (!n) {
		return 0;
	}
	new_string_copy(ex, res, dst, n);
	return 1;
}
static int _global_gmdate(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	char dst[BUFSIZ];
	char *s;
	int t, n;
	t = vars_toTime(ex, argc - 1, argv + 1, 1);
	if (argc < 1) {
		s = "f";
		n = 1;
	} else {
		exerrif(ex, !(s = var_toChars(ex, argv)),  "bad argument-0");
	}
	n = timetostr(dst, s, t, 1);
	if (!n) {
		return 0;
	}
	new_string_copy(ex, res, dst, n);
	return 1;
}

/************************************************************
 * HTTP / NETWORK
 ************************************************************/
static int _global_putheader(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	char *s;
	int i,n,m=0;
	exerrif(ex, !ex->hh, "no http header");
	exerrif(ex, argc < 1,  "bad arguments");
	for (i = 0; i < argc; i++) {
		if (var_toString(ex, argv + i, &s, &n)) {
			while ((n > 0) && isspace(s[n - 1])) n--;
			if (n > 0) {
				m += fprintf(ex->hh, "%.*s\n", n, s);
			}
		}
	}
	new_int(res, m);
	return 1;
}

static int _global_setcookie(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	char buf[BUFSIZ];
	char *k, *v;
	int d, n;
	
	exerrif(ex, !ex->hh, "no http header");
	exerrif(ex, argc < 1,  "bad arguments");
	exerrif(ex, !(k = var_toChars(ex, argv)), "bad key");
	
	if (!var_toString(ex, argv + 1, &v, &n)) {
		v = NULL;
	} else {
		encode(&v, v, n, 0);
	}
	
	d = (argc >= 3) ? var_toInt(ex, argv + 2) : 0;
	if (d) {
		timetostr(buf, "e", time(NULL) + (d * 86400), 1);
	}
	
	if (!v) {
		n = fprintf(ex->hh, "Set-Cookie: %s=\n", k);
	} else if (!d) {
		n = fprintf(ex->hh, "Set-Cookie: %s=%s\n", k, v);
	} else {
		n = fprintf(ex->hh, "Set-Cookie: %s=%s; expires=%s\n", k, v, buf);
	}
	if (v) free(v);
	
	new_int(res, n);
	
	return 1;
}

/************************************************************
 * 
 ************************************************************/
static int _global_sendmail(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	char *to, *data;
	
	data = NULL;
	
	exerrif(ex, argc < 1, "bad arguments");
	exerrif(ex, !(to = var_toChars(ex, argv)), "bad argument 0");
	exerrif(ex, (argc >= 2) && !(data = var_toChars(ex, argv + 1)), "bad argument 1");
	
	new_int(res, sendmail(to, data, ex->err));
	
	return 1;
}

/************************************************************
 * 
 ************************************************************/
static int _global_popmail(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	char *u, *s;
	int n;
	FILE* fp;
	
	s = NULL; /* password */
	
	exerrif(ex, (argc < 1) || !(u = var_toChars(ex, argv)), "bad arguments");
	exerrif(ex, (argc >= 2) && !(s = var_toChars(ex, argv + 1)), "bad password");
	
	fp = tmpfile();
	if (!popmail(u, s, fp, ex->err)) {
		fclose(fp);
		return 0;
	}
	
	n = ftell(fp);
	rewind(fp);
	s = calloc(n + 1, 1);
	fread(s, 1, n, fp);
	fclose(fp);
	new_string_with(ex, res, s, n);
	
	return 1;
}

/************************************************************
 * 
 ************************************************************/
static int _global_ftpsite(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	char *u, *p, *cmd, *s;
	
	p = cmd = NULL;
	
	exerrif(ex, argc < 2, "bad arguments");
	exerrif(ex, !(u = var_toChars(ex, argv)), "bad argument 0");
	
	argc--;
	argv++;
	for (; argc-- > 0; argv++) {
		exerrif(ex, !(s = var_toChars(ex, argv)), "bad argument");
		if (!cmd) cmd = s;
		else if (!p) {
			p = cmd;
			cmd = s;
		}
	}
	
	new_int(res, ftpsite(u, p, cmd, ex->err));
	
	return 1;
}

/************************************************************
 * 
 ************************************************************/
static int _global_httpget(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	var_t *ref;
	char *url, *head, *data, *s;
	int n, j;
	FILE* fp;
	char buf[BUFSIZ];
	
	ref = NULL;
	head = data = NULL;
	
	exerrif(ex, argc < 1, "bad arguments");
	exerrif(ex, !(url = var_toChars(ex, argv)), "bad argument 0");
	
	argc--;
	argv++;
	for (; argc-- > 0; argv++) {
		if (is_refer_var(argv)) {
			ref = argv;
		} else {
			exerrif(ex, !(s = var_toChars(ex, argv)), "bad header string");
			if (!head) head = s;
			if (!data) data = s;
		}
	}
	
	fp = tmpfile();
	if (!httpget(url, head, data, fp, ex->err)) {
		fclose(fp);
		return 0;
	}
	
	j = ftell(fp);
	rewind(fp);
	while (fgets(buf, BUFSIZ, fp)) {
		if (iscrlf(*buf)) {
			break;
		}
	}
	if (ref) {
		n = ftell(fp);
		rewind(fp);
		s = calloc(n + 1, 1);
		fread(s, 1, n, fp);
		new_string_with(ex, res, s, n);
		refer_set(ex, ref, res);
	}
	n = j - ftell(fp);
	s = calloc(n + 1, 1);
	fread(s, 1, n, fp);
	new_string_with(ex, res, s, n);
	
	fclose(fp);
	
	return 1;
}

/************************************************************
 *
 ************************************************************/
int _global_length(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	exerrif(ex, argc != 1,  "bad arguments");
	return var_getLength(ex, argv, res);
}
int _global_isNull(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	exerrif(ex, argc != 1,  "bad arguments");
	new_bool(res, !argv->p);
	return 1;
}
int _global_isBool(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	exerrif(ex, argc != 1,  "bad arguments");
	new_bool(res, argv->p == bool_proto);
	return 1;
}
int _global_isChar(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	exerrif(ex, argc != 1,  "bad arguments");
	new_bool(res, argv->p == char_proto);
	return 1;
}
int _global_isNumber(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	exerrif(ex, argc != 1,  "bad arguments");
	new_bool(res, (argv->p == int_proto) || (argv->p == float_proto));
	return 1;
}
int _global_isInt(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	exerrif(ex, argc != 1,  "bad arguments");
	new_bool(res, argv->p == int_proto);
	return 1;
}
int _global_isFloat(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	exerrif(ex, argc != 1,  "bad arguments");
	new_bool(res, argv->p == float_proto);
	return 1;
}
int _global_isString(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	exerrif(ex, argc != 1,  "bad arguments");
	new_bool(res, argv->p == string_proto);
	return 1;
}
int _global_isArray(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	exerrif(ex, argc != 1,  "bad arguments");
	new_bool(res, argv->p == array_proto);
	return 1;
}
int _global_isObject(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	exerrif(ex, argc != 1,  "bad arguments");
	new_bool(res, argv->p == object_proto);
	return 1;
}
int _global_toBool(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	exerrif(ex, argc != 1,  "bad arguments");
	new_bool(res, var_toBool(ex, argv));
	return 1;
}
int _global_toChar(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	exerrif(ex, argc != 1,  "bad arguments");
	new_char(res, var_toChar(ex, argv));
	return 1;
}
int _global_toInt(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	exerrif(ex, argc != 1,  "bad arguments");
	new_int(res, var_toInt(ex, argv));
	return 1;
}
int _global_toFloat(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	exerrif(ex, argc != 1,  "bad arguments");
	new_float(res, var_toFloat(ex, argv));
	return 1;
}
int _global_toNumber(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	double f;
	int i;
	exerrif(ex, argc != 1,  "bad arguments");
	if (var_toNumber(ex, argv, &f, &i) > 1) {
		new_float(res, f);
	} else {
		new_int(res, i);
	}
	return 1;
}
int _global_toString(exec_t* ex, int argc, var_t* argv, var_t* res)
{
	char *s;
	int n;
	exerrif(ex, argc != 1,  "bad arguments");
	exerrif(ex, !var_toString(ex, argv, &s, &n), "bad argument-0");
	new_string_copy(ex, res, s, n);
	return 1;
}

/************************************************************
 *
 ************************************************************/
int global_callFunc(exec_t* ex, int sym, int argc, var_t* argv, var_t* res)
{
	switch (sym) {
#include "_global_call.h"
	}
	return -1;
}

