/*
 * Copyright (c) 2007, to-do. All rights reserved.
 */
#include "util.h"
#include "var.h"
#include "_sym.h"
#include "_ext.h"

/*********************************************************
 * STRING UTILS
 *********************************************************/
static int isattrchr(int c)
{
	return ((c == '@') || (c == '$'));
}
static int iskeychr(int c)
{
	return (isalnum(c) || (c=='_') || (c=='-') || (c==':'));
}
static int keylen(char *s)
{
	int n;
	for (n=0; iskeychr(s[n]); n++) ;
	return n;
}
static int issamekey(char *s1, char *s2)
{
	int n;
	for (n=0; iskeychr(*s1) || iskeychr(*s2); n++) {
		if (tolower(*s1++) != tolower(*s2++)) return 0;
	}
	return n;
}
static int iswordchar(int c)
{
	return (isalnum(c) || (c=='_') || (c>=0x80 && c<=0xff));
}
static int isnumbers(char *s, int n)
{
	if ((*s=='-') || (*s=='+') || (*s=='.')) { s++; n--; }
	return ((n > 0) && isdigit(*s));
}

/****************************************************:
 * FIND NODE
 ****************************************************/
static char* skip_tag(char *s)
{
	int c;
	assert(*s++ == '<');
	if ((c = *s++) == '!') {
		if ((c = *s++) == '[') {
			if ((s = strstr(s, "]]>")) != NULL) return (s+3);
			/* bad CDATA */
		} else if ((c=='-') && (*s == c)) {
			if ((s = strstr(s, "-->")) != NULL) return (s+3);
			/* bad comment */
		} else {
			/* bad <! */
		}
	} else if (c=='?') {
		if ((s = strstr(s, "?>")) != NULL) return (s+2);
		/* bad <? */
	} else if (c=='/') {
		if ((s = strchr(s, '>')) != NULL) return (s+1);
		/* bad </ */
	} else if (iskeychr(c)) {
		while (((c = *s++) != 0) && (c != '>')) {
			if ((c=='"') || (c=='\'')) {
				while (*s && (*s++ != c)) ;
			}
		}
		if (c) return s;
		/* bad tag */
	}
	return NULL;
}

static int find_node(char *s, int i, int n, char *k, int cur, int r[4])
{
	char *f,*e,*z;
	int dir = 0;
	f = s;
	z = s + n;
	e = s + i;
	while ((e<z)
	&& ((s = strchr(e, '<')) != NULL)
	&& ((e = skip_tag(s)) != NULL)
	&& (e <= z)) {
		if ((*(s+1)=='/') && iskeychr(*(s+2))) {
			if (k && issamekey(s+2, k)) {
				if (--dir <= 0) {
					r[2]=s - f;
					r[3]=e - f;
					return 1;
				}
			} else {
				if (cur) {
					dir--;
				}
			}
		} else if (iskeychr(*(s+1))) {
			if (!k || issamekey(s+1, k)) {
				if (!k) k = s+1;
				if (*(e-2)=='/') {
					if (dir <= 0) {
						r[0]=s - f;
						r[1]=r[2]=r[3]=e - f;
						return 1;
					}
				} else {
					if (dir++ <= 0) {
						r[0]=s - f;
						r[1]=e - f;
					}
				}
			} else {
				if (cur) {
					if (*(e-2)=='/') {
						;
					} else {
						dir++;
					}
				}
			}
		}
	}
	if (dir > 0) {
		/* can't find terminate */
	}
	return 0;
}
static int find_attr(char *s, int i, int n, char *q, int r[4])
{
	char *f,*e,*k;
	int c;
	f = s;
	e = s + n;
	s = s + i;
	if ((*s++ != '<') || !iskeychr(*s++)) {
		/* bad tag */
		return 0;
	}
	while ((s<e) && iskeychr(*s)) s++;
	if (!isspace(*s)) return 0;
	while ((s<e) && isspace(*s)) s++;
	while ((s<e) && iskeychr(*s)) {
		k = s++;
		while ((s<e) && iskeychr(*s)) s++;
		while ((s<e) && isspace(*s)) s++;
		if ((s >= e) || (*s++ != '=')) {
			/* bad equal */
			return 0;
		}
		while ((s<e) && isspace(*s)) s++;
		if ((*s=='"')||(*s=='\'')) {
			c = *s++;
			r[1] = s - f;
			while ((s<e) && (*s != c)) s++;
			r[2] = s - f;
			if (*s == c) s++;
		} else {
			r[1] = s - f;
			while ((s<e) && !isspace(*s)
			&& (*s != '/') && (*s != '>')) s++;
			r[2] = s - f;
		}
		while ((s<e) && isspace(*s)) s++;
		if (issamekey(k, q)) {
			r[0] = k - f;
			r[3] = s - f;
			return 1;
		}
	}
	return 0;
}
static int find_word(char *s, int n, char *w, int m, int wo)
{
	char *e = s + n;
	while (s < e) {
		if (*s == '<') {
			if (!(s = skip_tag(s))) return 0;
		} else if (*s=='&') {
			s++;
			if (*s=='#') {
				s++;
				while (isdigit(*s)) s++;
			} else {
				while (isalpha(*s)) s++;
			}
			if (*s==';') s++;
		} else if (!iswordchar(*s)) {
			s++;
		} else {
			n = 1;
			while (((s+n)<e) && iswordchar(s[n])) n++;
			if (wo) {
				if ((n==m) && !strncasecmp(s, w, m)) {
					return m;
				}
			} else {
				for (; n>=m; s++,n--) {
					if (!strncasecmp(s, w, m)) {
						return m;
					}
				}
			}
			s += n;
		}
	}
	return 0;
}
static int search_words(char *s, int n, char *w, int m, int wo, int or)
{
	char *e;
	e = w + m;
	while (w<e) {
		if (!iswordchar(*w)) {
			w++;
		} else {
			m = 1;
			while (((w+m)<e) && iswordchar(w[m])) m++;
			if (m>=2) {
				if (find_word(s, n, w, m, wo)) {
					if (or) return 1;
				} else {
					if (!or) return 0;
				}
			}
			w += m;
		}
	}
	return (!or);
}

/****************************************************
 * STREAM
 ****************************************************/
typedef struct {
	FILE *fp;
	int sz;
} stream_t;

static void free_stream(void *p)
{
	FILE* fp;
	stream_t* st;
	if (!p) {
		return;
	}
	st = (stream_t*)p - 1;
	if ((fp = st->fp) != NULL) {
		assert(munmap(st, st->sz) == 0);
		assert(fclose(fp) == 0) ;
	} else {
		free(st);
	}
}
static void* alloc_stream(int sz)
{
	stream_t* st;
	FILE *fp;
	sz += sizeof(stream_t) + 1;
	fp = NULL;
	if (sz > 0xffff) {
		assert((fp = tmpfile()) != NULL);
		fseek(fp, sz, SEEK_SET);
		fputc(0, fp);
		rewind(fp);
		st = mmap(0, sz, PROT_READ|PROT_WRITE, MAP_SHARED, fileno(fp), 0);
		assert((st!=NULL) && (st!=MAP_FAILED));
	} else {	
		st = calloc(sz, 1);
	}
	st->fp = fp;
	st->sz = sz;
	return (st+1);
}
static void* realloc_stream(void *p, int sz)
{
	stream_t* st;
	FILE* fp;
	if (!p) {
		return alloc_stream(sz);
	}
	st = (stream_t*)p - 1;
	sz += sizeof(stream_t) + 1;
	if (sz <= st->sz) {
		return p;
	}
	if ((fp = st->fp) != NULL) {
		assert(munmap(st, st->sz) == 0);
		assert(fclose(fp) == 0);
		fseek(fp, sz, SEEK_SET);
		fputc(0, fp);
		rewind(fp);
		st = mmap(0, sz, PROT_READ|PROT_WRITE, MAP_SHARED, fileno(fp), 0);
		assert((st!=NULL) && (st!=MAP_FAILED));
	} else {
		st = realloc(st, sz);
	}
	st->fp = fp;
	st->sz = sz;
	return (st + 1);
}
static int stream_from_file(char *filename, char **xs, int *sz)
{
	FILE* fp;
	int n;
	if (!(fp = fopen(filename,"r"))) {
		return 0;
	}
	fseek(fp,0,SEEK_END);
	n = ftell(fp);
	rewind(fp);
	*xs = alloc_stream(n);
	*sz = fread(*xs, 1, n, fp);
	fclose(fp);
	return 1;
}
static int open_stream(char *s, char **xs, int *sz)
{
	if (!s || strchr(s, '<')) {
		*sz = s ? strlen(s) : 0;
		*xs = alloc_stream((*sz > BUFSIZ) ? *sz : BUFSIZ);
		if (*sz) memcpy(*xs, s, *sz);
		return 1;
	}
	return stream_from_file(s, xs, sz);
}
static int write_stream(char *s, int n, FILE* fp)
{
	char *f = "<?xml version=\"1.0\" encoding=\"utf-8\" ?>\n<xml>%.*s</xml>\n";
	char *e = s + n;
	while ((s < e) && isspace(*s)) s++;
	if (!strncmp(s, "<?", 2)) f = "%.*s\n";
	while ((s < e) && isspace(*(e - 1))) e--;
	return (((n = e - s) > 0) ? fprintf(fp, f, n, s) : 0);
}

/*************************************************************
 * RESIZE MEMORY
 *************************************************************/
static int trim_range(char *s, int pos, int len, char **rs, int *rn)
{
	char *e;
	e = s + len;
	s += pos;
	while ((s<e) && isspace(*(e-1))) e--;
	while ((s<e) && isspace(*s)) s++;
	*rs = s;
	*rn = e - s;
	return *rn;
}
static int init_range(char *s, int n, int *pos, int *len)
{
	char *f,*e;
	f = s;
	e = s + n;
	while ((s<e) && isspace(*(e-1))) e--;
	while ((s<e) && isspace(*s)) s++;
	*pos = s - f;
	*len = e - f;
	if (*s != '<') {
		/* no tag */
		return 0;
	}
	if (*(s+1) != '?') {
		/* no header, no root */
		return 1;
	}
	while ((s<e) && !(*e=='<' && *(e+1)=='/')) e--;
	while ((s<e) && !(*s=='<' && iskeychr(*(s+1)))) s++;
	if ((s>=e) || !issamekey(s+1,e+2)
	|| !(s = skip_tag(s)) || (s>e)) {
		/* bad root */
		return 0;
	}
	*pos = s - f;
	*len = e - f;
	return 1;
}

/*****************************************
 * QUERY
 *****************************************/
typedef struct query_t query_t;
struct query_t {
	char *name;
	int op;
	int not;
	char *key;
	int asnum;
	union {
		struct { char *s; int n; } s;
		int i;
		double f;
	} value;
};

static int parse_query(char *s, query_t *q, int qsz)
{
	int i,c;
	memset(q, 0, sizeof(query_t) * qsz);
	if (!s) {
		return 0;
	}
	for (i=0; isattrchr(*s) || iskeychr(*s); i++, q++) {
		assert( (i+1) < qsz );
		q->name = s++;
		while (iskeychr(*s)) s++;
		while (isspace(*s)) s++;
		if (*s == '[') {
			s++;
			while (isspace(*s)) s++;
			if (*s == ']') {
				q->op = ']';
			} else if ((*s == '-') || isdigit(*s)) {
				q->op = '[';
				q->value.i = (int)strtod(s, &s); /* first is 0 */
				if ((q->value.i > 0) && islower(*s)) { /* 1st,2nd,3rd,4th,,, */
					q->value.i--;
				}
			} else {
				if (*s=='!') {
					q->not = *s++; /* allow !hoge */
					while (isspace(*s)) s++;
				}
				if (isattrchr(*s) || iskeychr(*s)) {
					q->key = s++;
					while (iskeychr(*s)) s++;
					while (isspace(*s)) s++;
				}
				if (*s==']') {
					q->op = 'x'; /* check ex */
				} else {
					q->op = '~';
					while (*s && strchr("!=~|*?^$<>", *s)) {
						if ((*s == '=') && ((q->op=='<') || (q->op=='>'))) {
							q->not = *s++;
							q->op ^= 2; /* 3c<=>3E */
						} else if (*s=='!') {
							q->not = *s++;
						} else {
							q->op = *s++;
						}
					}
					while (isspace(*s)) s++;
					if ((*s=='"')||(*s=='\'')) {
						c = *s++;
						q->value.s.s = s;
						while (*s && (*s != c)) s++;
						q->value.s.n = s - q->value.s.s;
						if (*s == c) s++;
					} else {
						q->value.s.s = s;
						while (*s && !isspace(*s) && (*s != ']')) s++;
						q->value.s.n = s - q->value.s.s;
						if (strchr("=<>", q->op)) {
							q->asnum = isnumbers(q->value.s.s, q->value.s.n);
							if (q->asnum) {
								q->value.f = strtod(q->value.s.s, NULL);
							}
						}
					}
				}
			}
			while (*s && (*s != ']')) s++;
			if (*s==']') s++;
		}
		if ((*s == '.') || (*s == '/')) {
			s++;
		}
	}
	return i;
}

typedef struct range_t range_t;
struct range_t {
	int count;
	int next;
	int top;
	int pos;
	int len;
	int bot;
};

static int eval_query(char *xs, int pos, int len, query_t* q, range_t* r)
{
	char *s;
	int p[4], b, n;
	
	if (!r->count) {
		s = (q + 1)->name;
		r->next = !s ? 0 : (isattrchr(*s) ? -1 : 1);
		if (q->op == ']') {
			return 0;
		}
	} else {
		if (q->op == '[') {
			/* no more position */
			return 0;
		}
	}
	
	b = 0;
	while (!b) {
		
		if (isattrchr(*q->name)
		? !find_attr(
			xs, pos, len,
			q->name + 1,
			p)
		: !find_node(
			xs, pos, len,
			q->name,
			r->count > 0,
			p)
		) {
			if (q->op == '[') {
				if (q->value.i < 0) {
					if (r->count > 0) {
						return 1;
					}
				}
			}
			return 0;
		}
		
		r->top = p[0];
		if (r->next < 0) { /* next is attribute */
			r->pos = p[0];
			r->len = p[1];
		} else {
			r->pos = p[1];
			r->len = p[2];
		}
		r->bot = p[3];
		pos = p[3];
		r->count++;
		
		if (!q->op) {
			return 1;
		} else if (q->op == '[') {
			if (q->value.i < 0) {
				; /* continue */
			} else if (r->count > q->value.i) {
				return 1;
			}
		} else {
			s = "";
			n = 0;
			if (!q->key
			|| (isattrchr(*q->key)
			? find_attr(xs, p[0], p[1], q->key + 1, p)
			: find_node(xs, p[1], p[2], q->key, 1, p)
			)) {
				trim_range(xs, p[1], p[2], &s, &n);
			}
			switch (q->op) {
			case '~':
				b = search_words(s, n, q->value.s.s, q->value.s.n, 0, 0);
				break;
			case '|': /* or match */
				b = search_words(s, n, q->value.s.s, q->value.s.n, 0, 1);
				break;
			case '*': /* word match */
				b = search_words(s, n, q->value.s.s, q->value.s.n, 1, 0);
				break;
			case '?': /* word+or match */
				b = search_words(s, n, q->value.s.s, q->value.s.n, 1, 1);
				break;
			case '^': /* starts */
				b = (n >= q->value.s.n)
					&& (strncmp(s, q->value.s.s, q->value.s.n)==0);
				break;
			case '$': /* ends */
				b = (n >= q->value.s.n)
					&& (strncmp(s + (n - q->value.s.n), q->value.s.s, q->value.s.n)==0);
				break;
			case '=':
				if (q->asnum)
					b = strtod(s, NULL) == q->value.f;
				else
					b = (n == q->value.s.n) && (strncmp(s, q->value.s.s, n)==0);
				break;
			case '<':
				if (q->asnum)
					b = strtod(s, NULL) < q->value.f;
				else {
					/* n1 < n2 ? cmp < 1, n1 >= n2 ? cmp < 0 */
					b = strncmp(s, q->value.s.s,
						(n < q->value.s.n) ? n : q->value.s.n)
							< (n < q->value.s.n);
				}
				break;
			case '>':
				if (q->asnum)
					b = strtod(s, NULL) > q->value.f;
				else {
					/* n1 > n2 ? cmp >= 0, n1 <= n2 ? cmp >= 1 */
					b = strncmp(s, q->value.s.s,
						(n < q->value.s.n) ? n : q->value.s.n)
							>= (n <= q->value.s.n);
				}
				break;
			default:
				b = n > 0;
				break;
			}
			if (q->not) b = !b;
		}
	}
	
	return 1;
}

/*************************************************************
 * DELETE
 *************************************************************/
static int resize_range(char **xs, int *sz, int from, int to)
{
	int err = to - from;
	int move = *sz - from;
	if (!err) return 0;
	if (err > 0) {
		*xs = realloc_stream(*xs, *sz + err);
	}
	memmove(*xs + to, *xs + from, move);
	(*sz) += err;
	return err;
}
static int delete_range(char **xs, int *sz, int pos, int len, query_t* q)
{
	range_t r;
	int count, pos_z, len_z;
	
	count = 0;
	memset(&r, 0, sizeof(range_t));
	while (eval_query(*xs, pos, len, q, &r)) {
		pos_z = *sz - r.bot;
		len_z = *sz - len;
		if (!r.next) {
			resize_range(xs, sz, r.bot, r.top);
			count++;
		} else {
			count += delete_range(xs, sz, r.pos, r.len, q + 1);
		}
		pos = *sz - pos_z;
		len = *sz - len_z;
	}
	return count;
}
static int delete_stream(char **xs, int *sz, char *path)
{
	query_t q[32];
	int pos, len;
	
	if (!parse_query(path, q, 32)
	|| !init_range(*xs, *sz, &pos, &len)) {
		return 0;
	}
	return delete_range(xs, sz, pos, len, q);
}

/*************************************************************
 * MODIFY
 *************************************************************/
static int update_range(char **xs, int *sz,
	 int top, int pos, int len, int bot,
	 char *name, char *s, int n)
{
	if ((*s == '<') && issamekey(s + 1, name)) {
		resize_range(xs, sz, bot, top + n);
		memcpy(*xs + top, s, n);
	} else {
		resize_range(xs, sz, len, pos + n);
		memcpy(*xs + pos, s, n);
	}
	return 1;
}
static int insert_range(char **xs, int *sz, int pos, char *name, char *s, int n)
{
	int nameLen;
	if ((*s == '<') && issamekey(s + 1, name)) {
		resize_range(xs, sz, pos, pos + n);
		memcpy(*xs + pos, s, n);
	} else if (isattrchr(*name)) {
		assert( *(*xs + --pos) == '>' );
		if (*(*xs + (pos - 1)) == '/') pos--;
		nameLen = keylen(++name);
		resize_range(xs, sz, pos, pos + nameLen+4+n);
		*(*xs + pos
			+ sprintf(*xs + pos,
			" %.*s=\"%.*s",
			nameLen, name, n, s)
		) = '"';
	} else {
		nameLen = keylen(name);
		resize_range(xs, sz, pos, pos + (nameLen*2)+5+n);
		*(*xs + pos
			+ sprintf(*xs + pos,
			"<%.*s>%.*s</%.*s",
			nameLen, name, n, s, nameLen, name)
		) = '>';
	}
	return 1;
}
static int modify_range(char **xs, int *sz,
	int pos, int len, query_t* q, char* s, int n)
{
	range_t r;
	int count, pos_z, len_z;
	
	count = 0;
	memset(&r, 0, sizeof(range_t));
	while (eval_query(*xs, pos, len, q, &r)) {
		pos_z = *sz - r.bot;
		len_z = *sz - len;
		if (!r.next) {
			count += update_range(xs, sz,
				r.top, r.pos, r.len, r.bot,
				q->name, s, n);
		} else {
			count += modify_range(xs, sz, r.pos, r.len, q + 1, s, n);
		}
		pos = *sz - pos_z;
		len = *sz - len_z;
	}
	if (!count && !r.next) {
		if (!q->op || (q->op == ']')) {
			count += insert_range(xs, sz, len, q->name, s, n);
		}
	}
	return count;
}
static int modify_stream(char **xs, int *sz, char *path, char *s, int n)
{
	query_t q[32];
	int pos, len;
	
	if (!s || (n <= 0)) {
		s = "";
		n = 0;
	}
	if (!parse_query(path, q, 32)
	|| !init_range(*xs, *sz, &pos, &len)) {
		return 0;
	}
	return modify_range(xs, sz, pos, len, q, s, n);
}

/*************************************************************
 * SELECT
 *************************************************************/
typedef struct elem_t elem_t;
typedef struct order_t order_t;
typedef struct select_t select_t;

struct elem_t {
	char *s;
	int n;
	int prev;
	int next;
};

struct order_t {
	char *key;
	int not;
	int asnum;
};

struct select_t {
	elem_t * elems;
	order_t* order;
	int count;
	int min;
	int max;
};

static int parse_order(char *s, order_t* o, int sz)
{
	int i;
	memset(o, 0, sizeof(order_t) * sz);
	if (!s) return 0;
	for (i=0; *s; i++, o++) {
		assert( (i+1) < sz );
		o->asnum = -1;
		while (*s && !(isattrchr(*s) || iskeychr(*s))) {
			switch (*s++) {
			case '!': case '>': o->not = 1; break;
			case '+': case '=': o->asnum = 1; break;
			}
		}
		if (!*s) break;
		o->key = s++;
		while (isspace(*s) || (*s==',')) s++;
	}
	return i;
}
static int compare_elem(elem_t* e1, elem_t* e2, order_t* o)
{
	char *s1,*s2;
	int p[4],b,n1,n2;
	double f1,f2;
	for (b=0; !b && o->key; o++) {
		s1=s2="";
		n1=n2=0;
		if (isattrchr(*o->key)) {
			if (find_attr(e1->s, 0, e1->n, o->key+1, p))
			{ trim_range(e1->s, p[1], p[2], &s1, &n1); }
			if (find_attr(e2->s, 0, e2->n, o->key+1, p))
			{ trim_range(e2->s, p[1], p[2], &s2, &n2); }
		} else {
			if (find_node(e1->s, 0, e1->n, o->key, 1, p))
			{ trim_range(e1->s, p[1], p[2], &s1, &n1); }
			if (find_node(e2->s, 0, e2->n, o->key, 1, p))
			{ trim_range(e2->s, p[1], p[2], &s2, &n2); }
		}
		if (o->asnum < 0) {
			o->asnum = isnumbers(s1, n1);
		}
		if (o->asnum) {
			f1 = strtod(s1, NULL);
			f2 = strtod(s2, NULL);
			b = (f1 < f2) ? -1 : (f1 > f2);
		} else {
			b = strncmp(s1, s2, (n1 < n2) ? n1 : n2);
			if (!b) b = n1 - n2;
		}
		if (o->not) b = -b;
	}
	return b;
}
static int select_elem(char *xs, int pos, int len, select_t* sel)
{
	elem_t *e,*f;
	char *s;
	int i,j,n;
	
	if (!trim_range(xs, pos, len, &s, &n)) {
		return 0;
	}
	if (!sel->count) {
		sel->min = -1;
		sel->max = -1;
	}
	sel->elems = realloc_stream(sel->elems, (sel->count + 1) * sizeof(elem_t));
	e = sel->elems + (i = sel->count);
	e->s = s;
	e->n = n;
	e->prev = -1;
	e->next = -1;
	if (!sel->order) {
		if (sel->max >= 0) {
			(sel->elems + sel->max)->next = i;
			e->prev = sel->max;
		}
	} else {
		j = 0;
		while ((j>=0) && (j < i)) {
			f = sel->elems + j;
			if (compare_elem(e, f, sel->order) < 0) {
				e->next = j;
				j = f->prev;
				if (j == e->prev) {
					if (j >= 0) {
						(sel->elems + j)->next = i;
					}
					f->prev = i;
					break;
				}
			} else {
				e->prev = j;
				j = f->next;
				if (j == e->next) {
					if (j >= 0) {
						(sel->elems + j)->prev = i;
					}
					f->next = i;
					break;
				}
			}
		}
	}
	if (e->prev < 0) sel->min = i;
	if (e->next < 0) sel->max = i; 
	sel->count++;
	return 1;
}
static int select_range(char *xs, int pos, int len, query_t* q, select_t* sel)
{
	range_t r;
	int count;
	
	count = 0;
	memset(&r, 0, sizeof(range_t));
	while (eval_query(xs, pos, len, q, &r)) {
		if (!r.next) {
			count += select_elem(xs, r.pos, r.len, sel);
		} else {
			count += select_range(xs, r.pos, r.len, q+1, sel);
		}
		pos = r.bot;
	}
	return count;
}

static int select_stream(char *xs, int len, char *path, char *sortby, char **rs, int *rn)
{
	query_t q[32];
	order_t o[32];
	select_t sel;
	elem_t *e;
	char *name, *s;
	int i, nameLen, pos;
	
	memset(&sel, 0, sizeof(select_t));
	if (parse_order(sortby, o, 32)) {
		sel.order = o;
	}
	
	if (!parse_query(path, q, 32)
	|| !init_range(xs, len, &pos, &len)) {
		return 0;
	}
	
	if (!select_range(xs, pos, len, q, &sel)) {
		free_stream(sel.elems);
		return 0;
	}
	
	name = NULL;
	for (i=0; q[i].name != NULL; i++) {
		name = q[i].name;
	}
	assert(name != NULL);
	if (isattrchr(*name)) name++;
	nameLen = keylen(name);
	
	*rn = 0;
	for (i=0; i < sel.count; i++) {
		(*rn) += (nameLen * 2) + 5;
		(*rn) += (sel.elems + i)->n;
	}
	
	s = *rs = alloc_stream(*rn);
	
	i = sel.min;
	while (i >= 0) {
		e = sel.elems + i;
		s += sprintf(s,
			"<%.*s>%.*s</%.*s>",
			nameLen, name,
			e->n, e->s,
			nameLen, name
		);
		i = e->next;
	}
	
	assert((s - *rs) == *rn);
	
	free_stream(sel.elems);
	
	return sel.count;
}

/*************************************************************
 * FETCH FIRST
 *************************************************************/
static int fetch_range(char *xs, int pos, int len, query_t* q, char **vs, int *vn, char **ks, int *kn)
{
	range_t r;
	
	memset(&r, 0, sizeof(range_t));
	while (eval_query(xs, pos, len, q, &r)) {
		if (!r.next) {
			trim_range(xs, r.pos, r.len, vs, vn);
			if (ks) {
				*ks = xs + r.top + 1;
				*kn = keylen(*ks);
			}
			return r.bot;
		} else {
			return fetch_range(xs, r.pos, r.len, q+1, vs, vn, ks, kn);
		}
		pos = r.bot;
	}
	return 0;
}
static int fetch_stream(char *xs, int pos, int len, char *path, char **vs, int* vn, char **ks, int *kn)
{
	query_t q[32];
	int p[4], i, j;
	*vs = "";
	*vn = 0;
	if (ks) {
		*ks = "";
		*kn = 0;
	}
	if (path) {
		pos = 0;
	}
	if (!pos) {
		if (!init_range(xs, len, &pos, &len)) {
			return 0;
		}
	}
	if (!path || isdigit(*path)) {
		j = path ? atoi(path) : 0;
		for (i=0; i <= j; i++) {
			if (!find_node(xs, pos, len, NULL, 1, p)) {
				return 0;
			}
			pos = p[3];
		}
		trim_range(xs, p[1], p[2], vs, vn);
		if (ks) {
			*ks = xs + p[0] + 1;
			*kn = keylen(*ks);
		}
	} else {
		if (!parse_query(path, q, 32)) {
			return 0;
		}
		pos = fetch_range(xs, pos, len, q, vs, vn, ks, kn);
	}
	return pos;
}

/*************************************************************
 * COUNT
 *************************************************************/
static int count_range(char *xs, int pos, int len, query_t* q)
{
	range_t r;
	int count;
	
	count = 0;
	memset(&r, 0, sizeof(range_t));
	while (eval_query(xs, pos, len, q, &r)) {
		if (!r.next) {
			count++;
		} else {
			count += count_range(xs, r.pos, r.len, q+1);
		}
		pos = r.bot;
	}
	return count;
}
static int count_stream(char *xs, int len, char *path)
{
	query_t q[32];
	int pos;
	
	if (!parse_query(path, q, 32)
	|| !init_range(xs, len, &pos, &len)) {
		return 0;
	}
	return count_range(xs, pos, len, q);
}


/*************************************************************
 * INDEX
 *************************************************************/
static int index_range(char *xs, int pos, int len, query_t* q)
{
	range_t r;
	
	memset(&r, 0, sizeof(range_t));
	while (eval_query(xs, pos, len, q, &r)) {
		if (!r.next) {
			assert(r.count > 0);
			return (r.count - 1);
		} else {
			return index_range(xs, r.pos, r.len, q+1);
		}
		pos = r.bot;
	}
	return -1;
}
static int index_stream(char *xs, int len, char *path)
{
	query_t q[32];
	int pos;
	
	if (!parse_query(path, q, 32)
	|| !init_range(xs, len, &pos, &len)) {
		return 0;
	}
	return index_range(xs, pos, len, q);
}

/*************************************************************
 * MAX
 *************************************************************/
static int max_range(char *xs, int pos, int len, query_t* q, char **rs, int *rn, int *asnum)
{
	range_t r;
	char *s;
	int count, n;
	
	count = 0;
	memset(&r, 0, sizeof(range_t));
	while (eval_query(xs, pos, len, q, &r)) {
		if (!r.next) {
			if (trim_range(xs, r.pos, r.len, &s, &n)) {
			}
			if (*asnum < 0) {
				*asnum = isnumbers(*rs = s, *rn = n);
			} else if (*asnum
			? (strtod(s, NULL) > strtod(*rs, NULL))
			: (strncmp(s, *rs, (n < *rn) ? n : *rn) >= (n <= *rn))
			) {
				*rs = s; *rn = n;
			}
			count++;
		} else {
			count += max_range(xs, r.pos, r.len, q+1, rs, rn, asnum);
		}
		pos = r.bot;
	}
	return count;
}
static int max_stream(char *xs, int len, char *path, char **rs, int *rn)
{
	query_t q[32];
	int pos, asnum;
	
	*rs = "";
	*rn = 0;
	asnum = -1;
	if (!parse_query(path, q, 32)
	|| !init_range(xs, len, &pos, &len)) {
		return 0;
	}
	return max_range(xs, pos, len, q, rs, rn, &asnum);
}

/*************************************************************
 * MIN
 *************************************************************/
static int min_range(char *xs, int pos, int len, query_t* q, char **rs, int *rn, int *asnum)
{
	range_t r;
	char *s;
	int count,n;
	
	count = 0;
	memset(&r, 0, sizeof(range_t));
	while (eval_query(xs, pos, len, q, &r)) {
		if (!r.next) {
			if (trim_range(xs, r.pos, r.len, &s, &n)) {
			}
			if (*asnum < 0) {
				*asnum = isnumbers(*rs = s, *rn = n);
			} else if (*asnum
			? (strtod(s, NULL) < strtod(*rs, NULL))
			: (strncmp(s, *rs, (n < *rn) ? n : *rn) < (n < *rn))
			) {
				*rs = s; *rn = n;
			}
			count++;
		} else {
			count += min_range(xs, r.pos, r.len, q+1, rs, rn, asnum);
		}
		pos = r.bot;
	}
	return count;
}
static int min_stream(char *xs, int len, char *path, char **rs, int *rn)
{
	query_t q[32];
	int pos, asnum;
	
	*rs = "";
	*rn = 0;
	asnum = -1;
	if (!parse_query(path, q, 32)
	|| !init_range(xs, len, &pos, &len)) {
		return 0;
	}
	return min_range(xs, pos, len, q, rs, rn, &asnum);
}

/*************************************************************
 * SUM
 *************************************************************/
static int sum_range(char *xs, int pos, int len, query_t* q, double *sum)
{
	range_t r;
	char *s;
	int count, n;
	
	count = 0;
	memset(&r, 0, sizeof(range_t));
	while (eval_query(xs, pos, len, q, &r)) {
		if (!r.next) {
			if (trim_range(xs, r.pos, r.len, &s, &n)) {
				*sum += strtod(s, NULL);
			}
			count++;
		} else {
			count += sum_range(xs, r.pos, r.len, q+1, sum);
		}
		pos = r.bot;
	}
	return count;
}
static int sum_stream(char *xs, int len, char *path, double *sum)
{
	query_t q[32];
	int pos;
	
	*sum = 0;
	if (!parse_query(path, q, 32)
	|| !init_range(xs, len, &pos, &len)) {
		return 0;
	}
	return sum_range(xs, pos, len, q, sum);
}

/****************************************************
 * FORMAT
 ****************************************************/
static int fput_unentity(char *s, int n, FILE* fp)
{
	char *e = s + n;
	int c;
	n = 0;
	while ((s < e) && ((c = *s++) != 0)) {
		if (c == '&') {
			c = entity_to_c(&s);
		}
		fputc(c, fp);
		n++;
	}
	return n;
}
static int is_value_key(char **r, char **k)
{
	char *s = *r;
	if (*s++ == '{') {
		while (isspace(*s)) s++;
		if (iskeychr(*s)) {
			*k = s++;
			while (iskeychr(*s)) s++;
			while (isspace(*s)) s++;
			if (*s++ == '}') {
				*r = s;
				return 1;
			}
		}
	}
	return 0;
}
static int is_block_key(char **r, char **k)
{
	int c, t, m;
	char *s = *r;
	*k = NULL;
	if ((*s == '<') || (*s == '[')) {
		c = (*s++ == '<') ? '>' : ']';
		t = (*s == '/') ? (*s++ << 8) : 0;
		if (!strncmp(s, "for", m=3)
		|| !strncmp(s, "if", m=2)
		|| !strncmp(s, "not", m=3)) {
			t |= *s;
			s += m;
			if ((*s == ':') && iskeychr(*(++s))) {
				*k = s++;
				while (iskeychr(*s)) s++;
			} else if (!(t >> 8)) {
				return 0;
			}
			while (isspace(*s)) s++;
			if (*s++ == c) {
				*r = s;
				return t;
			}
		}
	}
	return 0;
}
static int format_range(char *xs, int pos, int len,
	char *s, int n, FILE* fp, char **env, int *envsz, int ml)
{
	char *brk,*next,*end,*tag,*pre,*key,*k;
	int p[4],dir,spn,sz,type,t;
	sz = dir = 0;
	end = s + n;
	pre = next = s;
	brk = ml ? "<{" : "[{";
	key = NULL;
	while ((next < end)
	&& ((next = strpbrk(next, brk)) != NULL)
	&& (next < end)) {
		tag = next;
		if (is_value_key(&next, &k)) {
			if (dir <= 0) {
				if (tag > pre) {
					sz += fwrite(pre, 1, tag - pre, fp);
				}
				if (find_node(s = xs, pos, len, k, 0, p)
				|| ((s = *env) && find_node(s, 0, *envsz, k, 1, p))) {
					if (trim_range(s, p[1], p[2], &s, &n)) {
						sz += ml ? fwrite(s, 1, n, fp) : fput_unentity(s, n, fp);
						if (!*env || !find_node(*env, 0, *envsz, k, 1, p)) {
							insert_range(env, envsz, *envsz, k, s, n);
						}
					}
				}
				pre = next;
			}
			continue;
		} else if ((t = is_block_key(&next, &k)) >> 8) {
			if (--dir <= 0) {
				if (!key || ((t & 0xff) != type)) {
					dir++; /* bad terminate */
					sz += fprintf(fp, "bad type: %c\n", t & 0xff);
				} else if (k && !issamekey(k, key)) {
					dir++; /* bad term key */
					sz += fprintf(fp, "bad key: %.*s\n", keylen(k), k);
				} else {
					while ((tag > pre) && isspace(*tag) && !iscrlf(*tag)) tag--;
					while ((next < end) && iscrlf(*next)) next++;
					if ((spn = tag - pre) > 0) {
						if (type == 'f') { /* for */
							p[0] = 0;
							p[3] = pos;
							while (find_node(xs, p[3], len, key, p[0] > 0, p)) {
								if (trim_range(xs, p[1], p[2], &s, &n)) {
									sz += format_range(s, 0, n, pre, spn, fp, env, envsz, ml);
								}
							}
						} else { /* if || not */
							if (find_node(xs, pos, len, key, 0, p) == (type == 'i')) {
								sz += format_range(xs, pos, len, pre, spn, fp, env, envsz, ml);
							}
						}
					}
					pre = next;
					key = NULL;
					dir = 0;
				}
			}
			continue;
		} else if (t) {
			if (dir++ <= 0) {
				while ((tag > pre) && isspace(*tag) && !iscrlf(*tag)) tag--;
				while ((next < end) && iscrlf(*next)) next++;
				if (tag > pre) {
					sz += fwrite(pre, 1, tag - pre, fp);
				}
				pre = next;
				key = k;
				type = t;
			}
			continue;
		} else {
			next++;
		}
	}
	if (dir <= 0) {
		if (end > pre) {
			sz += fwrite(pre, 1, end - pre, fp);
		}
	}
	return sz;
}
static int format_stream(char *xs, int len, char *s, FILE* fp)
{
	char *newsp, *env;
	int pos, envsz, n;
	
	newsp = 0;
	if (strpbrk(s, "<{[")) {
		n = strlen(s);
	} else if (stream_from_file(s, &s, &n)) {
		newsp = s;
	} else {
		return 0;
	}
	
	env = NULL;
	envsz = 0;
	
	if (!init_range(xs, len, &pos, &len)) {
		n = 0;
	} else {
		while ((n > 0) && isspace(*s)) { s++; n--; }
		n = format_range(xs, pos, len, s, n, fp, &env, &envsz, *s == '<');
		free_stream(env);
	}
	
	if (newsp) {
		free_stream(newsp);
	}
	
	return n;
}

/*************************************************************
 * TRACE
 *************************************************************/
static int trace_range(char *xs, int len, int level, FILE* fp)
{
	char *last,*s;
	int p[4],pos,rows,i,n;
	last = NULL;
	pos = rows = 0;
	while (find_node(xs, pos, len, NULL, 1, p)) {
		s = xs + p[0] + 1;
		assert( iskeychr(*s) );
		if (!last || !issamekey(s, last)) {
			last = s;
			for (i=0; i < level; i++) {
				fputc('\t', fp);
			}
			for (i=0; iskeychr(s[i]); i++) ;
			fprintf(fp, "%.*s", i, s);
			s += i;
			while (isspace(*s)) s++;
			n = 0;
			while (iskeychr(*s)) {
				fputs(!n ? " (" : ", ", fp);
				for (i=0; iskeychr(s[i]); i++) ;
				fprintf(fp, "%.*s", i, s);
				s += i;
				while (isspace(*s)) s++;
				if (*s == '=') {
					s++;
					while (isspace(*s)) s++;
					if ((*s=='"') || (*s=='\'')) {
						i=*s++;
						while (*s && (*s != i)) s++;
						if (*s == i) s++;
					} else {
						while (*s && !isspace(*s)
						&& (*s != '>') && (*s != '/')) s++;
					}
					while (isspace(*s)) s++;
				} else {
					fputs("!?", fp);
				}
				n++;
			}
			if (n) fputc(')',fp);
			fputc('\n', fp);
			rows++;
			if (trim_range(xs, p[1], p[2], &s, &n) > 0) {
				rows += trace_range(s, n, level+1, fp);
			}
		}
		pos = p[3];
	}
	return rows;
}
static int trace_stream(char *xs, int len, FILE* fp)
{
	return trace_range(xs, len, 0, fp);
}

/************************************************************
 * JSOBJECT TO XML
 ************************************************************/
static int entity_xml(char *d, char *s, int n)
{
	char *e, *es;
	int c;
	e = s + n;
	n=0;
	while ((s < e) && ((c = *s++) != 0)) {
		if ((es = c_to_entity(c)) != NULL) {
			if (d) strcpy(d + n, es);
			n += strlen(es);
		} else {
			if (d) d[n]=c;
			n++;
		}
	}
	if (d) d[n]=0;
	return n;
}
static int unentity_xml(char *d, char *s, int n)
{
	char *e = s + n;
	int c;
	n = 0;
	while ((s < e) && ((c = *s++) != 0)) {
		if (c == '&') {
			c = entity_to_c(&s);
		}
		d[n++] = c;
	}
	d[n] = 0;
	return n;
}
static int var_to_xml(exec_t* ex, var_t* v, char *dst, char *tag, int taglen)
{
	object_t* obj;
	each_t e;
	char *s;
	int i, n, len;
	len = 0;
	if (tag) {
		if (!dst) {
			len += (taglen * 2) + 5;
		} else {
			len += sprintf(dst + len, "<%.*s>", taglen, tag);
		}
	}
	if (v->p == object_proto) {
		obj = v->u.p;
		for (i = 0; i < obj->n; i++) {
			if (object_each(obj, i, &e)
			&& symbol_to_string(ex, e.k, &s, &n)) {
				len += var_to_xml(ex, &e.v, dst ? (dst + len) : NULL, s, n);
			}
		}
	} else if (var_toString(ex, v, &s, &n)) {
		len += entity_xml(dst ? (dst + len) : NULL, s, n);
	}
	if (tag && dst) {
		len += sprintf(dst + len, "</%.*s>", taglen, tag);
	}
	return len;
}
static int var_from_xml(exec_t* ex, var_t* v, char *xs, int xn)
{
	object_t *obj;
	char *s;
	int p[4], xi, n, k;
	if (!xs || !xn) return 0;
	if (!memchr(xs, '<', xn)) {
		n = unentity_xml(s = calloc(xn + 1, 1), xs, xn);
		new_string_with(ex, v, s, n);
		return 1;
	}
	if (!init_range(xs, xn, &xi, &xn)) {
		return 0;
	}
	obj = object_alloc(ex);
	while (find_node(xs, xi, xn, NULL, 1, p)) {
		if (((n = keylen(s = xs + p[0] + 1)) > 0)
		&& ((k = string_to_symbol(ex, s, n)) != 0)
		&& trim_range(xs, p[1], p[2], &s, &n)
		&& var_from_xml(ex, v, s, n)) {
			object_set(ex, obj, k, v);
		}
		xi = p[3];
	}
	new_object(ex, v, obj);
	return 1;
}

/************************************************************
 * XML JS INTERFACE
 ************************************************************/
typedef struct {
	char *s;
	int n;
	int i;
	char *f;
} xml_t;

int new_xml_from_string(exec_t* ex, var_t* v, char *s, int n, char *f)
{
	xml_t *x;
	v->p = NULL;
	x = (xml_t*)alloc_link(ex, sizeof(xml_t), xml_proto);
	if (f) {
		x->f = strdup(f);
	}
	x->s = s;
	x->n = n;
	v->p = xml_proto;
	v->u.p = (void*)x;
	return 1;
}

int new_xml_from_file(exec_t* ex, var_t* v, char *filename, char* mode)
{
	char *s;
	int n, w;
	
	w = mode && strpbrk(mode, "wc");
	if (!open_stream(filename, &s, &n)) {
		if (!w) {
			/* open error */
			return 0;
		}
		n = 0;
		assert((s = alloc_stream(BUFSIZ)) != NULL);
	}
	return new_xml_from_string(ex, v, s, n, w ? filename : NULL);
}

/************************************************************
 * PROTOTYPE FUNCTIONS
 ************************************************************/
void xml_clear(exec_t* ex, void *p)
{
	xml_t *x = p;
	if (x->f) {
		free(x->f);
		x->f = NULL;
	}
	if (x->s) {
		free_stream(x->s);
		x->s = NULL;
	}
	x->n = 0;
}
void xml_trace(exec_t* ex, void *p)
{
	xml_t *x = p;
	trace_stream(x->s, x->n, ex->out);
}
int xml_toString(exec_t* ex, var_t* v, char **s, int *n)
{
	xml_t *x = v->u.p;
	if (x->s && x->n) {
		*s = x->s;
		*n = x->n;
		return *n;
	}
	*s = NULL;
	*n = 0;
	return 0;
}
int xml_getLength(exec_t* ex, void *p)
{
	return ((xml_t*)p)->n;
}

/************************************************************
 *
 ************************************************************/
static int _xml_write(exec_t* ex, xml_t* x, int argc, var_t* argv, var_t* res)
{
	FILE *fp;
	char *fn;
	fn = NULL;
	fp = ex->out;
	exerrif(ex, !x->s, "bad xml");
	if (argc >= 1) {
		if (argv->p == file_proto) {
			fp = *(FILE**)(argv->u.p);
		} else {
			exerrif(ex, !(fn = var_toChars(ex, argv)), "bad argument 0");
			exerrif(ex, !(fp = fopen(fn, "w")), "fopen error");
		}
	}
	new_int(res, write_stream(x->s, x->n, fp));
	if (fn) {
		fclose(fp);
	}
	return 1;
}
static int _xml_close(exec_t* ex, xml_t* x, int argc, var_t* argv, var_t* res)
{
	FILE * fp;
	int r = 0;
	
	exerrif(ex, !x->s, "bad xml");
	exerrif(ex, argc != 0, "bad arguments");
	
	if (x->f) {
		if (!(fp = fopen(x->f, "w"))) {
			fprintf(ex->err, "fopen error: %s %s\n", x->f, strerror(errno));
		} else {
			r = write_stream(x->s, x->n, fp);
			fclose(fp);
		}
		free(x->f);
		x->f = NULL;
	} else {
		if (x->s) {
			r = x->n;
		}
	}
	
	free_stream(x->s);
	x->s = NULL;
	
	new_int(res, r);
	return 1;
}
static int _xml_count(exec_t* ex, xml_t* x, int argc, var_t* argv, var_t* res)
{
	char *path;
	exerrif(ex, !x->s, "bad xml");
	exerrif(ex, (argc != 1) || !(path = var_toChars(ex, argv)), "bad arguments");
	new_int(res, count_stream(x->s, x->n, path));
	return 1;
}
static int _xml_index(exec_t* ex, xml_t* x, int argc, var_t* argv, var_t* res)
{
	char *path;
	exerrif(ex, !x->s, "bad xml");
	exerrif(ex, (argc != 1) || !(path = var_toChars(ex, argv)), "bad arguments");
	new_int(res, index_stream(x->s, x->n, path));
	return 1;
}
static int _xml_min(exec_t* ex, xml_t* x, int argc, var_t* argv, var_t* res)
{
	char *path, *s;
	int n;
	exerrif(ex, !x->s, "bad xml");
	exerrif(ex, (argc != 1) || !(path = var_toChars(ex, argv)), "bad arguments");
	if (!min_stream(x->s, x->n, path, &s, &n) || !s || (n<=0)) {
		return 0;
	}
	if (isnumbers(s, n)) {
		new_float(res, strtod(s, NULL));
	} else {
		new_string_copy(ex, res, s, n);
	}
	return 1;
}
static int _xml_max(exec_t* ex, xml_t* x, int argc, var_t* argv, var_t* res)
{
	char *path, *s;
	int n;
	exerrif(ex, !x->s, "bad xml");
	exerrif(ex, (argc != 1) || !(path = var_toChars(ex, argv)), "bad arguments");
	if (!max_stream(x->s, x->n, path, &s, &n) || !s || (n<=0)) {
		return 0;
	}
	if (isnumbers(s, n)) {
		new_float(res, strtod(s, NULL));
	} else {
		new_string_copy(ex, res, s, n);
	}
	return 1;
}
static int _xml_sum(exec_t* ex, xml_t* x, int argc, var_t* argv, var_t* res)
{
	char *path;
	double f;
	exerrif(ex, !x->s, "bad xml");
	exerrif(ex, (argc != 1) || !(path = var_toChars(ex, argv)), "bad arguments");
	if (!sum_stream(x->s, x->n, path, &f)) {
		return 0;
	}
	new_float(res, f);
	return 1;
}
static int _xml_fetch(exec_t* ex, xml_t* x, int argc, var_t* argv, var_t* res)
{
	char *path, *s;
	int n;
	path = NULL;
	exerrif(ex, !x->s, "bad xml");
	exerrif(ex, (argc >= 1) && !(path = var_toChars(ex, argv)), "bad argument 0");
	if (!(x->i = fetch_stream(x->s, x->i, x->n, path, &s, &n, NULL, NULL))) {
		return 0;
	}
	if (!var_from_xml(ex, res, s, n)) {
		return 0;
	}
	return 1;
}
static int _xml_select(exec_t* ex, xml_t* x, int argc, var_t* argv, var_t* res)
{
	char *path, *sortby, *s;
	int n;
	path = sortby = NULL;
	exerrif(ex, !x->s, "bad xml");
	exerrif(ex, (argc < 1) || !(path = var_toChars(ex, argv)), "bad arguments");
	exerrif(ex, (argc >= 2) && !(sortby = var_toChars(ex, argv + 1)), "bad argument 1");
	if (!select_stream(x->s, x->n, path, sortby, &s, &n)) {
		return 0;
	}
	new_xml_from_string(ex, res, s, n, NULL);
	return 1;
}
static int _xml_delete(exec_t* ex, xml_t* x, int argc, var_t* argv, var_t* res)
{
	char *path;
	exerrif(ex, !x->s, "bad xml");
	exerrif(ex, (argc != 1) || !(path = var_toChars(ex, argv)), "bad arguments") ;
	new_int(res, delete_stream(&x->s, &x->n, path));
	return 1;
}
static int _xml_modify(exec_t* ex, xml_t* x, int argc, var_t* argv, var_t* res)
{
	char *path, *s;
	int n;
	s = NULL;
	n = 0;
	exerrif(ex, !x->s, "bad xml");
	exerrif(ex, (argc < 1) || !(path = var_toChars(ex, argv)), "bad arguments") ;
	if (argc >= 2) {
		exerrif(ex, !(n = var_to_xml(ex, argv + 1, NULL, NULL, 0)), "bad argument 0");
		assert((s = calloc(n + 1, 1)) != NULL);
		assert(var_to_xml(ex, argv + 1, s, NULL, 0) == n);
	}
	new_int(res, modify_stream(&x->s, &x->n, path, s, n));
	if (s) {
		free(s);
	}
	return 1;
}
static int _xml_formatf(exec_t* ex, xml_t* x, int argc, var_t* argv, var_t* res)
{
	char *path, *fn;
	FILE *fp;
	fp = ex->out;
	path = fn = NULL;
	
	exerrif(ex, !x->s, "bad xml");
	exerrif(ex, (argc < 1) || !(path = var_toChars(ex, argv)), "bad arguments") ;
	if (argc >= 2) {
		if ((argv + 1)->p == file_proto) {
			fp = *(FILE**)((argv + 1)->u.p);
		} else {
			exerrif(ex, !(fn = var_toChars(ex, argv + 1)), "bad argument 1");
			exerrif(ex, !(fp = fopen(fn, "w")), "fopen error");
		}
	}
	
	new_int(res, format_stream(x->s, x->n, path, fp));
	if (fn) {
		fclose(fp);
	}
	return 1;
}
static int _xml_formats(exec_t* ex, xml_t* x, int argc, var_t* argv, var_t* res)
{
	char *path, *s;
	FILE *fp;
	int n;
	path = NULL;
	exerrif(ex, !x->s, "bad xml");
	exerrif(ex, (argc != 1) || !(path = var_toChars(ex, argv)), "bad arguments") ;
	assert((fp = tmpfile()) != 0);
	n = format_stream(x->s, x->n, path, fp);
	s = calloc(n + 1, 1);
	rewind(fp);
	fread(s, 1, n, fp);
	fclose(fp);
	new_string_with(ex, res, s, n);
	return 1;
}

/************************************************************
 *
 ************************************************************/
int xml_callFunc(exec_t* ex, void* p, int sym, int argc, var_t* argv, var_t* res)
{
	switch (sym) {
#include "_xml_call.h"
	}
	return -1;
}
