/**
 * @file   init_phmm.c
 * @author Akinobu LEE
 * @date   Tue Feb 15 23:05:33 2005
 * 
 * <JA>
 * @brief  %HMM ե뤪HMMListޥåԥ󥰥եΥɤ߹ߤȽ
 * </JA>
 * 
 * <EN>
 * @brief  Load %HMM definition file and HMMList mapping file into memory and set it up.
 * </EN>
 * 
 * $Revision:$
 * 
 */
/*
 * Copyright (c) 1991-2005 Kawahara Lab., Kyoto University
 * Copyright (c) 2000-2005 Shikano Lab., Nara Institute of Science and Technology
 * Copyright (c) 2005      Julius project team, Nagoya Institute of Technology
 * All rights reserved
 */

#include <sent/stddefs.h>
#include <sent/htk_hmm.h>

/* initialize phoneme HMM structure */
/** 
 * Allocate memory for a new %HMM definition data.
 * 
 * @return pointer to the newly allocated %HMM definition data.
 */
HTK_HMM_INFO *
hmminfo_new()
{
  HTK_HMM_INFO *new;

  new = (HTK_HMM_INFO *)mymalloc(sizeof(HTK_HMM_INFO));

  return(new);
}

/** 
 * @brief Load HTK %HMM definition file and HMMList file, and setup phone %HMM information.
 *
 * First try ascii format, then try binary format.
 * 
 * @param hmminfo [out] pointer to store all the %HMM definition data.
 * @param hmmfilename [in] file name of HTK %HMM definition file, NULL if not.
 * @param namemapfile [in] file name of HMMList mapping file.
 */
void
init_hmminfo(HTK_HMM_INFO *hmminfo, char *hmmfilename, char *namemapfile)
{
  FILE *fp;
  boolean ok_p;

  ok_p = FALSE;

  /* read hmmdef file */
  j_printerr("Reading in HMM definition...");
  /* first, try ascii format */
  if ((fp = fopen_readfile(hmmfilename)) == NULL) {
    j_error("failed to open %s\n",hmmfilename);
  }
  if (rdhmmdef(fp, hmminfo) == TRUE) {
    ok_p = TRUE;
  }
  if (fclose_readfile(fp) < 0) {
    j_error("failed to close %s\n", hmmfilename);
  }
  if (ok_p == FALSE) {
    /* second, try binary format */
    if ((fp = fopen_readfile(hmmfilename)) == NULL) {
      j_error("failed to open %s\n",hmmfilename);
    }
    if (read_binhmm(fp, hmminfo) == TRUE) {
      ok_p = TRUE;
    }
    if (fclose_readfile(fp) < 0) {
      j_error("failed to close %s\n", hmmfilename);
    }
  }
  if (ok_p == FALSE) {
    j_error("failed to read %s\n", hmmfilename);
  }

  j_printerr("   defined HMMs: %5d\n", hmminfo->totalhmmnum);

  /* make mapping from logically named HMM to real defined HMM name */
  if (namemapfile != NULL) {
    /* use HMMList map file */
    if ((fp = fopen_readfile(namemapfile)) == NULL) {
      j_error("failed to open %s\n",namemapfile);
    }
    if (rdhmmlist(fp, hmminfo) == FALSE) {
      j_error("HMMList \"%s\" read error\n",namemapfile);
    }
    if (fclose_readfile(fp) < 0) {
      j_error("failed to close %s\n", namemapfile);
    }
    j_printerr("  logical names: %5d in HMMList\n", hmminfo->totallogicalnum);
  } else {
    /* add all names of physical HMMs as logical names */
    hmm_add_physical_to_logical(hmminfo);
    j_printerr("  logical names: %5d\n", hmminfo->totallogicalnum);
  }

  /* extract basephone */
  make_hmm_basephone_list(hmminfo);
  j_printerr("    base phones: %5d used in logical\n", hmminfo->basephone.num);

  /* Guess we need to handle context dependency */
  /* (word-internal CD is done statically, cross-word CD dynamically */
  if (guess_if_cd_hmm(hmminfo)) {
    hmminfo->is_triphone = TRUE;
  } else {
    hmminfo->is_triphone = FALSE;
  }

  hmminfo->sp = NULL;

  j_printerr("done\n");
}

/** 
 * Set up pause model.
 * 
 * @param hmminfo [i/o] %HMM definition data
 * @param spmodel_name [in] name string of short pause model
 */
void
htk_hmm_set_pause_model(HTK_HMM_INFO *hmminfo, char *spmodel_name)
{
  HMM_Logical *l;

  l = htk_hmmdata_lookup_logical(hmminfo, spmodel_name);
  if (l == NULL) {
    j_printerr("Warning: no model named as \"%s\"\n", spmodel_name);
  }
  hmminfo->sp = l;
}
