/*
 * Copyright (c) 1991-2003 Kyoto University
 * Copyright (c) 2000-2003 NAIST
 * All rights reserved
 */

/* backtrellis.c --- keep trellis at 1st pass and provide access to 2nd pass */
/* $BBh(B1$B%Q%9$N7k2L$r(B BACKTRELLIS $B$H$7$FJ]B8$7Bh(B2$B%Q%9$G;2>H$9$k(B */

/* $Id: backtrellis.c,v 1.7 2003/09/29 06:01:22 ri Exp $ */

#include <julius.h>

/* backtrellis $B$r=i4|2=$9$k(B($B5/F0;~$K(B1$B2s$@$1<B9T(B) */
/* initialize backtrellis (once on startup) */
void
bt_init(BACKTRELLIS *bt)
{
  bt->num  = NULL;
  bt->rw   = NULL;
  bt->list = NULL;
  bt->root = NULL;
}

/* backtrellis $B$r=`Hw$9$k(B ($BG'<13+;O;~$4$H$K<B9T(B) */
/* prepare backtrellis (beginning of each speech input) */
void
bt_prepare(BACKTRELLIS *bt)
{
  TRELLIS_ATOM *tre, *tmp;
  int t;

  /* free previous data */
  if (bt->root != NULL) free(bt->root);
  if (bt->rw != NULL) free(bt->rw);
  if (bt->num != NULL) free(bt->num);
  bt->num = NULL;
  bt->rw = NULL;
  bt->root = NULL;

  tre = bt->list;
  while (tre != NULL) {
    tmp = tre->next;
    free(tre);
    tre = tmp;
  }

  /* reset entry point */
  bt->list = NULL;
}  


/* $BBh(B1$B%Q%9$G=P8=$7$?C18l=*C<$N%H%l%j%9>pJs(B (tatom) $B$r%P%C%/%H%l%j%9(Bbt$B$K3JG<(B */
/* store trellis atom (= word end) to backtrellis */
/* $B$3$3$G$O3JG<$@$1$7$F!$Bh(B1$B%Q%9=*N;8e$K(B bt_relocate_rw() $B$G%U%l!<%`=g$K:FG[CV$9$k(B */
/* this function just store the new atom into list.
   they will be re-located per frame after 1st pass for quicker access in 2nd pass */
void
bt_store(BACKTRELLIS *bt, TRELLIS_ATOM *tatom)
{
#ifdef WORD_GRAPH
  tatom->within_wordgraph = FALSE;
#endif
  tatom->next = bt->list;
  bt->list = tatom;
  /*j_printf("  %3d: %15s %f < %3d %15s\n", tatom->endtime,
	 winfo->woutput[tatom->wid], tatom->backscore,
	 tatom->begintime, winfo->woutput[tatom->last_tre->wid]);*/
}

/* $BBh(B1$B%Q%9=*N;8e(B, $B3JG<$5$l$?C18l%H%l%j%9>pJs$r%U%l!<%`=g$K:FG[CV$9$k(B */
/* re-locate the stored atom lists per frame (will be called after 1st pass) */
void
bt_relocate_rw(BACKTRELLIS *bt)
{
  TRELLIS_ATOM *tre;
  int t;
  int totalnum, n;

  bt->num = (int *)mymalloc(sizeof(int) * bt->framelen);

  /* count number of trellis atom (= survived word end) for each frame */
  for (t=0;t<bt->framelen;t++) bt->num[t] = 0;
  totalnum = 0;
  for (tre=bt->list;tre;tre=tre->next) {
#ifdef SP_BREAK_CURRENT_FRAME
    /* the last frame (when triggered from sp to non-sp) should be discarded */
    if (tre->endtime >= bt->framelen) continue;
#endif      
    bt->num[tre->endtime]++;
    totalnum++;
  }
  /* if no atom found, return here with all bt->num[t] set to 0 */
  if (totalnum <= 0) return;
  
  /* allocate area */
  bt->root = (TRELLIS_ATOM **)mymalloc(sizeof(TRELLIS_ATOM *) * totalnum);
  bt->rw  = (TRELLIS_ATOM ***)mymalloc(sizeof(TRELLIS_ATOM **) * bt->framelen);
  n = 0;
  for (t=0;t<bt->framelen;t++) {
    if (bt->num[t] > 0) {
      bt->rw[t] = (TRELLIS_ATOM **)&(bt->root[n]);
      n += bt->num[t];
    }
  }
  /* then store the atoms */
  for (t=0;t<bt->framelen;t++) bt->num[t] = 0;
  for (tre=bt->list;tre;tre=tre->next) {
#ifdef SP_BREAK_CURRENT_FRAME
    /* the last frame (when triggered from sp to non-sp) should be discarded */
    if (tre->endtime >= bt->framelen) continue;
#endif      
    t = tre->endtime;
    bt->rw[t][bt->num[t]] = tre;
    bt->num[t]++;
  }
}


/* $B0J2<$N4X?t$O(B bt_relocate_rw $B<B9T8e$K$N$_;HMQ2DG=$H$J$k!%(B*/
/* functions below this line should be called after bt_relocate_rw() */

#ifdef SP_BREAK_CURRENT_FRAME
/* $BF~NO%;%0%a%s%H$NN>C<$K;D$C$?:GL`C18l2>@b$r<h$j=P$7(B, $B$=$l$i$r(B
 $BBh(B2$B%Q%9$K$*$1$k=i4|(B/$B:G=*2>@b$H$7$F%;%C%H$9$k(B */
/* extract the best word hypothesis on each end of input segment, and
   store them as start/end word in the 2nd pass */
void
set_terminal_words(BACKTRELLIS *bt)
{
  LOGPROB maxscore;
  int i,t;

  maxscore = LOG_ZERO;
  for(t=bt->framelen-1;t>=0;t--) {
    if (bt->num[t] > 0) break;
  }
  for(i=0;i<bt->num[t];i++) {
    if (maxscore < (bt->rw[t][i])->backscore) {
      maxscore = (bt->rw[t][i])->backscore;
      sp_break_2_begin_word = (bt->rw[t][i])->wid;
    }
  }
  maxscore = LOG_ZERO;
  for(t=0;t<bt->framelen;t++) {
    if (bt->num[t] > 0) break;
  }
  for(i=0;i<bt->num[t];i++) {
    if (maxscore < (bt->rw[t][i])->backscore) {
      maxscore = (bt->rw[t][i])->backscore;
      sp_break_2_end_word = (bt->rw[t][i])->wid;
    }
  }
#ifdef SP_BREAK_DEBUG
  printf("2nd pass begin word: %s\n",
	 (sp_break_2_begin_word == WORD_INVALID) ? "WORD_INVALID" : winfo->wname[sp_break_2_begin_word]);
  printf("2nd pass end word: %s\n",
	 (sp_break_2_end_word == WORD_INVALID) ? "WORD_INVALID" : winfo->wname[sp_break_2_end_word]);
#endif
}
#endif /* SP_BREAK_CURRENT_FRAME */

/* $BBh(B1$B%Q%9=*N;8e(B, $BBh(B2$B%Q%9$G$N@\B3$N$?$a$K!$@\B3E@$N>uBV$N=PNO3NN($r%9%3%"$+$i(B
   $B:9$70z$$$F$*$/(B */
/* the outprob on the trellis connection point should be discounted */
void
bt_discount_pescore(WCHMM_INFO *wchmm, BACKTRELLIS *bt, HTK_Param *param)
{
  int t,i;
  TRELLIS_ATOM *tre;
  
  for (t=0; t<bt->framelen; t++) {
    for (i=0; i<bt->num[t]; i++) {
      tre = bt->rw[t][i];
      tre->backscore -= outprob_style(wchmm, wchmm->wordend[tre->wid], tre->last_tre->wid, t, param);
    }
  }

}

/* bt_relocate_rw $B=*N;8e(B, $BC18l(BID$B$G%=!<%H$7$F$*$/(B */
/* sort trellis atoms on each frame by word ID */
static int
compare_wid(TRELLIS_ATOM **a, TRELLIS_ATOM **b)
{
  if ((*a)->wid > (*b)->wid) return 1;
  if ((*a)->wid < (*b)->wid) return -1;
  return 0;
}
void
bt_sort_rw(BACKTRELLIS *bt)
{
  int t;

  for (t=0;t<bt->framelen;t++) {
    qsort(bt->rw[t], bt->num[t], sizeof(TRELLIS_ATOM *),
	  (int (*)(const void *,const void *))compare_wid);
  }
}


/* $B0J2<$N4X?t$O;vA0$K(Bbt_sort_rw() $B$,8F$P$l$F$$$k$3$H(B($BBh(B2$B%Q%9MQ(B) */
/* functions below should be called after bt_sort_rw() */

/* $B;~9o(B t $B>e$NC18l=*C<%H%l%j%9>e$GC18l(B wkey $B$r8!:w$7$F7k2L$rJV$9(B */
/* $B$"$l$P$=$N%H%l%j%9C18l>pJs$X$N%]%$%s%?(B, $B$J$1$l$P(B NULL $B$rJV$9(B */
/* search trellis atom (= word end) with word ID 'wkey' on frame 't' */
/* return pointer to the found trellis atom, and NULL on failure */
TRELLIS_ATOM *
bt_binsearch_atom(BACKTRELLIS *bt, int t, WORD_ID wkey)
{
  /* do binary search */
  /* assume rw are ordered by wid */
  int left, right, mid;
  TRELLIS_ATOM *tmp;
#ifdef WPAIR
  int i;
  LOGPROB maxscore;
  TRELLIS_ATOM *maxtre;
#endif
  
  if (bt->num[t] == 0) return(NULL);
  
  left = 0;
  right = bt->num[t] - 1;
  while (left < right) {
    mid = (left + right) / 2;
    if ((bt->rw[t][mid])->wid < wkey) {
      left = mid + 1;
    } else {
      right = mid;
    }
  }
  tmp = bt->rw[t][left];
  if (tmp->wid == wkey) {
#ifdef WPAIR
    /* same word with different context will be found:
       most likely one will be returned */
    maxscore = LOG_ZERO;
    i = left;
    while (i >= 0 && (tmp = bt->rw[t][i])->wid == wkey) {
      if (maxscore < tmp->backscore) {
	maxscore = tmp->backscore;
	maxtre = tmp;
      }
      i--;
    }
    i = left;
    while (i < bt->num[t] && (tmp = bt->rw[t][i])->wid == wkey) {
      if (maxscore < tmp->backscore) {
	maxscore = tmp->backscore;
	maxtre = tmp;
      }
      i++;
    }

    tmp = maxtre;
#endif /* WORD_GRAPH */

    return(tmp);
  } else {
    return(NULL);
  }
}
