/*
 * Copyright (c) 1991-2003 Kyoto University
 * Copyright (c) 2000-2003 NAIST
 * All rights reserved
 */

/* record.c --- record recognized speech input into successive file */

/* $Id: record.c,v 1.4 2003/09/29 06:01:22 ri Exp $ */

/* filenames are YYYY.MMDD.HHMMSS.wav */

#include <julius.h>
#include <time.h>

#define TSTRLEN 18

static char tstr[TSTRLEN];
static char recfilename[MAXLINELEN], finalfilename[MAXLINELEN];
static FILE *fp;
static int totalnum;

/* make base filename string from current system time */
static void
timestring(char *t, int maxlen)
{
  time_t timep;
  struct tm *lmtm;

  time(&timep);
  lmtm = localtime(&timep);

  snprintf ( t, maxlen,"%04d.%02d%02d.%02d%02d%02d", 1900+lmtm->tm_year, 1+lmtm->tm_mon, lmtm->tm_mday, lmtm->tm_hour, lmtm->tm_min, lmtm->tm_sec);
}

/* make file pathname: [record_dirname]/[timestring()].wav */
static void
make_record_filename(char *buf, int buflen, char *filename)
{
  if (record_dirname == NULL) {
    j_error("no record directory specified??\n");
  }
  snprintf(buf, buflen, "%s/%s.wav", record_dirname, filename);
}

/* make temporary pathname: record_dirname/tmprecord.$(pid) */
static void
make_tmp_filename(char *buf, int buflen)
{
  snprintf(buf, buflen, "%s/tmprecord.%d", record_dirname, getpid());
}  

/* open temporary file for recording */
void
record_sample_open()
{
  make_tmp_filename(recfilename, MAXLINELEN);
  if ((fp = wrwav_open(recfilename, smpFreq)) == NULL) {
    perror("Error: record_sample_open");
    j_error("failed to open \"%s\" (temporary record file)\n", recfilename);
  }

  totalnum = 0;
}

/* write samples to file */
void
record_sample_write(SP16 *speech, int samplenum)
{

  if (wrwav_data(fp, speech, samplenum) == FALSE) {
    perror("Error: record_sample_write");
    j_error("failed to write samples to \"%s\"\n", recfilename);
  }

  /* make timestamp of system time when an input begins */
  /* the current temporal recording file will be renamed to this time-stamp filename */
  if (totalnum == 0) {		/* beginning of recording */
    timestring(tstr, TSTRLEN);
  }
  
  totalnum += samplenum;
}

/* close the current file */
void
record_sample_close()
{
  if (wrwav_close(fp) == FALSE) {
    perror("Error: record_sample_close");
  }

  /* now rename the temporary file to time-stamp filename */
  make_record_filename(finalfilename, MAXLINELEN, tstr);
  if (rename(recfilename, finalfilename) < 0) {
    perror("Error: record_sample_close");
    j_error("failed to move %s to %s\n", recfilename, finalfilename);
  }
  if (verbose_flag) {
    j_printerr("recorded to \"%s\" (%d bytes, %.2f sec.)\n", finalfilename, totalnum * sizeof(SP16), (float)totalnum / (float) smpFreq);
  }
}
