/*
 * Copyright (c) 1991-2003 Kyoto University
 * Copyright (c) 2000-2003 NAIST
 * All rights reserved
 */

/* search_bestfirst_v1.c --- hypothesis handling and scoring functions
   for backscan algorithm */

/* $Id: search_bestfirst_v1.c,v 1.9 2003/09/29 06:01:22 ri Exp $ */

#include <julius.h>
#ifndef PASS2_STRICT_IWCD

#undef TCD			/* enable debug messages */

/**********************************************************************/
/************ $B2>@b%N!<%I$N4pK\A`:n(B                         ************/
/************ basic functions for hypothesis node handling ************/
/**********************************************************************/

/* node $B$r2rJ|(B */
/* free node */
void
free_node(NODE *node)
{
  if (node == NULL) return;
  free(node->g);
  if (ccd_flag) free(node->g_prev);

  free(node);
}

/* src $B$+$i(B dst $B$X%3%T!<(B */
/* copy node from 'src' to 'dst' */
NODE *
cpy_node(NODE *dst, NODE *src)
{
  
  dst->next = src->next;
  dst->prev = src->prev;
  memcpy(dst->g, src->g, sizeof(LOGPROB) * peseqlen);
  memcpy(dst->seq, src->seq, sizeof(WORD_ID) * MAXSEQNUM);
#ifdef CM_SEARCH
#ifdef CM_MULTIPLE_ALPHA
  {
    int w;
    for(w=0;w<src->seqnum;w++) {
      memcpy(dst->cmscore[w], src->cmscore[w], sizeof(LOGPROB) * cm_alpha_num);
    }
  }     
#else
  memcpy(dst->cmscore, src->cmscore, sizeof(LOGPROB) * MAXSEQNUM);
#endif
#endif /* CM_SEARCH */
  dst->seqnum = src->seqnum;
  dst->score = src->score;
  dst->bestt = src->bestt;
  dst->estimated_next_t = src->estimated_next_t;
  dst->endflag = src->endflag;
#ifdef USE_DFA
  dst->state = src->state;
#endif
  dst->tre = src->tre;
  if (ccd_flag) {
    memcpy(dst->g_prev, src->g_prev, sizeof(LOGPROB)*peseqlen);
    dst->last_ph = src->last_ph;
#ifdef USE_NGRAM
    dst->lscore = src->lscore;
#endif
  }
#ifdef USE_NGRAM
  dst->totallscore = src->totallscore;
#endif
  return(dst);
}

/* $B?7$?$J2>@b%N!<%I$r3d$jIU$1(B */
/* allocate new node */
NODE *
newnode()
{
  NODE *tmp;
  int i;

  if ((tmp=(NODE *)mymalloc(sizeof(NODE)))==NULL) {
    j_error("can't malloc\n");
  }
  /*bzero(tmp,sizeof(NODE));*/
  tmp->next=NULL;
  tmp->prev=NULL;
  tmp->g = (LOGPROB *)mymalloc(sizeof(LOGPROB)*peseqlen);
  if (ccd_flag) {
    tmp->g_prev = (LOGPROB *)mymalloc(sizeof(LOGPROB)*peseqlen);
    tmp->last_ph = NULL;
#ifdef USE_NGRAM
    tmp->lscore = LOG_ZERO;
    tmp->totallscore = LOG_ZERO;
#endif
  }
  tmp->endflag = FALSE;
  tmp->seqnum = 0;
  for(i=0;i<peseqlen;i++) {
    tmp->g[i] = LOG_ZERO;
    if (ccd_flag) tmp->g_prev[i] = LOG_ZERO;
  }

  return(tmp);
}


/**********************************************************************/
/************ $BA08~$-%H%l%j%9E83+$HL`EY7W;;(B             ****************/
/************ expand trellis and update forward score *****************/
/**********************************************************************/

static LOGPROB **wordtrellis;	/* buffer for computing viterbi of a word */
static LOGPROB *g;
static HMM_Logical **phmmseq;
static int phmmlen_max;

/* $B#1C18lJ,$N%H%l%j%97W;;MQ$N%o!<%/%(%j%"(B wordtrellis[t][]$B$r3NJ](B */
/* allocate work area 'wordtrellis[t][]' for trellis computation of a word */
void
malloc_wordtrellis()
{
  int i;
  int maxwn;
  LOGPROB *p;

  maxwn = winfo->maxwn + 10;
  wordtrellis = (LOGPROB **)mymalloc(sizeof(LOGPROB *) * peseqlen);
  p = (LOGPROB *)mymalloc(sizeof(LOGPROB) * peseqlen * maxwn);
  for (i=0;i<peseqlen;i++) {
    wordtrellis[i] = p + i * maxwn;
  }
  wordtrellis[0][maxwn] = 1.0;

  g = (LOGPROB *)mymalloc(sizeof(LOGPROB) * peseqlen);

  phmmlen_max = winfo->maxwlen + 2;
  phmmseq = (HMM_Logical **)mymalloc(sizeof(HMM_Logical *) * phmmlen_max);
}
/* $B>e5-$N%o!<%/%(%j%"(B wordtrellis[t][] $B$r2rJ|(B */
/* free the 'wordtrellis[t][]' */
void
free_wordtrellis()
{
  free(wordtrellis[0]);
  free(wordtrellis);
  free(g);
  free(phmmseq);
}



/**********************************************************************/
/************ $B2>@b$NA08~$-L`EY7W;;(B                  *******************/
/************ compute forward score of a hypothesis *******************/
/**********************************************************************/

/* --- $BBh(B2$B%Q%9C5:w(B(RL)$B$K$*$1$kC18l4V(Btriphone$B$N07$$(B ---

   $B$3$N(B search_bestfirst_v1.c $B$G$O!$%G%3!<%G%#%s%0$N9bB.2=$N$?$a(B
   $B<!C18l$H$=$NA0$NC18l$N@\B3E@$K$D$$$F(B
   $B!VC18l4V2;AG%3%s%F%-%9%H$NCY1d=hM}!W$r9T$J$&(B:
   (a) $B?72>@b$N@8@.(B(next_word())$B$G$O!$<!C18l$N:G8e$N2;AG$N1&%3%s%F%-%9%H$N$_(B
       $B$,9MN8$5$l$k!%(B
   (b) $B$=$NC18l4V$N40A4$J2;AG4D6-0MB8@-$O!$$=$N2>@b$,$$$C$?$s%9%?%C%/$K(B
       $BF~$C$?8e$b$&0lEY(B POP $B$5$l$?$H$-$K(B scan_word() $B$K$F2~$a$F7W;;$9$k!%(B

   $B2>@b@8@.;~$K$O$9$Y$F$N@8@.2>@b$KBP$7$F0MB87W;;$r9T$J$:!$$"$H$G%9%3%"$,(B
   $B9b$/(B POP $B$5$l$?2>@b$K$D$$$F$N$_:F7W;;$r9T$J$&!%$3$N$?$a=hM}$,9bB.2=$5$l$k!%(B
   $B$?$@$72>@b%9%3%"7W;;(B(next_word())$B$K$*$$$F<!C18l@\B3ItJ,$N4D6-0MB8@-$,(B
   $B9MN8$5$l$J$$$N$G(B, $B%9%3%"$K<c43$N8m:9$,@8$8$k2DG=@-$,$"$k!%(B
   
   $B<BAu$K$D$$$F(B:
   (1) next_word() $B$G$O!$<!C18l$N:G8e$N2;AG$N$_$r1&%3%s%F%-%9%H(B(=$BE83+85(B
       $BC18l$N:G=i$N2;AG(B)$B$r9MN8$7$FJQ2=$5$;!$%H%l%j%9@\B3E@$N=PNO3NN($r5a$a$k!%(B
   (2) scan_word() $B$G$O!$?7C18lItJ,$H$b$&#1$DA0$NC18l$N:G=i$N2;AG$rJQ2=(B
       $B$5$;!$(Bscan $B$9$k!%$=$N$?$a?7C18lItJ,$@$1$G$J$/!$$=$N$b$&0l2;AGA0$^$G(B
       scan $B$NBP>]$H$J$k!%$3$N(B "1-phoneme backscan" $B$r9T$J$&$?$a(B,
       $B3F2>@b%N!<%I$O:G=*(BHMM$B>uBV$NA08~$-%9%3%"(B (NODE$B$K$*$1$k(B g[]) $B$@$1$G$J$/!$(B
       $B$=$N(B backscan $B3+;OE@(B($B$b$&#1$DA0$NC18l$N:G=i$N2;AG$N<jA0(B)$B$N%9%3%"(B
       $B$bJ]B8$7$F$*$/I,MW$,$"$k(B (NODE $B$K$*$1$k(B g_prev[])$B!%(B

   $B$J$*!$#12;AG$N$_$+$i$J$kC18l$G$O(B backscan $B3+;OE@$HC18l6-3&$,=E$J$k$3$H$r(B
   $B9MN8$9$kI,MW$,$"$k$?$a!$<BAu$O$b$&>/$7J#;($K$J$k!%(B
 */

/* --- handling of cross-word triphone in 2nd pass search (in RL direction) ---
   
   Here in search_bestfirst_v1.c, we use "delayed cross-word context handling"
   method for connection of next word and last word of the hypothesis
   for speeding up decoding:
   (a) only right context of the tail phone in the next word is considered
       when generating a new hypothesis (next_word()).
   (b) the whole context dependency will be fully computed when the
       hypothesis is once pushed to stack and later popped in scan_word().
   This method avoid computing full context-dependency handling for all
   generated hypothesis in next_word(), and only re-compute it after
   primising ones are popped from stack later.  This speeds up decoding.
   But since the context dependency is not considered in the total hypothesis
   score (computed in next_word()).

   To realize this,
   (1) in nextword(), the tail phone in the new word is modified considering
       the right context (= head phone in the last word of source hypothesis),
       and the outprob on the connection point between backtrellis and forward
       trellis is computed using the triphone.
   (2) In scan_word(), not only the new word but also the head phone in the
       previous word should be modified and re-scanned.
       To realize this '1-phoneme backscan' procedure, hypothesis nodes
       have to keep forward scores not only at the last HMM state (g[] in
       NODE), but also at the backscan restart point (= before the head
       phone in the previous word, g_prev[] in NODE).
       
   Note that the actual implementation becomes a little more complicated
   to handle 1-phoneme words...

*/

/* $BJ82>@b$NA08~$-L`EY$r99?7$9$k(B
   $B:G8e$N(B1$BC18l$NA08~$-%H%l%j%9$r7W;;$9$k(B */
/* compute forward score by expanding trellis of the last word */
void
scan_word(NODE *now, HTK_Param *param)
{
  int   i,t;
  HMM *whmm;
  A_CELL *ac;
  WORD_ID word;
  LOGPROB tmpmax, tmptmp, score1;
  int startt = 0;
  int wordhmmnum;
  LOGPROB tmpmax2 = LOG_ZERO;
  int phmmlen;
  HMM_Logical *ret, *wend;
  int store_point = 0;
  int crossword_point = 0;
  boolean back_rescan = FALSE;
  boolean node_exist_p;
  
  /* ----------------------- prepare HMM ----------------------- */

  if (ccd_flag) {
    /* $BD>A0$N2;AG$,$"$l$P!$$=$3$^$G$5$+$N$\$C$F(B scan $B$9$k(B */
    /* if there are any last phone, enable backscan */
    if (now->last_ph == NULL) {
      /* initial score: now->g[] */
      /* scan range: phones in now->seq[now->seqnum-1] */
      back_rescan = FALSE;
    } else {
      /* initial score: now->g_prev[] (1-phone before)*/
      /* scan range: phones in now->seq[now->seqnum-1] + now->last_ph */
      back_rescan = TRUE;
    }
  }
#ifdef TCD
  if (now->last_ph != NULL) {
    j_printf("inherited last_ph: %s\n", (now->last_ph)->name);
  } else {
    j_printf("no last_ph inherited\n");
  }
#endif

  /* scan $BHO0OJ,$N(BHMM$B$r=`Hw(B */
  /* prepare HMM of the scan range */
  word = now->seq[now->seqnum-1];

  if (ccd_flag) {

    if (back_rescan) {
      
      /* scan range: phones in now->seq[now->seqnum-1] + now->last_ph */
      
      phmmlen = winfo->wlen[word] + 1;
      if (phmmlen > phmmlen_max) {
	j_error("short of phmmlen\n");
      }
      for (i=0;i<phmmlen - 2;i++) {
	phmmseq[i] = winfo->wseq[word][i];
      }
      /* $B:G=*C18l$H(B last_ph $B4V$NC18l4V(Btriphone$B$r9MN8(B */
      /* consider cross-word context dependency between the last word and now->last_ph */
      wend = winfo->wseq[word][winfo->wlen[word]-1];
      ret = get_right_context_HMM(wend, now->last_ph->name, hmminfo);
      if (ret == NULL) {	/* triphone not found */
	/* fallback to the original bi/mono-phone */
	/* error if the original is pseudo phone (not explicitly defined
	   in hmmdefs/hmmlist) */
	/* exception: word with 1 phone (triphone may exist in the next expansion */
	if (winfo->wlen[word] > 1 && wend->is_pseudo) {
	  error_missing_right_triphone(wend, now->last_ph->name);
	}
	phmmseq[phmmlen-2] = wend;
      } else {
	phmmseq[phmmlen-2] = ret;
      }
      ret = get_left_context_HMM(now->last_ph, wend->name, hmminfo);
      if (ret == NULL) {
	/* fallback to the original bi/mono-phone */
	/* error if the original is pseudo phone (not explicitly defined
	   in hmmdefs/hmmlist) */
	if (now->last_ph->is_pseudo) {
	  error_missing_left_triphone(now->last_ph, wend->name);
	}
	phmmseq[phmmlen-1] = now->last_ph;
      } else {
	phmmseq[phmmlen-1] = ret;
      }

#ifdef TCD
      j_printf("w=");
      for(i=0;i<winfo->wlen[word];i++) {
	j_printf(" %s",(winfo->wseq[word][i])->name);
      }
      j_printf(" | %s\n", (now->last_ph)->name);
      j_printf("scan for:");
      for (i=0;i<phmmlen;i++) {
	j_printf(" %s", phmmseq[i]->name);
      }
      j_printf("\n");
#endif

      /* $BC18l(BHMM$B$r:n$k(B */
      /* make word HMM */
      whmm = new_make_word_hmm(hmminfo, phmmseq, phmmlen);
      
      /* backscan $B$J$N$G!$7W;;A0$N(B g[] $B=i4|CM$O(B now->g_prev[] $B$r;HMQ(B */
      /* As backscan enabled, the initial forward score g[] is set by
	 now->g_prev[] */
      for (t=0;t<peseqlen;t++) {
	g[t]=now->g_prev[t];

      }
      
      /* $B<!CJMQ$N(Bg_prev$B$r3JG<$9$k%N!<%I0LCV$r@_Dj(B */
      /* set where to store scores as new g_prev[] for the next backscan
	 in the HMM */
      store_point = hmm_logical_state_num(phmmseq[0]) - 2 - 1;
      /* scan$BCf$KD>A0C18l$H$3$NC18l$r$^$?$0>l=j$r@_Dj(B */
      /* set where is the connection point of the last word in the HMM */
      crossword_point = whmm->len - (hmm_logical_state_num(phmmseq[phmmlen-1]) - 2) - 1;
      
    } else {			/* not backscan mode */
      
      /* scan range: phones in now->seq[now->seqnum-1] */
      
#ifdef TCD
      j_printf("scan(org):");
      for (i=0;i<winfo->wlen[word];i++) {
	j_printf(" %s", (winfo->wseq[word][i])->name);
      }
      j_printf("\n");
#endif
      
      /* $BC18l(BHMM$B$r:n$k(B */
      /* make word HMM */
      whmm = new_make_word_hmm(hmminfo, winfo->wseq[word], winfo->wlen[word]);
      
      /* $B7W;;A0$N(B g[] $B=i4|CM$O(B now->g[] $B$r;HMQ(B */
      /* the initial forward score g[] is set by now->g[] */
      for (t=0;t<peseqlen;t++) {
	g[t]=now->g[t];
      }
      
      /* $B<!CJMQ$N(Bg_prev$B$r3JG<$9$k%N!<%I0LCV$r@_Dj(B */
      /* set where to store scores as new g_prev[] for the next backscan
	 in the HMM */
      store_point = hmm_logical_state_num(winfo->wseq[word][0]) - 2 - 1;

      /* scan$BCf$KD>A0C18l$H$3$NC18l$r$^$?$0>l=j$O!$$J$7(B */
      /* the connection point of the last word is not exist in the HMM */
      crossword_point = -1;
    }
    
  } else {			/* ccd_flag == FALSE */

    /* $B2;AG4D6-Hs0MB8$N>l9g$OC1=c$K:G=*C18lJ,$N(B HMM $B$r:n@.(B */
    /* for monophone: simple make HMM for the last word */
    whmm = new_make_word_hmm(hmminfo, winfo->wseq[word], winfo->wlen[word]);

    /* $B7W;;A0$N(B g[] $B=i4|CM$O(B now->g[] $B$r;HMQ(B */
    /* the initial forward score g[] is set by now->g[] */
    for (t=0;t<peseqlen;t++) {
      g[t]=now->g[t];
    }
    
  }

#ifdef TCD
  j_printf("whmm len	  = %d\n",whmm->len);
  j_printf("crossword_point = %d\n", crossword_point);
  j_printf("g[] store point = %d\n", store_point);
#endif

  wordhmmnum = whmm->len;
  if (wordhmmnum >= winfo->maxwn + 10) {
    j_error("scan_word: word too long\n");
  }

  /* ----------------------- do scan ----------------------- */
#if 0
  printf("-------\n");
  for(t=0;t<peseqlen;t++) {
    printf("%d %f %f\n",t,framemaxscore[t],g[t]);
  }
  printf("-------\n");
#endif
  
  /* scan$B3+;OE@$r8!:w(B -> startt$B$X(B*/
  /* search for the start frame -> set to startt */
  for(t = peseqlen-1; t >=0 ; t--) {
    if (
#ifdef SCAN_BEAM
	g[t] > framemaxscore[t] - scan_beam_thres &&
#endif
	g[t] > LOG_ZERO) {
      break;
    }
  }
  if (t < 0) {			/* no node has score > LOG_ZERO */
    for(t=0;t<peseqlen;t++) {
      if (ccd_flag) now->g_prev[t] = LOG_ZERO;
      now->g[t] = LOG_ZERO;
    }
    goto end_of_scan;
  }
  startt = t;
  
  /* clear [startt+1..peseqlen-1] */
  for(t=peseqlen-1;t>startt;t--) {
    if (ccd_flag) now->g_prev[t] = LOG_ZERO;
    now->g[t] = LOG_ZERO;
  }
  
  /* $B;~4V(B [startt] $B>e$NCM$r=i4|2=(B */
  /* initialize scores on frame [startt] */
  for(i=0;i<wordhmmnum-1;i++) wordtrellis[startt][i] = LOG_ZERO;
  wordtrellis[startt][wordhmmnum-1] = g[startt] + outprob(startt, &(whmm->state[wordhmmnum-1]), param);
  if (ccd_flag) now->g_prev[startt] = LOG_ZERO;
  now->g[startt] = LOG_ZERO;

  /* $B%a%$%s%k!<%W(B: startt $B$+$i;O$^$j(B 0 $B$K8~$+$C$F(B Viterbi $B7W;;(B */
  /* main loop: start from [startt], and compute Viterbi toward [0] */
  for(t=startt-1;t>=0;t--) {
    
    node_exist_p = FALSE;	/* TRUE if there is at least 1 survived node in this frame */

    /* $BC<$N%N!<%I(B [t][wordhmmnum-1]$B$O!$FbItA+0\(B $B$+(B g[]$B$N9b$$J}$K$J$k(B */
    /* the edge node [t][wordhmmnum-1] is either internal transitin or g[] */
    tmptmp = LOG_ZERO;
    for (ac=whmm->state[wordhmmnum-1].ac;ac;ac=ac->next) {
      score1 = wordtrellis[t+1][ac->arc] + ac->a;
      if (tmptmp < score1) tmptmp = score1;
    }
    tmpmax = (g[t] > tmptmp) ? g[t] : tmptmp;

    /* $BC<$N%N!<%I$N%9%3%"%(%s%Y%m!<%W%A%'%C%/(B: $B0lDjI}30$J$iMn$H$9(B */
    /* check if the edge node is within score envelope */
    if (
#ifdef SCAN_BEAM
	tmpmax <= framemaxscore[t] - scan_beam_thres ||
#endif
	tmpmax <= LOG_ZERO
	) {
      wordtrellis[t][wordhmmnum-1] = LOG_ZERO;
    } else {
      node_exist_p = TRUE;
      wordtrellis[t][wordhmmnum-1] = tmpmax + outprob(t, &(whmm->state[wordhmmnum-1]), param);
    }

    /* $B%N!<%I(B [t][wordhmmnum-2..0] $B$K$D$$$F%H%l%j%9$r7W;;(B */
    /* expand trellis for node [t][wordhmmnum-2..0] */
    for(i=wordhmmnum-2;i>=0;i--) {

      if (ccd_flag) {

	/* $B:GL`%Q%9$H:GL`%9%3%"(B tmpmax $B$r8+$D$1$k(B */
	/* tmpmax2 $B$O<!2sMQ(B g_prev[] $B$N$?$a$N:GBgCM(B($B<+8JA+0\$r=|$$$?:GBgCM(B) */
	/* find most likely path and the max score 'tmpmax' */
	/* 'tmpmax2' is max score excluding self transition, for next g_prev[] */
	if (i == store_point) {
	  tmpmax2 = LOG_ZERO;
	}
	tmpmax = LOG_ZERO;
	for (ac=whmm->state[i].ac;ac;ac=ac->next) {
	  score1 = wordtrellis[t+1][ac->arc] + ac->a;
	  if (i <= crossword_point && ac->arc > crossword_point) {
	    /* $B$3$l$OC18l$r1[$($kA+0\(B (backscan $B<B9T;~(B) */
	    /* this is a transition across word (when backscan is enabled) */
#ifdef USE_NGRAM
	    score1 += now->lscore; /* add N-gram LM score */
#else
	    score1 += penalty2;	/* add DFA insertion penalty */
#endif
	  }
	  if (i == store_point && i != ac->arc) {
	    if (tmpmax2 < score1) tmpmax2 = score1;
	  }
	  if (tmpmax < score1) tmpmax = score1;
	}

	/* $B%9%3%"%(%s%Y%m!<%W%A%'%C%/(B: $B0lDjI}30$J$iMn$H$9(B */
	/* check if score of this node is within the score envelope */
	if (
#ifdef SCAN_BEAM
	    tmpmax <= framemaxscore[t] - scan_beam_thres ||
#endif
	    tmpmax <= LOG_ZERO
	    ) {  /* invalid node */
	  wordtrellis[t][i] = LOG_ZERO;
	  if (i == store_point) now->g_prev[t] = LOG_ZERO;
	} else { /* survived node */
	  if (i == store_point) now->g_prev[t] = tmpmax2;
	  node_exist_p = TRUE;	/* at least one node survive in this frame */
	  /* compute output probability */
	  tmptmp = outprob(t, &(whmm->state[i]), param);
	  /* score of node [t][i] has been determined here */
	  wordtrellis[t][i] = tmpmax + tmptmp;
	}
	
      } else {			/* not triphone */

	/* backscan $BL5$7(B: store_point, crossword_point $B$OL54X78(B */
	/* no backscan: store_point, crossword_point ignored */
	tmpmax = LOG_ZERO;
	for (ac=whmm->state[i].ac;ac;ac=ac->next) {
	  score1 = wordtrellis[t+1][ac->arc] + ac->a;
	  if (tmpmax < score1) tmpmax = score1;
	}

	/* $B%9%3%"%(%s%Y%m!<%W%A%'%C%/(B: $B0lDjI}30$J$iMn$H$9(B */
	/* check if score of this node is within the score envelope */
	if (
#ifdef SCAN_BEAM
	    tmpmax <= framemaxscore[t] - scan_beam_thres ||
#endif
	    tmpmax <= LOG_ZERO
	    ) {
	  /* invalid node */
	  wordtrellis[t][i] = LOG_ZERO;
	} else {
	  /* survived node */
	  node_exist_p = TRUE;
	  /* score of node [t][i] has been determined here */
	  wordtrellis[t][i] = tmpmax + outprob(t, &(whmm->state[i]), param);
	}
	
      }
    } /* end of node loop */

    /* $B;~4V(B t $B$N(BViterbi$B7W;;=*N;!%A08~$-%9%3%"$O(Bscan$B$7$?C18l$N;OC<(B */
    /* Viterbi end for frame [t].  the forward score is the score of word
       beginning scanned */
    now->g[t] = wordtrellis[t][0];

    /* scan$B$7$?C18l$NBh#1%Q%9$G$N;OC<;~9o$h$j@h$^$G(B t $B$,?J$s$G$*$j!$$+$D(B
       $B$3$N(B t $B$K$*$$$F%9%3%"%(%s%Y%m!<%W$K$h$C$F@8$-;D$C$?%N!<%I$,0l$D$b(B
       $BL5$+$C$?$J$i$P(B,$B$3$N%U%l!<%`$G7W;;$rBG$A@Z$j$=$l0J>e@h(B([0..t-1])$B$O(B
       $B7W;;$7$J$$(B */
    /* if frame 't' already reached the beginning frame of scanned word
       in 1st pass and no node was survived in this frame (all nodes pruned
       by score envelope), terminate computation at this frame and
       do not computer further frame ([0..t-1]). */
    if (t < now->estimated_next_t && (!node_exist_p)) {
      /* clear the rest scores */
      for (i=t-1;i>=0;i--) {
	now->g[i] = LOG_ZERO;
	if (ccd_flag) now->g_prev[i] = LOG_ZERO;
      }
      /* terminate loop */
      break;
    }
    
  } /* end of time loop */
  
  if (debug2_flag) j_printf("scanned: [%3d-%3d]\n",t+1,startt);

 end_of_scan:
  /* $B<!2s(B backscan $B$N$?$a$N>pJs3JG<(B */
  /* store data for next backscan */
  if (ccd_flag) {
    if (store_point == wordhmmnum - 1) {
      /* last_ph$BL5$7!$$+$DC18l$N2;AGD9(B=1$B$N>l9g!"<!2s$N(B scan_word() $B$G(B
	 $BC18lA4BN$,$b$&0lEY:F7W;;$5$l$k!%$3$N>l9g(B,
	 g_prev $B$O!$$3$N(Bscan_word$B$r3+;O$9$kA0$N%9%3%"$rF~$l$F$*$/I,MW$,$"$k(B */
      /* if there was no 'last_ph' and the scanned word consists of only
	 1 phone, the whole word should be re-computed in the future scan_word().
	 So the next 'g_prev[]' should be the initial forward scores
	 before we begin Viterbi (= g[t]). */
      for (t = startt; t>=0; t--) {
	now->g_prev[t] = g[t];
      }
    }
    /* $B<!2s$N$?$a$K(B now->last_ph $B$r99?7(B */
    /* update 'now->last_ph' for future scan_word() */
    if (back_rescan) {
      now->last_ph = phmmseq[0];
    } else {
      now->last_ph = winfo->wseq[word][0];
    }
  }

  /* free work area */
  free_hmm(whmm);
#ifdef TCD
  j_printf("last_ph = %s\n", (now->last_ph)->name);
#endif
}


/**************************************************************************/
/*** $B?72>@b$NE83+$H%R%e!<%j%9%F%#%C%/$r7R$$$@A4BN%9%3%"$r7W;;(B           ***/
/*** expand new hypothesis and compute the total score (with heuristic) ***/
/**************************************************************************/

/* now $B$K<!C18l(B nword $B$r@\B3$7$F?7$7$$2>@b(B new $B$r@8@.$7!$$=$N(B
   ($B%R%e!<%j%9%F%#%C%/$r4^$`(B)$B2>@b%9%3%"$r5a$a$k(B */
/* generate a new hypothesis 'new' by connecting next word 'nword' to
   the source hypothesis 'now', and compute the total score of 'new' with
   heuristics */
void
next_word(NODE *now, NODE *new,	NEXTWORD *nword, HTK_Param *param, BACKTRELLIS *backtrellis)
{
  int   t;
  HMM_Logical *newphone;
  int lastword;
  int   i;
  LOGPROB a_value;
  int   startt;
  int word;
  LOGPROB totalscore, tmpp;
  TRELLIS_ATOM *tre;

  new->score = LOG_ZERO;

  word = nword->id;
  lastword=now->seq[now->seqnum-1];

  /* $BC18lJB$S!"(BDFA$B>uBVHV9f!"8@8l%9%3%"$r7Q>5!&99?7(B */
  /* inherit and update word sequence, DFA state and total LM score */
  for (i=0;i< now->seqnum;i++){	
    new->seq[i] = now->seq[i];
#ifdef CM_SEARCH
#ifdef CM_MULTIPLE_ALPHA
    memcpy(new->cmscore[i], now->cmscore[i], sizeof(LOGPROB) * cm_alpha_num);
#else
    new->cmscore[i] = now->cmscore[i];
#endif
#endif /* CM_SEARCH */
  }
  new->seq[i] = word;
  new->seqnum = now->seqnum+1;
#ifdef USE_DFA
  new->state = nword->next_state;
#endif
#ifdef USE_NGRAM
  new->totallscore = now->totallscore + nword->lscore;
#endif  

  if (ccd_flag) {
    
    /* $BE83+C18l$N@\B3E@$N2;AG(BHMM$B$r(Bnewphone$B$K%;%C%H$9$k!%(B
       $B852>@b(B now $B$H$NC18l4V$N2;AG4D6-0MB8@-$r9MN8$9$k(B */
    /* set the triphone at the connection point to 'newphone', considering
       cross-word context dependency to 'now' */
    newphone = get_right_context_HMM(winfo->wseq[word][winfo->wlen[word]-1], now->last_ph->name, hmminfo);
    if (newphone == NULL) {	/* triphone not found */
      /* fallback to the original bi/mono-phone */
      /* error if the original is pseudo phone (not explicitly defined
	 in hmmdefs/hmmlist) */
      /* exception: word with 1 phone (triphone may exist in the next expansion */
      if (winfo->wlen[word] > 1 && winfo->wseq[word][winfo->wlen[word]-1]->is_pseudo){
	error_missing_right_triphone(winfo->wseq[word][winfo->wlen[word]-1], now->last_ph->name);
      }
      newphone = winfo->wseq[word][winfo->wlen[word]-1];
    }
    
    /* $B852>@b$r(Bscan$B$7$?;~$NKvC<2;AG(BHMM -> $B?72>@b$ND>A02;AG(BHMM */
    /* inherit last_ph */
    new->last_ph = now->last_ph;
    
#ifdef USE_NGRAM
    /* set current LM score */
    new->lscore = nword->lscore;
#endif
    
    /* backscan$BMQ@\B3%]%$%s%H$N%9%3%"(B g_prev[] $B$r%3%T!<(B */
    /* copy g_prev[] that are scores at backscan connection point */
    for (t=0;t<peseqlen;t++) {
      new->g_prev[t] = now->g_prev[t];
    }
    
  } else {			/* not triphone */
    
    /* $BE83+C18l$N@\B3(B(=$B=*C<(B)$B$N2;AG(BHMM$B$r(Bnewphone$B$K%;%C%H(B */
    /* set the phone at the connection point to 'newphone' */
    newphone = winfo->wseq[word][winfo->wlen[word]-1];
  }

  /* a_value: $B@\B3E@$NA+0\3NN((B */
  /* a_value: transition probability of connection point */
  i = hmm_logical_state_num(newphone);
  a_value = (hmm_logical_trans(newphone))->a[i-2][i-1];

  /***************************************************************************/
  /* $BA08~$-(B($BBh#2%Q%9(B),$B8e$m8~$-(B($BBh#1%Q%9(B)$B%H%l%j%9$r@\B3$7:GL`@\B3E@$r8+$D$1$k(B */
  /* connect forward/backward trellis to look for the best connection time   */
  /***************************************************************************/
  
  startt = peseqlen-2;
  new->g[startt+1] = LOG_ZERO;

#ifdef WORD_GRAPH
  /*-----------------------------------------------------------------*/
  /* $BC18l%0%i%U%b!<%I$G$OF1$8%U%l!<%`$NF1$8C18l$G$bD>A0C18lKh$K%9%3(B
     $B%"$,0c$&!%(Bbestt $B$O(B $B8GDj(B */
  /* there are several trellis word with the same word ID in the same
     frame, with different word context in WORD_GRAPH mode.  And the
     best t is fixed (no re-esimation) */
  /*-----------------------------------------------------------------*/

  /* update new->g[t] */
  for(t=startt;t>=0;t--) {
    new->g[t] = now->g[t+1] + a_value
#ifdef USE_NGRAM
      + nword->lscore
#else
      + penalty2
#endif
      ;
  }
  
  /* $B:GL`@\B3E@$O(B, $B:G8e$K;2>H$7$?(BTRELLIS_ATOM$B$N=*C<$K8GDj(B */
  /* the best connection point is fixed to the end time of last trellis word (no re-alignment */
  new->bestt = (nword->tre)->endtime;
  /* $B<!2s$NE83+C18l7hDj$N$?$a$N?dDj;OC<;~4V$O!$>e5-$N;OC<(B */
  /* the estimated next beginning frame for the next word expansion
     corresponds to the endtime above (this is also fixed) */
  new->estimated_next_t = (nword->tre)->begintime - 1;

  /* $B=PNO3NN($r(B tmpp $B$K7W;;(B */
  /* compute output prob to 'tmpp' */
  if (newphone->is_pseudo) {
    tmpp = outprob_cd(new->bestt, &(newphone->body.pseudo->stateset[newphone->body.pseudo->state_num-2]), param);
  } else {
    tmpp = outprob_state(new->bestt, newphone->body.defined->s[newphone->body.defined->state_num-2], param);
  }

  /* $B2>@bA4BN$N%9%3%"$r7W;;(B */
  new->score = new->g[new->bestt] + (nword->tre)->backscore + tmpp;
  
  /* $BE83+C18l$N%H%l%j%9>e$N>pJs$rEAHB(B */
  new->tre  =  nword->tre;
  
#else  /* not WORD_GRAPH */

  /*-----------------------------------------------------------------*/
  /* $BC18l%H%l%j%9$rC5$7$F(B, $B<!C18l$N:GL`@\B3E@$rH/8+$9$k(B */
  /* determine the best connection time of the new word, seeking the word
     trellis */
  /*-----------------------------------------------------------------*/

  /* update new->g[t] */
  for(t=startt;t>=0;t--) {
    new->g[t] = now->g[t+1] + a_value
#ifdef USE_NGRAM
      + nword->lscore
#else
      + penalty2
#endif
      ;
  }

#ifdef USE_DFA
  if (!looktrellis_flag) {
    /* $B$9$Y$F$N%U%l!<%`$K$o$?$C$F:GL`$rC5$9(B */
    /* search for best trellis word throughout all frame */
    for(t = startt; t >= 0; t--) {
      tre = bt_binsearch_atom(backtrellis, t, (WORD_ID) word);
      if (tre == NULL) continue;
      if (newphone->is_pseudo) {
	tmpp = outprob_cd(t, &(newphone->body.pseudo->stateset[newphone->body.pseudo->state_num-2]), param);
      } else {
	tmpp = outprob_state(t, newphone->body.defined->s[newphone->body.defined->state_num-2], param);
      }
      totalscore = new->g[t] + tre->backscore + tmpp;
      if (new->score < totalscore) {
	new->score = totalscore;
	new->bestt = t;
	new->estimated_next_t = tre->begintime - 1;
      }
    }
  } else {
#endif /* USE_DFA */
      
  /* $B$3$NE83+C18l$N%H%l%j%9>e$N=*C<;~4V$NA08e$N$_%9%-%c%s$9$k(B
     $BA08e$KO"B3$7$FB8:_$9$k%U%l!<%`$K$D$$$F$N$_7W;;(B */
  /* search for best trellis word only around the estimated time */
  /* 1. search forward */
  for(t = (nword->tre)->endtime; t >= 0; t--) {
    tre = bt_binsearch_atom(backtrellis, t, (WORD_ID) word);
    if (tre == NULL) break;	/* go to 2 if the trellis word disappear */
    if (newphone->is_pseudo) {
      tmpp = outprob_cd(t, &(newphone->body.pseudo->stateset[newphone->body.pseudo->state_num-2]), param);
    } else {
      tmpp = outprob_state(t, newphone->body.defined->s[newphone->body.defined->state_num-2], param);
    }
    totalscore = new->g[t] + tre->backscore + tmpp;
    if (new->score < totalscore) {
      new->score = totalscore;
      new->bestt = t;
      new->estimated_next_t = tre->begintime - 1;
      new->tre = tre;
    }
  }
  /* 2. search backward */
  for(t = (nword->tre)->endtime + 1; t <= startt; t++) {
    tre = bt_binsearch_atom(backtrellis, t, (WORD_ID) word);
    if (tre == NULL) break;	/* end if the trellis word disapper */
    if (newphone->is_pseudo) {
      tmpp = outprob_cd(t, &(newphone->body.pseudo->stateset[newphone->body.pseudo->state_num-2]), param);
    } else {
      tmpp = outprob_state(t, newphone->body.defined->s[newphone->body.defined->state_num-2], param);
    }
    totalscore = new->g[t] + tre->backscore + tmpp;
    if (new->score < totalscore) {
      new->score = totalscore;
      new->bestt = t;
      new->estimated_next_t = tre->begintime - 1;
      new->tre = tre;
    }
  }

#ifdef USE_DFA
  }
#endif
  
#endif /* WORD_GRAPH */

}

/**********************************************************************/
/********** $B=i4|2>@b$N@8@.(B                 ****************************/
/********** generate an initial hypothesis ****************************/
/**********************************************************************/
void
start_word(
     NODE *new,
     NEXTWORD *nword,
     HTK_Param *param,
     BACKTRELLIS *backtrellis)
{
  HMM_Logical *newphone;
  WORD_ID word;
  TRELLIS_ATOM *tre = NULL;
  LOGPROB tmpp;
  int t;

  /* initialize data */
  word = nword->id;
  new->score = LOG_ZERO;
  new->seqnum = 1;
  new->seq[0] = word;

#ifdef USE_DFA
  new->state = nword->next_state;
#endif
#ifdef USE_NGRAM
  new->totallscore = nword->lscore;
#endif  

  /* cross-word triphone handling is not needed on startup */
  newphone = winfo->wseq[word][winfo->wlen[word]-1];
  if (ccd_flag) {
    new->last_ph = NULL;
#ifdef USE_NGRAM
    new->lscore = nword->lscore;
#endif
  }
  
#ifdef USE_NGRAM
  new->g[peseqlen-1] = nword->lscore;
#else  /* USE_DFA */
  new->g[peseqlen-1] = 0;
#endif
  
#ifdef WORD_GRAPH
  if (newphone->is_pseudo) {
    tmpp = outprob_cd(peseqlen-1, &(newphone->body.pseudo->stateset[newphone->body.pseudo->state_num-2]), param);
  } else {
    tmpp = outprob_state(peseqlen-1, newphone->body.defined->s[newphone->body.defined->state_num-2], param);
  }
  
  new->score = new->g[peseqlen-1] + (nword->tre)->backscore + tmpp;
  new->tre  =  nword->tre;
  new->bestt = peseqlen-1;
  new->estimated_next_t = (nword->tre)->begintime - 1;
  
#else

  for (t=peseqlen-1; t>=0; t--) {
    tre = bt_binsearch_atom(backtrellis, t, word);
    if (tre != NULL) {
      new->bestt = t;
      if (newphone->is_pseudo) {
	tmpp = outprob_cd(peseqlen-1, &(newphone->body.pseudo->stateset[newphone->body.pseudo->state_num-2]), param);
      } else {
	tmpp = outprob_state(peseqlen-1, newphone->body.defined->s[newphone->body.defined->state_num-2], param);
      }
      
      new->score = new->g[peseqlen-1] + tre->backscore + tmpp;
      new->estimated_next_t = tre->begintime - 1;
      new->tre = tre;
      break;
    }
  }
  if (tre == NULL) {		/* no word in backtrellis */
    new->score = LOG_ZERO;
  }
#endif /* WORD_GRAPH */
}



/* $B:G=*2>@b$N%9%3%"7W;;(B */
/* generate final hypothesis from 'now' to 'new' */
void
last_next_word(NODE *now, NODE *new, HTK_Param *param)
{
  cpy_node(new, now);
  /* $B:G=*%9%3%"$r@_Dj(B */
  /* update the final score */
  new->score = now->g[0];
}


#endif /* PASS2_STRICT_IWCD */
