/*
 * Copyright (c) 1991-2003 Kyoto University
 * Copyright (c) 2000-2003 NAIST
 * All rights reserved
 */

/* wrwav.c --- write waveform to WAVE file */

/* $Id: wrwav.c,v 1.2 2003/09/29 06:01:23 ri Exp $ */

#include <sent/stddefs.h>
#include <sent/speech.h>

static int totallen;

/* write data with endian conversion */
/* (all .wav datas are in little endian) */
static boolean
mywrite(void *buf, size_t unitbyte, int unitnum, FILE *fp)
{
  int tmp;
#ifdef WORDS_BIGENDIAN
  if (unitbyte > 1) swap_bytes(buf, unitbyte, unitnum);
#endif
  if ((tmp = myfwrite(buf, unitbyte, unitnum, fp)) < unitnum) {
    return(FALSE);
  }
#ifdef WORDS_BIGENDIAN
  if (unitbyte > 1) swap_bytes(buf, unitbyte, unitnum);
#endif
  return(TRUE);
}

/* open/create a WAVE file for writing, and write header */
/* return file pointer, NULL on failure */
#define MYWRITE(A,B,C,D)  if (!mywrite(A, B, C, D)) {j_printerr("wrwav_open: write failed\n"); fclose_writefile(fp); return NULL;}
FILE *
wrwav_open(char *filename, int sfreq)
{
  FILE *fp;
  unsigned int i;
  unsigned short s;

  /* open file for writing */
  if ((fp = fopen_writefile(filename)) == NULL) return NULL;

  /* write header */
  /* first 4 byte: 'R' 'I' 'F' 'F' */
  MYWRITE("RIFF", 1, 4, fp);
  /* 4 byte: byte num of rest: dummy for now */
  i = 0; MYWRITE(&i, 4, 1, fp);

  /* first part: WAVE format specifications */
  /* 8 byte: 'W' 'A' 'V' 'E' 'f' 'm' 't' ' ' */
  MYWRITE("WAVEfmt ", 1, 8, fp);
  /* 4byte: byte size of the next part (16 bytes here) */
  i = 16; MYWRITE(&i, 4, 1, fp);
  /* 2byte: data format */
  s = 1; MYWRITE(&s, 2, 1, fp);	/* PCM */
  /* 2byte: channel num */
  s = 1; MYWRITE(&s, 2, 1, fp);	/* mono */
  /* 4byte: sampling rate */
  i = sfreq; MYWRITE(&i, 4, 1, fp);
  /* 4byte: bytes per second */
  i = sfreq * sizeof(SP16); MYWRITE(&i, 4, 1, fp);
  /* 2bytes: bytes per frame ( = (bytes per sample) x channel ) */
  s = sizeof(SP16); MYWRITE(&s, 2, 1, fp);
  /* 2bytes: bits per sample */
  s = sizeof(SP16) * 8; MYWRITE(&s, 2, 1, fp);
  
  /* data part header */
  MYWRITE("data", 1, 4, fp);
  /* data length: dummy for now */
  i = 0; MYWRITE(&i, 4, 1, fp);

  totallen = 0;			/* reset total length */

  return(fp);
}

/* write sample data */
boolean
wrwav_data(FILE *fp, SP16 *buf, int len)
{
  boolean ret;
  ret = mywrite(buf, sizeof(SP16), len, fp);
  if (ret) totallen += len;
  return(ret);
}

/* close file */
boolean
wrwav_close(FILE *fp)
{
  unsigned int i;
  
  /* overwrite data length after recording */
  /* at 5-8(len+36), 41-44 (len) */
  if (fseek(fp, 40, SEEK_SET) != 0) { /* error */
    perror("wrwav_close");
    return(FALSE);
  }
  i = totallen * sizeof(SP16);
  if (!mywrite(&i, 4, 1, fp)) {
    j_printerr("wrwav_close: failed to re-write header\n");
    return(FALSE);
  }
  if (fseek(fp, 4, SEEK_SET) != 0) { /* error */
    perror("wrwav_close");
    return(FALSE);
  }
  i = totallen * sizeof(SP16) + 36;
  if (!mywrite(&i, 4, 1, fp)) {
    j_printerr("wrwav_close: failed to re-write header\n");
    return(FALSE);
  }
  
  /* close file */
  fclose_writefile(fp);

  return(TRUE);
}
