package org.junitdoc.ui.testcasetable;

import java.util.List;

import org.eclipse.jdt.core.ElementChangedEvent;
import org.eclipse.jdt.core.ICompilationUnit;
import org.eclipse.jdt.core.IElementChangedListener;
import org.eclipse.jdt.core.IJavaElementDelta;
import org.eclipse.jdt.core.JavaCore;
import org.eclipse.jdt.core.dom.CompilationUnit;
import org.eclipse.jdt.core.dom.MethodDeclaration;
import org.eclipse.jdt.core.dom.TextElement;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.IToolBarManager;
import org.eclipse.jface.action.Separator;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.CellLabelProvider;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.TreeNode;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.TreeViewerColumn;
import org.eclipse.jface.viewers.ViewerCell;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.TreeColumn;
import org.eclipse.ui.IActionBars;
import org.eclipse.ui.ISelectionListener;
import org.eclipse.ui.ISharedImages;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.part.ViewPart;
import org.junitdoc.JUnitDocPlugin;
import org.junitdoc.core.rewriter.ASTUtils;
import org.junitdoc.core.rewriter.JDTUtils;
import org.junitdoc.core.testcasetable.TestCase;
import org.junitdoc.core.testcasetable.TestCaseTableModel;
import org.junitdoc.ui.Const;
import org.junitdoc.ui.UIMessages;

public class TestCaseTableViewPart extends ViewPart implements
		IElementChangedListener {

	private TreeViewer viewer;

	private ICompilationUnit source;

	private TestCaseTableModel model;

	private Action action1;

	private Action action2;

	private ISelectionListener listener = new ISelectionListener() {
		public void selectionChanged(IWorkbenchPart workbenchPart,
				ISelection selection) {

			JUnitDocPlugin.debugLog("TestCaseTableViewPart.selectionChanged"); //$NON-NLS-1$
			JUnitDocPlugin.debugLog(selection.getClass());
			JUnitDocPlugin.debugLog(selection);

			final ICompilationUnit currentUnit = JDTUtils.getICompilationUnit();

			if (viewer.getControl() == null || viewer.getControl().isDisposed())
				return;

			Display d = viewer.getControl().getDisplay();
			if (d != null) {
				d.asyncExec(new Runnable() {
					public void run() {

						if (currentUnit != null && !currentUnit.equals(source)) {
							initialize(currentUnit);
						}
					}
				});
			}
		}
	};

	public void initialize(ICompilationUnit source) {
		JUnitDocPlugin.debugLog("initialize...."); //$NON-NLS-1$
		this.source = source;

		CompilationUnit ast = ASTUtils.createAST(source);
		refresh(ast);
	}

	public void refresh(CompilationUnit ast) {
		List<MethodDeclaration> methods = ASTUtils
				.findAnnotatedMethodDeclarations(ast, Const.TEST_ANNOTATION);

		synchronized (model) {
			model.parse(methods);
			viewer.refresh();
			viewer.expandAll();
		}
	}

	public void createPartControl(Composite parent) {
		viewer = new TreeViewer(parent, SWT.MULTI | SWT.H_SCROLL | SWT.V_SCROLL);
		viewer.getTree().setHeaderVisible(true);
		viewer.getTree().setLinesVisible(true);
		viewer.setAutoExpandLevel(TreeViewer.ALL_LEVELS);
		viewer.setContentProvider(new TestCaseTreeTableContentProvider());

		makeColumn();

		makeActions();
		contributeToActionBars();

		model = new TestCaseTableModel();
		viewer.setInput(model);

		JavaCore.addElementChangedListener(this);
		getSite().getPage().addPostSelectionListener(listener);
	}

	private void makeColumn() {
		// make Category column
		TreeViewerColumn viewerCol = new TreeViewerColumn(viewer, SWT.LEFT);
		viewerCol.setLabelProvider(new HeaderCellLabelProvider());

		TreeColumn col = viewerCol.getColumn();
		col.setText(UIMessages
				.getString("TestCaseTableViewPart.column.summary")); //$NON-NLS-1$
		col.setWidth(300);

		// make Steps columns
		makeTestCaseColumn(
				viewer,
				UIMessages.getString("TestCaseTableViewPart.column.steps"), new TestCaseCellLabelProvider() { //$NON-NLS-1$
					@Override
					protected String getText(TestCase value) {
						return textElementToString(value.getSteps());
					}
				});

		// make Expected Column
		makeTestCaseColumn(
				viewer,
				UIMessages
						.getString("TestCaseTableViewPart.columns.expectedresults"), new TestCaseCellLabelProvider() { //$NON-NLS-1$
					@Override
					protected String getText(TestCase value) {
						return textElementToString(value.getExpected());
					}
				});

	}

	private String textElementToString(List<TextElement> list) {
		if (list.size() == 1) {
			return list.get(0).getText();
		}

		StringBuilder sb = new StringBuilder();
		int index = 1;
		for (TextElement textElement : list) {
			sb.append(index + ". " + textElement.getText() + "\n"); //$NON-NLS-1$ //$NON-NLS-2$
			index++;
		}
		return sb.toString();
	}

	private void makeTestCaseColumn(TreeViewer viewer, String columnText,
			TestCaseCellLabelProvider provider) {
		TreeViewerColumn viewerCol = new TreeViewerColumn(viewer, SWT.LEFT);
		viewerCol.setLabelProvider(provider);
		TreeColumn col = viewerCol.getColumn();
		col.setText(columnText);
		col.setWidth(300);
		col.setResizable(true);
		col.setMoveable(false);
	}

	private class HeaderCellLabelProvider extends CellLabelProvider {

		@Override
		public void update(ViewerCell cell) {
			TreeNode node = (TreeNode) cell.getElement();
			cell.setText(node.getValue().toString());
		}
	}

	private abstract class TestCaseCellLabelProvider extends CellLabelProvider {

		@Override
		public void update(ViewerCell cell) {
			TreeNode treeNode = (TreeNode) cell.getElement();
			if (treeNode.hasChildren()) {
				return;
			}

			Object value = treeNode.getValue();
			if (value instanceof TestCase) {
				String text = getText((TestCase) value);
				cell.setText(text);
			}
		}

		abstract protected String getText(TestCase value);
	}

	private void contributeToActionBars() {
		IActionBars bars = getViewSite().getActionBars();
		fillLocalPullDown(bars.getMenuManager());
		fillLocalToolBar(bars.getToolBarManager());
	}

	private void fillLocalPullDown(IMenuManager manager) {
		manager.add(action1);
		manager.add(new Separator());
		manager.add(action2);
	}

	private void fillLocalToolBar(IToolBarManager manager) {
		manager.add(action1);
		manager.add(action2);
	}

	private void makeActions() {
		action1 = new Action() {
			public void run() {
				showMessage("Action 1 executed"); //$NON-NLS-1$
			}
		};
		action1.setText("Action 1"); //$NON-NLS-1$
		action1.setToolTipText("Action 1 tooltip"); //$NON-NLS-1$
		action1.setImageDescriptor(PlatformUI.getWorkbench().getSharedImages()
				.getImageDescriptor(ISharedImages.IMG_OBJS_INFO_TSK));

		action2 = new Action() {
			public void run() {
				showMessage("Action 2 executed"); //$NON-NLS-1$
			}
		};
		action2.setText("Action 2"); //$NON-NLS-1$
		action2.setToolTipText("Action 2 tooltip"); //$NON-NLS-1$
		action2.setImageDescriptor(PlatformUI.getWorkbench().getSharedImages()
				.getImageDescriptor(ISharedImages.IMG_OBJS_INFO_TSK));
	}

	private void showMessage(String message) {
		MessageDialog.openInformation(viewer.getControl().getShell(),
				"TestCase Table", message); //$NON-NLS-1$
	}

	/**
	 * Passing the focus request to the viewer's control.
	 */
	public void setFocus() {
		viewer.getControl().setFocus();
	}

	@Override
	public void dispose() {
		JavaCore.removeElementChangedListener(this);
		getSite().getPage().removeSelectionListener(listener);
		super.dispose();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.jdt.core.IElementChangedListener#elementChanged(org.eclipse.jdt.core.ElementChangedEvent)
	 */
	public void elementChanged(ElementChangedEvent event) {
		if ((event.getDelta().getFlags() & IJavaElementDelta.F_CONTENT) != 0) {
			return;
		}
		if ((event.getDelta().getFlags() & IJavaElementDelta.F_FINE_GRAINED) != 0) {
			return;
		}

		if ((event.getDelta().getFlags() & IJavaElementDelta.F_AST_AFFECTED) != 0) {
			JUnitDocPlugin.debugLog("AST AFFECTED"); //$NON-NLS-1$

			if (viewer.getControl() == null)
				return;

			final IJavaElementDelta delta = event.getDelta();

			Display d = viewer.getControl().getDisplay();
			if (d != null) {
				d.asyncExec(new Runnable() {
					public void run() {

						ICompilationUnit currentSource = JDTUtils
								.getICompilationUnit();

						if (currentSource != null) {

							handleChanges(delta);
						}
					}
				});
			}
		}
	}

	protected void handleChanges(IJavaElementDelta delta) {
		refresh(delta.getCompilationUnitAST());
	}
}