package org.junitdoc.core.decisiontable;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import org.eclipse.jdt.core.dom.ASTNode;
import org.eclipse.jdt.core.dom.AnnotationTypeDeclaration;
import org.junitdoc.core.rewriter.ASTUtils;
import org.junitdoc.ui.Const;

public class Type {

	private AnnotationTypeDeclaration annoType;

	private String name;

	private List<Type> children = new ArrayList<Type>();

	private int checkedCount;

	private int depth;

	public Type(AnnotationTypeDeclaration annoType, int depth) {
		this.annoType = annoType;
		this.name = annoType.getName().getFullyQualifiedName();
		this.depth = depth;
	}

	public void addChild(Type type) {
		children.add(type);
	}

	public String getName() {
		return name;
	}

	public AnnotationTypeDeclaration getAnnotationTypeDeclaration() {
		return annoType;
	}

	public String getFullName() {

		List<AnnotationTypeDeclaration> list = new ArrayList<AnnotationTypeDeclaration>();
		list.add(annoType);

		AnnotationTypeDeclaration parent = annoType;

		while (!isDecisionTableAnnotated(parent)) {
			ASTNode node = parent.getParent();

			if (node instanceof AnnotationTypeDeclaration) {
				AnnotationTypeDeclaration parentAnno = (AnnotationTypeDeclaration) node;
				list.add(parentAnno);

				parent = parentAnno;

			} else {
				throw new IllegalStateException("Illegal Decision Table Def.");
			}
		}

		StringBuilder sb = new StringBuilder();

		for (int i = list.size() - 1; i >= 0; i--) {
			sb.append(list.get(i).getName().getFullyQualifiedName());
			if (i > 0) {
				sb.append(".");
			}
		}

		return sb.toString();
	}

	private boolean isDecisionTableAnnotated(
			AnnotationTypeDeclaration parentAnno) {
		return ASTUtils.isAnnotated(parentAnno, Const.DECISIONTABLE_ANNOTATION);
	}

	public boolean hasChildren() {
		return children.size() > 0;
	}

	public List<Type> getChildren() {
		return children;
	}

	public int getDepth() {
		return depth;
	}

	public int getTotalDepth() {
		int rtn = 1;
		for (Type child : children) {
			rtn++;
			rtn += child.getTotalDepth();
		}
		return rtn;
	}

	public List<String> convertText(String prefix) {
		List<String> rtn = new ArrayList<String>();

		rtn.add(name);
		for (Type child : children) {
			rtn.addAll(child.convertText(prefix + prefix));
		}
		return rtn;
	}

	public Collection<? extends Type> getTypeList() {
		List<Type> rtn = new ArrayList<Type>();
		for (Type child : children) {
			rtn.add(child);
			rtn.addAll(child.getTypeList());
		}
		return rtn;
	}

	public int getCheckedCount() {
		return checkedCount;
	}

	public void addCheckedCount() {
		checkedCount++;
	}

	public int getCheckedTotalCount() {
		int rtn = checkedCount;
		for (Type child : children) {
			rtn += child.getCheckedTotalCount();
		}
		return rtn;
	}

	public void clearCheckedCount() {
		checkedCount = 0;
	}

}
