package org.junitdoc.ui.decisiontable;

import java.util.List;

import org.eclipse.jdt.core.ElementChangedEvent;
import org.eclipse.jdt.core.ICompilationUnit;
import org.eclipse.jdt.core.IElementChangedListener;
import org.eclipse.jdt.core.IJavaElementDelta;
import org.eclipse.jdt.core.dom.AnnotationTypeDeclaration;
import org.eclipse.jdt.core.dom.CompilationUnit;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.CTabFolder;
import org.eclipse.swt.custom.CTabItem;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Menu;
import org.junitdoc.JUnitDocPlugin;
import org.junitdoc.core.decisiontable.DecisionTableModel;
import org.junitdoc.core.rewriter.ASTUtils;
import org.junitdoc.core.rewriter.JDTUtils;
import org.junitdoc.ui.Const;

public class DecisionTablesTabPage implements IElementChangedListener {

	private Composite parent;
	private CTabFolder tabFolder;
	private MenuManager menuManager;

	public DecisionTablesTabPage(Composite parent) {
		this.parent = parent;
		tabFolder = new CTabFolder(parent, SWT.BORDER);
		tabFolder.setLayout(new GridLayout(1, true));
		tabFolder.setLayoutData(new GridData(GridData.FILL_BOTH));

		tabFolder.setTabHeight(25);
		tabFolder.setSimple(false);
	}

	public void initialize(ICompilationUnit source) {
		disposeTabItems();
		createTabItems(ASTUtils.createAST(source));
	}

	private void disposeTabItems() {
		for (CTabItem tabItem : tabFolder.getItems()) {
			tabItem.dispose();
		}
	}

	public void initialize(CompilationUnit ast) {
		createTabItems(ast);
	}

	private void createTabItems(CompilationUnit ast) {
		List<AnnotationTypeDeclaration> annos = ASTUtils
				.getAnnotationTypeDeclarationsWithAnnotation(ast,
						Const.DECISIONTABLE_ANNOTATION);

		// save current selected tab.
		CTabItem selection = tabFolder.getSelection();
		String title = null;
		if (selection != null) {
			title = selection.getText();
		}

		// disposeTabIfNecessary(annos);
		// dispose all tabItems here!
		disposeTabItems();
		createTabIfNecessary(annos);

		// select saved tab.
		CTabItem[] items = tabFolder.getItems();
		for (CTabItem tabItem : items) {
			if (tabItem.getText().equals(title)) {
				tabFolder.setSelection(tabItem);
				break;
			}
		}

		// If selected tab is disposed, let select first tab.
		if (tabFolder.getItemCount() > 0) {
			tabFolder.setSelection(0);
		}
	}

	private void disposeTabIfNecessary(List<AnnotationTypeDeclaration> annos) {

		for (CTabItem tabItem : tabFolder.getItems()) {

			boolean found = false;

			for (AnnotationTypeDeclaration anno : annos) {
				if (tabItem.getText().equals(
						anno.getName().getFullyQualifiedName())) {
					found = true;
					break;
				}
			}
			if (!found) {
				tabItem.dispose();
			}
		}
	}

	private void createTabIfNecessary(List<AnnotationTypeDeclaration> annos) {

		for (AnnotationTypeDeclaration anno : annos) {

			createDecisionTable(anno);

			// boolean found = false;
			//
			// for (CTabItem item : tabFolder.getItems()) {
			// if (item.getText().equals(
			// anno.getName().getFullyQualifiedName())) {
			// found = true;
			// break;
			// }
			// }
			// if (!found) {
			// createDecisionTable(anno);
			// }
		}
	}

	private void createDecisionTable(AnnotationTypeDeclaration anno) {
		CTabItem tabItem = new CTabItem(tabFolder, SWT.NONE);
		String testTargetMethodName = anno.getName().getFullyQualifiedName();
		tabItem.setText(testTargetMethodName);

		DecisionTableViewer viewer = new DecisionTableViewer(tabFolder,
				testTargetMethodName);

		Menu menu = menuManager.createContextMenu(viewer.getControl());
		viewer.getControl().setMenu(menu);

		viewer.initialize((CompilationUnit) anno.getRoot());

		tabItem.setControl(viewer.getControl());
		tabItem.setData(viewer);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.jdt.core.IElementChangedListener#elementChanged(org.eclipse.jdt.core.ElementChangedEvent)
	 */
	public void elementChanged(final ElementChangedEvent event) {
		// change TabFolder if necessary
		JUnitDocPlugin
				.debugLog("DecisionTablesTabPage elementChanged:" + event);

		JUnitDocPlugin.debugLog("source:" + event.getSource().getClass());

		if ((event.getDelta().getFlags() & IJavaElementDelta.F_CONTENT) != 0) {
			return;
		}
		if ((event.getDelta().getFlags() & IJavaElementDelta.F_FINE_GRAINED) != 0) {
			return;
		}

		if ((event.getDelta().getFlags() & IJavaElementDelta.F_AST_AFFECTED) != 0) {
			JUnitDocPlugin.debugLog("AST AFFECTED");

			if (getControl() == null)
				return;

			final IJavaElementDelta delta = event.getDelta();

			Display d = getControl().getDisplay();
			if (d != null) {
				d.asyncExec(new Runnable() {
					public void run() {

						ICompilationUnit currentSource = JDTUtils
								.getICompilationUnit();

						if (currentSource != null) {

							// eventHandler.handleChanges(delta);
							handleChanges(delta.getCompilationUnitAST(),
									currentSource);
						}
					}
				});
			}
		}
	}

	public CTabFolder getControl() {
		return tabFolder;
	}

	protected void handleChanges(CompilationUnit newAst, ICompilationUnit source) {
		initialize(newAst);
	}

	public void dispose() {
		tabFolder.dispose();
	}

	public String getSelectedTabName() {
		CTabItem tabItem = tabFolder.getItem(tabFolder.getSelectionIndex());
		return tabItem.getText();
	}

	public DecisionTableModel[] getDecisionTableModels() {
		CTabItem[] items = tabFolder.getItems();

		DecisionTableModel[] rtn = new DecisionTableModel[items.length];

		int index = 0;

		for (CTabItem tabItem : items) {
			Object data = tabItem.getData();

			DecisionTableViewer viewer = (DecisionTableViewer) data;
			rtn[index] = viewer.getModel();
			index++;
		}
		return rtn;
	}

	public void setMenuManager(MenuManager menuMgr) {
		this.menuManager = menuMgr;
	}
}