package org.junitdoc.core.util;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.jdt.core.ICompilationUnit;
import org.eclipse.jdt.core.IJavaElement;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.IMember;
import org.eclipse.jdt.core.IMethod;
import org.eclipse.jdt.core.IPackageFragment;
import org.eclipse.jdt.core.IPackageFragmentRoot;
import org.eclipse.jdt.core.IParent;
import org.eclipse.jdt.core.IType;
import org.eclipse.jdt.core.JavaModelException;
import org.eclipse.jdt.core.dom.ASTNode;
import org.eclipse.jdt.core.dom.ASTVisitor;
import org.eclipse.jdt.core.dom.AbstractTypeDeclaration;
import org.eclipse.jdt.core.dom.AnnotationTypeDeclaration;
import org.eclipse.jdt.core.dom.BodyDeclaration;
import org.eclipse.jdt.core.dom.CompilationUnit;
import org.eclipse.jdt.core.dom.MethodDeclaration;
import org.eclipse.jdt.core.dom.TypeDeclaration;
import org.junitdoc.JUnitDocPlugin;
import org.junitdoc.core.rewriter.ASTUtils;

public class JavaModelUtils {

	public static boolean isMarkedAnnotationWith(String annotation,
			IMember member, CompilationUnit ast) {

		ASTNode found = findASTNode(member, ast);

		if (found == null) {
			return false;
		}

		if (found instanceof BodyDeclaration) {
			BodyDeclaration body = (BodyDeclaration) found;

			return ASTUtils.isAnnotated(body, annotation);
		}
		return false;
	}

	public static MethodDeclaration findMethodDeclaration(IMethod method,
			CompilationUnit ast) {
		ASTNode found = findASTNode(method, ast);
		if (found instanceof MethodDeclaration) {
			return (MethodDeclaration) found;
		}
		return null;
	}

	public static AnnotationTypeDeclaration findAnnotationTypeDeclaration(
			IType type, CompilationUnit ast) {

		ASTNode found = findASTNode(type, ast);
		if (found instanceof AnnotationTypeDeclaration) {
			return (AnnotationTypeDeclaration) found;
		}
		return null;
	}

	public static TypeDeclaration findTypeDeclaration(IType type,
			CompilationUnit ast) {

		ASTNode found = findASTNode(type, ast);
		if (found instanceof TypeDeclaration) {
			return (TypeDeclaration) found;
		}
		return null;
	}

	public static ASTNode findASTNode(IMember member, CompilationUnit ast) {

		if (!member.exists()) {
			return null;
		}

		try {

			final int offset = member.getNameRange().getOffset();
			final int length = member.getNameRange().getLength();

			final ASTNode[] found = new ASTNode[1];

			ast.accept(new ASTVisitor() {

				@Override
				public boolean visit(MethodDeclaration node) {
					return findMethod(node);
				}

				private boolean findMethod(MethodDeclaration node) {
					if (found[0] != null) {
						return false;
					}
					if (match(node.getName())) {
						found[0] = node;
						return false;
					}
					return true;
				}

				@Override
				public boolean visit(AnnotationTypeDeclaration node) {
					return findType(node);
				}

				@Override
				public boolean visit(TypeDeclaration node) {
					return findType(node);
				}

				private boolean findType(AbstractTypeDeclaration typeDeclare) {
					if (found[0] != null) {
						return false;
					}
					if (match(typeDeclare.getName())) {
						found[0] = typeDeclare;
						return false;
					}
					return true;
				}

				private boolean match(ASTNode node) {
					return node.getStartPosition() == offset
							&& node.getLength() == length;
				}
			});

			if (found[0] != null) {
				return found[0];
			}

		} catch (JavaModelException e) {
			JUnitDocPlugin.errorLog(e);
		}

		return null;
	}

	public static List<ICompilationUnit> collectICompilationUnits(
			IJavaElement javaElement) throws JavaModelException {

		List<ICompilationUnit> rtn = new ArrayList<ICompilationUnit>();

		if (javaElement instanceof ICompilationUnit) {
			rtn.add((ICompilationUnit) javaElement);
		}

		// Very slow...why?
		if (javaElement instanceof IJavaProject) {
			IJavaProject project = (IJavaProject) javaElement;
			IPackageFragmentRoot[] roots = project.getAllPackageFragmentRoots();
			collectICompilationUnits(roots, rtn);

		} else if (javaElement instanceof IPackageFragmentRoot) {
			collectICompilationUnits(new IJavaElement[] { javaElement }, rtn);

		} else if (javaElement instanceof IPackageFragment) {

			IPackageFragment packageFragment = (IPackageFragment) javaElement;
			if (packageFragment.hasChildren()) {
				IJavaElement[] children = packageFragment.getChildren();
				collectICompilationUnits(children, rtn);
			}
		}

		return rtn;
	}

	private static void collectICompilationUnits(IJavaElement[] elements,
			List<ICompilationUnit> list) throws JavaModelException {

		for (IJavaElement element : elements) {
			if (element instanceof ICompilationUnit) {
				list.add((ICompilationUnit) element);

			} else if (element instanceof IParent) {
				IParent parent = (IParent) element;
				if (parent.hasChildren()) {
					IJavaElement[] children = parent.getChildren();
					collectICompilationUnits(children, list);
				}
			}
		}

	}
}
