package junkutil.common;

import junit.framework.Test;
import junit.framework.TestCase;
import junit.framework.TestSuite;
import junkutil.common.StringUtil;

/**
 * StringUtileXgB
 * @author Hiroyuki Shiota
 */
public class StringUtilTest extends TestCase {
	/**
	 * RXgN^
	 * @param name
	 */
	public StringUtilTest(String name) {
		super(name);
	}

	/**
	 * @return
	 */
	public static Test suite() {
		TestSuite test = new TestSuite("StringUtilTest");
		test.addTest(new TestSuite(StringUtilTest.class));
		return test;
	}

	/**
	 * @see junit.framework.TestCase#setUp()
	 */
	protected void setUp() throws Exception {
	}

	/**
	 * @see junit.framework.TestCase#tearDown()
	 */
	protected void tearDown() throws Exception {
	}

	/**
	 * @throws Exception
	 */
	public void testReplaceFirst() throws Exception {
		assertTrue(StringUtil.replaceFirst("aaa", "a", "b").equals("baa"));
		assertTrue(StringUtil.replaceFirst("bbb", "a", "b").equals("bbb"));
	}
	
	/**
	 * @throws Exception
	 */
	public void testReplaceAll() throws Exception {
		assertTrue(StringUtil.replaceAll("aaa", "a", "b").equals("bbb"));
		assertTrue(StringUtil.replaceAll("bbb", "a", "b").equals("bbb"));
	}
	
	/**
	 * @throws Exception
	 */
	public void testTrimLeft() throws Exception {
		assertTrue(StringUtil.trimLeft("  <-trimLeft  ").equals("<-trimLeft  "));
	}
	/**
	 * @throws Exception
	 */
	public void testTrimRight() throws Exception {
		assertTrue(StringUtil.trimRight("  trimRight->  ").equals("  trimRight->"));
	}
	/**
	 * @throws Exception
	 */
	public void testPadLeft() throws Exception {
		assertTrue(StringUtil.padLeft("<-padLeft", 12).equals("   <-padLeft"));
		assertTrue(StringUtil.padLeft("123", 5, '0').equals("00123"));
	}
	/**
	 * @throws Exception
	 */
	public void testPadRight() throws Exception {
		assertTrue(StringUtil.padRight("padRight->", 12).equals("padRight->  "));
		assertTrue(StringUtil.padRight("123", 5, '0').equals("12300"));
	}
	/**
	 * @throws Exception
	 */
	public void wrapBracket() throws Exception {
		assertTrue(StringUtil.wrapBracket("wraped", "*").equals("*wraped*"));
		assertTrue(StringUtil.wrapBracket("wraped", "<", ">").equals("<wraped>"));
	}
	/**
	 * @throws Exception
	 */
	public void testUnwrapBracket() throws Exception {
		assertTrue(StringUtil.unwrapBracket("*unwraped*", "*").equals("unwraped"));
		assertTrue(StringUtil.unwrapBracket("<unwraped>", "<", ">").equals("unwraped"));
		assertTrue(StringUtil.unwrapBracket("*unwraped*_", "*").equals("*unwraped*_"));
		assertTrue(StringUtil.unwrapBracket("<unwraped>_", "<", ">").equals("<unwraped>_"));
	}
	/**
	 * @throws Exception
	 */
	public void testGetRepeatedChars() throws Exception {
		assertTrue(StringUtil.getRepeatedChars('x', 0).equals(""));
		assertTrue(StringUtil.getRepeatedChars('x', 5).equals("xxxxx"));
		assertTrue(StringUtil.getRepeatedChars('x', 10).equals("xxxxxxxxxx"));
	}
	/**
	 * @throws Exception
	 */
	public void testGetRepeatedSpaces() throws Exception {
		assertTrue(StringUtil.getRepeatedSpaces(0).equals(""));
		assertTrue(StringUtil.getRepeatedSpaces(5).equals("     "));
		assertTrue(StringUtil.getRepeatedSpaces(10).equals("          "));
	}
	/**
	 * @throws Exception
	 */
	public void testEquals() throws Exception {
		String [] nul = null;
		String [] blank = new String[]{""};
		String [] strs = new String[]{"1","2","3"};
		
		assertTrue(StringUtil.equals(nul, nul));
		assertTrue(!StringUtil.equals(blank, nul));
		assertTrue(!StringUtil.equals(nul, blank));
		
		assertTrue(StringUtil.equals(blank, new String[]{""}));

		assertTrue(StringUtil.equals(strs, new String[]{"1","2","3"}));
		assertTrue(!StringUtil.equals(strs, new String[]{"1","2"}));
		assertTrue(!StringUtil.equals(strs, new String[]{"1","2","1"}));
		assertTrue(!StringUtil.equals(strs, new String[]{"1","2",null}));
		assertTrue(!StringUtil.equals(strs, new String[]{"3","2","1"}));

		assertTrue(StringUtil.equals(new String[]{"1","2","3"}, strs));
		assertTrue(!StringUtil.equals(new String[]{"1","2"}, strs));
		assertTrue(!StringUtil.equals(new String[]{"1","2","1"}, strs));
		assertTrue(!StringUtil.equals(new String[]{"1","2",null}, strs));
		assertTrue(!StringUtil.equals(new String[]{"3","2","1"}, strs));
	}
	/**
	 * @throws Exception
	 */
	public void testJoin() throws Exception {
		String [] strs = new String[]{"1","2","3"};
		
		assertTrue(StringUtil.join(strs, "").equals("123"));
		assertTrue(StringUtil.join(strs, ",").equals("1,2,3"));
		assertTrue(StringUtil.join(strs, "--").equals("1--2--3"));
	}
	/**
	 * @throws Exception
	 */
	public void testSplit() throws Exception {
		assertTrue(StringUtil.equals(StringUtil.split("", "+"), new String[]{""}));
		assertTrue(StringUtil.equals(StringUtil.split("+", "+"), new String[]{"",""}));
		assertTrue(StringUtil.equals(StringUtil.split("1+2+3=6", "+"), new String[]{"1","2","3=6"}));
		assertTrue(StringUtil.equals(StringUtil.split("+12+3=6+", "+"), new String[]{"","12","3=6", ""}));
	}
	/**
	 * @throws Exception
	 */
	public void testFuzzyEquals() throws Exception {
		assertTrue(StringUtil.fuzzyEquals("aabaa", "aaaa", 1));
		assertTrue(StringUtil.fuzzyEquals("aabaa", "aacaa", 2));
		assertTrue(!StringUtil.fuzzyEquals("aabaa", "aacaa", 1));
	}
	/**
	 * @throws Exception
	 */
	public void testToString() throws Exception {
		assertTrue(StringUtil.toString(new String[]{"a", "b", "c"}).equals("[a, b, c]"));
	}
	/**
	 * @throws Exception
	 */
	public void testEncodeURL() throws Exception
	{
		assertTrue(StringUtil.encodeURL("testeXg").equals("test%E3%83%86%E3%82%B9%E3%83%88%E8%A9%A6%E9%A8%93"));
		assertTrue(StringUtil.encodeURL("testeXg", "utf-8").equals("test%E3%83%86%E3%82%B9%E3%83%88%E8%A9%A6%E9%A8%93"));
		assertTrue(StringUtil.encodeURL("testeXg", "Shift_JIS").equals("test%83%65%83%58%83%67%8E%8E%8C%B1"));
	}
	/**
	 * @throws Exception
	 */
	public void testDecodeURL() throws Exception
	{
		assertTrue(StringUtil.decodeURL("test%E3%83%86%E3%82%B9%E3%83%88%E8%A9%A6%E9%A8%93").equals("testeXg"));
		assertTrue(StringUtil.decodeURL("test%E3%83%86%E3%82%B9%E3%83%88%E8%A9%A6%E9%A8%93", "utf-8").equals("testeXg"));
		assertTrue(StringUtil.decodeURL("test%83%65%83%58%83%67%8E%8E%8C%B1", "Shift_JIS").equals("testeXg"));
	}
	/**
	 * @throws Exception
	 */
	public void testEscapeHtml() throws Exception
	{
		assertTrue(StringUtil.escapeHtml("<a&b'c\"d>").equals("&lt;a&amp;b&#39;c&quot;d&gt;"));
	}
	/**
	 * @throws Exception
	 */
	public void testUnescapeHtml() throws Exception
	{
		assertTrue(StringUtil.unescapeHtml("&lt;a&amp;b&#39;c&quot;d&gt;").equals("<a&b'c\"d>"));
	}
	
	/**
	 * @throws Exception
	 */
	public void testMatch() throws Exception {
		assertTrue(StringUtil.match("foobarhoge", "^foo"));
		assertTrue(StringUtil.match("foobarhoge", "b[ab]r"));
		assertTrue(StringUtil.match("foobarhoge", "hoge$"));
	}
}
