import sys

from math import cos, sin, radians, pi, sqrt, floor, ceil, fabs

from os import path

from dxf_mod import DXF_Module, RoundOff

from matrix import Matrix

from dxf_class import DL_Extrusion, DL_Attributes, DL_LayerData, DL_DimensionData, DL_DimAlignedData,\
	DL_DimLinearData, DL_DimRadialData, DL_DimDiametricData, DL_DimAngularData, DL_DimAngular3PData, DL_LeaderData, \
	DL_LeaderVertexData, DL_HatchEdgeData, DL_HatchData, DL_HatchLoopData, DL_ImageData, DL_ImageDefData
#-------------------------------------------------------------
#constant definition
#-------------------------------------------------------------
M_PI = pi #3.1415926535897932384626433832795
DL_DXF_MAXGROUPCODE	= 1100
DL_UNKNOWN			=   0
DL_LAYER			  =  10
DL_BLOCK			  =  11
DL_ENDBLK			 =  12
DL_LINETYPE		   =  13
DL_SETTING			=  50
DL_ENTITY_POINT	   = 100
DL_ENTITY_LINE		= 101
DL_ENTITY_POLYLINE	= 102
DL_ENTITY_LWPOLYLINE  = 103
DL_ENTITY_VERTEX	  = 104
DL_ENTITY_SPLINE	  = 105
DL_ENTITY_KNOT		= 106
DL_ENTITY_CONTROLPOINT= 107
DL_ENTITY_ARC		 = 108
DL_ENTITY_CIRCLE	  = 109
DL_ENTITY_ELLIPSE	 = 110
DL_ENTITY_INSERT	  = 111
DL_ENTITY_TEXT		= 112
DL_ENTITY_MTEXT	   = 113
DL_ENTITY_DIMENSION   = 114
DL_ENTITY_LEADER	  = 115
DL_ENTITY_HATCH	   = 116
DL_ENTITY_ATTRIB	  = 117
DL_ENTITY_IMAGE	   = 118
DL_ENTITY_IMAGEDEF	= 119
DL_ENTITY_TRACE	   = 120
DL_ENTITY_SOLID	   = 121
DL_ENTITY_SEQEND	  = 122

#-------------------------------------------------------------
#Converts the given string into a double or returns the given 
#default valud (def) if value is [] or empty.
#-------------------------------------------------------------
def toReal(value, default=0.0):
	if value != '':
		ret = default
		if value.find(',') > -1:
			val = value.replace(',','.')
			ret = float(val)
		else:
		   ret = float(value)
		return ret
	else:
		return default

#-------------------------------------------------------------
#Converts the given string into an int or returns the given 
#default valud (def) if value is [] or empty.
#-------------------------------------------------------------
def toInt(value, default=0):
	if value != '': # and value[0] != '\0':
		return int(value)
	else:
		return default

class DL_CreationInterface:
	def __init__(self):
		self.attributes = DL_Attributes('', 0, 0, 'BYLAYER')
		self.extrusion = DL_Extrusion()
		self.LayerList = []
		self.VarDic = {}

	def delete(self):
		del self.attributes
		del self.extrusion
		for obj in self.LayerList:
			del obj
		del self.LayerList
		for k, v in self.VarDic.iteritems():
#			print k, v
			del v
		del self.VarDic

	def dump(self): #dump variales
		for k, v in self.VarDic.iteritems():
			print k, v

	def setExtrusion(self, dx, dy, dz, elevation):
		self.extrusion.setDirection(dx, dy, dz)
		self.extrusion.setElevation(elevation)

	#Sets the current attributes for entities.
	def setAttributes(self, attrib):
		self.attributes.layer, self.attributes.color, self.attributes.width, self.attributes.lineType \
			= attrib.layer, attrib.color, attrib.width, attrib.lineType
#		self.attributes = attrib
		del attrib
#-----------------------------------------------------
	def addLayer(self,  data):
		self.LayerList.append(data)

	#
	# Called for every vector variable in the DXF file (e.g. "$EXTMIN").
	#
	def setVariableVector(self, key, v1, v2, v3, code):
		self.VarDic[key] = [v1, v2, v3, code]

	#
	# Called for every string variable in the DXF file (e.g. "$ACADVER").
	#
	def setVariableString(self, key, value, code):
		self.VarDic[key] = [value, code]
	
	#
	# Called for every int variable in the DXF file (e.g. "$ACADMAINTVER").
	#
	def setVariableInt(self, key, value, code):
		self.VarDic[key] = [value, code]
	
	#
	# Called for every double variable in the DXF file (e.g. "$DIMEXO").
	#
	def setVariableDouble(self, key, value, code):
		self.VarDic[key] = [value, code]

	#
	# Called for every block. Note: all entities added after this
	# command go into this block until endBlock() is called.
	# see endBlock()
	#
	def addBlock(self,  data):
		return

	#Called to end the current block
	def endBlock(self):
		return

	#Called for every point
	def addPoint(self, data):
		return

	#Called for every line
	def addLine(self, data):
		return

	#Called for every arc
	def addArc(self, data):
		return

	#Called for every circle
	def addCircle(self, data):
		return

	#Called for every ellipse
	def addEllipse(self, data):
		return

	#Called for every polyline start
	def addPolyline(self, data):
		return

	#Called for every lwpolyline start
	def addLWPolyline(self, data):
		return

	#Called for every polyline vertex
	def addVertex(self, data):
		return
	
	#Called for every spline
	def addSpline(self, data):
		return
	
	#Called for every spline control point
	def addControlPoint(self, data):
		return
	
	#Called for every spline knot value
	def addKnot(self, data):
		return

	#Called for every insert.
	def addInsert(self, data):
		return
	
	#Called for every trace start
	def addTrace(self, data):
		return

	#Called for every solid start
	def addSolid(self, data):
		return

	#Called for every Multi Text entity.
	def addMText(self, data):
		return

	#
	# Called for additional text chunks for MTEXT entities.
	# The chunks come at 250 character in size each. Note that 
	# those chunks come <b>before</b> the actual MTEXT entity.
	#
	def addMTextChunk(self, text):
		return

	#Called for every Text entity.
	def addText(self, data):
		return

	#
	# Called for every aligned dimension entity. 
	#
	def addDimAlign(self, data, edata):
		return

	#
	# Called for every linear or rotated dimension entity. 
	#
	def addDimLinear(self, data, edata):
		return

	#
	# Called for every radial dimension entity. 
	#
	def addDimRadial(self, data, edata):
		return

	#
	# Called for every diametric dimension entity. 
	#
	def addDimDiametric(self, data, edata):
		return

	#
	# Called for every angular dimension (2 lines version) entity. 
	#
	def addDimAngular(self, data, edata):
		return

	#
	# Called for every angular dimension (3 points version) entity. 
	#
	def addDimAngular3P(self, data, edata):
		return
	
	#
	# Called for every leader start. 
	#
	def addLeader(self, data):
		return
	
	#
	# Called for every leader vertex 
	#
	def addLeaderVertex(self, data):
		return
	
	#
	# Called for every hatch entity. 
	#
	def addHatch(self, data):
		return
	
	#
	# Called for every image entity. 
	#
	def addImage(self, data):
		return

	#
	# Called for every image definition.
	#
	def linkImage(self, data):
		return

	#
	# Called for every hatch loop. 
	#
	def addHatchLoop(self, data):
		return

	#
	# Called for every hatch edge entity. 
	#
	def addHatchEdge(self, data):
		return

	#
	#Called when a SEQEND occurs (when a POLYLINE or ATTRIB is done)
	#
	def endSequence(self):
#		print 'DL_CreationInterface endSequence'
		return
		
	#Called after an entity has been completed.
	def endEntity(self):
#		print 'DL_CreationInterface endEntity'
		return

class DL_Dxf:
	def __init__(self, FileName):
		# Only the useful part of the group code
		self.groupCodeTmp = ''
		# ...same as integer
		self.groupCode = 0
		# Only the useful part of the group value
		self.groupValue = ''
		# Current entity type
		self.currentEntity = DL_UNKNOWN
		# Value of the current setting
		self.settingValue = ''
		# Key of the current setting (e.g. "$ACADVER")
		self.settingKey = ''
		# Stores the group codes
#-----------------------------------------------------------------------
#		self.values = [ [''] for i in range(DL_DXF_MAXGROUPCODE)]
#-----------------------------------------------------------------------
		self.values = {} #[''] * DL_DXF_MAXGROUPCODE
		# First call of this method. We initialize all group values in
		#  the first call.
		self.firstCall = 0
		# Attributes of the current entity (layer, color, width, line type)
		self.attrib = DL_Attributes('', 0, 0, 'BYLAYER')
		# library version. hex: 0x20003001 = 2.0.3.1
		self.libVersion = 0
		self.linebuf = []
		self.lpos = 0

		self.version = ''
		self.styleHandleStd = 0
		self.polylineLayer = ''
		self.vertices = []
		self.maxVertices = 0
		self.vertexIndex = -1

		self.vnum = -1	#polyline flag(>-1) and number of polyline vertex
		self.vlist = []	#vertex list
		self.closeFlag = 0

		self.knots = []
		self.maxKnots = 0
		self.knotIndex = -1

		self.controlPoints = []
		self.maxControlPoints = 0
		self.controlPointIndex = -1

		self.leaderVertices = []
		self.maxLeaderVertices = 0
		self.leaderVertexIndex = -1

		# array of hatch loops
		self.hatchLoops = []
		self.maxHatchLoops = 0
		self.hatchLoopIndex = -1
		# array in format [loop#][edge#]
		self.hatchEdges = []
		self.maxHatchEdges = []
		self.hatchEdgeIndex = []
		self.dropEdges = 'false'

		# Bulge for the next vertex.
		self.bulge = 0.0

		# Initialize Blender Module
		Dir,fName=path.split(FileName)
#		print Dir, fName
		ObjName, ext = fName.split('.')
#		print ObjName, ext
		self.dxfmod = DXF_Module(ObjName)
		self.ScanFile(FileName)
		#For Block Data
		self.CurBlockName = ''
		self.BlockList = {}

	def SetValue(self, key, data):
		if self.values.has_key(str(key)):
			del self.values[str(key)]
		self.values[str(key)] = data

	def GetValue(self, key):
		if self.values.has_key(str(key)):
			return self.values[str(key)]
		else:
			return ''

	def InitValues(self):
		self.values.clear()

	def Finish(self):
		self.dxfmod.zoom()
		self.values.clear()
		del self.values
		del self.linebuf
		del self.vertices
		del self.vlist
		del self.knots
		del self.controlPoints
		del self.leaderVertices
		for obj in self.hatchLoops:
			del obj
		del self.hatchLoops
		for obj in self.hatchEdges:
			del obj
		del self.hatchEdges
		del self.maxHatchEdges
		del self.hatchEdgeIndex
		for v in self.BlockList.itervalues():
			del v
		del self.BlockList

	def addSetting(self, creationInterface):
		c = -1
		for i in range(381):
			if self.GetValue(i) != '':
				c = i
				break
		# string
		if c>=0 and c<=9:
			creationInterface.setVariableString(self.settingKey,
												 self.GetValue(c), c)
		# vector
		elif c >= 10 and c <= 39:
			if c == 10:
				creationInterface.setVariableVector(
					self.settingKey,
					toReal(self.GetValue(c)),
					toReal(self.GetValue(c+10)),
					toReal(self.GetValue(c+20)),
					c)
		# double
		elif c >= 40 and c <= 59:
			creationInterface.setVariableDouble(self.settingKey,
												 toReal(self.GetValue(c)),
												 c)
		# int
		elif c >= 60 and c <= 99:
			creationInterface.setVariableInt(self.settingKey,
											  toInt(self.GetValue(c)),
											  c)
		# misc
		elif c >= 0:
			creationInterface.setVariableString(self.settingKey,
												 self.GetValue(c),
												 c)
#		print 'key[', c, ']=', self.settingKey, ', value=', self.GetValue(c)

	def addLayer(self, creationInterface):
		# correct some impossible attributes for layers:
		self.attrib = creationInterface.attributes
		if self.attrib.color == 256 or self.attrib.color == 0:
			self.attrib.color = 7

		if self.attrib.width < 0:
			self.attrib.width = 1

		if self.attrib.lineType != 'BYLAYER' or \
				self.attrib != 'BYBLOCK':
			self.attrib.lineType = 'CONTINUOUS'

		# add layer
		creationInterface.addLayer(DL_LayerData(self.GetValue(2),
									toInt(self.GetValue(70))))
#		print 'Layer ', self.GetValue(2)

	def addBlockList(self, type, data):
		child = self.BlockList[self.CurBlockName]
		child[1].append([type, data])

	def addBlock(self, creationInterface):
		d = [
		# Name:
		self.GetValue(2),
		# flags:
		toInt(self.GetValue(70)),
		# base point:
		toReal(self.GetValue(10)),
		toReal(self.GetValue(20)),
		toReal(self.GetValue(30))]
		creationInterface.addBlock(d)
		print 'Block ', d
		self.CurBlockName = self.GetValue(2)
		self.BlockList[self.CurBlockName] = [d, []]

	def endBlock(self, creationInterface):
		creationInterface.endBlock()
		print 'endBlock:', self.CurBlockName, 'elements num:', len(self.BlockList[self.CurBlockName][1])
		self.CurBlockName = ''

	def addPoint(self, creationInterface):
		d = [toReal(self.GetValue(10)),
			toReal(self.GetValue(20)),
			toReal(self.GetValue(30))]
		creationInterface.addPoint(d)
		print 'Point'

	def addLine(self, creationInterface):
		d = [toReal(self.GetValue(10)),
			toReal(self.GetValue(20)),
			toReal(self.GetValue(30)),
			toReal(self.GetValue(11)),
			toReal(self.GetValue(21)),
			toReal(self.GetValue(31))]
		creationInterface.addLine(d)
		if self.CurBlockName != '':
			self.addBlockList(DL_ENTITY_LINE, d)
		else:
			self.dxfmod.line(d)

	def addPolyline(self, creationInterface):
		pd = [toInt(self.GetValue(71)),
			toInt(self.GetValue(72)),
			toInt(self.GetValue(70))]
		creationInterface.addPolyline(pd)
		self.vnum = 0	#number of polyline vertex
		print 'Polyline:', pd
		self.closeFlag = pd[0]
		if self.vlist != []:
			del self.vlist

	def addLWPolyline(self, creationInterface):
		print 'addLWPolyline maxVertices=', self.maxVertices
		pd = [self.maxVertices,
			toInt(self.GetValue(71)),
			toInt(self.GetValue(72)),
			toInt(self.GetValue(70))]
		creationInterface.addLWPolyline(pd)
		print 'LWPolyline:', pd
		if pd[0] & 2 != 0:
			self.vertices.append([self.vertices[0][0], self.vertices[0][1], self.vertices[0][2], self.vertices[0][3]])
			self.maxVertices += 1
		for i in range(self.maxVertices):
			d = [self.vertices[i][0],
				self.vertices[i][1],
				self.vertices[i][2],
				self.vertices[i][3]]
			creationInterface.addVertex(d)
		creationInterface.endEntity()
		if self.CurBlockName != '':
			self.addBlockList(DL_ENTITY_LWPOLYLINE, self.vertices)
		else:
			self.dxfmod.polyline(self.vertices)

	def addVertex(self, creationInterface):
		d = [toReal(self.GetValue(10)),
			toReal(self.GetValue(20)),
			toReal(self.GetValue(30)),
			#self.bulge,
			toReal(self.GetValue(42))]

		#Bulge for the next vertex
		self.bulge = toReal(self.GetValue(42))

		creationInterface.addVertex(d)
		if self.vnum > -1:
#			print 'append Vertex--------------------'
#			self.vlist.append(DL_VertexData(d[0], d[1], d[2], d[3]))
			self.vlist.append([d[0], d[1], d[2]])
			self.vnum = self.vnum + 1

	def addSpline(self, creationInterface):
		sd = [
			toInt(self.GetValue(71), 3),
			toInt(self.GetValue(72), 0),
			toInt(self.GetValue(73), 0),
			toInt(self.GetValue(70), 4)]
		creationInterface.addSpline(sd)
		for i in range(self.maxControlPoints):
			d = [self.controlPoints[i][0],
				self.controlPoints[i][1],
				self.controlPoints[i][2]]
			creationInterface.addControlPoint(d)
		print 'Spline'
		if self.CurBlockName != '':
			self.addBlockList(DL_ENTITY_SPLINE, self.controlPoints)
		else:
			self.dxfmod.polyline(self.controlPoints)

	def arc2polyline(self, data, n_div = 32):
		incr = 2.0 * pi / float(n_div)
		th = radians(data[4])
		eang = radians(data[5])
		if fabs(eang -th)<= RoundOff:
			eang = th + 360.0
		if th > eang:
			eang += pi * 2.0
		loop_cnt = int(floor((eang - th) / incr))
		pt = [[0.0, 0.0, 0.0, 0.0] for i in range(loop_cnt)] 
		for i in range(loop_cnt):
			pt[i][0] = data[0] + data[3] * cos(th)
			pt[i][1] = data[1] + data[3] * sin(th)
			th += incr	  #ʳ٤ʬͤû
		if th < eang or fabs(eang - th) < RoundOff:
			pt.append([data[0] + data[3] * cos(eang), data[1] + data[3] * sin(eang), 0.0, 0.0])
		return pt

	def addArc(self, creationInterface):
		d = [toReal(self.GetValue(10)),
			toReal(self.GetValue(20)),
			toReal(self.GetValue(30)),
			toReal(self.GetValue(40)),
			toReal(self.GetValue(50)),
			toReal(self.GetValue(51))]
		creationInterface.addArc(d)
#		print 'Arc', d
		ptlist = self.arc2polyline(d, 32)
#		print ptlist
		if self.CurBlockName != '':
			#self.addBlockList(DL_ENTITY_ARC, d)
			self.addBlockList(DL_ENTITY_POLYLINE, ptlist)
		else:
			#self.dxfmod.arc(d)
			self.dxfmod.polyline(ptlist)
		del ptlist

	def circle2polyline(self, data, n_div = 32):
		loop_cnt = n_div + 1
		pt = [[0.0, 0.0, 0.0, 0.0] for i in range(loop_cnt)] 
		incr = 2.0 * pi / float(n_div)
		th = 0.0
		for i in range(loop_cnt):
			pt[i][0] = data[0] + data[3] * cos(th)
			pt[i][1] = data[1] + data[3] * sin(th)
			th += incr	  #ʳ٤ʬͤû
		return pt

	def addCircle(self, creationInterface):
		d = [toReal(self.GetValue(10)),
			toReal(self.GetValue(20)),
			toReal(self.GetValue(30)),
			toReal(self.GetValue(40))]
		creationInterface.addCircle(d)
#		print 'Circle', d
		ptlist = self.circle2polyline(d, 32)
#		print ptlist
		if self.CurBlockName != '':
			#self.addBlockList(DL_ENTITY_CIRCLE, d)
			self.addBlockList(DL_ENTITY_POLYLINE, ptlist)
		else:
			#self.dxfmod.circle(d)
			self.dxfmod.polyline(ptlist)
		del ptlist

	def ellipse2polyline(self, data, n_div = 32):
		incr = 2.0 * pi / float(n_div)
		ra = sqrt(data[3]*data[3]+data[4]*data[4])
		rb = data[6] * ra

		matr = Matrix(3,3)
		matr.init(0)
		matr[0][0] = data[3]/ra
		matr[0][1] = -data[4]/ra
		matr[1][0] = data[4]/ra
		matr[1][1] = data[3]/ra
		matr[2][2] = 1.0
		mat1 = Matrix(3,1)
		th = data[7]
		loop_cnt = int(floor((data[8] - data[7]) / incr))
#		print 'loop:', loop_cnt
		pt = [[0.0, 0.0, 0.0, 0.0] for i in range(loop_cnt)]
		for i in range(loop_cnt):
			mat1[0][0] = ra * cos(th)
			mat1[1][0] = rb * sin(th)
			mat1[2][0] = 1.0
			rmat = matr * mat1
			pt[i][0] = data[0] + rmat[0][0]
			pt[i][1] = data[1] + rmat[1][0]
			th += incr	  #ʳ٤ʬͤû
		if th < data[8] or fabs(data[8] - th) < RoundOff:
#			print th - incr, data[8], th
			mat1[0][0] = ra * cos(data[8])
			mat1[1][0] = rb * sin(data[8])
			mat1[2][0] = 1.0
			rmat = matr * mat1
			pt.append([data[0] + rmat[0][0], data[1] + rmat[1][0], 0.0, 0.0])
		return pt

	def addEllipse(self, creationInterface):
		d = [toReal(self.GetValue(10)),
			 toReal(self.GetValue(20)),
			 toReal(self.GetValue(30)),
			 toReal(self.GetValue(11)),
			 toReal(self.GetValue(21)),
			 toReal(self.GetValue(31)),
			 toReal(self.GetValue(40), 1.0),
			 toReal(self.GetValue(41), 0.0),
			 toReal(self.GetValue(42), 2*M_PI)]
		creationInterface.addEllipse(d)
#		print 'Ellipse', d
		ptlist = self.ellipse2polyline(d, 32)
		if self.CurBlockName != '':
			#self.addBlockList(DL_ENTITY_ELLIPSE, d)
			self.addBlockList(DL_ENTITY_POLYLINE, ptlist)
		else:
			#self.dxfmod.ellipse(d)
			self.dxfmod.polyline(ptlist)
		del ptlist

	def Transform(self, type, base, d, data):
		#ư
		matm = Matrix(4,4)
		matm.init(0)
		matm[0][0] = 1.0
		matm[1][1] = 1.0
		matm[2][2] = 1.0
		matm[3][3] = 1.0
		matm[0][3] = -base[0]
		matm[1][3] = -base[1]
		matm[2][3] = -base[2]
		#
		mats = Matrix(4,4)
		mats.init(0)
		mats[0][0] = d[4]
		mats[1][1] = d[5]
		mats[2][2] = d[6]
		mats[3][3] = 1.0
		#ž
		ang = radians(d[7])
		matr = Matrix(4,4)
		matr.init(0)
		matr[0][0] = cos(ang)
		matr[0][1] = -sin(ang)
		matr[1][0] = sin(ang)
		matr[1][1] = cos(ang)
		matr[2][2] = 1.0
		matr[3][3] = 1.0
		#ư
		matm1 = Matrix(4,4)
		matm1.init(0)
		matm1[0][0] = 1.0
		matm1[1][1] = 1.0
		matm1[2][2] = 1.0
		matm1[3][3] = 1.0
		matm1[0][3] = d[1] / d[4] + base[0]
		matm1[1][3] = d[2] / d[5] + base[1]
		matm1[2][3] = d[3] + base[2]
		tmat = matm * mats * matr * matm1
		mat1 = Matrix(4,1)
		result = []
		if type == DL_ENTITY_LINE:
			result = [[0.0] for i in range(len(data))]
			mat1[0][0] = data[0]
			mat1[1][0] = data[1]
			mat1[2][0] = data[2]
			mat1[3][0] = 1.0
			rmat = tmat * mat1
			result[0] = rmat[0][0]
			result[1] = rmat[1][0]
			result[2] = rmat[2][0]

			mat1[0][0] = data[3]
			mat1[1][0] = data[4]
			mat1[2][0] = data[5]
			rmat = tmat * mat1
			result[3] = rmat[0][0]
			result[4] = rmat[1][0]
			result[5] = rmat[2][0]
		elif type == DL_ENTITY_ARC:
			result = [[0.0] for i in range(len(data))]
			#
			result[3] *= d[4]
			#ưž
			mat1[0][0] = data[0]
			mat1[1][0] = data[1]
			mat1[2][0] = data[2]
			mat1[3][0] = 1.0
			rmat = tmat * mat1
			result[0] = rmat[0][0]
			result[1] = rmat[1][0]
			result[2] = rmat[2][0]
			result[3] = data[3]
			result[4] = data[4]
			result[5] = data[5]
		elif type == DL_ENTITY_CIRCLE:
			result = [[0.0] for i in range(len(data))]
			#
			result[3] *= d[4]
			#ưž
			mat1[0][0] = data[0]
			mat1[1][0] = data[1]
			mat1[2][0] = data[2]
			mat1[3][0] = 1.0
			rmat = tmat * mat1
			result[0] = rmat[0][0]
			result[1] = rmat[1][0]
			result[2] = rmat[2][0]
			result[3] = data[3]
		elif type == DL_ENTITY_ELLIPSE:
			result = [[0.0] for i in range(len(data))]
			#ưž
			mat1[0][0] = data[0]
			mat1[1][0] = data[1]
			mat1[2][0] = data[2]
			mat1[3][0] = 1.0
			rmat = tmat * mat1
			result[0] = rmat[0][0]
			result[1] = rmat[1][0]
			result[2] = rmat[2][0]
			result[3] = data[3]
			mat1[0][0] = data[3]
			mat1[1][0] = data[4]
			mat1[2][0] = data[5]
			rmat = tmat * mat1
			result[3] = rmat[0][0]
			result[4] = rmat[1][0]
			result[5] = rmat[2][0]
			result[6] = data[6]
			result[7] = data[7]
			result[8] = data[8]
		elif type == DL_ENTITY_POLYLINE or type == DL_ENTITY_LWPOLYLINE:
			result = [[0.0, 0.0, 0.0, 0.0] for i in range(len(data))]
			for i, pt in enumerate(data):
				mat1[0][0] = pt[0]
				mat1[1][0] = pt[1]
				mat1[2][0] = pt[2]
				mat1[3][0] = 1.0
				rmat = tmat * mat1
				result[i][0] = rmat[0][0]
				result[i][1] = rmat[1][0]
				result[i][2] = rmat[2][0]
				if len(pt) == 4:
					result[i][3] = pt[3]
		elif type == DL_ENTITY_SPLINE:
			result = [[0.0, 0.0, 0.0] for i in range(len(data))]
			for i, pt in enumerate(data):
				mat1[0][0] = pt[0]
				mat1[1][0] = pt[1]
				mat1[2][0] = pt[2]
				mat1[3][0] = 1.0
				rmat = tmat * mat1
				result[i][0] = rmat[0][0]
				result[i][1] = rmat[1][0]
				result[i][2] = rmat[2][0]
		elif type == DL_ENTITY_TRACE:
			result = [[0.0, 0.0, 0.0] for i in range(len(data))]
			for i, pt in enumerate(data):
				mat1[0][0] = pt[0]
				mat1[1][0] = pt[1]
				mat1[2][0] = pt[2]
				mat1[3][0] = 1.0
				rmat = tmat * mat1
				result[i][0] = rmat[0][0]
				result[i][1] = rmat[1][0]
				result[i][2] = rmat[2][0]
		elif type == DL_ENTITY_SOLID:
			result = [[0.0, 0.0, 0.0] for i in range(len(data))]
			for i, pt in enumerate(data):
				mat1[0][0] = pt[0]
				mat1[1][0] = pt[1]
				mat1[2][0] = pt[2]
				mat1[3][0] = 1.0
				rmat = tmat * mat1
				result[i][0] = rmat[0][0]
				result[i][1] = rmat[1][0]
				result[i][2] = rmat[2][0]
		return result

	def addInsert(self, creationInterface):
		d = [self.GetValue(2),
			# insertion point
			toReal(self.GetValue(10), 0.0),
			toReal(self.GetValue(20), 0.0),
			toReal(self.GetValue(30), 0.0),
			# scale:
			toReal(self.GetValue(41), 1.0),
			toReal(self.GetValue(42), 1.0),
			toReal(self.GetValue(43), 1.0),
			# angle:
			toReal(self.GetValue(50), 0.0),
			# cols / rows:
			toInt(self.GetValue(70), 1),
			toInt(self.GetValue(71), 1),
			# spacing:
			toReal(self.GetValue(44), 0.0),
			toReal(self.GetValue(45), 0.0)]

		creationInterface.addInsert(d)
		base = [self.BlockList[self.GetValue(2)][0][2], self.BlockList[self.GetValue(2)][0][3], self.BlockList[self.GetValue(2)][0][4]]
		print 'Insert', d, 'base:', base
		for type, data in self.BlockList[self.GetValue(2)][1]:
			data1 = self.Transform(type, base, d, data)
			if type == DL_ENTITY_LINE:
				self.dxfmod.line(data1)
			elif type == DL_ENTITY_ARC:
				self.dxfmod.arc(data1)
			elif type == DL_ENTITY_CIRCLE:
				self.dxfmod.circle(data1)
			elif type == DL_ENTITY_POLYLINE or type == DL_ENTITY_LWPOLYLINE or type == DL_ENTITY_SPLINE:
				self.dxfmod.polyline(data1)
			elif type == DL_ENTITY_TRACE:
				self.dxfmod.solid(data1)
			elif type == DL_ENTITY_SOLID:
				self.dxfmod.solid(data1)
			if d[0] == 'A$C7DB721AC' :
				print data1

	def addMText(self, creationInterface):
		print 'MText'

	def addText(self, creationInterface):
		d = [
			# insertion point
			toReal(self.GetValue(10), 0.0),
			toReal(self.GetValue(20), 0.0),
			toReal(self.GetValue(30), 0.0),
			# alignment point
			toReal(self.GetValue(11), 0.0),
			toReal(self.GetValue(21), 0.0),
			toReal(self.GetValue(31), 0.0),
			# height
			toReal(self.GetValue(40), 2.5),
			# x scale
			toReal(self.GetValue(41), 1.0),
			# generation flags
			toInt(self.GetValue(71), 0),
			# h just
			toInt(self.GetValue(72), 0),
			# v just
			toInt(self.GetValue(73), 0),
			# text
			self.GetValue(1),
			# style
			self.GetValue(7),
			# angle
			(toReal(self.GetValue(50), 0.0)*2*M_PI)/360.0]
		creationInterface.addText(d)
#		print 'Text:', self.GetValue(1)

	def addAttrib(self, creationInterface):
		d = [
		# insertion point
		toReal(self.GetValue(10), 0.0),
		toReal(self.GetValue(20), 0.0),
		toReal(self.GetValue(30), 0.0),
		# alignment point
		toReal(self.GetValue(11), 0.0),
		toReal(self.GetValue(21), 0.0),
		toReal(self.GetValue(31), 0.0),
		# height
		toReal(self.GetValue(40), 2.5),
		# x scale
		toReal(self.GetValue(41), 1.0),
		# generation flags
		toInt(self.GetValue(71), 0),
		# h just
		toInt(self.GetValue(72), 0),
		# v just
		toInt(self.GetValue(74), 0),
		# text
		self.GetValue(1),
		# style
		self.GetValue(7),
		# angle
		(toReal(self.GetValue(50), 0.0)*2*M_PI)/360.0]

		creationInterface.addText(d)
		print 'Attrib'

	def getDimData(self):
		#generic dimension data:
		return DL_DimensionData(
			# def point
			toReal(self.GetValue(10), 0.0),
			toReal(self.GetValue(20), 0.0),
			toReal(self.GetValue(30), 0.0),
			# text middle point
			toReal(self.GetValue(11), 0.0),
			toReal(self.GetValue(21), 0.0),
			toReal(self.GetValue(31), 0.0),
			# type
			toInt(self.GetValue(70), 0),
			# attachment point
			toInt(self.GetValue(71), 5),
			# line sp. style
			toInt(self.GetValue(72), 1),
			# line sp. factor
			toReal(self.GetValue(41), 1.0),
			# text
			self.GetValue(1),
			# style
			self.GetValue(3),
			# angle
			toReal(self.GetValue(53), 0.0))

	def addDimLinear(self, creationInterface):
		print 'DimLinear'
		d = self.getDimData()
		# horizontal / vertical / rotated dimension:
		dl = DL_DimLinearData(
			# definition point 1
			toReal(self.GetValue(13), 0.0),
			toReal(self.GetValue(23), 0.0),
			toReal(self.GetValue(33), 0.0),
			# definition point 2
			toReal(self.GetValue(14), 0.0),
			toReal(self.GetValue(24), 0.0),
			toReal(self.GetValue(34), 0.0),
			# angle
			toReal(self.GetValue(50), 0.0),
			# oblique
			toReal(self.GetValue(52), 0.0))
		creationInterface.addDimLinear(d, dl)

	def addDimAligned(self, creationInterface):
		print 'DimAligned'
		d = self.getDimData()

		# aligned dimension:
		da = DL_DimAlignedData(
			# extension point 1
			toReal(self.GetValue(13), 0.0),
			toReal(self.GetValue(23), 0.0),
			toReal(self.GetValue(33), 0.0),
			# extension point 2
			toReal(self.GetValue(14), 0.0),
			toReal(self.GetValue(24), 0.0),
			toReal(self.GetValue(34), 0.0))
		creationInterface.addDimAlign(d, da)

	def addDimRadial(self, creationInterface):
		print 'DimRadial'
		d = self.getDimData()

		dr = DL_DimRadialData(
			# definition point
			toReal(self.GetValue(15), 0.0),
			toReal(self.GetValue(25), 0.0),
			toReal(self.GetValue(35), 0.0),
			# leader length:
			toReal(self.GetValue(40), 0.0))
		creationInterface.addDimRadial(d, dr)

	def addDimDiametric(self, creationInterface):
		print 'DimDiametric'
		d = self.getDimData()

		# diametric dimension:
		dr = DL_DimDiametricData(
			# definition point
			toReal(self.GetValue(15), 0.0),
			toReal(self.GetValue(25), 0.0),
			toReal(self.GetValue(35), 0.0),
			# leader length:
			toReal(self.GetValue(40), 0.0))
		creationInterface.addDimDiametric(d, dr)

	def addDimAngular(self, creationInterface):
		print 'DimAngular'
		d = self.getDimData()

		# angular dimension:
		da = DL_DimAngularData(
			# definition point 1
			toReal(self.GetValue(13), 0.0),
			toReal(self.GetValue(23), 0.0),
			toReal(self.GetValue(33), 0.0),
			# definition point 2
			toReal(self.GetValue(14), 0.0),
			toReal(self.GetValue(24), 0.0),
			toReal(self.GetValue(34), 0.0),
			# definition point 3
			toReal(self.GetValue(15), 0.0),
			toReal(self.GetValue(25), 0.0),
			toReal(self.GetValue(35), 0.0),
			# definition point 4
			toReal(self.GetValue(16), 0.0),
			toReal(self.GetValue(26), 0.0),
			toReal(self.GetValue(36), 0.0))
		creationInterface.addDimAngular(d, da)

	def addDimAngular3P(self, creationInterface):
		print 'DimAngular3P'
		d = self.getDimData()

		# angular dimension (3P):
		da = DL_DimAngular3PData(
			# definition point 1
			toReal(self.GetValue(13), 0.0),
			toReal(self.GetValue(23), 0.0),
			toReal(self.GetValue(33), 0.0),
			# definition point 2
			toReal(self.GetValue(14), 0.0),
			toReal(self.GetValue(24), 0.0),
			toReal(self.GetValue(34), 0.0),
			# definition point 3
			toReal(self.GetValue(15), 0.0),
			toReal(self.GetValue(25), 0.0),
			toReal(self.GetValue(35), 0.0))
		creationInterface.addDimAngular3P(d, da)

	def addLeader(self, creationInterface):
		print 'Leader'
		le = DL_LeaderData(
			# arrow head flag
			toInt(self.GetValue(71), 1),
			# leader path type
			toInt(self.GetValue(72), 0),
			# Leader creation flag
			toInt(self.GetValue(73), 3),
			# Hookline direction flag
			toInt(self.GetValue(74), 1),
			# Hookline flag
			toInt(self.GetValue(75), 0),
			# Text annotation height
			toReal(self.GetValue(40), 1.0),
			# Text annotation width
			toReal(self.GetValue(41), 1.0),
			# Number of vertices in leader
			toInt(self.GetValue(76), 0)
		)
		creationInterface.addLeader(le)
		for i in range(self.maxLeaderVertices):
			d = DL_LeaderVertexData(self.leaderVertices[i][0],
								  self.leaderVertices[i][1],
								  self.leaderVertices[i][2])
			creationInterface.addLeaderVertex(d)
		if self.CurBlockName != '':
			self.addBlockList(DL_ENTITY_SPLINE, self.leaderVertices)
		else:
			self.dxfmod.polyline(self.leaderVertices)

	def addHatch(self, creationInterface):
		hd = DL_HatchData(toInt(self.GetValue(91), 1),
						toInt(self.GetValue(70), 0),
						toReal(self.GetValue(41), 1.0),
						toReal(self.GetValue(52), 0.0),
						self.GetValue(2))
		creationInterface.addHatch(hd)
		for l in range(self.maxHatchLoops):
			ld = DL_HatchLoopData(self.maxHatchEdges[l])
			creationInterface.addHatchLoop(ld)
			for b in range(self.maxHatchEdges[l]):
				creationInterface.addHatchEdge(self.hatchEdges[l][b])
				#draw hatch data
#				d = [self.hatchEdges[l][b].x1, self.hatchEdges[l][b].y1, 0.0, self.hatchEdges[l][b].x2, self.hatchEdges[l][b].y2, 0.0]
#				self.dxfmod.line(d)
		creationInterface.endEntity()
		self.currentEntity = DL_UNKNOWN
#		print 'Hatch', hd

	def addImage(self, creationInterface):
		print 'Image'
		id = DL_ImageData(# pass ref insead of name we don't have yet
			self.GetValue(340),
			# ins point:
			toReal(self.GetValue(10), 0.0),
			toReal(self.GetValue(20), 0.0),
			toReal(self.GetValue(30), 0.0),
			# u vector:
			toReal(self.GetValue(11), 1.0),
			toReal(self.GetValue(21), 0.0),
			toReal(self.GetValue(31), 0.0),
			# v vector:
			toReal(self.GetValue(12), 0.0),
			toReal(self.GetValue(22), 1.0),
			toReal(self.GetValue(32), 0.0),
			# image size (pixel):
			toInt(self.GetValue(13), 1),
			toInt(self.GetValue(23), 1),
			# brightness, contrast, fade
			toInt(self.GetValue(281), 50),
			toInt(self.GetValue(282), 50),
			toInt(self.GetValue(283), 0))

		creationInterface.addImage(id)
		creationInterface.endEntity()
		currentEntity = DL_UNKNOWN

	def addImageDef(self, creationInterface):
		print 'ImageDef'
		id = DL_ImageDefData(# handle
			self.GetValue(5),
			self.GetValue(1))

		creationInterface.linkImage(id)
		creationInterface.endEntity()
		self.currentEntity = DL_UNKNOWN

	def addTrace(self, creationInterface):
		td = [[0.0,0.0,0.0] for i in range(4)]
		for k in range(4):
			td[k][0] = toReal(self.GetValue(10 + k))
			td[k][1] = toReal(self.GetValue(20 + k))
			td[k][2] = toReal(self.GetValue(30 + k))
		creationInterface.addTrace(td)
		if self.CurBlockName != '':
			self.addBlockList(DL_ENTITY_TRACE, td)
		else:
			self.dxfmod.solid(sd)
		print 'Trace'

	def addSolid(self, creationInterface):
		sd = [[0.0,0.0,0.0] for i in range(4)]
		for k in range(4):
			print k, ' ', self.GetValue(10 + k), ' ', self.GetValue(20 + k), ' ', self.GetValue(30 + k)
			sd[k][0] = toReal(self.GetValue(10 + k))
			sd[k][1] = toReal(self.GetValue(20 + k))
			sd[k][2] = toReal(self.GetValue(30 + k))
		creationInterface.addSolid(sd)
		if sd[2] == sd[3]:
			print 'Solid 2==3\n'
			del sd[3]
		if self.CurBlockName != '':
			self.addBlockList(DL_ENTITY_SOLID, sd)
		else:
			self.dxfmod.solid(sd)

	#Called when a SEQEND occurs (when a POLYLINE or ATTRIB is done)
	def endSequence(self, creationInterface):
		if self.vnum > -1:
			if self.closeFlag & 2 != 0:
				self.vlist.append([self.vlist[0][0], self.vlist[0][1], self.vlist[0][2]])
			if self.CurBlockName != '':
				self.addBlockList(DL_ENTITY_POLYLINE, self.vlist)
			else:
				self.dxfmod.polyline(self.vlist)
			print 'end of polyline--------------------ĺ:', self.vnum
			del self.vlist
			self.vlist = []
			self.vnum = -1
			self.closeFlag = 0
		creationInterface.endSequence()

	#Called after an entity has been completed.
	def endEntity(self, creationInterface):
		creationInterface.endEntity()

	def handleMTextData(self, creationInterface):
		if self.groupCode == 3:
			creationInterface.addMTextChunk(self.groupValue)
			return 'true'
		return 'false'

	def handleLWPolylineData(self, creationInterface):
		print 'handleLWPolylineData'
		#Allocate LWPolyline vertices (group code 90):
		if self.groupCode == 90:
			self.maxVertices = toInt(self.groupValue)
			if self.maxVertices>0:
				if self.vertices != []:
					del self.vertices
				self.vertices = [[0.0, 0.0, 0.0, 0.0] for i in range(self.maxVertices)]
			self.vertexIndex=-1
			return 'true'
		# Compute LWPolylines vertices (group codes 10/20/30/42):
		elif self.groupCode == 10 or self.groupCode == 20 or self.groupCode==30 or self.groupCode==42:
			if self.vertexIndex < self.maxVertices-1 and self.groupCode == 10:
				self.vertexIndex = self.vertexIndex + 1
			if self.groupCode <= 30:
				if self.vertexIndex >= 0 and self.vertexIndex < self.maxVertices:
					self.vertices[self.vertexIndex][self.groupCode/10-1] = toReal(self.groupValue)
			elif self.groupCode == 42 and self.vertexIndex < self.maxVertices:
				self.vertices[self.vertexIndex][3] = toReal(self.groupValue)
			return 'true'
		return 'false'

	def handleSplineData(self, creationInterface):
		# Allocate Spline knots (group code 72):
		if self.groupCode == 72:
			self.maxKnots = toInt(self.groupValue)
			if self.maxKnots>0:
				if self.knots != []:
					del self.knots
				self.knots = [0.0] * self.maxKnots
			self.knotIndex=-1
			return 'true'
		# Allocate Spline control points (group code 73):
		elif self.groupCode == 73:
			self.maxControlPoints = toInt(self.groupValue)
			if self.maxControlPoints>0:
				if self.controlPoints != []:
					del self.controlPoints
				self.controlPoints = [[0.0, 0.0, 0.0] for i in range(self.maxControlPoints)]
			controlPointIndex=-1
			return 'true'
		# Compute spline knot vertices (group code 40):
		elif self.groupCode == 40:
			if self.knotIndex < self.maxKnots-1:
				self.knotIndex = self.knotIndex + 1
				self.knots[self.knotIndex] = toReal(self.groupValue)
			return 'true'
		# Compute spline control points (group codes 10/20/30):
		elif self.groupCode == 10 or self.groupCode == 20 or self.groupCode == 30:
			if self.controlPointIndex < self.maxControlPoints-1 and self.groupCode==10:
				self.controlPointIndex = self.controlPointIndex + 1
			if self.controlPointIndex>=0 and self.controlPointIndex<self.maxControlPoints:
				self.controlPoints[self.controlPointIndex][self.groupCode/10-1] = toReal(self.groupValue)
			return 'true'
		return 'false'

	def handleLeaderData(self, creationInterface):
		# Allocate Leader vertices (group code 76):
		if self.groupCode==76:
			self.maxLeaderVertices = toInt(self.groupValue)
			if self.maxLeaderVertices>0:
				if self.leaderVertices != []:
					del self.leaderVertices
				self.leaderVertices = [[0.0, 0.0, 0.0] for i in range(self.maxLeaderVertices)]
			self.leaderVertexIndex=-1
			return 'true'

		# Compute Leader vertices (group codes 10/20/30):
		elif self.groupCode == 10 or self.groupCode == 20 or self.groupCode == 30:
			if self.leaderVertexIndex < self.maxLeaderVertices-1 and self.groupCode==10:
				self.leaderVertexIndex = self.leaderVertexIndex + 1
			if self.groupCode<=30:
				if self.leaderVertexIndex>=0 and self.leaderVertexIndex<self.maxLeaderVertices:
					self.leaderVertices[self.leaderVertexIndex][self.groupCode/10-1] = toReal(self.groupValue)
			return 'true'
		return 'false'

	def handleHatchData(self, creationInterface):
		#staticˤ
		firstPolylineStatus = 0

		# Allocate hatch loops (group code 91):
		if self.groupCode==91 and toInt(self.groupValue)>0:
			#print 'allocating ' , toInt(self.groupValue) , ' loops\n'
			if self.hatchLoops!= []:
				for i in range(self.maxHatchLoops):
					self.hatchLoops[0]
				del self.hatchLoops
				self.hatchLoops = []
			
			if self.maxHatchEdges!= []:
				del self.maxHatchEdges
				self.maxHatchEdges = []
			
			if self.hatchEdgeIndex!= []:
				del self.hatchEdgeIndex
				self.hatchEdgeIndex = []
			
			if self.hatchEdges!= []:
#				print '[', self.hatchEdges,']', self.maxHatchLoops
				for i in range(self.maxHatchLoops):
					self.hatchEdges[0]
				del self.hatchEdges
				self.hatchEdges = []
			
			self.maxHatchLoops = toInt(self.groupValue)

			#print 'maxHatchLoops: ' , maxHatchLoops , '\n'

			if self.maxHatchLoops>0:
				self.hatchLoops = [[]] * self.maxHatchLoops
				self.maxHatchEdges = [0] * self.maxHatchLoops
				self.hatchEdgeIndex = [0] * self.maxHatchLoops
				self.hatchEdges = [[]] * self.maxHatchLoops
				#print 'new hatchEdges[' , self.maxHatchLoops , ']\n'
#				for i in range(self.maxHatchLoops):
#					self.hatchEdges[i] = []
					#print 'hatchEdges[' , i , '] = []\n'
#					self.maxHatchEdges[i] = 0
				
				self.hatchLoopIndex = -1
				self.dropEdges = 'false'
			
			#print 'done\n'
			return 'true'
		

		# Allocate hatch edges, group code 93
		if self.groupCode == 93 and toInt(self.groupValue) > 0:
			if self.hatchLoopIndex < self.maxHatchLoops-1 and self.hatchLoops!= [] and\
					self.maxHatchEdges != [] and self.hatchEdgeIndex != [] and\
					self.hatchEdges!= []:
				#print '  allocating ' , toInt(groupValue) , ' edges\n'
				self.dropEdges = 'false'
				self.hatchLoopIndex = self.hatchLoopIndex + 1
				self.hatchLoops[self.hatchLoopIndex]\
					= DL_HatchLoopData(toInt(self.groupValue))

				self.maxHatchEdges[self.hatchLoopIndex] = toInt(self.groupValue)
				self.hatchEdgeIndex[self.hatchLoopIndex] = -1
				for i in range(toInt(self.groupValue)):
					self.hatchEdges[self.hatchLoopIndex].append(DL_HatchEdgeData())
				#print 'hatchEdges[' , self.hatchLoopIndex , '] = new '
				#  , toInt(self.groupValue) , '\n'
				firstPolylineStatus = 0
			else:
				#print 'dropping ' , toInt(self.groupValue) , ' edges\n'
				self.dropEdges = 'true'
			#print 'done\n'
			return 'true'
		

		# Init hatch edge for non-polyline boundary (group code 72)
#		print '-->', self.GetValue(92), '<---', len(self.GetValue(92))
		if self.hatchEdges != [] and \
				self.hatchEdgeIndex != [] and \
				self.maxHatchEdges != [] and \
				self.hatchLoopIndex >=0 and \
				self.hatchLoopIndex < self.maxHatchLoops and \
				self.hatchEdgeIndex[self.hatchLoopIndex] < \
				self.maxHatchEdges[self.hatchLoopIndex] and \
				(toInt(self.GetValue(92))&2) ==0 and \
				self.groupCode==72 and \
				self.dropEdges=='false':

			#print 'Init hatch edge for non-polyline boundary\n'
			#print 'hatchLoopIndex: ' , self.hatchLoopIndex , '\n'
			#print 'maxHatchLoops: ' , self.maxHatchLoops , '\n'

			self.hatchEdgeIndex[self.hatchLoopIndex] = self.hatchEdgeIndex[self.hatchLoopIndex] + 1

			#print '  init edge: type: '
			#, toInt(self.groupValue)
			#, ' index: ' , self.hatchEdgeIndex[self.hatchLoopIndex] , '\n'

			self.hatchEdges[self.hatchLoopIndex][self.hatchEdgeIndex[self.hatchLoopIndex]].type = toInt(self.groupValue)
			self.hatchEdges[self.hatchLoopIndex][self.hatchEdgeIndex[self.hatchLoopIndex]].defined = 'false'

			#print 'done\n'
			return 'true'
		

		# Handle hatch edges for non-polyline boundaries
		#   (group codes 10, 20, 11, 21, 40, 50, 51, 73)
		if self.dropEdges == 'false' and \
				self.hatchEdges != [] and \
				self.hatchEdgeIndex != [] and \
				self.hatchLoopIndex >=0 and \
				self.hatchLoopIndex < self.maxHatchLoops and \
				self.hatchEdges[self.hatchLoopIndex] != [] and \
				self.hatchEdgeIndex[self.hatchLoopIndex] >=0 and \
				self.hatchEdgeIndex[self.hatchLoopIndex] < \
				self.maxHatchEdges[self.hatchLoopIndex] and \
				((toInt(self.GetValue(92))&2)==0) and \
				(self.groupCode ==10 or self.groupCode ==20 or \
				 self.groupCode ==11 or self.groupCode ==21 or \
				 self.groupCode ==40 or self.groupCode ==50 or \
				 self.groupCode ==51 or self.groupCode ==73):

			#print 'Handle hatch edge for non-polyline boundary\n'
			#print '  found edge data: ' , self.groupCode , '\n'
			#print '	 value: ' , toReal(self.groupValue) , '\n'

			# can crash:
			#print '	 defined: '
			#   , int(self.hatchEdges[self.hatchLoopIndex])
			#   [self.hatchEdgeIndex[self.hatchLoopIndex]].defined , '\n'

			#print '92 flag: '' , self.GetValue(92) , ''\n'
			#print '92 flag (int): '' , int(self.GetValue(92)) , ''\n'

			if self.hatchEdges[self.hatchLoopIndex][self.hatchEdgeIndex[self.hatchLoopIndex]].defined == 'false':
				if self.hatchEdges[self.hatchLoopIndex][self.hatchEdgeIndex[self.hatchLoopIndex]].type==1:
					if self.groupCode == 10:
						self.hatchEdges[self.hatchLoopIndex][self.hatchEdgeIndex[self.hatchLoopIndex]].x1 \
						= toReal(self.groupValue)
					elif self.groupCode == 20:
						self.hatchEdges[self.hatchLoopIndex][self.hatchEdgeIndex[self.hatchLoopIndex]].y1 \
						= toReal(self.groupValue)
					elif self.groupCode == 11:
						self.hatchEdges[self.hatchLoopIndex][self.hatchEdgeIndex[self.hatchLoopIndex]].x2 \
						= toReal(self.groupValue)
					elif self.groupCode == 21:
						self.hatchEdges[self.hatchLoopIndex][self.hatchEdgeIndex[self.hatchLoopIndex]].y2 \
						= toReal(self.groupValue)
						self.hatchEdges[self.hatchLoopIndex][self.hatchEdgeIndex[self.hatchLoopIndex]].defined = 'true'

				if self.hatchEdges[self.hatchLoopIndex][self.hatchEdgeIndex[self.hatchLoopIndex]].type==2:
					if self.groupCode == 10:
						self.hatchEdges[self.hatchLoopIndex][self.hatchEdgeIndex[self.hatchLoopIndex]].cx \
						= toReal(self.groupValue)
					elif self.groupCode == 20:
						self.hatchEdges[self.hatchLoopIndex][self.hatchEdgeIndex[self.hatchLoopIndex]].cy \
						= toReal(self.groupValue)
					elif self.groupCode == 40:
						self.hatchEdges[self.hatchLoopIndex][self.hatchEdgeIndex[self.hatchLoopIndex]].radius \
						= toReal(self.groupValue)
					elif self.groupCode == 50:
						self.hatchEdges[self.hatchLoopIndex][self.hatchEdgeIndex[self.hatchLoopIndex]].angle1 \
						= toReal(self.groupValue)/360.0*2*M_PI
					elif self.groupCode == 51:
						self.hatchEdges[self.hatchLoopIndex][self.hatchEdgeIndex[self.hatchLoopIndex]].angle2 \
						= toReal(self.groupValue)/360.0*2*M_PI
					elif self.groupCode == 73:
						self.hatchEdges[self.hatchLoopIndex][self.hatchEdgeIndex[self.hatchLoopIndex]].ccw \
						= toInt(self.groupValue)
						self.hatchEdges[self.hatchLoopIndex][self.hatchEdgeIndex[self.hatchLoopIndex]].defined = 'true'
			return 'true'
		return 'false'

	def addEntity(self, creationInterface):
		if self.currentEntity == DL_SETTING:
			self.addSetting(creationInterface)
		elif self.currentEntity == DL_LAYER:
			self.addLayer(creationInterface)
		elif self.currentEntity == DL_BLOCK:
			self.addBlock(creationInterface)
		elif self.currentEntity == DL_ENDBLK:
			self.endBlock(creationInterface)
		elif self.currentEntity == DL_ENTITY_POINT:
			self.addPoint(creationInterface)
		elif self.currentEntity == DL_ENTITY_LINE:
			self.addLine(creationInterface)
		elif self.currentEntity == DL_ENTITY_POLYLINE:
			self.addPolyline(creationInterface)
		elif self.currentEntity == DL_ENTITY_LWPOLYLINE:
			self.addLWPolyline(creationInterface)
		elif self.currentEntity == DL_ENTITY_VERTEX:
			self.addVertex(creationInterface)
		elif self.currentEntity == DL_ENTITY_SPLINE:
			self.addSpline(creationInterface)
		elif self.currentEntity == DL_ENTITY_ARC:
			self.addArc(creationInterface)
		elif self.currentEntity == DL_ENTITY_CIRCLE:
			self.addCircle(creationInterface)
		elif self.currentEntity == DL_ENTITY_ELLIPSE:
			self.addEllipse(creationInterface)
		elif self.currentEntity == DL_ENTITY_INSERT:
			self.addInsert(creationInterface)
		elif self.currentEntity == DL_ENTITY_MTEXT:
			self.addMText(creationInterface)
		elif self.currentEntity == DL_ENTITY_TEXT:
			self.addText(creationInterface)
		elif self.currentEntity == DL_ENTITY_ATTRIB:
			self.addAttrib(creationInterface)
		elif self.currentEntity == DL_ENTITY_DIMENSION:
			type = toInt(self.GetValue(70)) & 0x07
			if type == 0:
				self.addDimLinear(creationInterface)
			elif type == 1:
				self.addDimAligned(creationInterface)
			elif type == 2:
				self.addDimAngular(creationInterface)
			elif type == 3:
				self.addDimDiametric(creationInterface)
			elif type == 4:
				self.addDimRadial(creationInterface)
			elif type == 5:
				self.addDimAngular3P(creationInterface)
		elif self.currentEntity == DL_ENTITY_LEADER:
			self.addLeader(creationInterface)
		elif self.currentEntity == DL_ENTITY_HATCH:
			self.addHatch(creationInterface)
		elif self.currentEntity == DL_ENTITY_IMAGE:
			self.addImage(creationInterface)
		elif self.currentEntity == DL_ENTITY_IMAGEDEF:
			self.addImageDef(creationInterface)
		elif self.currentEntity == DL_ENTITY_TRACE:
			self.addTrace(creationInterface)
		elif self.currentEntity == DL_ENTITY_SOLID:
			self.addSolid(creationInterface)
		elif self.currentEntity == DL_ENTITY_SEQEND:
			self.endSequence(creationInterface)


	def setCurrentEntity(self, creationInterface):
#		print 'currentEntity = ', self.currentEntity, 'groupValue=', self.groupValue
		prevEntity = self.currentEntity
		#Read DXF settings:
		if self.groupValue[0] == '$':
#			print self.groupValue
			self.currentEntity = DL_SETTING
			self.settingKey = self.groupValue
		#Read Layers:
		elif self.groupValue == 'LAYER':
			self.currentEntity = DL_LAYER
		#Read Blocks:
		elif self.groupValue == 'BLOCK':
			self.currentEntity = DL_BLOCK
		elif self.groupValue == 'ENDBLK':
			self.currentEntity = DL_ENDBLK
		#Read entities:
		elif self.groupValue == 'POINT':
			self.currentEntity = DL_ENTITY_POINT
		elif self.groupValue == 'LINE':
			self.currentEntity = DL_ENTITY_LINE
		elif self.groupValue == 'POLYLINE':
			self.currentEntity = DL_ENTITY_POLYLINE
		elif self.groupValue == 'LWPOLYLINE':
			self.currentEntity = DL_ENTITY_LWPOLYLINE
		elif self.groupValue == 'VERTEX':
			self.currentEntity = DL_ENTITY_VERTEX
		elif self.groupValue == 'SPLINE':
			self.currentEntity = DL_ENTITY_SPLINE
		elif self.groupValue == 'ARC':
			self.currentEntity = DL_ENTITY_ARC
		elif self.groupValue == 'ELLIPSE':
			self.currentEntity = DL_ENTITY_ELLIPSE
		elif self.groupValue == 'CIRCLE':
			self.currentEntity = DL_ENTITY_CIRCLE
		elif self.groupValue == 'INSERT':
			self.currentEntity = DL_ENTITY_INSERT
		elif self.groupValue == 'TEXT':
			self.currentEntity = DL_ENTITY_TEXT
		elif self.groupValue == 'MTEXT':
			self.currentEntity = DL_ENTITY_MTEXT
		elif self.groupValue == 'ATTRIB':
			self.currentEntity = DL_ENTITY_ATTRIB
		elif self.groupValue == 'DIMENSION':
			self.currentEntity = DL_ENTITY_DIMENSION
		elif self.groupValue == 'LEADER':
			self.currentEntity = DL_ENTITY_LEADER
		elif self.groupValue == 'HATCH':
			self.currentEntity = DL_ENTITY_HATCH
		elif self.groupValue == 'IMAGE':
			self.currentEntity = DL_ENTITY_IMAGE
		elif self.groupValue == 'IMAGEDEF':
			self.currentEntity = DL_ENTITY_IMAGEDEF
		elif self.groupValue == 'TRACE':
		   self.currentEntity = DL_ENTITY_TRACE
		elif self.groupValue == 'SOLID':
		   self.currentEntity = DL_ENTITY_SOLID
		elif self.groupValue == 'SEQEND':
			self.currentEntity = DL_ENTITY_SEQEND
		else:
			self.currentEntity = DL_UNKNOWN

		#end of old style POLYLINE entity
		if prevEntity == DL_ENTITY_VERTEX and self.currentEntity != DL_ENTITY_VERTEX:
			self.endEntity(creationInterface)

	def ScanFile(self, PathName):
		f=open(PathName, 'r')
		self.linebuf = f.readlines()
		f.close()

	def getDxfGroups(self):
#		print self.linebuf[self.lpos]
		token = self.linebuf[self.lpos].replace('\n','')
		token = token.strip()
#		print 'code=', token
		self.groupCode = toInt(token)

		self.lpos = self.lpos + 1
		token = self.linebuf[self.lpos].replace('\n','')
		self.groupValue = token.strip()
#		print 'value=', self.groupValue
		self.lpos = self.lpos + 1

	def processFile(self):
	#	FileName = 'C:/Documents and Settings/zero/My Documents/Visual Studio 2005/Projects/BlenderScripts/data/JDPC001N.dxf'
	#	FileName = 'data/JDPC001N.dxf'
		FileName = ''

		creationInterface = DL_CreationInterface()

		while self.lpos <len(self.linebuf)-1 :
			self.getDxfGroups()
			if self.groupCode == 999:
				print ':', self.groupValue, '\n'
			elif self.groupCode == 0 or self.groupCode == 9 :
#				print 'lineno: ', self.lpos
				# defaults to layer '0'
				name = '0'
				if len(self.GetValue(8)) > 0 :
					name = self.GetValue(8)
				# default to BYLAYER
				width = -1
				# Compatibillity with qcad1
	#			print repr(self.GetValue(39)), type(self.GetValue(39))
				if len(self.GetValue(39)) > 0 and len(self.GetValue(370)) > 0:
					width = toInt(self.GetValue(39))
				# since autocad 2002
				elif len(self.GetValue(370)) > 0:
					width = toInt(self.GetValue(370))
				color = 256
				if len(self.GetValue(62)) > 0:
					color = toInt(self.GetValue(62))

				linetype = 'BYLAYER'
				if len(self.GetValue(6)) > 0:
					linetype = self.GetValue(6)

				attrib = DL_Attributes(self.GetValue(8),  # layer
										color,			  # color
										width,			  # width
										linetype)		  # linetype
				creationInterface.setAttributes(attrib)

				creationInterface.setExtrusion(toReal(self.GetValue(210)),
												toReal(self.GetValue(220)),
												toReal(self.GetValue(230), 1.0),
												toReal(self.GetValue(30)))
				#---------------------------------------------------
				self.addEntity(creationInterface)
				#ꥻå
				self.InitValues()
				self.settingValue = ''
				self.settingKey = ''
				self.setCurrentEntity(creationInterface)
			else:
				# Group code does not indicate start of new entity or setting,
				# so this group must be continuation of data for the current
				# one.
				if self.groupCode < DL_DXF_MAXGROUPCODE:
					handled = 'false'
					if self.currentEntity == DL_ENTITY_MTEXT:
						handled = self.handleMTextData(creationInterface)
					elif self.currentEntity == DL_ENTITY_LWPOLYLINE:
						handled = self.handleLWPolylineData(creationInterface)
					elif self.currentEntity == DL_ENTITY_SPLINE:
						handled = self.handleSplineData(creationInterface)
					elif self.currentEntity == DL_ENTITY_LEADER:
						handled = self.handleLeaderData(creationInterface)
					elif self.currentEntity == DL_ENTITY_HATCH:
						handled = self.handleHatchData(creationInterface)
					if handled == 'false':
						#Normal group / value pair:
						self.SetValue(self.groupCode, self.groupValue)
		self.Finish()
		creationInterface.delete()
		del creationInterface
