<?php
/**
* MemoPositionTableAccess.class.php memopositiontableDAO
* PHP versions：4.4.2
* @パッケージ：-
* @作者：K.Shimomura
* @作成日/作成者：2006/09/01 (IIS)K.Shimomura
* @更新日/更新者：
* @バージョン：1.0.0
*
* Copyright (c) 2006 Iwate Information System Co.,Ltd. All Rights Reserved.       
* You may see the License at LICENSE_EN.txt(English) or LICENSE_JA.txt(Japanese). 
*/ 
//データクラス
require_once("class/db/tabledata/MemoPositionData.class.php");
//データオブジェクト定義
require_once("class/db/dbaccess/DataObject.php");
//データアクセスクラス
require_once("class/db/dbaccess/KakikoDbAccess.class.php");
//継承テーブルアクセスクラス
require_once("class/db/dbaccess/TableAccess.class.php");


class MemoPositionTableAccess extends TableAccess {

	/**
	 *	クラス変数
	 */
	var $prefix;				// プレフィックス


    /**
     * コンストラクタ
     * @引数： (I) directryname  - string ディレクトリ名称
     */
    function MemoPositionTableAccess($directryname) {

		// 基底クラスのコンストラクタ呼び出し
    	$this->TableAccess();

    	$this->setTableName($directryname."_memopositiontable");
    	$this->setRecordSet(null);

		// プレフィックス設定
		$this->prefix = $directryname;
    }

    /**
     * 関数概要：メモ位置情報データ全件取得クエリ発行
     * 処理詳細：メモ位置情報テーブルデータの全件取得クエリを生成する
     *
     * @引数： (I) db       - object DBオブジェクト
     * @戻値： boolean true :正常
     *      ：         false:異常
     * @作成日/作成者：2006/09/01 (IIS)K.Shimomura
     * @更新日/更新者：
     * @更新内容：
     */
	function selectAllData($db) {

		$dba = new KakikoDBAccess($db, $this->getTablename());

		if ($dba -> dba_select() != true) {
			trigger_error($this->o_resource->getValue("D16_001_01"), E_USER_ERROR);
			return false;
		}
		$this->setRecordSet($dba->dba_getRecordSet());

		// 解放
		$dba->dba_free();

		return true;
	}

    /**
     * 関数概要：メモ位置情報キー指定データ取得クエリ発行
     * 処理詳細：メモ位置情報テーブルキー指定データの取得クエリを生成する
     *
     * @引数： (I) db       - object  DBオブジェクト
     *      ： (I) key      - integer キー内容(メモID)
     * @戻値： boolean true :正常
     *      ：         false:異常
     * @作成日/作成者：2006/09/01 (IIS)K.Shimomura
     * @更新日/更新者：
     * @更新内容：
     */
	function selectFindByKeyData($db, $key) {

		$dba = new KakikoDBAccess($db, $this->getTablename());

		// 条件指定
		$dba->dba_whereAdd("", "memoid = ".$key);

		if ($dba -> dba_select() != true) {
			trigger_error($this->o_resource->getValue("D16_002_01"), E_USER_ERROR);
			return false;
		}

		$this->setRecordSet($dba->dba_getRecordSet());

		// 解放
		$dba->dba_free();

		return true;
	}

   /**
    * 関数名：メモ位置テーブル最大最小座標取得
    * @引数：(I) $db	- DBオブジェクト
    *        (I) $imiSpaceId	- 対象意味空間ID
    * @戻値：$bool		- true/false
    * @作成日/作成者：2006/09/01 (IIS) N.Soma
    * @更新日/更新者：
    */	 	 
	function selectMaxMinPosition($db, $imiSpaceId) {

		$dba = new KakikoDBAccess($db, $this->getTablename());

		// カラムフィールド指定		
		$dba->dba_fieldAdd("max(coordinatex) as maxx");	//最大X座標
		$dba->dba_fieldAdd("max(coordinatey) as maxy");	//最大Y座標
		$dba->dba_fieldAdd("min(coordinatex) as minx");	//最小X座標
		$dba->dba_fieldAdd("min(coordinatey) as minx");	//最小X座標
		
		// 条件指定
		$dba->dba_whereAdd("", "imispaceid = ".$dba->query_str_convert($imiSpaceId));

		if ($dba -> dba_select() != true) {
			trigger_error(
				"ImiMemoPositionTableAccess-selectMaxMinPositionでエラーが発生しました。
				\n確認してください。", 
				E_CORE_ERROR);
			return false;
		}

		$this->setRecordSet($dba->dba_getRecordSet());

		// 解放
		$dba->dba_free();

		return true;
	}


    /**
     * 関数概要：メモ位置情報データ挿入
     * 処理詳細：メモ位置情報テーブルデータの挿入クエリを生成する
     *
     * @引数： (I) db       - object DBオブジェクト
     *      ： (I) dataobj  - object メモ位置情報データオブジェクト
     * @戻値： boolean true :正常
     *      ：         false:異常
     * @作成日/作成者：2006/09/01 (IIS)K.Shimomura
     * @更新日/更新者：
     * @更新内容：
     */
	function insertData($db, $dataobj) {

		$dba = new KakikoDBAccess($db, $this->getTablename());

		if (is_object($dataobj) && get_class($dataobj) == MEMOPOSITION_DATAOBJ) {

			// メモID
//			$dataobj->setMemoID($dataobj->getMemoID());
			$dataobj->setMemoID('default');
			// メモ種類ID
			$dataobj->setMemoTypeID($dataobj->getMemoTypeID());
			// 経度(X)
			$dataobj->setLongitude($dataobj->getLongitude());
			// 緯度(Y)
			$dataobj->setLatitude($dataobj->getLatitude());
			// 削除フラグ
			if (is_null($dataobj->getDeleteFlg()) != true) {
				$dataobj->setDeleteFlg(
					$dba->query_str_convert($dataobj->getDeleteFlg()));
			}
			// 登録日時
			$dataobj->setAddDateTime(
				$dba->query_str_convert($dataobj->getAddDateTime()));
			// 更新日時
			$dataobj->setUpdateDateTime(
				$dba->query_str_convert($dataobj->getUpdateDateTime()));

			if ($dba->dba_insert($dataobj) != true) {
				trigger_error($this->o_resource->getValue("D16_003_01"), E_USER_ERROR);
				return false;
			}
		} else {
			trigger_error($this->o_resource->getValue("D16_003_02"), E_USER_ERROR);
			return false;
		}

		// 解放
		$dba->dba_free();

		return true;
	}

	/**
	 *	関数名：メモ位置テーブルキー指定データ更新
     * @引数：(I) $db               - DBオブジェクト
     *        (I) $targetDataobj    - 対象列(条件列/更新列)指定用データオブジェクト
     *        (I) $valDataobj       - 設定値(条件値/更新値)指定用データオブジェクト
     * @戻値：$bool			- true/false
     * @作成日/作成者：2006/09/01 (IIS) K.Shimomura
     * @更新日/更新者：
     * @使用方法：whereキーとして使用したい列は$targetDataobjの該当変数をWHERE_KEYに指定し、条件値を$valDataobjの該当変数に使用する。
	 *            (プライマリキー以外にWHERE_KEYを指定しても無視されます)
     *            更新したい列は$targetDataobjの該当変数をVAL_TARGETに指定し、更新値を$valDataobjの該当変数に使用する。
     *            NULLで更新したい時は$valDataobjの該当変数にNULL_CHARを指定する。
     */
	function updateByKeyData($db, $targetDataobj, $valDataobj) {

		$whereflg = false;
		$dba = new KakikoDBAccess($db, $this->getTablename());

		if ((is_object($targetDataobj) && get_class($targetDataobj) == MEMOPOSITION_DATAOBJ) && 
			(is_object($valDataobj) && get_class($valDataobj) == MEMOPOSITION_DATAOBJ)) {

			// 条件および更新内容指定

			// メモID
			if ($targetDataobj->getMemoID() === WHERE_KEY) {
				$dba->dba_whereAdd("", "memoid = ".$valDataobj->getMemoID());
				$whereflg = true;
			} elseif ($targetDataobj->getMemoID() === VAL_TARGET) {
				// メモIDはDBでのシーケンスナンバーを使用しているので
				// 変えられると困る
//				$dba->dba_updateAdd(
//						"memoid = "
//							.$valDataobj->getMemoID());
			}
			// メモ種類ID
			if ($targetDataobj->getMemoTypeID() === VAL_TARGET) {
				$dba->dba_updateAdd(
						"memotypeid = "
							.$valDataobj->getMemoTypeID());
			}
			// 経度(X)
			if ($targetDataobj->getLongitude() === VAL_TARGET) {
				$dba->dba_updateAdd(
						"longitude = "
							.$valDataobj->getLongitude());
			}
			// 緯度(Y)
			if ($targetDataobj->getLatitude() === VAL_TARGET) {
				$dba->dba_updateAdd(
						"latitude = "
							.$valDataobj->getLatitude());
			}
			// 削除フラグ
			if ($targetDataobj->getDeleteFlg() === VAL_TARGET) {
					$dba->dba_updateAdd(
							"deleteflg = "
								.$dba->query_str_convert($valDataobj->getDeleteFlg()));
			}
			// 登録日時
			if ($targetDataobj->getAddDateTime() === VAL_TARGET) {
				$dba->dba_updateAdd(
						"adddatetime = "
							.$dba->query_str_convert($valDataobj->getAddDateTime()));
			}
			// 更新日時
			if ($targetDataobj->getUpdateDateTime() === VAL_TARGET) {
				$dba->dba_updateAdd(
						"updatedatetime = "
							.$dba->query_str_convert($valDataobj->getUpdateDateTime()));
			}

			// 条件指定確認
			if ($whereflg == false) {
				trigger_error($this->o_resource->getValue("D16_004_01"), E_USER_ERROR);
				return false;
			}
			//SQL実行
			if ($dba -> dba_update() != true) {
				trigger_error($this->o_resource->getValue("D16_004_02"), E_USER_ERROR);
				return false;
			}
		} else {
			trigger_error($this->o_resource->getValue("D16_004_03"), E_USER_ERROR);
			return false;
		}

		// 解放
		$dba->dba_free();

		return true;

	}

    /**
     * 関数概要：メモ位置情報キー指定データ削除
     * 処理詳細：メモ位置情報テーブルキー指定データの削除クエリを生成する
     *
     * @引数： (I) db       - object  DBオブジェクト
     *      ： (I) key      - integer キー内容(メモID)
     * @戻値： boolean true :正常
     *      ：         false:異常
     * @作成日/作成者：2006/09/01 (IIS)K.Shimomura
     * @更新日/更新者：
     * @更新内容：
     */
	function deleteByKeyData($db, $key) {

		$dba = new KakikoDBAccess($db, $this->getTablename());

		// 条件指定
		$dba->dba_whereAdd("", "memoid = ".$key);

		if ($dba -> dba_delete() != true) {
			trigger_error($this->o_resource->getValue("D16_005_01"), E_USER_ERROR);
			return false;
		}

		// 解放
		$dba->dba_free();

		return true;
	}

    /**
     * 関数概要：一覧データ取得
     * 処理詳細：レコードセットより、メモ位置情報リストを生成する
     *
     * @引数： 
     * @戻値： array   メモ位置情報リスト
     * @作成日/作成者：2006/09/01 (IIS)K.Shimomura
     * @更新日/更新者：
     * @更新内容：
     */
    function getDataList() {
        
		$memoposition_array = array();

		$rs = $this->getRecordSet();

		while($row = $rs->fetchrow(DB_FETCHMODE_ASSOC)) {

			$memopositiondata = new MemoPositionData();

			$memopositiondata->setMemoID(isset($row['memoid'])?$row['memoid']:null);
			$memopositiondata->setMemoTypeID(isset($row['memotypeid'])?$row['memotypeid']:null);
			$memopositiondata->setLongitude(isset($row['longitude'])?$row['longitude']:null);
			$memopositiondata->setLatitude(isset($row['latitude'])?$row['latitude']:null);
			$memopositiondata->setDeleteFlg(isset($row['deleteflg'])?$row['deleteflg']:null);
			$memopositiondata->setAddDateTime(isset($row['adddatetime'])?$row['adddatetime']:null);
			$memopositiondata->setUpdateDateTime(isset($row['updatedatetime'])?$row['updatedatetime']:null);

			array_push($memoposition_array, $memopositiondata);
		}
 		return $memoposition_array;
    }

    /**
     * 関数概要：単体データ取得
     * 処理詳細：レコードセットより、一件のメモ位置情報データを生成する
     *
     * @引数： 
     * @戻値： array   メモ位置情報データ型オブジェクト
     * @作成日/作成者：2006/09/01 (IIS)K.Shimomura
     * @更新日/更新者：
     * @更新内容：
     */
    function getOneData() {

		$rs = $this->getRecordSet();

		$row = $rs->fetchrow(DB_FETCHMODE_ASSOC);

		$memopositiondata = new MemoPositionData();

		$memopositiondata->setMemoID(isset($row['memoid'])?$row['memoid']:null);
		$memopositiondata->setMemoTypeID(isset($row['memotypeid'])?$row['memotypeid']:null);
		$memopositiondata->setLongitude(isset($row['longitude'])?$row['longitude']:null);
		$memopositiondata->setLatitude(isset($row['latitude'])?$row['latitude']:null);
		$memopositiondata->setDeleteFlg(isset($row['deleteflg'])?$row['deleteflg']:null);
		$memopositiondata->setAddDateTime(isset($row['adddatetime'])?$row['adddatetime']:null);
		$memopositiondata->setUpdateDateTime(isset($row['updatedatetime'])?$row['updatedatetime']:null);

 		return $memopositiondata;
    }

    /**
     * 関数概要：メモ位置テーブルキー指定重複チェック
     * 処理詳細：メモ位置テーブルデータをキー指定で重複しているかをチェック
     *
     * @引数： (I) db       - object  DBオブジェクト
     *      ： (I) key      - integer キー内容(メモID)
     * @戻値： boolean true :データなし
     *      ：         false:データあり
     * @作成日/作成者：2006/09/01 (IIS)K.Shimomura
     * @更新日/更新者：
     * @更新内容：
     */
	function checkFindByKeyData($db, $key) {

		$rtn = true;

		// キー指定データの取得
		$this->selectFindByKeyData($db, $key);

		if ($this->getRecordCount() > 0) {
			$rtn = false;	
		}

		return $rtn;
	}
}
?>