<?php
/**
* MapAdminService.class.php マップ管理者用DBアクセスサービス
* PHP versions：4.4.2
* @パッケージ：-
* @作者：K.Shimomura
* @作成日/作成者：2006/09/01 (IIS)K.Shimomura
* @更新日/更新者：
* @バージョン：1.0.0
*
* Copyright (c) 2006 Iwate Information System Co.,Ltd. All Rights Reserved.       
* You may see the License at LICENSE_EN.txt(English) or LICENSE_JA.txt(Japanese). 
*/

require_once("Date.php");                                  					// PEAR Date
require_once("class/db/dbaccess/AdminUserTableAccess.class.php");
require_once("class/db/dbaccess/BaseMapJoinMapDetailTableAccess.class.php");
require_once("class/db/dbaccess/BaseMapTableAccess.class.php");
require_once("class/db/dbaccess/MapTableAccess.class.php");
require_once("class/db/dbaccess/MapDetailTableAccess.class.php");
require_once("class/db/dbaccess/LayerPriorityJoinLayerTableAccess.class.php");
require_once("class/db/dbaccess/LayerPriorityTableAccess.class.php");
require_once("class/db/dbaccess/LayerTableAccess.class.php");
require_once("class/db/dbaccess/UserJoinLayerTableAccess.class.php");
require_once("class/db/dbaccess/UserJoinLayerUserAndLayerTableAccess.class.php");
require_once("class/db/dbaccess/UserTableAccess.class.php");
require_once("class/db/service/KakikoService.class.php");
require_once("class/db/tabledata/BaseMapData.class.php");
require_once("class/db/tabledata/LayerPriorityData.class.php");
require_once("class/db/tabledata/MapDetailData.class.php");
require_once("class/xml/RSSController.class.php");
require_once("function/KakikoFile.php");


class MapAdminService extends KakikoService {

    /**
     * コンストラクタ
     * @引数： (I) directryname  - string ディレクトリ名称
     */
    function MapAdminService($directryname) {

		// 基底クラスのコンストラクタ呼び出し
    	$this->KakikoService($directryname);

		// DB接続情報の設定
		$this->setDBConnectInfo();

    }

    /**
     * 関数概要：管理者ユーザテーブルキー指定重複チェック
     * 処理詳細：管理者ユーザテーブルに対してキー指定で重複データチェックを
     *         ：行うクエリ作成を呼び出す
     *
     * @引数： (I) key       - string 管理者ユーザテーブルキー(ユーザID)
     * @戻値： boolean true :データなし
     *      ：         false:データあり
     * @作成日/作成者：2006/09/01 (IIS)K.Shimomura
     * @更新日/更新者：
     * @更新内容：
     */
	function checkAdminUserDataByKey($key) {

		// DBオブジェクトの生成
		$this->createDBObject();

		$adminuser_access = new AdminUserTableAccess();

		// 管理ユーザデータの全件取得
		$rtn = $adminuser_access->checkFindByKeyData($this->o_db, $key);

		$adminuser_access->free();

		// DB接続の切断
		$this->disconnectDB();

		return $rtn;
	}

    /**
     * 関数概要：マップ管理者情報データ取得
     * 処理詳細：管理者ユーザデータの取得を行うクエリ作成を呼び出す
     *
     * @引数： (I) userid    - ユーザID
     * @戻値： array   管理者ユーザデータクラス
     *      ：         DBエラーの場合はnull
     * @作成日/作成者：2006/09/01 (IIS)K.Shimomura
     * @更新日/更新者：
     * @更新内容：
     */
	function getMapAdminInfoData($userid) {

		// DBオブジェクトの生成
		$this->createDBObject();

		// SQL:	SELECT adminusertable.userid, 
		//				adminusertable.mailaddress, 
		//				adminusertable.username,
		//				adminusertable.password
		//		FROM   adminusertable
		//		WHERE  adminusertable.userid = ?

		// 管理者ユーザテーブルDBアクセス
		$table_access = new AdminUserTableAccess();

		// クエリー発行
		$rtn = $table_access->selectFindByKeyData($this->o_db, $userid);
		if ($rtn != true) {
			$table_access->free();
			$this->disconnectDB();
			trigger_error($this->o_resource->getValue("S2_001_01"), E_USER_ERROR);
			return null;
		}

		if ($table_access->getRecordCount() == 0){
			return null;
		}

		// データオブジェクトの取得
		$array_data = $table_access->getOneData();

		// 解放
		$table_access->free();

		// DB接続の切断
		$this->disconnectDB();

		return $array_data;
	}

    /**
     * 関数概要：コンテンツ環境情報データ取得
     * 処理詳細：マップデータの取得を行うクエリ作成を呼び出す
     *
     * @引数： (I) userid    - ユーザID
     * @戻値： array   マップデータクラス
     *      ：         DBエラーの場合はnull
     * @作成日/作成者：2006/09/01 (IIS)K.Shimomura
     * @更新日/更新者：
     * @更新内容：
     */
	function getContentsEnvInfoData($userid) {

		// DBオブジェクトの生成
		$this->createDBObject();

		// SQL:	SELECT maptable.directryname, 
		//				maptable.userid,
		//				maptable.publiccontents,
		//				maptable.addattestlayer,
		//				maptable.contentsname,
		//				maptable.floatingboxcolor,
		//				maptable.floatingboxfontcolor, 
		//				maptable.floatingboxtransparency,
		//				maptable.autologin,
		//				maptable.layercount,
		//				maptable.memotypecount
		//		FROM   maptable
		//		WHERE  maptable.userid = ?

		// マップテーブルDBアクセス
		$table_access = new MapTableAccess();

		// クエリー発行
		$rtn = $table_access->selectFindByKeyData($this->o_db, $userid);
		if ($rtn != true) {
			$table_access->free();
			$this->disconnectDB();
			trigger_error($this->o_resource->getValue("S2_002_01"), E_USER_ERROR);
			return null;
		}

		if ($table_access->getRecordCount() == 0){
			return null;
		}
		
		// データオブジェクトの取得
		$array_data = $table_access->getOneData();

		// 解放
		$table_access->free();

		// DB接続の切断
		$this->disconnectDB();

		return $array_data;
	}

    /**
     * 関数概要：優先順位設定レイヤーデータ取得
     * 処理詳細：レイヤーテーブルとレイヤー優先順位テーブルをリンクしたデータの取得を
     *         ：行うクエリ作成を呼び出す
     *
     * @引数： 
     * @戻値： array   レイヤーリンクレイヤー優先順位データクラス
     *      ：         DBエラーの場合はnull
     * @作成日/作成者：2006/09/01 (IIS)K.Shimomura
     * @更新日/更新者：
     * @更新内容：
     */
	function getSettingPriorityLayerData() {

		// DBオブジェクトの生成
		$this->createDBObject();

		// レイヤー・レイヤー優先順位ジョインテーブルDBアクセス
		$table_access = new LayerPriorityJoinLayerTableAccess($this->directryname);

		// クエリー発行
		$rtn = $table_access->selectLayerExistLayerPriorityListData($this->o_db);
		if ($rtn != true) {
			$table_access->free();
			$this->disconnectDB();
			trigger_error($this->o_resource->getValue("S2_003_01"), E_USER_ERROR);
			return null;
		}

		// データオブジェクトの取得
		$array_data = $table_access->getDataList();

		// 解放
		$table_access->free();

		// DB接続の切断
		$this->disconnectDB();

		return $array_data;
	}

    /**
     * 関数概要：優先順位未設定レイヤーデータ取得
     * 処理詳細：レイヤーにマップ管理者が設定するレイヤー順位が設定されていないデータの取得を
     *         ：行うクエリ作成を呼び出す
     *
     * @引数： 
     * @戻値： array   レイヤーデータクラス
     *      ：         DBエラーの場合はnull
     * @作成日/作成者：2006/09/01 (IIS)K.Shimomura
     * @更新日/更新者：
     * @更新内容：
     */
	function getNoSettingPriorityLayerData() {

		// DBオブジェクトの生成
		$this->createDBObject();

		// レイヤーテーブルDBアクセス
		$table_access = new LayerTableAccess($this->directryname);

		// クエリー発行
		$rtn = $table_access->selectLayerNotExistLayerPriorityListData($this->o_db);
		if ($rtn != true) {
			$table_access->free();
			$this->disconnectDB();
			trigger_error($this->o_resource->getValue("S2_004_01"), E_USER_ERROR);
			return null;
		}

		// データオブジェクトの取得
		$array_data = $table_access->getDataList();

		// 解放
		$table_access->free();

		// DB接続の切断
		$this->disconnectDB();

		return $array_data;
	}

    /**
     * 関数概要：レイヤーデータリスト取得
     * 処理詳細：レイヤーテーブルとユーザテーブルをリンクしたデータの取得を
     *         ：行うクエリ作成を呼び出す
     *
     * @引数： (I) type      - string 取得状態フラグ 0:認証済み
     *      ：                                       1:許可待ち
     *      ： (I) layerName - string 検索レイヤー名称
     *      ： (I) userName  - string 検索ユーザ名称
     * @戻値： array   レイヤーリンクユーザデータクラス
     *      ：         DBエラーの場合はnull
     * @作成日/作成者：2006/09/01 (IIS)K.Shimomura
     * @更新日/更新者：
     * @更新内容：
     */
	function getLayerDataList($type, $layerName, $userName) {

		// DBオブジェクトの生成
		$this->createDBObject();

	    // クエリー条件(状態フラグ)
	    $status_array = $type == 0 
	    		? array(KAKIKOMAP_STATUSFLG_ON, KAKIKOMAP_STATUSFLG_OFF) 
	    									: array(KAKIKOMAP_STATUSFLG_WAIT);

		// SQL:	SELECT directry_usertable.userid, 
		//				directry_usertable.username,
		//				directry_usertable.mailaddress,
		//				directry_layertable.layerid,
		//				directry_layertable.layername,
		//				directry_layertable.adddatetime
		//				directry_layertable.statusflg 
		//		FROM   directry_usertable
		//			INNER JOIN directry_layertable
		//				ON directry_usertable.userid = directry_layertable.userid
		//		WHERE  directry_layertable.statusflg IN (?, ?)
		//			AND  directry_layertable.layername LIKE '%?%'
		//			AND  directry_usertable.username LIKE '%?%'
		//		ORDER BY directry_layertable.userid,
		//			directry_layertable.layerid

		// ユーザ・レイヤージョインテーブルDBアクセス
		$table_access = new UserJoinLayerTableAccess($this->directryname);

		// クエリー発行
		$rtn = $table_access->selectAdminUserLayerListData(
							$this->o_db, $status_array, $layerName, $userName);
		if ($rtn != true) {
			$table_access->free();
			$this->disconnectDB();
			trigger_error($this->o_resource->getValue("S2_005_01"), E_USER_ERROR);
			return null;
		}

		// データオブジェクトの取得
		$array_data = $table_access->getDataList();

		// 解放
		$table_access->free();

		// DB接続の切断
		$this->disconnectDB();

		return $array_data;

	}

    /**
     * 関数概要：ユーザデータリスト取得
     * 処理詳細：ユーザテーブルとレイヤーテーブルとレイヤー優先順位テーブルを
     *         ：リンクしたデータの取得を行うクエリ作成を呼び出す
     *
     * @引数： (I) layerName - string 検索レイヤー名称
     *      ： (I) userName  - string 検索ユーザ名称
     * @戻値： array   ユーザリンクレイヤーリンクレイヤー優先順位データクラス
     *      ：         DBエラーの場合はnull
     * @作成日/作成者：2006/09/01 (IIS)K.Shimomura
     * @更新日/更新者：
     * @更新内容：
     */
	function getUserDataList($layerName, $userName) {

		// DBオブジェクトの生成
		$this->createDBObject();

	    // クエリー条件(ユーザ種別)
	    $usertype_array = array(KAKIKOMAP_USERTYPE_LAYER, KAKIKOMAP_USERTYPE_NORMAL);

		// SQL:	SELECT DISTINCT directry_usertable.userid, 
		//				directry_usertable.username,
		//				directry_usertable.mailaddress,
		//				directry_usertable.adddatetime,
		//				directry_usertable.statusflg,
		//		FROM   directry_usertable
		//			LEFT JOIN directry_layerusertable
		//				ON directry_usertable.userid = directry_layerusertable.userid
		//			LEFT JOIN directry_layertable
		//				ON directry_layertable.layerid = directry_layerusertable.layerid
		//		WHERE  directry_usertable.usertype IN ('2','3')
		//			AND  directry_layertable.layername LIKE '%?%'
		//			AND  directry_usertable.username LIKE '%?%'
		//		ORDER BY directry_usertable.userid

		// ユーザ・レイヤー・レイヤー優先順位ジョインテーブルDBアクセス
		$table_access = new UserJoinLayerUserAndLayerTableAccess($this->directryname);

		// クエリー発行
		$rtn = $table_access->selectAdminUserInfoListData(
							$this->o_db, $usertype_array, $layerName, $userName);
		if ($rtn != true) {
			$table_access->free();
			$this->disconnectDB();
			trigger_error($this->o_resource->getValue("S2_006_01"), E_USER_ERROR);
			return null;
		}

		// データオブジェクトの取得
		$array_data = $table_access->getDataList();

		// 解放
		$table_access->free();

		// DB接続の切断
		$this->disconnectDB();

		return $array_data;

	}

	/**
	 *	関数名：ユーザテーブルデータキー更新
     * @引数：(I) targetDataobj    - 対象列(条件列/更新列)指定用データオブジェクト
     *        (I) valDataobj       - 設定値(条件値/更新値)指定用データオブジェクト
     * @戻値：bool			- true  正常更新
     *                      - false 異常
     * @作成日/作成者：2006/09/01 (IIS) K.Shimomura
     * @更新日/更新者：
     * @使用方法：ユーザテーブルデータをキー(ユーザID)で更新を行います。
	 *            
     */
	function updateUserDataByKey($targetDataobj, $valDataobj) {

		// DBオブジェクトの生成
		$this->createDBObject();

		// トランザクション開始
		$this->o_db->autoCommit(false);

		// ユーザDBアクセスインスタンス生成
		$table_access = new UserTableAccess($this->directryname);

		// ユーザデータ更新
		$rtn = $table_access->updateByKeyData($this->o_db, $targetDataobj, $valDataobj);
		if ($rtn != true) {
			$this->o_db->rollback();
			$table_access->free();
			$this->disconnectDB();
			trigger_error($this->o_resource->getValue("S2_007_01"), E_USER_ERROR);

			return false;
		}

		// 解放
		$table_access->free();

		// トランザクションコミット
		$this->o_db->commit();

		// DB接続の切断
		$this->disconnectDB();
		
		return true;

	}

	/**
	 *	関数名：レイヤーテーブルデータキー更新
     * @引数：(I) targetDataobj    - 対象列(条件列/更新列)指定用データオブジェクト
     *        (I) valDataobj       - 設定値(条件値/更新値)指定用データオブジェクト
     * @戻値：bool			- true  正常更新
     *                      - false 異常
     * @作成日/作成者：2006/09/01 (IIS) K.Shimomura
     * @更新日/更新者：
     * @使用方法：レイヤーテーブルデータをキー(ユーザID・レイヤーID)で更新を行います。
	 *            
     */
	function updateLayerDataByKey($targetDataobj, $valDataobj) {

		// DBオブジェクトの生成
		$this->createDBObject();

		// トランザクション開始
		$this->o_db->autoCommit(false);

		// ユーザDBアクセスインスタンス生成
		$table_access = new LayerTableAccess($this->directryname);

		// ユーザデータ更新
		$rtn = $table_access->updateByKeyData($this->o_db, $targetDataobj, $valDataobj);

		if ($rtn != true) {
			$this->o_db->rollback();
			$table_access->free();
			$this->disconnectDB();
			trigger_error($this->o_resource->getValue("S2_008_01"), E_USER_ERROR);
			return false;
		}

		// 解放
		$table_access->free();

		// トランザクションコミット
		$this->o_db->commit();

		// DB接続の切断
		$this->disconnectDB();
		
		return true;

	}

	/**
	 *	関数名：コンテンツ環境情報設定
     * @引数：(I) targetMapDataobj    - 対象列(条件列/更新列)指定用データオブジェクト
     *        (I) valMapDataobj       - 設定値(条件値/更新値)指定用データオブジェクト
     *        (I) priorityLayerData   - 優先順位設定レイヤーデータリスト
     * @戻値：bool			- true  正常更新
     *                      - false 異常
     * @作成日/作成者：2006/09/01 (IIS) K.Shimomura
     * @更新日/更新者：
     * @使用方法：コンテンツ環境情報の設定を行います。
	 *            
     */
	function setContentsEnv($targetMapDataobj, $valMapDataobj, $priorityLayerData) {

		// DBオブジェクトの生成
		$this->createDBObject();

		// トランザクション開始
		$this->o_db->autoCommit(false);

		// マップDBアクセスインスタンス生成
		$maptable_access = new MapTableAccess();

		// マップデータ更新
		$rtn = $maptable_access->updateByKeyData($this->o_db, $targetMapDataobj, $valMapDataobj);
		if ($rtn != true) {
			$this->o_db->rollback();
			$maptable_access->free();
			$this->disconnectDB();
			trigger_error($this->o_resource->getValue("S2_009_01"), E_USER_ERROR);

			return false;
		}

		// レイヤー優先順位DBアクセスインスタンス生成
		$layerprioritytable_access = new LayerPriorityTableAccess($this->directryname);

		// レイヤー優先順位データ全削除
		$rtn = $layerprioritytable_access->deleteAllData($this->o_db);
		if ($rtn != true) {
			$this->o_db->rollback();
			$maptable_access->free();
			$layerprioritytable_access->free();
			$this->disconnectDB();
			trigger_error($this->o_resource->getValue("S2_009_02"), E_USER_ERROR);

			return false;
		}

		// 優先順位設定レイヤーデータ追加
		if (!is_null($priorityLayerData)) {
			foreach ($priorityLayerData as $key=>$value) {

				// レイヤー優先順位データ型オブジェクト
				$priority_data = new LayerPriorityData();
	
				$priority_data->setPriorityOrder($key);
				$priority_data->setLayerID($value);
				$priority_data->setAddDateTime(date('Y/m/d H:i:s'));
				$priority_data->setUpdateDateTime(date('Y/m/d H:i:s'));
	
				// データ追加
				$rtn = $layerprioritytable_access->insertData($this->o_db, $priority_data);
				if ($rtn != true) {
					$this->o_db->rollback();
					$maptable_access->free();
					$layerprioritytable_access->free();
					$this->disconnectDB();
					trigger_error($this->o_resource->getValue("S2_009_03"), E_USER_ERROR);

					return false;
				}
			}
		}

		// 解放
		$maptable_access->free();
		$layerprioritytable_access->free();

		// トランザクションコミット
		$this->o_db->commit();

		// DB接続の切断
		$this->disconnectDB();

		return true;

	}

	/**
	 *	関数名：レイヤー削除
     * @引数：(I) userid    - ユーザID
     *        (I)$layerid   - レイヤーID
     * @戻値：bool			- true  正常登録
     *                      - false 異常
     * @作成日/作成者：2006/09/01 (IIS) K.Shimomura
     * @更新日/更新者：
     * @使用方法：レイヤーの削除を行います。
	 *            
     */
	function deleteLayerByKey($userid, $layerid) {

		// DBオブジェクトの生成
		$this->createDBObject();

		// トランザクション開始
		$this->o_db->autoCommit(false);

        // -----------------------------------------------
        // RSSファイルの削除
        // -----------------------------------------------
        $rss_control_obj = new RSSController($layerid, $this->directryname);
        $rss_control_obj->deleteRSSFile();

		// レイヤーテーブルデータ削除処理
		$table_access = new LayerTableAccess($this->directryname);

		$rtn = $table_access->deleteByKeyData($this->o_db, $userid, $layerid);
		if ($rtn != true) {
			$this->o_db->rollback();
			$table_access->free();
			$this->disconnectDB();
			trigger_error($this->o_resource->getValue("S2_010_01"), E_USER_ERROR);

			return false;
		}

		// 解放
		$table_access->free();

		// トランザクションコミット
		$this->o_db->commit();

		// DB接続の切断
		$this->disconnectDB();

		return true;

	}

	/**
	 *	関数名：ユーザ削除
     * @引数：(I) userid    - ユーザID
     * @戻値：bool			- true  正常登録
     *                      - false 異常
     * @作成日/作成者：2006/09/01 (IIS) K.Shimomura
     * @更新日/更新者：
     * @使用方法：ユーザの削除を行います。
	 *            
     */
	function deleteUserByKey($userid) {

		// DBオブジェクトの生成
		$this->createDBObject();

		// トランザクション開始
		$this->o_db->autoCommit(false);

		// ユーザテーブルデータ削除処理
		$table_access = new UserTableAccess($this->directryname);

		$rtn = $table_access->deleteByKeyData($this->o_db, $userid);
		if ($rtn != true) {
			$this->o_db->rollback();
			$table_access->free();
			$this->disconnectDB();
			trigger_error($this->o_resource->getValue("S2_011_01"), E_USER_ERROR);

			return false;
		}

		// 解放
		$table_access->free();

		// トランザクションコミット
		$this->o_db->commit();

		// DB接続の切断
		$this->disconnectDB();

		return true;

	}

    /**
     * 関数概要：ベースマップ情報データ取得
     * 処理詳細：ベースマップテーブルとマップ詳細テーブルを
     *         ：リンクしたデータの取得を行うクエリ作成を呼び出す
     *
     * @引数： 
     * @戻値： array   ベースマップリンクマップ詳細データクラス
     *      ：         DBエラーの場合はnull
     * @作成日/作成者：2006/09/01 (IIS)K.Shimomura
     * @更新日/更新者：
     * @更新内容：
     */
	function getBaseMapInfoData() {

		// DBオブジェクトの生成
		$this->createDBObject();

		// ベースマップジョインマップ詳細テーブルDBアクセス
		$table_access = new BaseMapJoinMapDetailTableAccess();

		// クエリー発行
		$rtn = $table_access->selectMapLinkMapDetailListData($this->o_db, $this->directryname);
		if ($rtn != true) {
			$table_access->free();
			$this->disconnectDB();
			trigger_error($this->o_resource->getValue("S2_012_01"), E_USER_ERROR);
			return null;
		}

		// データオブジェクトの取得
		$array_data = $table_access->getDataList();

		// DB接続の切断
		$this->disconnectDB();

		return $array_data;
	}

	/**
	 *	関数名：ベースマップ設定(GIS使用無し)
     * @引数：(I) $input_mapno    - データ入力地図番号格納配列
     *        (I) $post_data      - ポストデータ格納連想配列
     * @戻値：$bool			- true  正常処理
     *                      - false 異常
     * @作成日/作成者：2006/09/01 (IIS) K.Shimomura
     * @更新日/更新者：
     * @使用方法：ベースマップ(GIS使用無し)の設定を行います。
	 *            引数のポストデータには、'P_START_X'=>左下座標X
	 *                                    'P_START_Y'=>左下座標Y
	 *                                    'P_END_X'=>中心座標X
	 *                                    'P_END_Y'=>中心座標Y
	 *                                    'P_USEGIS'=>GIS使用区分
	 *                                    'P_MAPSIZE_HORIZONTAL'=>地図サイズ横
	 *                                    'P_MAPSIZE_VERTICAL'=>地図サイズ縦
	 *                                    'P_MOVEMAP_SPHERE'=>地図移動幅
	 *                                    'P_MAPSTATUS'=>画像ファイルステータス
	 *                                    'P_MAPIMAGE'=>画像ファイル区分
	 *                                    'P_LISTNAME'=>表示名
	 *                                    'P_INI_DEF'=>初期表示選択区分
	 *                                    'P_EX_DEF'=>外部参照選択区分
	 *                                    'P_PIXEL'=>ピクセル数
	 *                                    'P_DELCHECK'=>削除選択区分
	 *　　　　　　が必ず設定されていること。
     */
	function setBaseMap_GIS_Off($input_mapno, $post_data) {

		// DBオブジェクトの生成
		$this->createDBObject();

		// トランザクション開始
		$this->o_db->autoCommit(false);

		// ベースマップテーブル更新内容の設定

		// 条件及びNULL等のターゲット設定
		$targetBaseMapDataObj = new BaseMapData();

		// 更新条件
		$targetBaseMapDataObj->setDirectryName(WHERE_KEY);

		// 更新項目ターゲット設定
		$targetBaseMapDataObj->setUseGIS(VAL_TARGET);
		$targetBaseMapDataObj->setStartX(VAL_TARGET);
		$targetBaseMapDataObj->setStartY(VAL_TARGET);
		$targetBaseMapDataObj->setEndX(VAL_TARGET);
		$targetBaseMapDataObj->setEndY(VAL_TARGET);
		$targetBaseMapDataObj->setMapSizeHorizontal(VAL_TARGET);
		$targetBaseMapDataObj->setMapSizeVertical(VAL_TARGET);
		$targetBaseMapDataObj->setDiffX(NULL_CHAR);
		$targetBaseMapDataObj->setDiffY(NULL_CHAR);
		$targetBaseMapDataObj->setMoveMapSphere(VAL_TARGET);
		$targetBaseMapDataObj->setUpdateDateTime(VAL_TARGET);

		// 更新項目設定
		$valBaseMapDataObj = new BaseMapData();

		$valBaseMapDataObj->setDirectryName($this->directryname);
		$valBaseMapDataObj->setUseGIS($post_data['P_USEGIS']);
		$valBaseMapDataObj->setStartX($post_data['P_START_X']);
		$valBaseMapDataObj->setStartY($post_data['P_START_Y']);
		$valBaseMapDataObj->setEndX($post_data['P_END_X']);
		$valBaseMapDataObj->setEndY($post_data['P_END_Y']);
		$valBaseMapDataObj->setMapSizeHorizontal($post_data['P_MAPSIZE_HORIZONTAL']);
		$valBaseMapDataObj->setMapSizeVertical($post_data['P_MAPSIZE_VERTICAL']);
		$valBaseMapDataObj->setMoveMapSphere($post_data['P_MOVEMAP_SPHERE']);
		$valBaseMapDataObj->setUpdateDateTime(date('Y/m/d H:i:s'));

		// ベースマップDBアクセスインスタンス生成
		$table_access = new BaseMapTableAccess();

		// ベースマップデータ更新
		$rtn = $table_access->updateByKeyData($this->o_db, $targetBaseMapDataObj, $valBaseMapDataObj);
		if ($rtn != true) {
			$this->o_db->rollback();
			$table_access->free();
			$this->disconnectDB();
			trigger_error($this->o_resource->getValue("S2_013_01"), E_USER_ERROR);

			return false;
		}

		// 解放
		$table_access->free();

		// 地図番号分繰り返し
		foreach($input_mapno as $i) {

			// マップ詳細DBアクセスインスタンス生成
			$table_access = new MapDetailTableAccess();

			// 削除項目
			if (isset($post_data['P_DELCHECK'][$i])) {
			
				// マップ詳細データ存在チェック
				$rtn = $table_access->checkFindByKeyData($this->o_db, $this->directryname, $post_data['P_DELCHECK'][$i]);
				if (!$rtn) {
					// データがある場合は、削除処理
					$rtn = $table_access->deleteByKeyData($this->o_db, $this->directryname, $post_data['P_DELCHECK'][$i]);
					if ($rtn != true) {
						$this->o_db->rollback();
						$table_access->free();
						$this->disconnectDB();
						trigger_error($this->o_resource->getValue("S2_013_02"), E_USER_ERROR);
			
						return false;
					}
				}
			} else {
				// 更新項目
			
				// マップ詳細データ存在チェック
				$rtn = $table_access->checkFindByKeyData($this->o_db, $this->directryname, $i);
				if (!$rtn) {
					// データがある場合は、更新処理
					
					// 条件及びNULL等のターゲット設定
					$targetMapDetailDataObj = new MapDetailData();
			
					// 更新条件
					$targetMapDetailDataObj->setDirectryName(WHERE_KEY);
					$targetMapDetailDataObj->setMapNumber(WHERE_KEY);
			
					// 更新項目ターゲット設定
					$targetMapDetailDataObj->setListName(VAL_TARGET);
					$targetMapDetailDataObj->setInitialList(VAL_TARGET);
					$targetMapDetailDataObj->setOutsideReference(VAL_TARGET);
					$targetMapDetailDataObj->setPixel(VAL_TARGET);
					$targetMapDetailDataObj->setUpdateDateTime(VAL_TARGET);

					// 更新項目設定
					$valMapDetailDataObj = new MapDetailData();

					$valMapDetailDataObj->setDirectryName($this->directryname);
					$valMapDetailDataObj->setMapNumber($i);

					$valMapDetailDataObj->setListName($post_data['P_LISTNAME'][$i]);
					$valMapDetailDataObj->setInitialList(($post_data['P_INI_DEF']==$i)?KAKIKOMAP_SELECTED_TRUE:KAKIKOMAP_SELECTED_FALSE);
					$valMapDetailDataObj->setOutsideReference(($post_data['P_EX_DEF']==$i)?KAKIKOMAP_SELECTED_TRUE:KAKIKOMAP_SELECTED_FALSE);
					$valMapDetailDataObj->setPixel($post_data['P_PIXEL'][$i]);
					$valMapDetailDataObj->setUpdateDateTime(date('Y/m/d H:i:s'));

					// ベースマップデータ更新
					$rtn = $table_access->updateByKeyData($this->o_db, $targetMapDetailDataObj, $valMapDetailDataObj);
					if ($rtn != true) {
						$this->o_db->rollback();
						$table_access->free();
						$this->disconnectDB();
						trigger_error($this->o_resource->getValue("S2_013_03"), E_USER_ERROR);
			
						return false;
					}
				} else {
					// データがない場合は、追加処理

					// マップ詳細データ型オブジェクト
					$mapDetailDataObj = new MapDetailData();

					$mapDetailDataObj->setDirectryName($this->directryname);
					$mapDetailDataObj->setMapNumber($i);
					$mapDetailDataObj->setListName($post_data['P_LISTNAME'][$i]);
					$mapDetailDataObj->setInitialList(($post_data['P_INI_DEF']==$i)?KAKIKOMAP_SELECTED_TRUE:KAKIKOMAP_SELECTED_FALSE);
					$mapDetailDataObj->setOutsideReference(($post_data['P_EX_DEF']==$i)?KAKIKOMAP_SELECTED_TRUE:KAKIKOMAP_SELECTED_FALSE);
					$mapDetailDataObj->setPixel($post_data['P_PIXEL'][$i]);
					$mapDetailDataObj->setAddDateTime(date('Y/m/d H:i:s'));
					$mapDetailDataObj->setUpdateDateTime(date('Y/m/d H:i:s'));

					// データ追加
					$rtn = $table_access->insertData($this->o_db, $mapDetailDataObj);
					if ($rtn != true) {
						$this->o_db->rollback();
						$table_access->free();
						$this->disconnectDB();
						trigger_error($this->o_resource->getValue("S2_013_04"), E_USER_ERROR);
					
						return false;
					}
				}
			}
			// 解放
			$table_access->free();
		}

		// 初期表示チェック
		// ※初期表示選択状態のデータが削除されたときにチェックを上位地図番号のデータにする

		// マップ詳細DBアクセスインスタンス生成
		$table_access = new MapDetailTableAccess();

		// 初期表示選択データチェック
		$rtn = $table_access->checkInitialListDataExist($this->o_db, $this->directryname);
		if ($rtn) {
			// データがない場合は、更新処理
			$rtn = $table_access->updateInitialList($this->o_db, $this->directryname);
			if ($rtn != true) {
				$this->o_db->rollback();
				$table_access->free();
				$this->disconnectDB();
				trigger_error($this->o_resource->getValue("S2_013_05"), E_USER_ERROR);
	
				return false;
			}
		}

		// 外部参照チェック
		// ※外部参照選択状態のデータが削除されたときにチェックを上位地図番号のデータにする

		// 外部参照選択データ追加
		$rtn = $table_access->checkOutsideReferenceDataExist($this->o_db, $this->directryname);
		if ($rtn) {
			// データがない場合は、更新処理
			$rtn = $table_access->updateOutsideReference($this->o_db, $this->directryname);
			if ($rtn != true) {
				$this->o_db->rollback();
				$table_access->free();
				$this->disconnectDB();
				trigger_error($this->o_resource->getValue("S2_013_06"), E_USER_ERROR);
	
				return false;
			}
		}

		// 解放
		$table_access->free();

	    // 削除時の地図番号の繰上げ処理
	    $mapdata_flg = false;
	    for ($i = count($post_data['P_MAPSTATUS']); $i > 0 ; $i--) {

			// マップ詳細DBアクセスインスタンス生成
			$table_access = new MapDetailTableAccess();

			// マップ詳細データ存在チェック
			$rtn = $table_access->checkFindByKeyData($this->o_db, $this->directryname, $i);
			if (!$rtn) {
				// データがある場合は、更新処理
	            // データありフラグ
	            $mapdata_flg = true;
			} else {
				if ($mapdata_flg == true) {
					$rtn = $table_access->updateChangeMapNumber($this->o_db, $this->directryname, $i);
					if ($rtn != true) {
						$this->o_db->rollback();
						$table_access->free();
						$this->disconnectDB();
						trigger_error($this->o_resource->getValue("S2_013_07"), E_USER_ERROR);
			
						return false;
					}
				}
			}
	    }

		// ---------------------------------------
		// 画像関連処理
		// ---------------------------------------
		// 画像ファイル切り出し処理
		if ($this->mapImageSlice($post_data) == false) {
		    $this->o_db->rollback();
		    trigger_error($this->o_resource->getValue("S2_013_08"), E_USER_ERROR);
		    return false;
		}
		
		// 画像ファイルコピー
		if ($this->mapImageCopy() == false) {
		    $this->o_db->rollback();
		    trigger_error($this->o_resource->getValue("S2_013_09"), E_USER_ERROR);
		    return false;
		}
		
		// 画像保存フォルダ内の状態更新(削除/番号繰上げ)
		if ($this->mapImageDelete($input_mapno, $post_data) == false) {
		    $this->o_db->rollback();
		    trigger_error($this->o_resource->getValue("S2_013_10"), E_USER_ERROR);
		    return false;
		}
		
		// マップファイルの削除
		if ($this->mapFileDelete() == false) {
		    $this->o_db->rollback();
		    trigger_error($this->o_resource->getValue("S2_013_11"), E_USER_ERROR);
		    return false;
		}

		// トランザクションコミット
		$this->o_db->commit();

		// DB接続の切断
		$this->disconnectDB();

		return true;
	}

	/**
	 *	関数名：ベースマップ設定(GIS使用有り)
     * @引数：(I) $post_data      - ポストデータ格納連想配列
     * @戻値：$bool			- true  正常処理
     *                      - false 異常
     * @作成日/作成者：2006/09/01 (IIS) K.Shimomura
     * @更新日/更新者：
     * @使用方法：ベースマップ(GIS使用有り)の設定を行います。
	 *            引数のポストデータには、'P_START_X'=>左下座標X
	 *                                    'P_START_Y'=>左下座標Y
	 *                                    'P_END_X'=>中心座標X
	 *                                    'P_END_Y'=>中心座標Y
	 *                                    'P_USEGIS'=>GIS使用区分
	 *                                    'P_MAPSIZE_HORIZONTAL'=>地図サイズ横
	 *                                    'P_MAPSIZE_VERTICAL'=>地図サイズ縦
	 *                                    'P_MOVEMAP_SPHERE'=>地図移動幅
	 *                                    'P_DIFF_X'=>メモ中心表示地図の表示サイズX
	 *                                    'P_DIFF_Y'=>メモ中心表示地図の表示サイズY
	 *　　　　　　が必ず設定されていること。
     */
	function setBaseMap_GIS_On($post_data) {

		// DBオブジェクトの生成
		$this->createDBObject();

		// トランザクション開始
		$this->o_db->autoCommit(false);

		// ベースマップテーブル更新内容の設定

		// 条件及びNULL等のターゲット設定
		$targetBaseMapDataObj = new BaseMapData();

		// メモ中心表示地図の表示サイズ
		$diffx = (isset($post_data['P_DIFF_X']) && $post_data['P_DIFF_X'] != "")?$post_data['P_DIFF_X']:null;
		$diffy = (isset($post_data['P_DIFF_Y']) && $post_data['P_DIFF_Y'] != "")?$post_data['P_DIFF_Y']:null;

		// 更新条件
		$targetBaseMapDataObj->setDirectryName(WHERE_KEY);

		// 更新項目ターゲット設定
		$targetBaseMapDataObj->setUseGIS(VAL_TARGET);
		$targetBaseMapDataObj->setStartX(VAL_TARGET);
		$targetBaseMapDataObj->setStartY(VAL_TARGET);
		$targetBaseMapDataObj->setEndX(VAL_TARGET);
		$targetBaseMapDataObj->setEndY(VAL_TARGET);
		$targetBaseMapDataObj->setMapSizeHorizontal(VAL_TARGET);
		$targetBaseMapDataObj->setMapSizeVertical(VAL_TARGET);
		if (is_null($diffx)) {
			$targetBaseMapDataObj->setDiffX(NULL_CHAR);
		} else {
			$targetBaseMapDataObj->setDiffX(VAL_TARGET);
		}
		if (is_null($diffy)) {
			$targetBaseMapDataObj->setDiffY(NULL_CHAR);
		} else {
			$targetBaseMapDataObj->setDiffY(VAL_TARGET);
		}
		$targetBaseMapDataObj->setMoveMapSphere(VAL_TARGET);
		$targetBaseMapDataObj->setUpdateDateTime(VAL_TARGET);

		// 更新項目設定
		$valBaseMapDataObj = new BaseMapData();

		$valBaseMapDataObj->setDirectryName($this->directryname);
		$valBaseMapDataObj->setUseGIS($post_data['P_USEGIS']);
		$valBaseMapDataObj->setStartX($post_data['P_START_X']);
		$valBaseMapDataObj->setStartY($post_data['P_START_Y']);
		$valBaseMapDataObj->setEndX($post_data['P_END_X']);
		$valBaseMapDataObj->setEndY($post_data['P_END_Y']);
		$valBaseMapDataObj->setMapSizeHorizontal($post_data['P_MAPSIZE_HORIZONTAL']);
		$valBaseMapDataObj->setMapSizeVertical($post_data['P_MAPSIZE_VERTICAL']);
		$valBaseMapDataObj->setMoveMapSphere($post_data['P_MOVEMAP_SPHERE']);
		$valBaseMapDataObj->setUpdateDateTime(date('Y/m/d H:i:s'));
		$valBaseMapDataObj->setDiffX($diffx);
		$valBaseMapDataObj->setDiffY($diffy);

		// ベースマップDBアクセスインスタンス生成
		$table_access = new BaseMapTableAccess();

		// ベースマップデータ更新
		$rtn = $table_access->updateByKeyData($this->o_db, $targetBaseMapDataObj, $valBaseMapDataObj);
		if ($rtn != true) {
			$this->o_db->rollback();
			$table_access->free();
			$this->disconnectDB();
			trigger_error($this->o_resource->getValue("S2_014_01"), E_USER_ERROR);

			return false;
		}

		// 解放
		$table_access->free();

        // ---------------------------------------
        // マップ詳細管理管理テーブルデータ削除処理
        // ---------------------------------------

		// マップ詳細DBアクセスインスタンス生成
		$table_access = new MapDetailTableAccess();

		$rtn = $table_access->deleteByDirectryNameData($this->o_db, $this->directryname);
		if ($rtn != true) {
			$this->o_db->rollback();
			$table_access->free();
			$this->disconnectDB();
			trigger_error($this->o_resource->getValue("S2_014_02"), E_USER_ERROR);
	
			return false;
		}

		// 解放
		$table_access->free();

	    // ---------------------------------------
	    // 画像関連処理
	    // ---------------------------------------

		// ファイル消去
	    if ($this->mapImageDelete_GIS() == false) {
	        $o_db->db->rollback();
	        trigger_error($this->o_resource->getValue("S2_014_03"), E_USER_ERROR);
	        return false;
	    }

        // 一時フォルダ内マップファイル存在チェック
		if ($this->mapFileTempCheck() == true) {
		    // マップファイル削除
            if ($this->mapFileDelete() == false) {
                $o_db->db->rollback();
                trigger_error($this->o_resource->getValue("S2_014_04"), E_USER_ERROR);
                return false;
            }

            // マップファイルコピー
            if ($this->mapFileCopy() == false) {
                $o_db->db->rollback();
                trigger_error($this->o_resource->getValue("S2_014_05"), E_USER_ERROR);
                return false;
            }
        }

		// トランザクションコミット
		$this->o_db->commit();

		// DB接続の切断
		$this->disconnectDB();

		return true;
	}


//#################################################################################################
//                                        以下、サービス内使用関数定義
//#################################################################################################

	/**
	* 関数概要：イメージファイルの切り出し処理
	* 処理詳細：アップロードされたファイルを切り出す
	*
	* @引数： (I) $post_data      - ポストデータ格納連想配列
	* @戻値： boolean TRUE :正常
	*      ：         FALSE:異常
	* @作成日/作成者：2006/09/01 (IIS)H.Keta
	* @更新日/更新者：
	* @更新内容：
	*/
	function mapImageSlice($post_data) {
	
	    // 戻り値の初期化
	    $return = true;
	
	    // イメージ処理クラス
	    $o_kakikoimage = new KakikoImage();
	
	    // アップロード先パス
	    $uploadpath = "../../contents/".$this->directryname."/img/map/";
	    $uploadpath_tmp = $uploadpath."temp/";
	
	    // 一時フォルダのハンドル取得
	    if ($dir = opendir($uploadpath_tmp)) {
	        // フォルダ内のファイルを全て取得
	        while (($file = readdir($dir)) != false) {
	            // 親フォルダ/自フォルダ以外を処理
	            if ($file != "." && $file != ".." && $return != false) {
	                if (is_dir($uploadpath_tmp.$file)) {
	                    // ---------------------------
	                    // フォルダの場合(GIS無し時)
	                    // ---------------------------
	                    // 地図番号のファイル名のフォルダを処理
	                    if (ereg("^[0-9]{3}$", $file)) {
	                        // ステータスが新規/変更以外は処理しない
	                        if ($post_data['P_MAPIMAGE'][(integer)$file] == 2) {
		                        // サブフォルダのハンドル取得
		                        if ($s_dir = opendir($uploadpath_tmp.$file)) {
		                            // サブフォルダ内のファイルを全て取得
		                            while (($s_file = readdir($s_dir)) != false) {
		                                // ファイルのみ処理
		                                if ($s_file != "." && $s_file != ".." && !is_dir($uploadpath_tmp.$file."/".$s_file) && $return != false) {
		                                    // イメージ処理クラスパラメータセット
		                                    $param = array(
		                                                    "filename" => $uploadpath_tmp.$file."/".$s_file,          // 切り出し元ファイルパス
		                                                    "savepath" => $uploadpath_tmp.$file."/",                  // 切り出し後ファイル格納フォルダ
		                                                    "sliceW"   => $post_data['P_MAPSIZE_HORIZONTAL'],      // 切り出しサイズ(幅)
		                                                    "sliceH"   => $post_data['P_MAPSIZE_VERTICAL'],        // 切り出しサイズ(高さ)
		                                                    "rim"      => "0"                                         // のりしろサイズ
		                                                   );
		                                    $o_kakikoimage->setParam($param);
		                                    if ($o_kakikoimage->start() === true) {
		                                        // 切り出し処理が正常に終了した場合はファイルを削除
		                                        $return = unlink($uploadpath_tmp.$file."/".$s_file);
		                                    } else {
		                                        $return = false;
		                                    }
		                                }
		                            }
		                        }
		                    } else {
		                        // 新規/変更以外のフォルダは削除
		                        $return = kakikoDelete($uploadpath_tmp.$file);
		                    }
	                    }
	                }
	            }
	        }
	        // ハンドル開放
	        closedir($dir);
	    } else {
	        $return = false;
	    }
	    return $return;	
	}

	/**
	* 関数概要：非GIS版マップ移動
	* 処理詳細：一時フォルダから本環境フォルダへ非GIS版マップを移動
	*
	* @引数： なし
	* @戻値： boolean TRUE :正常
	*      ：         FALSE:異常
	* @作成日/作成者：2006/09/01 (IIS)H.Keta
	* @更新日/更新者：
	* @更新内容：
	*/
	function mapImageCopy() {
	
	    // 戻り値の初期化
	    $return = true;
	
	    // アップロード先パス
	    $uploadpath = "../../contents/".$this->directryname."/img/map/";
	    $uploadpath_tmp = $uploadpath."temp/";
	
	    // 一時フォルダのハンドル取得
	    if ($dir = opendir($uploadpath_tmp)) {
	        // フォルダ内のファイルを全て取得
	        while (($file = readdir($dir)) != false) {
	            // 親フォルダ・自フォルダは対象外
	            if ($file != "." && $file != ".." && $return != false) {
	                if (file_exists($uploadpath.$file)) {
	                    // 同一ファイル/フォルダが存在する場合
	                    $return = kakikoDelete($uploadpath.$file, false);
	                    if ($return == true) {
	                        $return = kakikoCopy($uploadpath_tmp.$file, $uploadpath.$file);
	                    }
	                } else {
	                    // 同一ファイル/フォルダが存在しない場合
	                    if (is_dir($uploadpath_tmp.$file)) {
	                        mkdir($uploadpath.$file);
	                    }
	                    $return = kakikoCopy($uploadpath_tmp.$file, $uploadpath.$file);
	                }
	            }
	        }
	        // ハンドル開放
	        closedir($dir);
	    } else {
	        $return = false;
	    }
	    return $return;
	}

	/**
	* 関数概要：非GIS版マップの削除処理
	* 処理詳細：非GIS版マップの削除する
	*
	* @引数： (I) input_mapno     - 入力項目地図番号配列
	*      ： (I) $post_data      - ポストデータ格納連想配列
	* @戻値： boolean TRUE :正常
	*      ：         FALSE:異常
	* @作成日/作成者：2006/09/01 (IIS)H.Keta
	* @更新日/更新者：
	* @更新内容：
	*/
	function mapImageDelete($input_mapno, $post_data) {
	
	    // アップロード先パス
	    $uploadpath = "../../contents/".$this->directryname."/img/map/";
	
	    // フォルダ削除処理
	    foreach($input_mapno as $i) {
	        // 削除項目
	        if (isset($post_data['P_DELCHECK'][$i])) {
	            if (file_exists(sprintf($uploadpath."%03d", $i))) {
	                if (kakikoDelete(sprintf($uploadpath."%03d", $i)) == false) {
	                    return false;
	                }
	            }
	        }
	    }
	
	    // フォルダ名繰上げ処理
	    $mapdata_flg = false;
	    for ($i = count($post_data['P_MAPSTATUS']); $i > 0 ; $i--) {
	        // データ存在チェック
	        if (file_exists(sprintf($uploadpath."%03d", $i))) {
	            // データありフラグ
	            $mapdata_flg = true;
	        } else {
	            if ($mapdata_flg == true) {
	                // 上位にデータが存在する場合、地図番号を繰り上げる
	                for ($s = $i + 1; $s <= count($post_data['P_MAPSTATUS']); $s++) {
	                    if (file_exists(sprintf($uploadpath."%03d", $s))) {
		                    if (rename(sprintf($uploadpath."%03d", $s), sprintf($uploadpath."%03d", $s - 1)) == false) {
		                        return false;
		                    }
	                    }
	                }
	                $i++;    // カウンタを戻す
	            }
	        }
	    }
	    return true;
	}

	/**
	* 関数概要：非GIS版マップの削除処理
	* 処理詳細：非GIS版マップの削除する
	*
	* @引数： 
	* @戻値： boolean TRUE :正常
	*      ：         FALSE:異常
	* @作成日/作成者：2006/09/01 (IIS)K.Shimomura
	* @更新日/更新者：
	* @更新内容：
	*/
	function mapImageDelete_GIS() {

	    // アップロード先パス
	    $uploadpath = "../../contents/".$this->directryname."/img/map/";

	    // フォルダ削除処理
	    for ($i = 1; $i <= 10; $i++) {

	        // 削除項目
	        if (file_exists(sprintf($uploadpath."%03d", $i))) {
	            if (kakikoDelete(sprintf($uploadpath."%03d", $i)) == false) {
	                return false;
	            }
	        }
	    }
	    return true;
	
	}

	/**
	* 関数概要：一時フォルダ内GIS版マップファイルチェック
	* 処理詳細：一時フォルダよりGIS版マップファイルがあるかをチェックする
	*
	* @引数： なし
	* @戻値： boolean TRUE :ファイルあり
	*      ：         FALSE:ファイルなし
	* @作成日/作成者：2006/09/01 (IIS)K.Shimomura
	* @更新日/更新者：
	* @更新内容：
	*/
	function mapFileTempCheck() {
	
	    // 戻り値の初期化
	    $return = false;
	
	    // 一時フォルダパス
	    $uploadpath_tmp = "../../contents/".$this->directryname."/img/map/temp/";
	
	    // 一時フォルダのハンドル取得
	    if ($dir = opendir($uploadpath_tmp)) {
	        // フォルダ内のファイルを全て取得
	        while (($file = readdir($dir)) != false) {
	            // 親フォルダ・自フォルダは対象外
	            if ($file != "." && $file != ".." && $return != true) {
					if (is_file($uploadpath_tmp.$file)) {
						if (strrchr($file, ".") == ".map") {
							$return = true;
						}
					}
	            }
	        }
	        // ハンドル開放
	        closedir($dir);
	    } else {
	        $return = false;
	    }
	    return $return;
	
	}

	/**
	* 関数概要：GIS版マップファイル移動
	* 処理詳細：一時フォルダから本環境フォルダへGIS版マップファイル移動
	*
	* @引数： なし
	* @戻値： boolean TRUE :正常
	*      ：         FALSE:異常
	* @作成日/作成者：2006/09/01 (IIS)K.Shimomura
	* @更新日/更新者：
	* @更新内容：
	*/
	function mapFileCopy() {
	
	    // 戻り値の初期化
	    $return = true;
	
	    // アップロード先パス
	    $uploadpath = "../../contents/".$this->directryname."/img/map/";
	    $uploadpath_tmp = $uploadpath."temp/";
	
	    // 一時フォルダのハンドル取得
	    if ($dir = opendir($uploadpath_tmp)) {
	        // フォルダ内のファイルを全て取得
	        while (($file = readdir($dir)) != false) {
	            // 親フォルダ・自フォルダは対象外
	            if ($file != "." && $file != ".." && $return != false) {
	                if (file_exists($uploadpath.$file)) {
	                    // 同一ファイル/フォルダが存在する場合
	                    $return = kakikoDelete($uploadpath.$file, false);
	                    if ($return == true) {
	                        $return = kakikoCopy($uploadpath_tmp.$file, $uploadpath.$file);
	                    }
	                } else {
	                    // 同一ファイル/フォルダが存在しない場合
	                    if (is_dir($uploadpath_tmp.$file)) {
	                        mkdir($uploadpath.$file);
	                    }
	                    $return = kakikoCopy($uploadpath_tmp.$file, $uploadpath.$file);
	                }
	            }
	        }
	        // ハンドル開放
	        closedir($dir);
	    } else {
	        $return = false;
	    }
	    return $return;
	
	}

	/**
	* 関数概要：GIS版マップファイルの削除処理
	* 処理詳細：GIS版マップファイルの削除処理
	*
	* @引数： (I) input_mapno: array  入力項目地図番号配列
	* @戻値： boolean TRUE :正常
	*      ：         FALSE:異常
	* @作成日/作成者：2006/09/01 (IIS)K.Shimomura
	* @更新日/更新者：
	* @更新内容：
	*/
	function mapFileDelete() {
	
	    // アップロード先パス
	    $uploadpath = "../../contents/".$this->directryname."/img/map/";
	
		// マップファイルの削除
	    if ($dir = opendir($uploadpath)) {
	        // フォルダ内のファイルを全て取得
	        while (($file = readdir($dir)) != false) {
	            // 親フォルダ・自フォルダは対象外
	            if ($file != "." && $file != "..") {
					if (is_file($uploadpath.$file)) {
						if (strrchr($file, ".") == ".map") {
							$return = kakikoDelete($uploadpath.$file, false);
						}
					}
	            }
	        }
	        // ハンドル開放
	        closedir($dir);
	    }
	    return true;
	}

}

?>
