<?php
/**
* KakikoFile.php 処理概要：ファイル処理
* PHP versions：4.4.2
* @パッケージ：-
* @作者：h.keta
* @作成日/作成者：2006/09/01 (IIS)h.keta
* @更新日/更新者：
* @バージョン：1.0.0
*
* Copyright (c) 2006 Iwate Information System Co.,Ltd. All Rights Reserved.       
* You may see the License at LICENSE_EN.txt(English) or LICENSE_JA.txt(Japanese). 
*/ 

/**
* 関数概要：ファイルコピー処理
* 処理詳細：引数のファイルをコピーする
*           フォルダが指定された場合は、以下のファイルも全てコピーする
*
* @引数： (I) from  - string コピー元となるファイル(フォルダ)パス
*         (I) to    - string コピー先となるフォルダ(フォルダ)パス
*                            ※コピー元にファイルを指定した場合、コピー先もファイルを指定すること
* @戻値：  bool : true(正常) false(異常)
* @作成日/作成者：2006/09/01 (IIS)H.Keta
* @更新日/更新者：
* @使用方法：
*      ・ファイルをコピーする場合
*          kakikoCopy("c:/temp/aaa.txt", "c:/test/bbb.txt");
*          ※実行結果：c:/test/bbb.txt
*
*      ・フォルダをコピーする場合
*          kakikoCopy("c:/temp", "c:/test");
*          ※実行結果：c:/test
*            ・testが存在しない場合は作成
*            ・temp以下を全てコピー(サブフォルダ含む)
*
*      ※全角のファイル名/フォルダ名は使用不可
*
* @更新内容：
*/
function kakikoCopy($from, $to){

    // コピー元が存在しない場合はエラー
    if (!file_exists($from)) {
        return false;
    }
    
    // 戻り値(初期化)
    $return = true;
    
    // ------------------------
    // コピー処理開始
    // ------------------------
    if (is_dir($from)) {
        // --------------
        // フォルダの場合
        // --------------
        // コピー先のフォルダが存在しない場合は作成
        if (!file_exists($to)) {
            $return = mkdir($to);
        }
        if ($return == true) {
            // フォルダ内ファイルコピー処理
            $return = dirCopyAll($from, $to);
        }
    } else {
        // --------------
        // ファイルの場合
        // --------------
        $return = copy($from, $to);
    }

    // 結果
    return $return;

}

/**
* 関数概要：ファイル(フォルダ)コピー処理
* 処理詳細：引数のファイル(フォルダ)をコピーする
*           フォルダが指定された場合は、以下のファイルも全てコピーする
*
* @引数： (I) from  - string コピー元となるファイル(フォルダ)パス
*         (I) to    - string コピー先となるフォルダ(フォルダ)パス
*                            ※コピー元にファイルを指定した場合、コピー先もファイルを指定すること
* @戻値： boolean TRUE :正常
*                 FALSE:異常
* @作成日/作成者：2006/09/01 (IIS)H.Keta
* @更新日/更新者：
* @更新内容：
*/
function dirCopyAll($from, $to){

    // 戻り値(初期化)
    $return = true;

    // 対象ディレクトリのハンドル取得
    if ($dir = opendir($from)) {
        // ディレクトリ内のファイル/フォルダを全て取得
        while (($file = readdir($dir)) != false && $return != false) {
            // 親フォルダ/自フォルダ以外を処理
            if ($file != "." && $file != "..") {
                // ファイル/フォルダを判定
                if (is_dir($from."/".$file)) {
                    // SVNのフォルダは作成しない(バージョン管理ファイルがまぎれないように。念のため)
                    if ($file != ".svn") {
                        // フォルダが存在しない場合は作成
                        if (!file_exists($to."/".$file)) {
                            $return = mkdir($to."/".$file);
                        }
                        // フォルダ内ファイルコピー処理(再帰呼出)
                        if ($return == true) {
                            $return = dirCopyAll($from."/".$file, $to."/".$file);
                        }
                    }
                } else {
                    // ファイルコピー
                    $return = copy($from."/".$file, $to."/".$file);
                }
            }
        }
        // ハンドル開放
        closedir($dir);
    }

    // 結果
    return $return;

}

/**
* 関数概要：ファイル削除処理
* 処理詳細：引数のファイルを削除する
*           フォルダが指定された場合は、以下のファイルも全て削除する
*
* @引数： (I) path   - string  削除対象なるファイル(フォルダ)パス
*      ： (I) alldel - boolean true:指定パスフォルダも削除 false:指定パスフォルダ以下を削除
* @戻値： boolean TRUE :正常
*                 FALSE:異常
* @作成日/作成者：2006/09/01 (IIS)H.Keta
* @更新日/更新者：
* @更新内容：
*/
function kakikoDelete($path, $alldel=true){

    // 削除先が存在しない場合はエラー
    if (!file_exists($path)) {
        return false;
    }
    
    // 戻り値(初期化)
    $return = true;
    
    // ------------------------
    // 削除処理開始
    // ------------------------
    if (is_dir($path)) {
        // --------------
        // フォルダの場合
        // --------------
        $return = dirDeleteAll($path);
        if ($alldel == true && $return == true) {
            $return = rmdir($path);
        }
    } else {
        // --------------
        // ファイルの場合
        // --------------
        $return = unlink($path);
    }

    // 結果
    return $return;

}

/**
* 関数概要：フォルダ削除処理
* 処理詳細：引数のフォルダを削除する
*
* @引数： (I) path   - string  削除対象なるフォルダパス
* @戻値： boolean TRUE :正常
*                 FALSE:異常
* @作成日/作成者：2006/09/01 (IIS)H.Keta
* @更新日/更新者：
* @更新内容：
*/
function dirDeleteAll($path){

    // 戻り値(初期化)
    $return = true;

    // 対象ディレクトリのハンドル取得
    if ($dir = opendir($path)) {
        // ディレクトリ内のファイル/フォルダを全て取得
        while (($file = readdir($dir)) != false && $return != false) {
            // 親フォルダ/自フォルダ以外を処理
            if ($file != "." && $file != "..") {
                // ファイル/フォルダを判定
                if (is_dir($path."/".$file)) {
                    // フォルダ内ファイル削除処理(再帰呼出)
                    $return = dirDeleteAll($path."/".$file);
                    // フォルダを削除
                    $return = rmdir($path."/".$file);
                } else {
                    // ファイル削除
                    $return = unlink($path."/".$file);
                }
            }
        }
        // ハンドル開放
        closedir($dir);
    }

    // 結果
    return $return;

}

?>
