unit untColorButton;

interface 

uses 
  Windows, Classes, Controls, Messages, ExtCtrls, Dialogs, Graphics; 

type
  TColorButton = class(TCustomPanel) 
  public 
    constructor Create(AOwner: TComponent); override; 
    destructor  Destroy; override; 
  private 
    FIsDown      : Boolean; 
    FIsClicked   : Boolean; 
    FColor       : TColor; 
    FColorDialog : TColorDialog; 
    FOnClick     : TNotifyEvent; 
    function  GetColor : TColor; 
    procedure SetColor(Color : TColor); 
    procedure WMPaint(var Msg:TWMPaint);       message WM_PAINT; 
    procedure WMMouseDown(var Msg: TMessage);  message WM_LBUTTONDOWN; 
    procedure WMMouseUp(var Msg: TMessage);    message WM_LBUTTONUP; 
    procedure CMMouseEnter(var Msg: TMessage); message CM_MOUSEENTER; 
    procedure CMMouseLeave(var Msg: TMessage); message CM_MOUSELEAVE; 
    procedure CMEnabledChanged(var Message: TMessage); message CM_ENABLEDCHANGED; 
    procedure WMSetFocus(var Message: TMessage); message WM_SETFOCUS; 
    procedure WMKillFocus(var Message: TMessage); message WM_KILLFOCUS; 
    procedure CNKeyDown(var Message: TMessage); message CN_KEYDOWN; 
    procedure CNKeyUp(var Message: TMessage); message CN_KEYUP; 
  published 
    property OnClick : TNotifyEvent read FOnClick write FOnClick; 
    property Color : TColor read GetColor write SetColor; 
  end; 

implementation 

uses 
  types, forms; 

function GetShiftState: TShiftState; 
var 
  KeyState: TKeyboardState; 
begin 
  GetKeyboardState(KeyState); 
  Result := []; 
  if KeyState[VK_MENU]    shr 7 = 1 then Include(Result, ssAlt); 
  if KeyState[VK_SHIFT]   shr 7 = 1 then Include(Result, ssShift); 
  if KeyState[VK_CONTROL] shr 7 = 1 then Include(Result, ssCtrl); 
  if KeyState[VK_LBUTTON] shr 7 = 1 then Include(Result, ssLeft); 
  if KeyState[VK_RBUTTON] shr 7 = 1 then Include(Result, ssRight); 
  if KeyState[VK_MBUTTON] shr 7 = 1 then Include(Result, ssMiddle); 
end; 

function TColorButton.GetColor : TColor; 
begin 
  Result := FColor; 
end; 

procedure TColorButton.SetColor(Color : TColor); 
begin 
  FColor := Color; 
end; 

procedure TColorButton.WMPaint(var Msg:TWMPaint); 
const 
  PADDING : Integer = 3; 
var 
  PenColor  : TColor; 
  Rect      : TRect; 
  FocusRect : TRect; 
begin 
  inherited; 

  Canvas.Pen.Style := psSolid; 

  Rect.Left   := PADDING; 
  Rect.Top    := PADDING; 
  Rect.Right  := Width - PADDING; 
  Rect.Bottom := Height - PADDING; 

  if Enabled then 
    PenColor := clBtnText 
  else 
    PenColor := clGrayText; 

  Canvas.Pen.Color   := PenColor; 
  Canvas.Brush.Color := FColor; 

  if Focused or FIsClicked then 
  begin 
    FocusRect.Left   := Rect.Left   - 1; 
    FocusRect.Top    := Rect.Top    - 1; 
    FocusRect.Right  := Rect.Right  + 1; 
    FocusRect.Bottom := Rect.Bottom + 1; 
    Canvas.DrawFocusRect(FocusRect); 
  end; 

  Canvas.Rectangle(RECT); 
end; 

procedure TColorButton.WMMouseDown(var Msg: TMessage); 
begin 
  FIsDown := True; 
  FIsClicked := True; 

  Self.BevelOuter := bvLowered; 

  SetFocus; 
  inherited; 
end; 

procedure TColorButton.WMMouseUp(var Msg: TMessage); 
begin 
  Self.BevelOuter := bvRaised; 


  if FIsDown then 
  begin 
    FColorDialog.Color := FColor; 
    if FColorDialog.Execute then 
    begin 
      FColor := FColorDialog.Color; 
      Repaint; 
    end; 
  end; 

  FIsDown    := False; 
  FIsClicked := False; 

  inherited; 

  if Assigned(FOnClick) then 
    FOnClick(Self); 
end; 

procedure TColorButton.CMMouseEnter(var Msg: TMessage); 
begin 
  if FIsClicked then 
  begin 
    FIsDown := True; 
    Self.BevelOuter := bvLowered; 
  end; 

  inherited; 
end; 

procedure TColorButton.CMMouseLeave(var Msg: TMessage); 
begin 
  if FIsClicked then 
  begin 
    Self.BevelOuter := bvRaised; 
    FIsDown := False; 
  end; 

  inherited; 
end; 

procedure TColorButton.CMEnabledChanged(var Message: TMessage); 
begin 
  inherited; 
  Repaint; 
end; 

procedure TColorButton.WMSetFocus(var Message: TMessage); 
const
  PADDING : Integer = 2; 
begin 
  inherited; 
  Repaint; 
end; 

procedure TColorButton.WMKillFocus(var Message: TMessage); 
begin 
  Repaint; 
end; 

procedure TColorButton.CNKeyDown(var Message: TMessage); 
begin 
  case Message.WParam of 
    VK_SPACE : WMMouseDown(Message); 
  else 
    inherited; 
  end; 
end; 

procedure TColorButton.CNKeyUp(var Message: TMessage); 
begin 
  case Message.WParam of 
    VK_SPACE : WMMouseUp(Message); 
  else 
    inherited; 
  end; 
end; 

constructor TColorButton.Create(AOwner : TComponent); 
begin 
  inherited Create(AOwner); 

  Parent := TWinControl(AOwner); 
  Width   := 32; 
  Height  := 16; 
  BevelWidth  := 1; 
  BorderWidth := 0; 
  BevelInner  := bvNone; 
  BevelOuter  := bvRaised; 
  Visible     := True; 

  FIsDown    := False; 
  FIsClicked := False; 
  FColor     := clBtnFace; 

  FColorDialog := TColorDialog.Create(self); 
end; 

destructor TColorButton.Destroy; 
begin 
  FColorDialog.Free; 

  inherited Destroy; 
end; 

end.
