// -*- Mode: C++; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*-

//
//  Copyright (C) 2002-2003 Hiroyuki Ikezoe
//  Copyright (C) 2003 Takuro Ashie
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2, or (at your option)
//  any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

#include "kz-mozembed.h"

#include <string.h>
#include <gtkmozembed.h>
#include <gtkmozembed_internal.h>

#include "kazehakase.h"
#include "kz-mozwrapper.h"
#include "kz-mozprogresslistener.h"
#include "gobject-utils.h"
#include "intl.h"
#include "mozilla.h"
#include "mozilla-prefs.h"
#include "kz-link.h"

#include <xpcom/nsCOMPtr.h>
#include <dom/nsIDOMDocument.h>
#include <content/nsIDocumentViewer.h>
#include <webbrwsr/nsIWebBrowser.h>
#include <content/nsIDocument.h>
#include <dom/nsIDOMMouseEvent.h>
#include <dom/nsIDOMKeyEvent.h>
#include <dom/nsIDOMEventTarget.h>
#include <dom/nsIDOMHTMLElement.h>
#include <dom/nsIDOMNamedNodeMap.h>
#include <dom/nsIDOMNSHTMLElement.h>  
#include <webbrowserpersist/nsIWebBrowserPersist.h>
#include <necko/nsNetUtil.h>
#include <find/nsIWebBrowserFind.h>
#include <dom/nsIDOMNSDocument.h>
#include <dom/nsIDOMNSEvent.h>
#include <docshell/nsIDocShell.h>
#include <docshell/nsIDocShellTreeItem.h>
#include <docshell/nsIDocShellTreeOwner.h>
#include <dom/nsIDOMNodeList.h>
#include <dom/nsIDOMWindow.h>
#include <content/nsISelection.h>
#include <dom/nsIDOMRange.h>
#include <find/nsIWebBrowserFind.h>
#include <necko/nsNetUtil.h>
#include <uconv/nsICharsetConverterManager.h>
#if MOZILLA_SNAPSHOT < 10
#	include <uconv/nsICharsetConverterManager2.h>
#endif
#include <dom/nsIDOMWindow.h>
#include <content/nsISelection.h>
#include <nsISHistory.h>
#include <nsIHistoryEntry.h>
#include <nsISHEntry.h>
#include <nsIWebNavigation.h>
#include <nsCWebBrowserPersist.h>

struct _KzMozEmbedPriv
{
	KzMozWrapper *wrapper;
	gint size_inited;
	gint cur_requests;
	gint total_requests;
	
	gboolean lock;

	/* for navigation link */
	gchar *link_next;
	gchar *link_prev;
	gchar *link_index;
	gchar *link_contents;
	gchar *link_start;
	
	/* for alternate rss */
	gchar *link_rss;
};

enum {
	FAVICON_SIGNAL,
	LAST_SIGNAL
};

static void kz_moz_embed_class_init    (KzMozEmbedClass *klass);
static void kz_moz_embed_init          (KzMozEmbed *kzembed);
static void kz_moz_embed_destroy       (GtkObject *object);
static void kz_moz_embed_finalize      (GObject    *object);
static void kz_moz_embed_realize       (GtkWidget *widget);
static void kz_moz_embed_unrealize     (GtkWidget *widget);
static void kz_moz_embed_size_allocate (GtkWidget *widget,
					GtkAllocation *allocation);
static void kz_moz_embed_title         (GtkMozEmbed *embed);
static void kz_moz_embed_location      (GtkMozEmbed *embed);
static void kz_moz_embed_net_start     (GtkMozEmbed *embed);
static gint kz_moz_embed_open_uri      (GtkMozEmbed *embed, const char *uri);
static void kz_moz_embed_size_to       (GtkMozEmbed *embed,
					gint width, gint height);
static void kz_moz_embed_visibility    (GtkMozEmbed *embed,
					gboolean visibility);
static void kz_moz_embed_net_stop      (GtkMozEmbed *embed);
static void kz_moz_embed_net_state_all (GtkMozEmbed *embed,
					const char  *aURI,
					gint         state,
					guint        status);

static void kz_moz_embed_get_head_link (KzMozEmbed *kzembed);

static void kz_moz_embed_navigation_link_free(KzMozEmbed *kzembed);

static gint kz_moz_embed_dom_key_down  (GtkMozEmbed *embed, gpointer event);
static gint kz_moz_embed_dom_key_up    (GtkMozEmbed *embed, gpointer event);
static gint kz_moz_embed_dom_key_press (GtkMozEmbed *embed, gpointer event);

static gchar    *mozilla_get_attribute      (nsIDOMNode *node,
					     gchar *attribute);
static gboolean  mozilla_get_link_from_node (nsIDOMDocument *domDoc,
					     nsIDOMNode *node,
					     gchar **url,
					     gchar **inner_html);
static glong     mozilla_set_event_context  (KzMozEmbed *kzembed,
					     nsIDOMEventTarget *target,
					     KzEmbedEvent *info);

static gboolean  mozilla_get_head_link_from_node (nsIDOMDocument *domDoc,
						  nsIDOMNode *node,
						  gchar **url);

static GtkMozEmbedClass *parent_class = NULL;

static gint kz_moz_embed_signals[LAST_SIGNAL] = {0};

KZ_OBJECT_GET_TYPE(kz_moz_embed, "KzMozEmbed", KzMozEmbed,
		   kz_moz_embed_class_init, kz_moz_embed_init,
		   GTK_TYPE_MOZ_EMBED)


static void
kz_moz_embed_class_init (KzMozEmbedClass *klass)
{
	GObjectClass *gobject_class;
	GtkObjectClass *object_class;
	GtkWidgetClass *widget_class;
	GtkMozEmbedClass *moz_embed_class;

	parent_class    = (GtkMozEmbedClass *) g_type_class_peek_parent (klass);

	gobject_class   = (GObjectClass *) klass;
	object_class    = (GtkObjectClass *) klass;
	widget_class    = (GtkWidgetClass *) klass;
	moz_embed_class = (GtkMozEmbedClass *) klass;

	// GObject
	gobject_class->finalize = kz_moz_embed_finalize;

	// GtkObject signals
	object_class->destroy = kz_moz_embed_destroy;
 
	// widget class
	widget_class->realize         = kz_moz_embed_realize;
	widget_class->unrealize       = kz_moz_embed_unrealize;
	widget_class->size_allocate   = kz_moz_embed_size_allocate;
 
	// GtkMozEmbedSignals
	moz_embed_class->net_state_all   = kz_moz_embed_net_state_all;
#if 0
	moz_embed_class->dom_mouse_over  = kz_moz_embed_dom_mouse_over;
	moz_embed_class->dom_mouse_out   = kz_moz_embed_dom_mouse_out;
	moz_embed_class->net_state       = kz_moz_embed_net_state;
	moz_embed_class->progress        = kz_moz_embed_progress;
	moz_embed_class->progress_all    = kz_moz_embed_progress_all;
	moz_embed_class->js_status       = kz_moz_embed_js_status;
	moz_embed_class->destroy_browser = kz_moz_embed_destroy_browser;
#endif

	moz_embed_class->title           = kz_moz_embed_title;
	moz_embed_class->location        = kz_moz_embed_location;
	moz_embed_class->net_start       = kz_moz_embed_net_start;
	moz_embed_class->net_stop        = kz_moz_embed_net_stop;
	moz_embed_class->open_uri        = kz_moz_embed_open_uri;
	moz_embed_class->size_to         = kz_moz_embed_size_to;
	moz_embed_class->visibility      = kz_moz_embed_visibility;
	moz_embed_class->dom_key_press   = kz_moz_embed_dom_key_press;
	moz_embed_class->dom_key_down    = kz_moz_embed_dom_key_down;
	moz_embed_class->dom_key_up      = kz_moz_embed_dom_key_up;

	// our own signals
	klass->favicon = NULL;

	kz_moz_embed_signals[FAVICON_SIGNAL]
		= g_signal_new ("favicon",
				G_TYPE_FROM_CLASS (klass),
				G_SIGNAL_RUN_LAST,
				G_STRUCT_OFFSET (KzMozEmbedClass, favicon),
				NULL, NULL,
				g_cclosure_marshal_VOID__STRING,
				G_TYPE_NONE, 1,
				G_TYPE_STRING);
}


static void
kz_moz_embed_init (KzMozEmbed *kzembed)
{
	// widgets
	kzembed->popup_window          = NULL;

	// status
	kzembed->location              = NULL;
	kzembed->title                 = NULL;
	kzembed->load_started          = 0;
	kzembed->load_percent          = 0;
	kzembed->bytes_loaded          = 0;
	kzembed->max_bytes_loaded      = 0;
	kzembed->is_loading            = FALSE;
	kzembed->load_status_message   = NULL;

	// priv
	kzembed->priv = g_new0(KzMozEmbedPriv, 1);
	kzembed->priv->wrapper        = NULL;
	kzembed->priv->size_inited    = FALSE;
	kzembed->priv->total_requests = 0;
	kzembed->priv->cur_requests   = 0;
	kzembed->priv->link_index     = NULL;
	kzembed->priv->link_contents  = NULL;
	kzembed->priv->link_start     = NULL;
	kzembed->priv->link_next      = NULL;
	kzembed->priv->link_prev      = NULL;

	kz_moz_embed_load_url(kzembed, "about:blank");
}

GtkWidget *
kz_moz_embed_new (const gchar *url)
{
	KzMozEmbed *kzembed = KZ_MOZ_EMBED(g_object_new(KZ_TYPE_MOZ_EMBED, NULL));

#if 0
	//
	// set the chrome type so it's stored in the object
	//
	g_moz_embed_set_chrome_mask(GTK_MOZ_EMBED(kzembed->mozembed),
				    actualChromeMask);
#endif
	kz_moz_embed_load_url(kzembed, url);

	return GTK_WIDGET(kzembed);
}

static void
kz_moz_embed_destroy (GtkObject *object)
{
	KzMozEmbed *kzembed = KZ_MOZ_EMBED(object);

	g_return_if_fail(KZ_IS_MOZ_EMBED(kzembed));

	g_free(kzembed->location);
	kzembed->location = NULL;
	g_free(kzembed->title);
	kzembed->title = NULL;

	if (GTK_OBJECT_CLASS(parent_class)->destroy)
		GTK_OBJECT_CLASS(parent_class)->destroy(object);
}


KZ_OBJECT_FINALIZE (kz_moz_embed, KzMozEmbed)


static void
kz_moz_embed_realize (GtkWidget *widget)
{
	if (GTK_WIDGET_CLASS(parent_class)->realize)
		GTK_WIDGET_CLASS(parent_class)->realize(widget);

	KzMozEmbed *kzembed = KZ_MOZ_EMBED(widget);

	if (!kzembed->priv->wrapper)
	{
		kzembed->priv->wrapper = new KzMozWrapper;
		nsresult rv = kzembed->priv->wrapper->Init(kzembed);
		if (NS_FAILED(rv))
		{
			g_error("KzMozEmbed: Faild to init KzMozWrapper!");
		}
	}
}


static void
kz_moz_embed_unrealize (GtkWidget *widget)
{
	KzMozEmbed *kzembed = KZ_MOZ_EMBED(widget);

	kzembed->priv->size_inited = FALSE;

	if (kzembed->priv->wrapper)
	{
		kzembed->priv->wrapper->Destroy();
		delete kzembed->priv->wrapper;
		kzembed->priv->wrapper = NULL;
	}
	
	kz_moz_embed_navigation_link_free(kzembed);
	
	if (GTK_WIDGET_CLASS(parent_class)->unrealize)
		GTK_WIDGET_CLASS(parent_class)->unrealize(widget);
}


static void
kz_moz_embed_navigation_link_free(KzMozEmbed *kzembed)
{	
	if (kzembed->priv->link_index)
		g_free(kzembed->priv->link_index);
	if (kzembed->priv->link_contents)
		g_free(kzembed->priv->link_contents);
	if (kzembed->priv->link_start)
		g_free(kzembed->priv->link_start);
	if (kzembed->priv->link_prev)
		g_free(kzembed->priv->link_prev);
	if (kzembed->priv->link_next)
		g_free(kzembed->priv->link_next);
	if (kzembed->priv->link_rss)
		g_free(kzembed->priv->link_rss);
	
	kzembed->priv->link_index     = NULL;
	kzembed->priv->link_contents  = NULL;
	kzembed->priv->link_start     = NULL;
	kzembed->priv->link_prev      = NULL;
	kzembed->priv->link_next      = NULL;
	kzembed->priv->link_rss       = NULL;
}


static void
kz_moz_embed_size_allocate(GtkWidget *widget, GtkAllocation *allocation)
{
	KzMozEmbed *kzembed = KZ_MOZ_EMBED(widget);

	g_return_if_fail(GTK_IS_WIDGET(widget));

	if (!GTK_WIDGET_REALIZED(widget)) return;

	//if (kzembed->priv->size_inited && !GTK_WIDGET_MAPPED(widget)) return;
	if (!GTK_WIDGET_MAPPED(widget)) return;

	if (GTK_WIDGET_CLASS(parent_class)->size_allocate)
		GTK_WIDGET_CLASS(parent_class)->size_allocate(widget, allocation);

	kzembed->priv->size_inited = TRUE;
}


void
kz_moz_embed_load_url (KzMozEmbed *kzembed, const gchar *url)
{
	gchar *start_page = NULL, *str;

	g_return_if_fail(KZ_IS_MOZ_EMBED(kzembed));

	str = KZ_CONF_GET_STR("Global", "startup_page");

	if (url && *url)
	{
		start_page = g_strdup(url);
	}
	else if (str)
	{
		start_page = g_strdup(str);
	}
	else
	{
		start_page = g_strdup("about:blank");
	}

	if (kz_moz_embed_get_lock(KZ_MOZ_EMBED(kzembed)))
	{
		GtkMozEmbed *newembed = NULL;
		g_signal_emit_by_name(G_OBJECT(kzembed), 
				      "new-window",
				      &newembed, 0);
		gtk_moz_embed_load_url(newembed, start_page);
	}
	else
		gtk_moz_embed_load_url(GTK_MOZ_EMBED(kzembed), start_page);

	g_free(str);
	g_free(start_page);
}


void
kz_moz_embed_view_source (KzMozEmbed *kzembed)
{
	gchar *url;
	
	g_return_if_fail(KZ_IS_MOZ_EMBED(kzembed));

	url = g_strdup_printf("view-source:%s", kz_moz_embed_get_location(kzembed));
	
	kz_moz_embed_load_url(kzembed, url);
	
	g_free(url);
}


const gchar *
kz_moz_embed_get_title (KzMozEmbed *kzembed)
{
	g_return_val_if_fail(KZ_IS_MOZ_EMBED(kzembed), NULL);
	return kzembed->title;
}

gchar *
kz_moz_embed_ensure_title (KzMozEmbed *kzembed)
{
	g_return_val_if_fail(KZ_IS_MOZ_EMBED(kzembed), NULL);

	if (kzembed->title && *kzembed->title)
		return g_strdup(kzembed->title);

	if (kzembed->location && *kzembed->location)
	{
		if (kz_moz_embed_is_loading(kzembed))
		{
			return g_strdup_printf(_("Loading %s ..."),
					       kzembed->location);
		}
		else
		{
			return g_strdup(kzembed->location);
		}
	}
	else
	{
		if (kz_moz_embed_is_loading(kzembed))
			return g_strdup(_("Loading..."));
	}

	return g_strdup(_("No title"));
}

gchar * 
kz_moz_embed_get_link_message (KzMozEmbed *kzembed)
{
	g_return_val_if_fail(KZ_MOZ_EMBED(kzembed), NULL);

	gchar *message;
	nsXPIDLString  uMessage;

	*getter_Copies(uMessage) = gtk_moz_embed_get_link_message_unichar
						(GTK_MOZ_EMBED(kzembed));
	message = g_strdup(NS_ConvertUCS2toUTF8(uMessage).get());

	return message;
}


gdouble
kz_moz_embed_get_progress (KzMozEmbed *kzembed)
{
	gdouble progress;

	g_return_val_if_fail(KZ_MOZ_EMBED(kzembed), 0.0);

	if (kzembed->priv->total_requests <= 0 ||
	    kzembed->priv->cur_requests <= 0)
	{
		return 0.0;
	}

	progress = (gdouble) kzembed->priv->cur_requests
		/ (gdouble) kzembed->priv->total_requests;

	if (progress > 1.0)
		return 1.0;

	return progress;
}


const gchar *
kz_moz_embed_get_location (KzMozEmbed *kzembed)
{
	g_return_val_if_fail(KZ_IS_MOZ_EMBED(kzembed), NULL);
	if (kzembed->location != NULL && !strncmp(kzembed->location, "about:blank", 11))
		return "";

	return kzembed->location;
}

static void
kz_moz_embed_title (GtkMozEmbed *embed)
{
	g_return_if_fail(KZ_IS_MOZ_EMBED(embed));

	KzMozEmbed *kzembed = KZ_MOZ_EMBED(embed);

	g_free(kzembed->title);

	nsXPIDLString  uTitle;
	*getter_Copies(uTitle) = gtk_moz_embed_get_title_unichar(embed);
	kzembed->title = g_strdup (NS_ConvertUCS2toUTF8(uTitle).get());
}

static void
kz_moz_embed_location (GtkMozEmbed *embed)
{
	KzMozEmbed *kzembed = KZ_MOZ_EMBED(embed);

	g_return_if_fail(KZ_IS_MOZ_EMBED(kzembed));

	g_free(kzembed->location);
	kzembed->location = gtk_moz_embed_get_location(embed);
}


static void
kz_moz_embed_net_start (GtkMozEmbed *embed)
{
	KzMozEmbed *kzembed = KZ_MOZ_EMBED(embed);

	g_return_if_fail(KZ_IS_MOZ_EMBED(kzembed));

	if (parent_class->net_start)
		parent_class->net_start(embed);

	kzembed->is_loading = TRUE;
}

static void
kz_moz_embed_net_stop (GtkMozEmbed *embed)
{
	KzMozEmbed *kzembed = KZ_MOZ_EMBED(embed);

	g_return_if_fail(KZ_IS_MOZ_EMBED(kzembed));

	if (parent_class->net_stop)
		parent_class->net_stop(embed);

	kzembed->is_loading = FALSE;

	/* First free previous link */
	kz_moz_embed_navigation_link_free(kzembed);

	kz_moz_embed_get_head_link(kzembed);
}

static void
kz_moz_embed_net_state_all (GtkMozEmbed *embed, const char *aURI,
			    gint state, guint status)
{
	KzMozEmbed *kzembed = KZ_MOZ_EMBED(embed);

	g_return_if_fail(KZ_IS_MOZ_EMBED(kzembed));

	if (state & GTK_MOZ_EMBED_FLAG_IS_NETWORK)
	{
		kzembed->priv->total_requests = 0;
		kzembed->priv->cur_requests = 0;
#if 0
                if (state & GTK_MOZ_EMBED_FLAG_START)
                {
		}
                else if (state & EMBED_STATE_STOP)
                {
                        kzembed->priv->total_requests = 0;
                        kzembed->priv->cur_requests = 0;
		} 
#endif
	}
        if (state & GTK_MOZ_EMBED_FLAG_IS_REQUEST)
        {
                if (state & GTK_MOZ_EMBED_FLAG_START)
                        kzembed->priv->total_requests ++;
                else if (state & GTK_MOZ_EMBED_FLAG_STOP)
                        kzembed->priv->cur_requests ++;
        }

	if (parent_class->net_state_all)
		parent_class->net_state_all(embed, aURI, state, status);
}

static gint
kz_moz_embed_open_uri (GtkMozEmbed *embed, const char *uri)
{
	if (parent_class->open_uri)
		parent_class->open_uri(embed, uri);

	if (!strncmp(uri, "mailto:", 7))
	{
		return TRUE;
	}

	return FALSE;
}

static void
kz_moz_embed_size_to (GtkMozEmbed *embed, gint width, gint height)
{
	gtk_widget_set_size_request(GTK_WIDGET(embed), width, height);
	gtk_widget_queue_resize(GTK_WIDGET(embed));
}

static gint
kz_moz_embed_dom_key_down (GtkMozEmbed *embed, gpointer event)
{
	return FALSE;
}

static gint
kz_moz_embed_dom_key_up (GtkMozEmbed *embed, gpointer event)
{
	return FALSE;
}

static gint
kz_moz_embed_dom_key_press (GtkMozEmbed *embed, gpointer event)
{
	return FALSE;
}

static void
kz_moz_embed_visibility (GtkMozEmbed *embed, gboolean visibility)
{
	GtkWidget *parent = NULL;

	parent = gtk_widget_get_parent(GTK_WIDGET(embed));
	g_return_if_fail(parent != NULL);
	
	if (visibility) 
	{
		gtk_widget_show(GTK_WIDGET(embed));
		gtk_widget_show(parent);
	}
	else
	{
		gtk_widget_hide(GTK_WIDGET(embed));
		gtk_widget_hide(parent);
	}
}

gboolean
kz_moz_embed_can_cut_selection (KzMozEmbed *kzembed)
{
	g_return_val_if_fail(KZ_IS_MOZ_EMBED(kzembed), FALSE);

	if (!kzembed->priv->wrapper) return TRUE;

	PRBool retval;
	nsresult rv = kzembed->priv->wrapper->CanCutSelection(&retval);

	if (NS_FAILED(rv)) return FALSE;

	return retval;
}

gboolean
kz_moz_embed_can_copy_selection (KzMozEmbed *kzembed)
{
	g_return_val_if_fail(KZ_IS_MOZ_EMBED(kzembed), FALSE);

	if (!kzembed->priv->wrapper) return TRUE;

	PRBool retval;
	nsresult rv = kzembed->priv->wrapper->CanCopySelection(&retval);

	if (NS_FAILED(rv)) return FALSE;

	return retval;
}

gboolean
kz_moz_embed_can_paste (KzMozEmbed *kzembed)
{
	g_return_val_if_fail(KZ_IS_MOZ_EMBED(kzembed), FALSE);

	if (!kzembed->priv->wrapper) return TRUE;

	PRBool retval;
	nsresult rv = kzembed->priv->wrapper->CanPaste(&retval);

	if (NS_FAILED(rv)) return FALSE;

	return retval;
}

void
kz_moz_embed_cut_selection (KzMozEmbed *kzembed)
{
	g_return_if_fail(KZ_IS_MOZ_EMBED(kzembed));
	g_return_if_fail(kzembed->priv->wrapper);

	kzembed->priv->wrapper->CutSelection();
}

void
kz_moz_embed_copy_selection (KzMozEmbed *kzembed)
{
	g_return_if_fail(KZ_IS_MOZ_EMBED(kzembed));
	g_return_if_fail(kzembed->priv->wrapper);

	kzembed->priv->wrapper->CopySelection();
}

void
kz_moz_embed_paste (KzMozEmbed *kzembed)
{
	g_return_if_fail(KZ_IS_MOZ_EMBED(kzembed));
	g_return_if_fail(kzembed->priv->wrapper);

	kzembed->priv->wrapper->Paste();
}

void
kz_moz_embed_select_all (KzMozEmbed *kzembed)
{
	g_return_if_fail(KZ_IS_MOZ_EMBED(kzembed));
	g_return_if_fail(kzembed->priv->wrapper);

	kzembed->priv->wrapper->SelectAll();
}

gboolean
kz_moz_embed_find (KzMozEmbed *embed, const char *keyword,
		   gboolean backward)
{
	g_return_val_if_fail(KZ_IS_MOZ_EMBED(embed), FALSE);
	g_return_val_if_fail(keyword, FALSE);

	nsCOMPtr<nsIWebBrowser> web;
	gtk_moz_embed_get_nsIWebBrowser(GTK_MOZ_EMBED(embed),
					getter_AddRefs(web));
	if (!web) return FALSE;

	nsresult rv;
	nsCOMPtr<nsIWebBrowserFind> finder(do_GetInterface(web));
	finder->SetSearchString(NS_ConvertUTF8toUCS2(keyword).get());
	finder->SetFindBackwards(backward);
	finder->SetWrapFind(TRUE);
	finder->SetEntireWord(TRUE);
	finder->SetSearchFrames(TRUE);
	finder->SetMatchCase(FALSE);
	PRBool did_find;
	rv = finder->FindNext(&did_find);

	return NS_SUCCEEDED(rv) && did_find ? TRUE : FALSE;
}

gboolean
kz_moz_embed_selection_is_collapsed (KzMozEmbed *kzembed)
{
	g_return_val_if_fail(KZ_IS_MOZ_EMBED(kzembed), TRUE);

	if (!kzembed->priv->wrapper) return TRUE;

	nsresult rv;

	nsCOMPtr<nsISelection> selection;
	rv = kzembed->priv->wrapper->GetSelection(getter_AddRefs(selection));
	if (!selection) return TRUE;

	PRBool collapsed;
	rv = selection->GetIsCollapsed(&collapsed);
	if (NS_FAILED(rv)) return TRUE;

	return collapsed;
}


gboolean
kz_moz_embed_get_links (KzMozEmbed *kzembed, GList **list,
			gboolean selected_only)
{
	g_return_val_if_fail(KZ_IS_MOZ_EMBED(kzembed), FALSE);
	g_return_val_if_fail(kzembed->priv->wrapper, FALSE);
	g_return_val_if_fail(list, FALSE);

	// get selection
	nsresult rv;
	nsCOMPtr<nsISelection> selection;
	rv = kzembed->priv->wrapper->GetSelection(getter_AddRefs(selection));
	if (NS_FAILED(rv)) return FALSE;

	// get all anchor nodes in the document.
	nsCOMPtr<nsIDOMDocument> domDoc;
        rv = kzembed->priv->wrapper->GetMainDomDocument(getter_AddRefs(domDoc));
        if (NS_FAILED(rv) || !domDoc) return FALSE;

	nsAutoString tagname;
	tagname.AssignWithConversion("a");
	nsCOMPtr<nsIDOMNodeList> nodeList;
	rv = domDoc->GetElementsByTagName(tagname,
					  getter_AddRefs(nodeList));
        if (NS_FAILED(rv) || !domDoc) return FALSE;

	PRUint32 num;
	rv = nodeList->GetLength(&num);
	if (NS_FAILED(rv) || num < 1) return FALSE;

	// store links to GList
	nsCOMPtr<nsIDOMNode> node;
	for (PRUint32 i = 0; i < num; i++)
	{
		rv = nodeList->Item(i, getter_AddRefs(node));
		if (NS_FAILED(rv) || !node) continue;

		// check whether the selection contains these nodes or not.
		if (selected_only)
		{
			PRBool contains;
			selection->ContainsNode(node, PR_TRUE, &contains);
			if (!contains) continue;
		}

		KzLink *kzlink = kz_link_new();
		mozilla_get_link_from_node(domDoc, node,
					   &kzlink->url,
					   &kzlink->title);
		if (kzlink->url && *kzlink->url)
			*list = g_list_append(*list, kzlink);
		else
			kz_link_unref(kzlink);
	}

	return TRUE;
}


void
kz_moz_embed_add_ui (KzMozEmbed *kzembed, EggMenuMerge *ui, KzUILevel level)
{
	g_warning("kz_moz_embed_add_ui() is not implemented yet");
}


void
kz_moz_embed_remove_ui (KzMozEmbed *kzembed)
{
	g_warning("kz_moz_embed_remove_ui() is not implemented yet");
}


glong
kz_moz_embed_get_key_event_info(KzMozEmbed *kzembed, gpointer event,
				KzEmbedEventKey **info_ret)
{
	KzEmbedEventKey *info;
	info = (KzEmbedEventKey *) kz_embed_event_new(KZ_EMBED_EVENT_KEY);
	*info_ret = info;

	nsresult result;

	nsIDOMKeyEvent *aKeyEvent = (nsIDOMKeyEvent*) event;

     	nsCOMPtr<nsIDOMEventTarget> OriginalTarget;

     	nsCOMPtr<nsIDOMNSEvent> aEvent = do_QueryInterface(aKeyEvent);
     	if (!aEvent) return KZ_CONTEXT_NONE;

	PRUint32 code;
	aKeyEvent->GetKeyCode(&code);
	code = info->key;

	aKeyEvent->GetCharCode(&code);
	code = info->char_code;

	PRBool mod_key;
	info->modifier = 0;
	aKeyEvent->GetAltKey(&mod_key);
        if (mod_key) info->modifier |= KZ_ALT_KEY;

	aKeyEvent->GetShiftKey(&mod_key);
        if (mod_key) info->modifier |= KZ_SHIFT_KEY;

	aKeyEvent->GetMetaKey(&mod_key);
        if (mod_key) info->modifier |= KZ_META_KEY;

	aKeyEvent->GetCtrlKey(&mod_key);
        if (mod_key) info->modifier |= KZ_CTRL_KEY;

     	result = aEvent->GetOriginalTarget(getter_AddRefs(OriginalTarget));

     	if (NS_FAILED(result) || !OriginalTarget) return KZ_CONTEXT_NONE;

     	nsCOMPtr<nsIDOMNode> OriginalNode = do_QueryInterface(OriginalTarget);
     	if (!OriginalNode) return KZ_CONTEXT_NONE;

     	nsString nodename;
     	OriginalNode->GetNodeName(nodename);

     	if (nodename.EqualsWithConversion("xul:thumb") ||
	    nodename.EqualsWithConversion("xul:slider"))
	{
       		return KZ_CONTEXT_NONE;
	}

     	nsCOMPtr<nsIDOMEventTarget> target;
     	result = aKeyEvent->GetTarget(getter_AddRefs(target));
     	if (NS_FAILED(result) || !target) return KZ_CONTEXT_NONE;

	return mozilla_set_event_context(kzembed, target, (KzEmbedEvent *) info);
}


glong
kz_moz_embed_get_mouse_event_info(KzMozEmbed *kzembed, gpointer event,
				  KzEmbedEventMouse **info_ret)
{
	KzEmbedEventMouse *info;
	info = (KzEmbedEventMouse *) kz_embed_event_new(KZ_EMBED_EVENT_MOUSE);
	*info_ret = info;

	nsresult result;

	nsIDOMMouseEvent *aMouseEvent = (nsIDOMMouseEvent*)event;

     	nsCOMPtr<nsIDOMEventTarget> OriginalTarget;

     	nsCOMPtr<nsIDOMNSEvent> aEvent = do_QueryInterface(aMouseEvent);
     	if (!aEvent) return KZ_CONTEXT_NONE;

	PRUint16 button;
	aMouseEvent->GetButton(&button);
	info->button = button;

	PRBool mod_key;
	info->modifier = 0;
	aMouseEvent->GetAltKey(&mod_key);
        if (mod_key) info->modifier |= KZ_ALT_KEY;

	aMouseEvent->GetShiftKey(&mod_key);
        if (mod_key) info->modifier |= KZ_SHIFT_KEY;

	aMouseEvent->GetMetaKey(&mod_key);
        if (mod_key) info->modifier |= KZ_META_KEY;

	aMouseEvent->GetCtrlKey(&mod_key);
        if (mod_key) info->modifier |= KZ_CTRL_KEY;

	PRInt32 pos;
	aMouseEvent->GetClientX(&pos);
	info->x = pos;
	aMouseEvent->GetClientY(&pos);
	info->y = pos;

     	result = aEvent->GetOriginalTarget(getter_AddRefs(OriginalTarget));

     	if (NS_FAILED(result) || !OriginalTarget) return KZ_CONTEXT_NONE;

     	nsCOMPtr<nsIDOMNode> OriginalNode = do_QueryInterface(OriginalTarget);
     	if (!OriginalNode) return KZ_CONTEXT_NONE;

     	nsString nodename;
     	OriginalNode->GetNodeName(nodename);

     	if (nodename.EqualsWithConversion ("xul:thumb") ||
	    nodename.EqualsWithConversion ("xul:slider"))
	{
       		return KZ_CONTEXT_NONE;
	}

     	nsCOMPtr<nsIDOMEventTarget> target;
     	result = aMouseEvent->GetTarget(getter_AddRefs(target));
     	if (NS_FAILED(result) || !target) return KZ_CONTEXT_NONE;

	return mozilla_set_event_context(kzembed, target, (KzEmbedEvent *) info);
}


static void
kz_moz_embed_get_head_link (KzMozEmbed *kzembed)
{
	nsresult rv;
	nsCOMPtr<nsIDOMDocument> domDoc;
        rv = kzembed->priv->wrapper->GetMainDomDocument(getter_AddRefs(domDoc));
        if (NS_FAILED(rv) || !domDoc) return;

	nsAutoString tagname;
	tagname.AssignWithConversion("link");
	nsCOMPtr<nsIDOMNodeList> nodeList;
	rv = domDoc->GetElementsByTagName(tagname,
					  getter_AddRefs(nodeList));
        if (NS_FAILED(rv) || !domDoc) return;
	PRUint32 num;
	rv = nodeList->GetLength(&num);
	if (NS_FAILED(rv) || num < 1) return;

	nsCOMPtr<nsIDOMNode> node;
	for (PRUint32 i = 0; i < num; i++)
	{
		gchar *url = NULL;
		rv = nodeList->Item(i, getter_AddRefs(node));
		if (NS_FAILED(rv) || !node) continue;
		
		char *relattr =  mozilla_get_attribute(node, "rel");
		if (!relattr) continue;

		mozilla_get_head_link_from_node(domDoc, node, &url);

		nsAutoString rela;
		rela.AssignWithConversion(relattr);
		nsCString relc;
		relc.AssignWithConversion(rela);
		char *rel = ToNewCString(relc);
		if (!strcasecmp(rel, "SHORTCUT ICON") ||
		    !strcasecmp(rel, "ICON"))
		{
			g_signal_emit(G_OBJECT(kzembed), 
				      kz_moz_embed_signals[FAVICON_SIGNAL],
				      0, url);
		}
		else if(!strcasecmp(rel, "ALTERNATE"))
		{
			kzembed->priv->link_rss = g_strdup(url);
		}
		else if(!strcasecmp(rel, "INDEX"))
		{
			kzembed->priv->link_index = g_strdup(url);
		}
		else if(!strcasecmp(rel, "CONTENTS"))
		{
			kzembed->priv->link_contents = g_strdup(url);
		}
		else if (!strcasecmp(rel, "PREV"))
		{
			kzembed->priv->link_prev = g_strdup(url);
		}
		else if (!strcasecmp(rel, "NEXT"))
		{
			kzembed->priv->link_next = g_strdup(url);
		}
		else if (!strcasecmp(rel, "START"))
		{
			kzembed->priv->link_start = g_strdup(url);
		}
		g_free(url);
		g_free(rel);
		g_free(relattr);
	}
}


// Nautilus CREDITS here
static gchar *
mozilla_get_attribute (nsIDOMNode *node, gchar *attribute)
{
     	nsresult result;

	nsCOMPtr<nsIDOMNamedNodeMap> attributes;
	result = node->GetAttributes(getter_AddRefs(attributes));
	if (!NS_SUCCEEDED (result) || !attributes) return NULL;

	nsAutoString attr; 

	attr.AssignWithConversion(attribute);

	nsCOMPtr<nsIDOMNode> attrNode;
	result = attributes->GetNamedItem (attr, getter_AddRefs(attrNode));
	if (!NS_SUCCEEDED(result) || !attrNode)  return NULL;

	nsAutoString nodeValue;

	result = attrNode->GetNodeValue(nodeValue);
	if (!NS_SUCCEEDED(result))  return NULL;

	return ToNewCString(nodeValue);
}


static gboolean
mozilla_get_link_from_node (nsIDOMDocument *domDoc, nsIDOMNode *node,
			    gchar **url, gchar **inner_html)
{
	if (inner_html) *inner_html = NULL;
	if (url) *url = NULL;

	// get url
	char *hrefattr =  mozilla_get_attribute(node, "href");
	if (!hrefattr) return FALSE;

	nsAutoString hrefa;
	hrefa.AssignWithConversion(hrefattr);

	nsCString hrefc,linkc;
	hrefc.AssignWithConversion(hrefa);

	nsIURI *baseURI;
	nsCOMPtr<nsIDocument> doc = do_QueryInterface(domDoc);
	nsresult rv;
#if MOZILLA_SNAPSHOT > 11
	baseURI = doc->GetBaseURL();
#elif MOZILLA_SNAPSHOT > 9
	rv = doc->GetBaseURL(&baseURI);
#else
	rv = doc->GetBaseURL(baseURI);
#endif	
	rv = baseURI->Resolve(hrefc,linkc);
	*url = ToNewCString(linkc);

	g_free(hrefattr);

	// get title
	nsAutoString nodename;
	node->GetNodeName(nodename);
	nsCOMPtr<nsIDOMNamedNodeMap> attributes;
	node->GetAttributes(getter_AddRefs(attributes));

	nsCOMPtr<nsIDOMNode> hrefNode;
	nsAutoString href; 
	href.AssignWithConversion("href");
	attributes->GetNamedItem(href, getter_AddRefs(hrefNode));
	if (!hrefNode) return FALSE;

	nsAutoString linkhtml;
	nsCOMPtr<nsIDOMNSHTMLElement> nsElement;

	nsCOMPtr<nsIDOMHTMLElement> element;
	element = do_QueryInterface(node);

	nsElement = do_QueryInterface(element);
	if (!nsElement) return FALSE;

	rv = nsElement->GetInnerHTML(linkhtml);
	if (NS_SUCCEEDED(rv) &&
	    NS_ConvertUCS2toUTF8(linkhtml).get()) 
	{
		*inner_html = g_strdup(NS_ConvertUCS2toUTF8(linkhtml).get());
	}

	return TRUE;
}


static glong
mozilla_set_event_context (KzMozEmbed *kzembed,
			   nsIDOMEventTarget *target,
			   KzEmbedEvent *info)
{
	nsresult result;

	nsCOMPtr<nsIDOMNode> node = do_QueryInterface(target);
	if (!node) return KZ_CONTEXT_NONE;

     	nsCOMPtr<nsIDOMDocument> domDoc;
     	result = node->GetOwnerDocument(getter_AddRefs(domDoc));
     	if (!NS_SUCCEEDED (result) || !domDoc) return KZ_CONTEXT_NONE;

     	nsCOMPtr<nsIDocument> doc = do_QueryInterface(domDoc);
     	if(!doc) return KZ_CONTEXT_NONE;

     	nsCOMPtr<nsIDOMNSDocument> nsDoc = do_QueryInterface(domDoc);
     	if (!nsDoc) return KZ_CONTEXT_NONE;

	nsIURI *baseURI;
#if MOZILLA_SNAPSHOT > 11
	baseURI = doc->GetBaseURL();
#elif MOZILLA_SNAPSHOT > 9
	result = doc->GetBaseURL(&baseURI);
#else
	result = doc->GetBaseURL(baseURI);
#endif
	if (NS_FAILED(result) || !baseURI) return KZ_CONTEXT_NONE;

     	nsString mime;
     	nsDoc->GetContentType(mime);
     	if (mime.EqualsWithConversion("text/xul"))  return KZ_CONTEXT_NONE;

	PRUint32 flags = KZ_CONTEXT_NONE;

	// check framed page
	nsCOMPtr<nsIDOMDocument> mainDocument;
	result = kzembed->priv->wrapper->GetMainDomDocument (getter_AddRefs(mainDocument));
	if (domDoc != mainDocument)
	{
		flags |= KZ_CONTEXT_FRAME;
		nsCAutoString url;
		baseURI->GetSpec(url);		
		info->frame_src = g_strdup(url.get());
	}
	
	// check whether the node is in the selection or not
	nsCOMPtr<nsISelection> selection;
	result = kzembed->priv->wrapper->GetSelection(getter_AddRefs(selection));
	if (selection)
	{
		PRBool contains;
		selection->ContainsNode(node, PR_TRUE, &contains);
		if (contains)
			flags |= KZ_CONTEXT_SELECTION;
	}

	// Get other context
	nsCOMPtr<nsIDOMHTMLElement> element;

     	do {
		PRUint16 type;
		node->GetNodeType(&type);

		element = do_QueryInterface(node);
		if (element)
		{
			nsAutoString tag;
			element->GetTagName(tag);

			if (tag.EqualsWithConversion("input", PR_TRUE))
			{
				flags |= KZ_CONTEXT_INPUT;
			}
			else if (tag.EqualsWithConversion("textarea", PR_TRUE))
			{
				flags |= KZ_CONTEXT_INPUT;
			}
			else if (tag.EqualsWithConversion("img", PR_TRUE))
			{
				flags |= KZ_CONTEXT_IMAGE;

				char *src = mozilla_get_attribute(node, "src");
				if (!src)  return KZ_CONTEXT_NONE;

			     	nsAutoString srca;
			     	srca.AssignWithConversion(src);

			     	nsCString srcc,imgc;
			     	srcc.AssignWithConversion(srca);

			     	result = baseURI->Resolve(srcc, imgc);
			     	g_free(src);

			     	info->img = ToNewCString(imgc);

			     	if (!info->img) return KZ_CONTEXT_NONE;
			}
			else
			{
				flags |= KZ_CONTEXT_OTHER;
			}

			nsCOMPtr<nsIDOMNamedNodeMap> attributes;
			node->GetAttributes(getter_AddRefs(attributes));
			if (attributes)
			{
				nsCOMPtr<nsIDOMNode> hrefNode;
				nsAutoString href; 
			     	href.AssignWithConversion("href");
				attributes->GetNamedItem(href, getter_AddRefs(hrefNode));
				if (hrefNode)
				{
					flags |= KZ_CONTEXT_LINK;

					mozilla_get_link_from_node(domDoc, node,
								   &info->link,
								   &info->linktext);
					if (!info->link)
					{
						g_free(info->linktext);
						return KZ_CONTEXT_NONE;
					}
					break;
				}
			}
		}

		nsCOMPtr<nsIDOMNode> parentNode;
		node->GetParentNode(getter_AddRefs(parentNode));

		if (!parentNode)
		{
			node = nsnull;
			flags |= KZ_CONTEXT_DOCUMENT;
			break;
		}
		node = parentNode;
	} while (node);

	info->context = flags;

	return flags;
}


static gboolean
mozilla_get_head_link_from_node (nsIDOMDocument *domDoc, nsIDOMNode *node,
				 gchar **url)
{
	if (url) *url = NULL;

	// get url
	char *hrefattr =  mozilla_get_attribute(node, "href");
	if (!hrefattr)
		return FALSE;
	nsAutoString hrefa;
	hrefa.AssignWithConversion(hrefattr);

	nsCString hrefc,linkc;
	hrefc.AssignWithConversion(hrefa);

	nsIURI *baseURI;
	nsCOMPtr<nsIDocument> doc = do_QueryInterface(domDoc);
	nsresult rv;
#if MOZILLA_SNAPSHOT > 11
	baseURI = doc->GetBaseURL();
#elif MOZILLA_SNAPSHOT > 9
	rv = doc->GetBaseURL(&baseURI);
#else
	rv = doc->GetBaseURL(baseURI);
#endif	
	rv = baseURI->Resolve(hrefc,linkc);
	*url = ToNewCString(linkc);

	g_free(hrefattr);
	return TRUE;
}


void
kz_moz_embed_reload(KzMozEmbed *kzembed, gint32 flags)
{
	g_return_if_fail(KZ_IS_MOZ_EMBED(kzembed));
	gtk_moz_embed_reload(GTK_MOZ_EMBED(kzembed), flags);
}


void
kz_moz_embed_stop_load(KzMozEmbed *kzembed)
{
	g_return_if_fail(KZ_IS_MOZ_EMBED(kzembed));
	gtk_moz_embed_stop_load(GTK_MOZ_EMBED(kzembed));
}


void
kz_moz_embed_go_back(KzMozEmbed *kzembed)
{
	g_return_if_fail(KZ_IS_MOZ_EMBED(kzembed));
	gtk_moz_embed_go_back(GTK_MOZ_EMBED(kzembed));
}


void
kz_moz_embed_go_forward(KzMozEmbed *kzembed)
{
	g_return_if_fail(KZ_IS_MOZ_EMBED(kzembed));
	gtk_moz_embed_go_forward(GTK_MOZ_EMBED(kzembed));
}


gboolean
kz_moz_embed_can_go_back(KzMozEmbed *kzembed)
{
	g_return_val_if_fail(KZ_IS_MOZ_EMBED(kzembed), FALSE);
	return gtk_moz_embed_can_go_back(GTK_MOZ_EMBED(kzembed));
}

gboolean
kz_moz_embed_can_go_forward(KzMozEmbed *kzembed)
{
	g_return_val_if_fail(KZ_IS_MOZ_EMBED(kzembed), FALSE);
	return gtk_moz_embed_can_go_forward(GTK_MOZ_EMBED(kzembed));
}

gboolean
kz_moz_embed_can_go_index(KzMozEmbed *kzembed)
{
	g_return_val_if_fail(KZ_IS_MOZ_EMBED(kzembed), FALSE);
	if(kzembed->priv->link_index)
		return TRUE;
	return FALSE;
}

gboolean
kz_moz_embed_can_go_contents(KzMozEmbed *kzembed)
{
	g_return_val_if_fail(KZ_IS_MOZ_EMBED(kzembed), FALSE);
	if(kzembed->priv->link_contents)
		return TRUE;
	return FALSE;
}

gboolean
kz_moz_embed_can_go_start(KzMozEmbed *kzembed)
{
	g_return_val_if_fail(KZ_IS_MOZ_EMBED(kzembed), FALSE);
	if(kzembed->priv->link_start)
		return TRUE;
	return FALSE;
}
gboolean
kz_moz_embed_can_go_prev(KzMozEmbed *kzembed)
{
	g_return_val_if_fail(KZ_IS_MOZ_EMBED(kzembed), FALSE);
	if(kzembed->priv->link_prev)
		return TRUE;
	return FALSE;
}

gboolean
kz_moz_embed_can_go_next(KzMozEmbed *kzembed)
{
	g_return_val_if_fail(KZ_IS_MOZ_EMBED(kzembed), FALSE);
	if(kzembed->priv->link_next)
		return TRUE;
	return FALSE;
}

void
kz_moz_embed_go_index (KzMozEmbed *kzembed)
{
	g_return_if_fail(KZ_IS_MOZ_EMBED(kzembed));
	g_return_if_fail(kzembed->priv->link_index);

	kz_moz_embed_load_url(kzembed,
			      kzembed->priv->link_index);
}

void
kz_moz_embed_go_contents (KzMozEmbed *kzembed)
{
	g_return_if_fail(KZ_IS_MOZ_EMBED(kzembed));
	g_return_if_fail(kzembed->priv->link_contents);

	kz_moz_embed_load_url(kzembed,
			      kzembed->priv->link_contents);
}

void
kz_moz_embed_go_start (KzMozEmbed *kzembed)
{
	g_return_if_fail(KZ_IS_MOZ_EMBED(kzembed));
	g_return_if_fail(kzembed->priv->link_start);

	kz_moz_embed_load_url(kzembed,
			      kzembed->priv->link_start);
}

void
kz_moz_embed_go_next (KzMozEmbed *kzembed)
{
	g_return_if_fail(KZ_IS_MOZ_EMBED(kzembed));
	g_return_if_fail(kzembed->priv->link_next);

	kz_moz_embed_load_url(kzembed,
			      kzembed->priv->link_next);
}

void
kz_moz_embed_go_prev (KzMozEmbed *kzembed)
{
	g_return_if_fail(KZ_IS_MOZ_EMBED(kzembed));
	g_return_if_fail(kzembed->priv->link_prev);

	kz_moz_embed_load_url(kzembed,
			      kzembed->priv->link_prev);
}


void
kz_moz_embed_copy_page (KzMozEmbed   *kzembed, KzMozEmbed   *dkzembed)
{
	KzMozWrapper *dWrapper = KZ_MOZ_EMBED(dkzembed)->priv->wrapper;;
	KzMozWrapper *sWrapper = KZ_MOZ_EMBED(kzembed)->priv->wrapper;;

        nsresult rv;

        nsCOMPtr<nsISupports> pageDescriptor;
        rv = sWrapper->GetPageDescriptor(getter_AddRefs(pageDescriptor));
        if (!pageDescriptor || NS_FAILED(rv)) return;

        rv = dWrapper->LoadDocument(pageDescriptor, 0);
}


gboolean
kz_moz_embed_shistory_copy (KzMozEmbed *source,
			    KzMozEmbed *dest,
			    gboolean back_history,
			    gboolean forward_history,
			    gboolean set_current)
{
	nsresult rv;
	KzMozWrapper *s_wrapper = KZ_MOZ_EMBED(source)->priv->wrapper;
	KzMozWrapper *d_wrapper = KZ_MOZ_EMBED(dest)->priv->wrapper;
	
	rv = s_wrapper->CopyHistoryTo (d_wrapper, back_history, forward_history, set_current);

	return NS_SUCCEEDED(rv) ? TRUE : FALSE;
}


gboolean
kz_moz_embed_get_lock (KzMozEmbed *kzembed)
{
	g_return_val_if_fail(KZ_IS_MOZ_EMBED(kzembed), FALSE);
	return kzembed->priv->lock;
}


void
kz_moz_embed_set_lock (KzMozEmbed *kzembed, gboolean lock)
{
	g_return_if_fail(KZ_IS_MOZ_EMBED(kzembed));
	kzembed->priv->lock = lock;
}


gboolean 
impl_save (KzMozEmbed *kzembed)
{
	nsresult rv;
	nsAutoString s;
	char *filename = "poincare.ikezoe.net";
	KzMozWrapper *wrapper = NULL;
	PRUint32 persistFlags = 0;
/*
	g_object_ref (persist);
	
	g_object_get (persist,
		      "source", &uri,        
		      "dest", &filename,
		      "flags", &flags,
		      "embed", &embed,
		      "max_size", &max_size,
		      NULL);
*/	
	g_return_val_if_fail (filename != NULL, FALSE);
	
	nsCOMPtr<nsIWebBrowserPersist> bpersist = 
		do_CreateInstance(NS_WEBBROWSERPERSIST_CONTRACTID);
	if (!bpersist) return FALSE;

	nsCOMPtr<nsIURI> linkURI;
	linkURI = nsnull;

        nsCOMPtr<nsILocalFile> file;
       	rv = NS_NewLocalFile(NS_ConvertUTF8toUCS2(filename), PR_TRUE, getter_AddRefs(file)); 
        if (NS_FAILED(rv) || !file) return FALSE;

	nsCOMPtr<nsILocalFile> path;
	char *datapath;
	datapath = g_strconcat (filename, ".content", NULL);
	NS_NewLocalFile(NS_ConvertUTF8toUCS2(datapath), PR_TRUE, getter_AddRefs(path));
	g_free (datapath);
	
	wrapper = kzembed->priv->wrapper;
	g_return_val_if_fail (wrapper != NULL, FALSE);	
/*
	nsCOMPtr<nsIDOMWindow> parent;
	parent = nsnull;
	
	if (embed)
	{
		wrapper->GetDOMWindow (getter_AddRefs (parent));
	}
*/
	persistFlags = nsIWebBrowserPersist::PERSIST_FLAGS_REPLACE_EXISTING_FILES;

	size_t len = strlen(filename);
	if((filename[len-1] == 'z' && filename[len-2] == 'g') ||
	   (filename[len-1] == 'Z' && filename[len-2] == 'G'))
	{
                persistFlags |= nsIWebBrowserPersist::PERSIST_FLAGS_NO_CONVERSION;
	}

	persistFlags |= nsIWebBrowserPersist::PERSIST_FLAGS_BYPASS_CACHE;
	bpersist->SetPersistFlags (persistFlags);

	KzMozProgressListener *aProgress = new KzMozProgressListener ();
	aProgress->Init(kzembed);

	g_return_val_if_fail (wrapper != NULL, FALSE);
		
	nsCOMPtr<nsIDOMDocument> DOMDocument;

	rv = wrapper->GetMainDomDocument (getter_AddRefs(DOMDocument));
	if (NS_FAILED(rv) || !DOMDocument) return FALSE;

	nsCOMPtr<nsIDocument> document =
		do_QueryInterface (DOMDocument, &rv);
	if (NS_FAILED(rv) || !document) return FALSE;

	nsCOMPtr<nsIURI> inURI;
	nsCAutoString sURI;

	rv = wrapper->GetDocumentUrl(sURI);
      	rv = NS_NewURI(getter_AddRefs(inURI), sURI);
/*
  	aProgress->InitForPersist (bpersist, parent,
                   	                   uri, file, 
                           	           ACTION_OBJECT_NOTIFY,
					   persist,
					   (flags & EMBED_PERSIST_SHOW_PROGRESS));
*/
	bpersist->SetProgressListener (aProgress);
	
	rv = bpersist->SaveDocument (DOMDocument, file, path, nsnull, 0, 0);
	if (NS_FAILED(rv)) return FALSE;

	return TRUE;
}
