///////////////////////////////////////////////////////////////////////////////
// cmd_ui_updater.hpp
//
//  Concepts:
//   CmdUIMap is...
//    MPL Sequence of Entry.
//   Entry has...
//    a static member function, process
//

#ifndef KETCHUP_CMD_UI_UPDATER_H
#define KETCHUP_CMD_UI_UPDATER_H

#ifdef _MSC_VER
#pragma once
#endif

#include <boost/call_traits.hpp>
#include <boost/spirit/fusion/sequence/tuple.hpp>
#include <boost/spirit/fusion/sequence/get.hpp>
#include <cassert>

#include "init_cbSize.hpp"
//#include "message_processor.hpp"
#include "detail/gen_process_statement.hpp"


namespace ketchup {
	
///////////////////////////////////////////////////////////////////////////////
// cmd_ui interfaces
//
template< class T >
struct get_cmd_ui_map;

struct cmd_ui
{
private:
	UINT m_nID;

public:
	explicit cmd_ui(UINT nID) : m_nID(nID) { }
	UINT get_id() const { return m_nID; }
	virtual void enable(bool = true) { };
	virtual void set_check(int = 1) { };
	virtual void set_radio(bool bOn = true) { set_check(bOn); };
	virtual void set_text(LPCTSTR) { };
	virtual void set_default(bool = true) { };
	virtual bool is_clingy() { return false; };
	virtual ~cmd_ui() { }
};

	namespace detail {

	struct cmd_ui_updater_base
	{
	private:
		bool b_cmd_ui_handler;

	protected:
		bool is_cmd_ui_handled() const { return b_cmd_ui_handler; }
		void set_cmd_ui_handled(bool b_) { b_cmd_ui_handler = b_; }
	};

	///////////////////////////////////////////////////////////////////////////////
	// menu_cmd_ui
	//
	struct menu_cmd_ui : cmd_ui
	{
	private:
		CMenuHandle m_menu;
		UINT m_nIndex;
		bool m_clingy;

	public:
		HMENU get_menu() const { return m_menu; }
		UINT get_index() const { return m_nIndex; }

		explicit menu_cmd_ui(UINT nID, HMENU hMenu, UINT nIndex, bool clingy)
			: cmd_ui(nID), m_menu(hMenu), m_nIndex(nIndex), m_clingy(clingy)
		{
			assert(m_menu.IsMenu());
		}

		virtual void enable(bool bOn = true)
		{
			m_menu.EnableMenuItem(m_nIndex, MF_BYPOSITION |
				(bOn ? MF_ENABLED : (MF_DISABLED | MF_GRAYED)));
		}

		virtual void set_check(int nCheck = 1)
		{
			m_menu.CheckMenuItem(m_nIndex, MF_BYPOSITION |
				(nCheck ? MF_CHECKED : MF_UNCHECKED));
		}

		virtual void set_radio(bool bOn = true)
		{
			set_check(bOn ? 1 : 0); // not yet
		}

		virtual void set_text(LPCTSTR lpszText)
		{
			assert(lpszText != NULL);
			// get current menu state so it doesn't change
			UINT nState = m_menu.GetMenuState(m_nIndex, MF_BYPOSITION);
			nState &= ~(MF_BITMAP|MF_OWNERDRAW|MF_SEPARATOR);

			// set menu text
			init_cbSize<MENUITEMINFO>::type mii;
			mii.fMask = MIIM_ID;
			ATLVERIFY(m_menu.GetMenuItemInfo(m_nIndex, TRUE, &mii));
			ATLVERIFY(m_menu.ModifyMenu(m_nIndex, MF_BYPOSITION |
				MF_STRING | nState, mii.wID, lpszText));
		}

		virtual void set_default(bool bOn = true)
		{
			// place checkmark next to menu item
			if (bOn)
				m_menu.SetMenuDefaultItem(m_nIndex, TRUE);
		}

		virtual bool is_clingy()
		{
			return m_clingy;
		}
	};

	///////////////////////////////////////////////////////////////////////////////
	// toolbar_cmd_ui
	//
	struct toolbar_cmd_ui : cmd_ui
	{
		CToolBarCtrl m_wndToolBar;

		explicit toolbar_cmd_ui(UINT nID, HWND hWnd)
			: cmd_ui(nID), m_wndToolBar(hWnd)
		{
			assert(m_wndToolBar.IsWindow());
		}

		// overrides
		virtual void enable(bool bOn = true)
		{
			// See: See: MFC7::CToolCmdUI::Enable
			UINT nOldState = m_wndToolBar.GetState(get_id());
			UINT nNewState = nOldState;
			assert(nNewState != -1);
			if (!bOn) {
				nNewState &= ~TBSTATE_ENABLED;
				// WINBUG: If a button is currently pressed and then is disabled
				// COMCTL32.DLL does not unpress the button, even after the mouse
				// button goes up!  We work around this bug by forcing TBBS_PRESSED
				// off when a button is disabled.
				nNewState &= ~TBSTATE_CHECKED;
			}
			else {
				nNewState |= TBSTATE_ENABLED;
			}

			ATLVERIFY(m_wndToolBar.SetState(get_id(), nNewState));
		}

		virtual void set_check(int nCheck = 1)
		{
			assert(nCheck >= 0 && nCheck <= 2); // 0=>off, 1=>on, 2=>indeterminate

			init_cbSize<TBBUTTONINFO>::type tbb;
			tbb.dwMask = TBIF_STATE | TBIF_STYLE;
			ATLVERIFY( m_wndToolBar.GetButtonInfo(get_id(), &tbb) != -1 );

			// add check style
			tbb.fsStyle = tbb.fsStyle | TBSTYLE_CHECK;

			// reset state and...
			tbb.fsState = tbb.fsState &
						~(TBSTATE_CHECKED | TBSTATE_INDETERMINATE);

			if (nCheck == 1)
				tbb.fsState |= TBSTATE_CHECKED;
			else if (nCheck == 2)
				tbb.fsState |= TBSTATE_INDETERMINATE;

			ATLVERIFY(m_wndToolBar.SetButtonInfo(get_id(), &tbb));
		}
	}; // toolbar_cmd_ui

	///////////////////////////////////////////////////////////////////////////////
	// childwindow_cmd_ui
	//
	struct childwindow_cmd_ui : cmd_ui
	{
		CWindow m_wndChild;
		explicit childwindow_cmd_ui(UINT nID, HWND hWndChild)
			: cmd_ui(nID), m_wndChild(hWndChild)
		{
			assert(m_wndChild.IsWindow());
		}

		virtual void enable(bool bOn = true)
		{
			// See: MFC7::CCmdUI::Enable
			// if control has the focus, move the focus before disabling
			if (!bOn && (::GetFocus() == m_wndChild))
			{
				CWindow wndParent = m_wndChild.GetParent();
				wndParent.SendMessage(WM_NEXTDLGCTL, 0, FALSE);
			}
			m_wndChild.EnableWindow(bOn);
		}

		virtual void set_check(int nCheck = 1)
		{
			// we can only check buttons or controls acting like buttons
			if (m_wndChild.SendMessage(WM_GETDLGCODE) & DLGC_BUTTON)
				m_wndChild.SendMessage(BM_SETCHECK, nCheck);
		}

		virtual void set_text(LPCTSTR lpszText)
		{
			assert(lpszText != NULL);
			m_wndChild.SetWindowText(lpszText);
		}
	};

	///////////////////////////////////////////////////////////////////////////////
	// multipanestatusbar_cmd_ui
	// Note:
	//  default win32-pane-architecture missing pane-sensitive id
	//
	template< class MultiPaneStatusBarCtrl >
	struct multipanestatusbar_cmd_ui : cmd_ui
	{
		MultiPaneStatusBarCtrl& m_wndStatusBar;
		int m_nIndex;

		explicit multipanestatusbar_cmd_ui(UINT nPaneID, MultiPaneStatusBarCtrl& wnd, int nIndex)
			: cmd_ui(nPaneID), m_wndStatusBar(wnd), m_nIndex(nIndex)
		{
			assert(m_wndStatusBar.IsWindow());
		}
	
		virtual void set_check(int nCheck = 1)
		{
			ATL::CString str;
			int nOldStyle = 0;
			m_wndStatusBar.GetText(m_nIndex, str, &nOldStyle);

			int nNewStyle = nOldStyle & ~SBT_POPOUT;
			if (nCheck != 0)
				nNewStyle |= SBT_POPOUT;

			m_wndStatusBar.SetText(m_nIndex, str, nNewStyle);
		}

		virtual void set_text(LPCTSTR lpstrText)
		{
			m_wndStatusBar.SetPaneText(get_id(), lpstrText);
		}
	};

	} // detail

///////////////////////////////////////////////////////////////////////////////
// cmd_ui_updater
//
template< class Derived >
struct cmd_ui_updater : detail::cmd_ui_updater_base
{
protected:
	typedef boost::fusion::tuple<Derived&, cmd_ui&> process_args_type;
	typedef typename boost::call_traits<process_args_type>::reference process_args_ref;
	typedef typename boost::call_traits<process_args_type>::param_type process_args_param_type;

	// shortcuts
	static Derived& get_derived(process_args_param_type xs)
	{
		return boost::fusion::get<0>(xs);
	}

	static cmd_ui& get_cmd_ui(process_args_param_type xs)
	{
		return boost::fusion::get<1>(xs);
	}

	template< class CmdUIMap >
	bool update_cmd_ui(cmd_ui& ui)
	{
		Derived& derived = *static_cast<Derived*>(this);
		bool old = derived.is_cmd_ui_handled(); // for recursive call
		typedef typename detail::gen_process_statement<CmdUIMap, process_args_ref>::type statement;
		process_args_type xs(derived, ui);
		bool ret = statement::process(xs);
		derived.set_cmd_ui_handled(old);
		return ret;
	}
	
	///////////////////////////////////////////////////////////////////////////////
	// update_menu_cmd_ui
	//
	template< class CmdUIMap >
	void update_menu_cmd_ui(CMenuHandle menu)
	{
		assert(menu.IsMenu());
		// See: MFC7::CFrameWnd::OnInitMenuPopup
		int count = menu.GetMenuItemCount();
		for (int i = 0; i < count; ++i)
		{
			UINT nID = menu.GetMenuItemID(i);
			if (nID == 0)
				continue; // menu separator or invalid cmd - ignore it

			if (nID == (UINT)-1)
			{
				// possibly a popup menu, route to first item of that popup
				CMenuHandle sub_menu = menu.GetSubMenu(i);
				if (sub_menu.m_hMenu == NULL ||
					(nID = sub_menu.GetMenuItemID(0)) == 0 ||
					nID == (UINT)-1)
				{
					continue;       // first item of popup can't be routed to
				}
				detail::menu_cmd_ui ui(nID, menu, i, true);
				update_cmd_ui<CmdUIMap>(ui); // popups
			}
			else
			{
				// normal menu item
				detail::menu_cmd_ui ui(nID, menu, i, false);
				update_cmd_ui<CmdUIMap>(ui);
			}

			// adjust for menu deletions and additions
			int new_count = menu.GetMenuItemCount();
			if (new_count < count)
			{
				i -= (count - new_count);
				while (i < new_count &&
					menu.GetMenuItemID(i) == nID)
				{
					++i;
				}
			}
			count = new_count;
		} // for
	}

	///////////////////////////////////////////////////////////////////////////////
	// update_menubar_cmd_ui
	//
	// who needs?

	///////////////////////////////////////////////////////////////////////////////
	// update_commandbar_cmd_ui
	//
	// who needs?

	///////////////////////////////////////////////////////////////////////////////
	// update_toolbar_cmd_ui
	//
	template< class CmdUIMap >
	void update_toolbar_cmd_ui(HWND hWndToolBar)
	{
		// See: MFC7::CToolBar::OnUpdateCmdUI
		CToolBarCtrl wndToolBar(hWndToolBar);
		assert(wndToolBar.IsWindow());

		int nIndexMax = wndToolBar.GetButtonCount();
		for (int i = 0; i < nIndexMax; ++i)
		{
			// get buttons state
			TBBUTTON button;
			wndToolBar.GetButton(i, &button);
			UINT nID = button.idCommand;
			detail::toolbar_cmd_ui ui(nID, wndToolBar);

			// ignore separators
			if (!(button.fsStyle & TBSTYLE_SEP))
				update_cmd_ui<CmdUIMap>(ui);
		}
	}

	///////////////////////////////////////////////////////////////////////////////
	// update_childwindow_cmd_ui
	//
	template< class CmdUIMap >
	void update_childwindow_cmd_ui(HWND hWndParent, UINT nID)
	{
		CWindow wndParent(hWndParent);
		assert(wndParent.IsWindow());
		CWindow wndChild = wndParent.GetDlgItem(nID);
		assert(wndChild.IsWindow());
		detail::childwindow_cmd_ui ui(nID, wndChild);
		update_cmd_ui<CmdUIMap>(ui);
	}

	///////////////////////////////////////////////////////////////////////////////
	// update_multipanestatusbar_cmd_ui
	//
	template< class CmdUIMap, class MultiPaneStatusBarCtrl >
	void update_multipanestatusbar_cmd_ui(MultiPaneStatusBarCtrl& wndStatusBar)
	{
		assert(wndStatusBar.IsWindow());

		int nIndexMax = wndStatusBar.m_nPanes;
		for (int i = 0; i < nIndexMax; ++i)
		{
			// get buttons state
			int nID = wndStatusBar.m_pPane[i];
			detail::multipanestatusbar_cmd_ui<MultiPaneStatusBarCtrl> ui(nID, wndStatusBar, i);
			update_cmd_ui<CmdUIMap>(ui);
		}
	}

	///////////////////////////////////////////////////////////////////////////////
	// cmd_ui_handler
	//
	template< UINT id, void (Derived::*func)(cmd_ui&) >
	struct cmd_ui_handler
	{
		static bool process(process_args_ref xs)
		{
			if (get_cmd_ui(xs).is_clingy())
				return false;

			if (get_cmd_ui(xs).get_id() == id) {
				get_derived(xs).set_cmd_ui_handled(true);
				(get_derived(xs).*func)(get_cmd_ui(xs));
				if (get_derived(xs).is_cmd_ui_handled())
					return true;
			}
			return false;
		}
	};

	///////////////////////////////////////////////////////////////////////////////
	// cmd_ui_handler_popup
	//
	template< UINT id, void (Derived::*func)(cmd_ui&) >
	struct cmd_ui_handler_popup
	{
		static bool process(process_args_ref xs)
		{
			if (!get_cmd_ui(xs).is_clingy())
				return false;

			if (get_cmd_ui(xs).get_id() == id) {
				get_derived(xs).set_cmd_ui_handled(true);
				(get_derived(xs).*func)(get_cmd_ui(xs));
				if (get_derived(xs).is_cmd_ui_handled())
					return true;
			}
			return false;
		}
	};

	///////////////////////////////////////////////////////////////////////////////
	// cmd_ui_handler_empty
	//
	template< UINT id, void (Derived::*func)(cmd_ui&) >
	struct cmd_ui_handler_empty
	{
		static bool process(process_args_ref xs)
		{
			return false;
		}
	};

	///////////////////////////////////////////////////////////////////////////////
	// cmd_ui_handler_set_default
	//
	template< UINT id >
	struct cmd_ui_handler_set_default
	{
		static bool process(process_args_ref xs)
		{
			if (get_cmd_ui(xs).is_clingy())
				return false;

			if (get_cmd_ui(xs).get_id() == id) {
				get_cmd_ui(xs).set_default(true);
			}
			return false;
		}
	};

	///////////////////////////////////////////////////////////////////////////////
	// future issue
	//
	template< class CmdUIMap >
	struct chain_cmd_ui_map
	{
		typedef typename Derived::arguments_type args_type;
		static bool process(args_type xs)
		{
			if (Derived::get_uMsg(xs) == WM_INITMENUPOPUP) {
				Derived::get_derived(xs).update_menu_cmd_ui<UpdateCmdUIMap>((HMENU)Derived::get_wParam(xs));
			}
			return false;
		}
	};
};

} // namespace ketchup

///////////////////////////////////////////////////////////////////////////////
// KETCHUP_CHAIN_CMD_UI_MAP for ATL's BEGIN_MSG_MAP
// 
// 	if (uMsg == WM_INITMENUPOPUP) update_menu_cmd_ui<CmdUIMap>((HMENU)wParam);

#define KETCHUP_CHAIN_CMD_UI_MAP(CmdUIMap) {\
	if (uMsg == WM_INITMENUPOPUP)\
  	update_menu_cmd_ui<CmdUIMap>((HMENU)wParam);\
}\
/**/

#endif

