///////////////////////////////////////////////////////////////////////////////
// ketchup/message_processor.hpp
//
//  Copyright 2005 MB@unknown.jp.
//
// This file is a part of the *ketchup*.
// The use and distribution terms for this software are covered by the
// Common Public License 1.0 (http://opensource.org/licenses/cpl.php)
// which can be found in the file CPL.TXT at the root of this distribution.
// By using this software in any fashion, you are agreeing to be bound by
// the terms of this license. You must not remove this notice, or
// any other, from this software.
//
//  Revision History
//   05 May 2005 Initial version

///////////////////////////////////////////////////////////////////////////////
// Concepts:
//
//  MessageMap is...
//   a MPL Sequence of Entry.
//  Entry is...
//   a type which has a static member function,
//   bool process(Args);
//

#ifndef KETCHUP_MESSAGE_PROCESSOR_H
#define KETCHUP_MESSAGE_PROCESSOR_H

#ifdef _MSC_VER
#pragma once
#endif

#include <boost/call_traits.hpp>
#include <boost/concept_check.hpp>
#include <boost/static_assert.hpp>
#include <boost/type_traits/is_same.hpp>
#include <boost/mpl/apply.hpp>
#include <boost/mpl/bool.hpp>
#include <boost/mpl/eval_if.hpp>
#include <boost/mpl/is_sequence.hpp>
#include <boost/mpl/placeholders.hpp>
#include <boost/spirit/fusion/sequence/get.hpp>
#include <boost/spirit/fusion/sequence/tuple.hpp>
#include <atlmisc.h>

#include "detail/gen_process_statement.hpp"

namespace ketchup {

	template< class T >
	struct get_message_map;

	namespace detail {

	template< class Derived >
	struct gen_arguments_type
	{
		typedef boost::fusion::tuple<
			Derived&, HWND, UINT, WPARAM, LPARAM, LRESULT&, DWORD, BOOL&
		> type;
	};

	template< class Derived >
	struct gen_arguments_reference
	{
		typedef typename gen_arguments_type<Derived>::type args_type;
		typedef typename boost::call_traits<args_type>::reference type;
	};

	template< class Derived >
	struct gen_arguments_param_type
	{
		typedef typename gen_arguments_type<Derived>::type args_type;
		typedef typename boost::call_traits<args_type>::param_type type;
	};

	///////////////////////////////////////////////////////////////////////////////
	// gen_message_processor_base
	//
	template< class Derived, class HasCrackedHandlers >
	struct gen_message_processor_base
	{
		typedef typename gen_arguments_type<Derived>::type arguments_type;
		typedef typename gen_arguments_reference<Derived>::type arguments_reference;

		struct base
		{
			template< class MessageMap >
			BOOL process_window_message(HWND hWnd, UINT uMsg, WPARAM wParam, LPARAM lParam, LRESULT& lResult, DWORD dwMsgMapID)
			{
				Derived& derived = *static_cast<Derived*>(this);
				BOOL bHandled = TRUE;
				arguments_type xs(derived, hWnd, uMsg, wParam, lParam, lResult, dwMsgMapID, bHandled);

				typedef typename gen_process_statement<MessageMap, arguments_reference>::type statement;
				return statement::process(xs) ? TRUE : FALSE;
			};

			typedef base type;
		};	

		struct base_for_cracked_handlers
		{
			bool b_msg_handled;
			bool is_msg_handled() { return b_msg_handled; }
			void set_msg_handled(bool b_) { b_msg_handled = b_; }

			template< class MessageMap >
			BOOL process_window_message(HWND hWnd, UINT uMsg, WPARAM wParam, LPARAM lParam, LRESULT& lResult, DWORD dwMsgMapID)
			{
				Derived& derived = *static_cast<Derived*>(this);
				BOOL bHandled = TRUE;
				arguments_type xs(derived, hWnd, uMsg, wParam, lParam, lResult, dwMsgMapID, bHandled);

				bool old = derived.is_msg_handled();	// for recursive calls
				typedef typename gen_process_statement<MessageMap, arguments_reference>::type statement;
				BOOL ret = statement::process(xs) ? TRUE : FALSE;
   			derived.set_msg_handled(old);
				return ret;
			};

			typedef base_for_cracked_handlers type;
		};

		typedef typename boost::mpl::eval_if<HasCrackedHandlers,
			base_for_cracked_handlers,
			base
		>::type type;
	};

	} // namespace detail 


///////////////////////////////////////////////////////////////////////////////
// message_processor
//
template< class Derived, bool has_cracked_handlers = false >
struct message_processor :
	detail::gen_message_processor_base< Derived, boost::mpl::bool_<has_cracked_handlers> >::type
{
protected:
	typedef typename detail::gen_arguments_type<Derived>::type arguments_type;
	typedef typename detail::gen_arguments_reference<Derived>::type arguments_reference;
	typedef typename detail::gen_arguments_param_type<Derived>::type arguments_param_type;

	///////////////////////////////////////////////////////////////////////////////
	// shortcuts
	//
	static Derived& get_derived(arguments_param_type xs)
	{
		return boost::fusion::get<0>(xs);
	}

	static HWND get_hWnd(arguments_param_type xs)
	{
		return boost::fusion::get<1>(xs);
	}

	static UINT get_uMsg(arguments_param_type xs)
	{
		return boost::fusion::get<2>(xs);
	}

	static WPARAM get_wParam(arguments_param_type xs)
	{
		return boost::fusion::get<3>(xs);
	}

	static LPARAM get_lParam(arguments_param_type xs)
	{
		return boost::fusion::get<4>(xs);
	}

	static LRESULT& get_lResult(arguments_param_type xs)
	{
		return boost::fusion::get<5>(xs);
	}

	static DWORD get_dwMsgMapID(arguments_param_type xs)
	{
		return boost::fusion::get<6>(xs);
	}

	static BOOL& get_bHandled(arguments_param_type xs)
	{
		return boost::fusion::get<7>(xs);
	}

	static Derived& get_this(arguments_param_type xs)
	{
		return get_derived(xs);
	}

protected:
	///////////////////////////////////////////////////////////////////////////////
	// process_window_message_to
	//
	template< class Wnd >
	static bool process_window_message_to(Wnd& wnd, arguments_reference xs, DWORD dwMsgMapID = 0)
	{
		return wnd.ProcessWindowMessage(get_hWnd(xs), get_uMsg(xs), get_wParam(xs), get_lParam(xs), get_lResult(xs), dwMsgMapID)
			== TRUE;
	}

	///////////////////////////////////////////////////////////////////////////////
	// Conditions of MessageMap
	//
	// Note: Default parameters in nested templates will be ignored by VC7.1.
	//
	#include "detail/atlwin_h.ipp"

	///////////////////////////////////////////////////////////////////////////////
	// empty_handler
	//
	struct empty_handler
	{
		static bool process(arguments_reference xs)
		{
			boost::ignore_unused_variable_warning(xs);
			return false;
		}
	};

	///////////////////////////////////////////////////////////////////////////////
	// chain_msg
	//
	template< class MessageMap, DWORD msgMapID >
	struct chain_msg_alt
	{
		static bool process(arguments_reference xs)
		{
			return get_derived(xs).process_window_message<MessageMap>(get_hWnd(xs), get_uMsg(xs), get_wParam(xs), get_lParam(xs), get_lResult(xs), msgMapID)
				== TRUE;
		}
	};

	template< class MessageMap >
	struct chain_msg : chain_msg_alt<MessageMap, 0> { };

	///////////////////////////////////////////////////////////////////////////////
	// chain_msg_map_alt
	//
	template< class BaseMessageProcessor, class MessageMap, DWORD msgMapID >
	struct chain_msg_map_alt
	{
		static bool process(arguments_reference xs)
		{
			return get_derived(xs).BaseMessageProcessor::process_window_message<MessageMap>(get_hWnd(xs), get_uMsg(xs), get_wParam(xs), get_lParam(xs), get_lResult(xs), msgMapID)
				== TRUE;
		}
	};

	template< class BaseMessageProcessor, class MessageMap >
	struct chain_msg_map : chain_msg_map_alt<BaseMessageProcessor, MessageMap, 0> { };

	///////////////////////////////////////////////////////////////////////////////
	// chain_msg_map_alt_Atl
	//
	template< class theChainClass, DWORD msgMapID >
	struct chain_msg_map_alt_Atl
	{
		static bool process(arguments_reference xs)
		{
			return get_derived(xs).theChainClass::ProcessWindowMessage(get_hWnd(xs), get_uMsg(xs), get_wParam(xs), get_lParam(xs), get_lResult(xs), msgMapID)
				== TRUE;
		}
	};

	template< class theChainClass >
	struct chain_msg_map_Atl : chain_msg_map_alt_Atl<theChainClass, 0> { };

	///////////////////////////////////////////////////////////////////////////////
	// alt_msg_map_member
	//  define CMessageMap& CMainFrame::get_cmdBar() { return &m_wndCmdBar; } and pass to...
	//
	template< CMessageMap& (Derived::*func)() >
	struct chain_msg_map_member
	{
		static bool process(arguments_reference xs)
		{
			return process_window_message_to((get_derived(xs).*func)(), xs, 0);
		}
	};
	// or define in your CMainFrame if you want chain to non-CMessageMap...
	/*
	struct chain_msg_map_m_wndCmdBar
	{
		static bool process(arguments_reference xs)
		{
			return process_window_message_to(get_this(xs).m_just_a_map, xs, 3);
		}
	};
	*/	
	///////////////////////////////////////////////////////////////////////////////
	// chain_msg_map_alt_member
	//
	template< CMessageMap& (Derived::*func)(), DWORD dwMsgMapID >
	struct chain_msg_map_alt_member
	{
		static bool process(arguments_reference xs)
		{
			return process_window_message_to((get_derived(xs).*func)(), xs, dwMsgMapID);
		}
	};

	///////////////////////////////////////////////////////////////////////////////
	// alt_msg_map
	//
	template< DWORD msgMapID, class MessageMap >
	struct alt_msg_map
	{
		static bool process(arguments_reference xs)
		{
			if (get_dwMsgMapID(xs) == msgMapID) {
					return get_derived(xs).process_window_message<MessageMap>(get_hWnd(xs), get_uMsg(xs), get_wParam(xs), get_lParam(xs), get_lResult(xs), get_dwMsgMapID(xs))
						== TRUE;
			}
			else
				return false; // pass to other alternates
		}
	};

	///////////////////////////////////////////////////////////////////////////////
	// forward_notifications
	//
	struct forward_notifications
	{
		static bool process(arguments_reference xs)
		{
			get_bHandled(xs) = TRUE;
			get_lResult(xs) = get_derived(xs).ForwardNotifications(get_uMsg(xs), get_wParam(xs), get_lParam(xs), get_bHandled(xs));
			if(get_bHandled(xs))
				return true;
		}
	};

	///////////////////////////////////////////////////////////////////////////////
	// default_reflection_handler
	//
	struct default_reflection_handler
	{
		static bool process(arguments_reference xs)
		{
			return Derived::DefaultReflectionHandler(get_hWnd(xs), get_uMsg(xs), get_wParam(xs), get_lParam(xs), get_lResult(xs))
				== TRUE;
		}
	};

	///////////////////////////////////////////////////////////////////////////////
	// cracked_handlers
	//
	#define RASCONNSTATE enum tagRASCONNSTATE // to avoid a undefined error
	#include "detail/atlcrack_h.ipp"


}; // struct message_processor

} // namespace ketchup

///////////////////////////////////////////////////////////////////////////////
// KETCHUP_CHAIN_MSG_MAP for ATL's BEGIN_MSG_MAP
// 
#define KETCHUP_CHAIN_MSG_MAP(MessageMap) {\
	if (process_window_message<MessageMap>(hWnd, uMsg, wParam, lParam, lResult, dwMsgMapID))\
		return TRUE;\
}

#endif
