///////////////////////////////////////////////////////////////////////////////
// ketchup/message_processor.hpp
//
#pragma once

#include <boost/mpl/fold.hpp>
#include <boost/mpl/eval_if.hpp>
#include <boost/mpl/placeholders.hpp>

///////////////////////////////////////////////////////////////////////////////
// local macros
//
#define KETCHUP_process_params \
	Derived& derived, HWND hWnd, UINT uMsg, WPARAM wParam, LPARAM lParam, LRESULT& lResult, DWORD dwMsgMapID, BOOL& bHandled

#define KETCHUP_process_args \
	derived, hWnd, uMsg, wParam, lParam, lResult, dwMsgMapID, bHandled

#include "detail/gen_message_processor_base.hpp"
#include "handler/message_handlers.hpp"
#include "handler/cracked_handlers.hpp"
#include "handler/cmd_ui_handlers.hpp"
#include "utility/ignore_unused_variables_warning.hpp"

namespace ketchup {

///////////////////////////////////////////////////////////////////////////////
// message_processor
//
template< class Derived >
struct message_processor :
	detail::gen_message_processor_base<Derived>::type,
	handler::message_handlers<Derived>,
	handler::cracked_handlers<Derived>,
	handler::cmd_ui_handlers<Derived>
{
protected:
	///////////////////////////////////////////////////////////////////////////////
	// empty_handler
	//
	struct empty_handler
	{
		static bool process(KETCHUP_process_params)
		{
			utility::ignore_unused_variables_warning(KETCHUP_process_args);
			return false;
		}
	};

	///////////////////////////////////////////////////////////////////////////////
	// chain_msg
	//
	template< class MessageMap, DWORD msgMapID >
	struct chain_msg_alt
	{
		static bool process(KETCHUP_process_params)
		{
			utility::ignore_unused_variables_warning(KETCHUP_process_args);
			return derived.process_window_message<MessageMap>(hWnd, uMsg, wParam, lParam, lResult, msgMapID) == TRUE;
		}
	};

	template< class MessageMap >
	struct chain_msg : chain_msg_alt<MessageMap, 0> { };

	///////////////////////////////////////////////////////////////////////////////
	// chain_msg_map_alt
	//
	template< class ChainClass, DWORD msgMapID >
	struct chain_msg_map_alt
	{
		static bool process(KETCHUP_process_params)
		{
			utility::ignore_unused_variables_warning(KETCHUP_process_args);
			return derived.ChainClass::ProcessWindowMessage(hWnd, uMsg, wParam, lParam, lResult, msgMapID) == TRUE;
		}
	};

	template< class ChainClass >
	struct chain_msg_map : chain_msg_map_alt<ChainClass, 0>
	{
	};

	///////////////////////////////////////////////////////////////////////////////
	// alt_msg_map_member
	//  define CMessageMap& CMainFrame::get_cmdBar() { return &m_wndCmdBar; } and pass to...
	//
	template< CMessageMap& (Derived::*func)(), DWORD msgMapID >
	struct chain_msg_map_alt_member
	{
		static bool process(KETCHUP_process_params)
		{
			return (derived.*func)().ProcessWindowMessage(hWnd, uMsg, wParam, lParam, lResult, msgMapID) == TRUE;
		}
	};
	// or define in your CMainFrame if you want chain to non-CMessageMap...
	/*
	struct chain_msg_map_m_wndCmdBar
	{
		static bool process(KETCHUP_process_params)
		{
			return m_wndCmdBar.ProcessWindowMessage(hWnd, uMsg,
				wParam, lParam, lResult, msgMapID) == TRUE;
		}
	};
	*/	
	///////////////////////////////////////////////////////////////////////////////
	// chain_msg_map_alt_member
	//
	template< CMessageMap& (Derived::*func)() >
	struct chain_msg_map_member : chain_msg_map_alt_member<func,0>
	{
	};

	///////////////////////////////////////////////////////////////////////////////
	// alt_msg_map
	//
	template< DWORD msgMapID, class MessageMap >
	struct alt_msg_map
	{
		static bool process(KETCHUP_process_params)
		{
			if (dwMsgMapID == msgMapID) {
					return derived.process_window_message<MessageMap>(hWnd, uMsg, wParam, lParam, lResult, msgMapID) == TRUE;
			}
			else
				return false; // pass to other alternates
		}
	};

	///////////////////////////////////////////////////////////////////////////////
	// forward_notifications
	//
	struct forward_notifications
	{
		static bool process(KETCHUP_process_params)
		{
			bHandled = TRUE;
			lResult = derived.ForwardNotifications(uMsg, wParam, lParam, bHandled);
			if(bHandled)
				return true;
		}
	};

	///////////////////////////////////////////////////////////////////////////////
	// default_reflection_handler
	//
	struct default_reflection_handler
	{
		static bool process(KETCHUP_process_params)
		{
			return Derived::DefaultReflectionHandler(hWnd, uMsg, wParam, lParam, lResult) == TRUE;
		}
	};

	///////////////////////////////////////////////////////////////////////////////
	// chain_client_commands
	//
	struct chain_client_commands
	{
		static bool process(KETCHUP_process_params)
		{
			utility::ignore_unused_variables_warning(KETCHUP_process_args);

			if (uMsg == WM_COMMAND)
			{
				HWND hWnd = derived.m_hWndClient;
				if (hWnd != NULL)
					::SendMessage(hWnd, uMsg, wParam, lParam);
				return false;
			}

			return false;
		}
	};

	///////////////////////////////////////////////////////////////////////////////
	// chain_mdi_child_commands
	//
	struct chain_mdi_child_commands
	{
		static bool process(KETCHUP_process_params)
		{
			utility::ignore_unused_variables_warning(KETCHUP_process_args);

			if (uMsg == WM_COMMAND)
			{
				HWND hWndChild = derived.MDIGetActive();
				if (hWndChild != NULL)
					::SendMessage(hWndChild, uMsg, wParam, lParam);
				return false;
			}
			return false;
		}
	};

}; // struct message_processor

} // namespace ketchup

///////////////////////////////////////////////////////////////////////////////
// cleaning up local macros
// 
#undef KETCHUP_process_params
#undef KETCHUP_process_args

///////////////////////////////////////////////////////////////////////////////
// KETCHUP_CHAIN_MSG_MAP for ATL's BEGIN_MSG_MAP
// 
#define KETCHUP_CHAIN_MSG_MAP(MessageMap) {\
	if (process_window_message<MessageMap>(hWnd, uMsg, wParam, lParam, lResult, dwMsgMapID))\
		return TRUE;\
}
