/*
 * The MIT License
 *
 * Copyright 2013 Dra0211.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
package kinugasa.game;

import java.awt.EventQueue;
import java.awt.Graphics2D;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.RenderingHints;
import java.awt.Toolkit;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.awt.image.BufferStrategy;
import java.awt.image.BufferedImage;
import java.util.List;
import kinugasa.contents.graphics.ImageUtil;
import kinugasa.inputDevice.KGPDllLoader;
import kinugasa.inputDevice.KeyConnection;
import kinugasa.inputDevice.MouseConnection;
import kinugasa.util.Versions;
import kinugasa.util.annotations.Internal;

/**
 * Kinugasa : KinugasaGame : ߊ}Q[t[[N̊jƂȂQ[NX̊{̎ł.
 * <br>
 * ߊ}t[[NgpăQ[\zɂ́ÃNXpQ[NXpӂ܂B<br>
 * <br>
 * @version 1.0.0 - 2013/04/20_18:54:20<br>
 * @author Dra0211<br>
 */
public abstract class KinugasaGame extends Game {

	/** Q[\EChEł. */
	private AWTGameWindow window;
	/** OtBbNXReLXg̃obt@ł. */
	private BufferStrategy graphicsBuffer;
	/** t[ɂăEChEɕ`悷邽߂
	 * OtBbNXReLXgł. */
	private Graphics2D graphicsContext;
	/** _O̕ił. */
	private RenderingHints renderingHints;
	/** Q[sXbhł. */
	private GameLoop loop;
	/** Q[FPSǗNXł. */
	private GameTimeManager gameTimeManager;
	/** `掞ɃOtBbNXReLXgNbsO邽߂
	 * EChE̓̈ł. */
	private Rectangle clippingRectangle;

	/**
	 * Q[쐬܂.̃RXgN^Ăяoł́AQ[͋N܂.
	 * @param conf Q[̐ݒ𑗐M܂BEChẼTCYFPS̐ݒ肪܂܂܂B<br>
	 */
	protected KinugasaGame(GameConfig conf) {
		super(conf.getTitle());
		window = new AWTGameWindow();
		renderingHints = conf.getRenderingHints();

		window.getCloseEventStorage().addAll(conf.getCloseEvetList());
		window.addWindowListener(new WindowAdapter() {
			@Override
			public void windowClosing(WindowEvent e) {
				List<WindowCloseEvent> eventList = window.getCloseEventStorage().asSortedList();
				for (int i = 0, size = eventList.size(); i < size; i++) {
					if (!eventList.get(i).close()) {
						return;
					}
				}
				gameExit();
			}
		});

		window.setTitle(conf.getTitle());
		window.setIconImage(conf.getIcon());
		window.setBackground(conf.getBackColor());
		window.setLocation(conf.getLocation());
		window.setSize(conf.getSize());
		window.setResizable(false);

		PlayerConstants playerConstants = PlayerConstants.getInstance();
		playerConstants.setUsingGamepad(conf.isUseGamePad());
		if (playerConstants.isUsingGamepad()) {
			KGPDllLoader.load();
		}

		playerConstants.setUsingKeyboard(conf.isUseKeyboard());
		if (playerConstants.isUsingKeyboard()) {
			KeyConnection.setListener(window);
		}

		playerConstants.setUsingMouse(conf.isUseMouseCursor());
		if (playerConstants.isUsingMouse()) {
			MouseConnection.setListener(window);
		} else {
			BufferedImage cursorImage = ImageUtil.newImage(16, 16);
			window.setCursor(Toolkit.getDefaultToolkit().createCustomCursor(cursorImage, new Point(), "game cursor"));
		}
		this.fps = conf.getFps();
	}

	/**
	 * ȈՃRXgN^ł.
	 * EChETCYwiFstartUp\bhŐݒ肷Kv܂B<br>
	 * @param name Q[EChẼ^CgƂȂ镶łBQ[̖OƂĂݒ肳܂B<br>
	 */
	public KinugasaGame(String name) {
		this(new GameConfig(name));
	}
	/** gameTimeMangerɐݒ肷FPS̍őlł. */
	private int fps;
	/** Q[̋NƏI1ɐ邽߂̃tOł. */
	private boolean started = false;

	@Override
	protected final void gameStart() throws IllegalStateException {
		if (started) {
			throw new IllegalStateException("game[" + name + "] is already started.");
		}
		loop = new GameLoop(this, gameTimeManager = new GameTimeManager(fps));
		EventQueue.invokeLater(new Runnable() {
			@Override
			public void run() {
				try {
					startUp();
				} catch (Throwable ex) {
					System.out.println("ERROR : " + ex);
					ex.printStackTrace();
					System.exit(1);
				}
				window.setVisible(true);
				window.createBufferStrategy(2);
				graphicsBuffer = window.getBufferStrategy();
				clippingRectangle = window.getInternalBounds();
				loop.start();
				started = true;
			}
		});
		System.out.println("> " + Versions.PROJECT_NAME + " " + Versions.LATEST + " [" + getName() + "] game start");
	}

	@Override
	public final void gameExit() throws IllegalStateException {
		if (!started) {
			throw new IllegalStateException("game[" + name + "] is not start.");
		}
		if (loop != null && loop.isStarted()) {
			loop.end();
		}
		try {
			dispose();
		} catch (Throwable ex) {
			System.out.println("ERROR : " + ex);
			ex.printStackTrace();
			System.exit(1);
		}
		window.dispose();
		System.out.println("> [" + getName() + "] game exit");
		System.exit(0);
	}

	@Internal
	@Override
	final void repaint() {
		graphicsContext = (Graphics2D) graphicsBuffer.getDrawGraphics();
		graphicsContext.setClip(clippingRectangle);
		graphicsContext.clearRect(clippingRectangle.x, clippingRectangle.y,
				clippingRectangle.width, clippingRectangle.height);
		graphicsContext.setRenderingHints(renderingHints);
		draw(graphicsContext);
		graphicsContext.dispose();
		if (graphicsBuffer.contentsRestored()) {
			repaint();
		}
		graphicsBuffer.show();
		if (graphicsBuffer.contentsLost()) {
			repaint();
		}
	}

	@Override
	protected abstract void startUp();

	@Override
	protected abstract void dispose();

	@Override
	protected abstract void update();

	@Override
	protected abstract void draw(Graphics2D g);

	@Override
	public final GameWindow getWindow() {
		return window;
	}

	@Override
	public final GameTimeManager getGameTimeManager() {
		return gameTimeManager;
	}

	@Override
	public String toString() {
		return "kinugasaGame : name=[" + name + "]";
	}

	/**
	 * o[Wo͂܂.
	 * @param args R}hC͖܂B<br>
	 */
	public static void main(String[] args) {
		System.out.println(Versions.PROJECT_NAME + " " + Versions.LATEST);
	}
}
