/****************************************************************************
 * KONOHA COPYRIGHT, LICENSE NOTICE, AND DISCRIMER
 *
 * Copyright (c) 2005-2008, Kimio Kuramitsu <kimio at ynu.ac.jp>
 *           (c) 2008-      Konoha Software Foundation
 * All rights reserved.
 *
 * You may choose one of the following two licenses when you use konoha.
 * See www.konohaware.org/license.html for further information.
 *
 * (1) GNU General Public License 2.0      (with    KONOHA_UNDER_GPL2)
 * (2) Konoha Software Foundation License 1.0
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER
 * OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 ****************************************************************************/

/* ************************************************************************ */

#include"commons.h"

/* ************************************************************************ */

#ifdef __cplusplus
extern "C" {
#endif

#define SELF STRUCT_InputStream

/* ======================================================================== */
/* [drivers] */

static knh_inptr_t* knh_InputStream_open__nop(Ctx *ctx, knh_bytes_t n);
static int knh_InputStream_getc__nop(Ctx *ctx, knh_inptr_t *ptr);
static size_t knh_InputStream_read__nop(Ctx *ctx, knh_inptr_t *ptr, char *buf, size_t bufsiz);
static void knh_InputStream_close__nop(Ctx *ctx, knh_inptr_t *ptr);
static knh_inptr_t* knh_InputStream_open__FILE(Ctx *ctx, knh_bytes_t file);
static int knh_InputStream_getc__FILE(Ctx *ctx, knh_inptr_t *ptr);
static size_t knh_InputStream_read__FILE(Ctx *ctx, knh_inptr_t *ptr, char *buf, size_t bufsiz);
static void knh_InputStream_close__FILE(Ctx *ctx, knh_inptr_t *ptr);
static int knh_InputStream_getc__Bytes(Ctx *ctx, knh_inptr_t *ptr);
static size_t knh_InputStream_read__Bytes(Ctx *ctx, knh_inptr_t *ptr, char *buf, size_t bufsiz);
static void knh_InputStream_close__Bytes(Ctx *ctx, knh_inptr_t *ptr);

/* ======================================================================== */

static knh_inptr_apis_t apis_inptr__nop = {
	knh_InputStream_open__nop,
	knh_InputStream_getc__nop,
	knh_InputStream_read__nop,
	knh_InputStream_close__nop
};

static knh_inptr_apis_t apis_inptr__FILE = {
	knh_InputStream_open__FILE,
	knh_InputStream_getc__FILE,
	knh_InputStream_read__FILE,
	knh_InputStream_close__FILE
};

static knh_inptr_apis_t apis_inptr__stdin = {
	knh_InputStream_open__nop,
	knh_InputStream_getc__FILE,
	knh_InputStream_read__FILE,
	knh_InputStream_close__nop
};

static knh_inptr_apis_t apis_inptr__Bytes = {
	knh_InputStream_open__nop,
	knh_InputStream_getc__Bytes,
	knh_InputStream_read__Bytes,
	knh_InputStream_close__Bytes
};

#if defined(KNH_USING_INPUTSTREAM_HTTP)
static knh_inptr_apis_t apis_inptr__http = {
	knh_InputStream_open__http,
	knh_InputStream_getc__http,
	knh_InputStream_read__http,
	knh_InputStream_close__http
};
#endif

/* ======================================================================== */
/* [structs] */

void
knh_InputStream_struct_init(Ctx *ctx, knh_InputStream *b, int init, Object *cs)
{
	b->inptr = NULL;
	KNH_INITv(b->ba, KNH_NULL);
	b->bapos = 0;
	b->baend = 0;
	b->apis = apis_inptr__nop;
	KNH_INITv(b->bconv, KNH_NULL);
	KNH_INITv(b->enc, TS_ENCODING);
	KNH_INITv(b->urn, TS_DEVNULL);
	b->size    = 0;
	b->line    = 1;
	b->prev    = '\n';
	b->fileid   = 0;
}

/* ------------------------------------------------------------------------ */

#define _knh_InputStream_struct_copy   NULL

/* ------------------------------------------------------------------------ */

#define _knh_InputStream_struct_compare  NULL

/* ------------------------------------------------------------------------ */

void
knh_InputStream_struct_traverse(Ctx *ctx, knh_InputStream *b, f_traverse gc)
{
	if(IS_SWEEP(gc) && b->inptr != NULL) {
		b->apis.fclose(ctx, b->inptr);
		b->inptr = NULL;
	}
	gc(ctx, b->ba);
	gc(ctx, b->enc);
	gc(ctx, b->bconv);
	gc(ctx, b->urn);
}

/* ======================================================================== */
/* [constructors] */

InputStream *
new_InputStream(Ctx *ctx, String *urn, knh_inptr_t *inptr, knh_inptr_apis_t drv)
{
	InputStream* o =
		(InputStream*)new_Object__RAW(ctx, FLAG_InputStream, CLASS_InputStream, sizeof(knh_InputStream));
	knh_InputStream_struct_init(ctx, DP(o), 0, NULL);
	DP(o)->inptr = inptr;
	if(DP(o)->inptr == NULL) {
		DP(o)->apis = apis_inptr__nop;
	}
	else {
		DP(o)->apis = drv;
	}
	if(urn == NULL || IS_NULL(urn)) {
		KNH_SETv(ctx, DP(o)->urn, TS_EMPTY);
	}
	else {
		KNH_SETv(ctx, DP(o)->urn, urn);
	}
	return o;
}

/* ======================================================================== */
/* [methods] */

void knh_InputStream_open(Ctx *ctx, InputStream *b, String *urn)
{
	KNH_SETv(ctx, DP(b)->urn, urn);
	if(knh_String_startsWith(urn, STEXT("file:"))) {
		DP(b)->apis = apis_inptr__FILE;
	}
#if defined(KNH_USING_INPUTSTREAM__http)
	else if(knh_String_startsWith(urn, STEXT("http:"))) {
		DP(b)->apis = apis_inptr__http;
	}
#endif
	else if(knh_bytes_index(knh_String_tobytes(urn), ':') != -1) {
		DP(b)->apis = apis_inptr__FILE;
	}
	DP(b)->inptr = DP(b)->apis.fopen(ctx, knh_String_tobytes(urn));
	if(DP(b)->inptr == NULL) {
		DP(b)->apis = apis_inptr__nop;
	}
}

/* ------------------------------------------------------------------------ */

int knh_InputStream_getc(Ctx *ctx, InputStream *b)
{
	int ch = DP(b)->apis.fgetc(ctx, DP(b)->inptr);
	if(ch != EOF) {
		DP(b)->size++;
		if(ch == '\n') {
			if(DP(b)->prev != '\r') {
				DP(b)->line++;
			}
		}else if(ch == '\r') {
			DP(b)->line++;
		}
		DP(b)->prev = ch;
	}
	return ch;
}

/* ------------------------------------------------------------------------ */

size_t
knh_InputStream_read(Ctx *ctx, InputStream *b, char *buf, size_t bufsiz)
{
	size_t size = DP(b)->apis.fread(ctx, DP(b)->inptr, buf, bufsiz);
	DP(b)->size += size;
	return size;
}
/* ------------------------------------------------------------------------ */
/* @method String InputStream.readLine() */

String* knh_InputStream_readLine(Ctx *ctx, InputStream *b)
{
	int ch;
	knh_wbuf_t cb = knh_Context_wbuf(ctx);
	while((ch = knh_InputStream_getc(ctx, b)) != EOF) {
		if(ch == '\r') {
			return new_String__wbufconv(ctx, cb, DP(b)->bconv);
		}
		if(ch == '\n') {
			if(DP(b)->prev == '\r') continue;
		}
		knh_Bytes_putc(ctx, cb.ba, ch);
	}
	if(knh_wbuf_size(cb) != 0) {
		return new_String__wbufconv(ctx, cb, DP(b)->bconv);
	}
	return (String*)KNH_NULL;
}

/* ------------------------------------------------------------------------ */
/* @method void InputStream.close() */

INLINE
void knh_InputStream_close(Ctx *ctx, InputStream *b)
{
	DP(b)->apis.fclose(ctx, DP(b)->inptr);
	DP(b)->inptr = NULL;
	DP(b)->apis = apis_inptr__nop;
}

/* ======================================================================== */
/* [nop] */

static
knh_inptr_t* knh_InputStream_open__nop(Ctx *ctx, knh_bytes_t n)
{
	return NULL;
}

/* ------------------------------------------------------------------------ */

static
int knh_InputStream_getc__nop(Ctx *ctx, knh_inptr_t *ptr)
{
	return EOF;
}

/* ------------------------------------------------------------------------ */

static size_t knh_InputStream_read__nop(Ctx *ctx, knh_inptr_t *ptr, char *buf, size_t bufsiz)
{
	return 0;
}

/* ------------------------------------------------------------------------ */

static void knh_InputStream_close__nop(Ctx *ctx, knh_inptr_t *ptr)
{

}

/* ======================================================================== */
/* [FILE] */

static knh_inptr_t* knh_InputStream_open__FILE(Ctx *ctx, knh_bytes_t file)
{
	char buf[FILENAME_BUFSIZ];
	knh_format_ospath(buf, sizeof(buf), file);
	{
		FILE *fp = fopen(buf, "r");
		if(fp == NULL) {
			KNH_THROWf(ctx, "IO!!: cannot open: %s\n", buf);
			return NULL;
		}
		return (knh_inptr_t*)fp;
	}
}

/* ------------------------------------------------------------------------ */

static int knh_InputStream_getc__FILE(Ctx *ctx, knh_inptr_t *ptr)
{
	FILE *fp = (FILE*)ptr;
	return fgetc(fp);
}

/* ------------------------------------------------------------------------ */

static
size_t knh_InputStream_read__FILE(Ctx *ctx, knh_inptr_t *ptr, char *buf, size_t bufsiz)
{
	FILE *fp = (FILE*)ptr;
	return fread(buf, bufsiz, 1, fp);
}

/* ------------------------------------------------------------------------ */

static
void knh_InputStream_close__FILE(Ctx *ctx, knh_inptr_t *ptr)
{
	FILE *fp = (FILE*)ptr;
	fclose(fp);
}

/* ------------------------------------------------------------------------ */

InputStream *new_InputStream__FILE(Ctx *ctx, FILE *fp)
{
	InputStream* o =
		(InputStream*)new_Object__RAW(ctx, FLAG_InputStream, CLASS_InputStream, sizeof(knh_InputStream));
	knh_InputStream_struct_init(ctx, DP(o), 0, NULL);

	DP(o)->inptr = (knh_inptr_t*)fp;
	if(fp == stdin) {
		DP(o)->apis = apis_inptr__stdin;
		KNH_SETv(ctx, DP(o)->urn, TS_DEVSTDIN);
	}
	else {
		DP(o)->apis = apis_inptr__FILE;
	}
	return o;
}

/* ======================================================================== */
/* [Bytes] */

static
int knh_InputStream_getc__Bytes(Ctx *ctx, knh_inptr_t *ptr)
{
	InputStream *b = (InputStream*)ptr;
	if(IS_NULL(DP(b)->ba) || !(DP(b)->bapos < DP(b)->baend)) return EOF;
	knh_uchar_t *p = (knh_uchar_t*)knh_Bytes_tochar(DP(b)->ba);
	int ch = p[DP(b)->bapos];
	DP(b)->bapos++;
	return ch;
}

/* ------------------------------------------------------------------------ */

static
size_t knh_InputStream_read__Bytes(Ctx *ctx, knh_inptr_t *ptr, char *buf, size_t bufsiz)
{
	InputStream *b = (InputStream*)ptr;
	if(IS_NULL(DP(b)->ba) || !(DP(b)->bapos < DP(b)->baend)) return 0;
	char *p = knh_Bytes_tochar(DP(b)->ba);
	size_t psize = DP(b)->baend - DP(b)->bapos;
	if (bufsiz < psize) psize = bufsiz;
	knh_memcpy(buf, p, psize);
	DP(b)->bapos += psize;
	return psize;
}

/* ------------------------------------------------------------------------ */

static
void knh_InputStream_close__Bytes(Ctx *ctx, knh_inptr_t *ptr)
{
	InputStream *b = (InputStream*)ptr;
	KNH_SETv(ctx, DP(b)->ba, KNH_NULL);
	DP(b)->bapos = 0;
	DP(b)->baend = 0;
}

/* ------------------------------------------------------------------------ */

InputStream *new_InputStream__Bytes(Ctx *ctx, Bytes *ba, size_t s, size_t e)
{
	InputStream* o =
		(InputStream*)new_Object__RAW(ctx, FLAG_InputStream, CLASS_InputStream, sizeof(knh_InputStream));
	knh_InputStream_struct_init(ctx, DP(o), 0, NULL);
	DP(o)->inptr = (knh_inptr_t*)o;
	KNH_SETv(ctx, DP(o)->ba, ba);
	KNH_ASSERT(e <= knh_Bytes_size(ba));
	KNH_ASSERT(s <= e);
	DP(o)->bapos   = s;
	DP(o)->baend   = e;
	DP(o)->apis = apis_inptr__Bytes;
	return o;
}

/* ======================================================================== */
/* [mappings] */


/* ------------------------------------------------------------------------ */

#ifdef __cplusplus
}
#endif
